/* -*- C -*-
 *
 * Program:	ximap
 * File:        ximap.c -- stuff for the session rather than a particular 
 *                         mailbox.
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 */
#include <string.h>
#include <netdb.h>
#include <signal.h>
#include <ctype.h>
#include <pwd.h>

#include <stdio.h>
#include <memory.h>

#include <Client/osdep.h>
#include <Client/mail.h>
#include <Client/misc.h>
 
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <X11/Xaw/Box.h>
#include <X11/Xaw/Form.h>

#ifdef MOTIF
#include <X11/Xm/Text.h>
#else
#include <X11/Xaw/AsciiText.h>
#endif

#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Label.h>

#include "structures.h"

#include "modal.h"
#include "message.h"
#include "globals.h"
#include "buttons.h"
#include "resources.h"
#include "util.h"
#include "ximap.h"
#include "mailbox.h"
#include "readwin.h"
#include "mboxlist.h"
#include "search.h"
#include "compose.h"
#include "browserutil.h"

#define default_check "2.0"
#define default_view  "20"
#define default_rings "1"
#define default_length "79"
#define default_serverPort "143"
#define default_insertPrefix ">"
#define default_printCommand "lpr"
#define default_string ""
#define default_insertWrapCol "0"

static Boolean default_false = FALSE;
static Boolean default_true = TRUE;

static XtResource ximap_resources[]= {
{ XtNdefaultMailbox, XtCDefaultMailbox, 
      XtRString, sizeof(char*),
      XtOffset(AppResources*, default_mailbox), 
      XtRImmediate, (void*) default_string },
{ XtNdefaultHost, XtCDefaultHost, 
      XtRString, sizeof(char*),
      XtOffset(AppResources*, default_host), 
      XtRImmediate, (void*) default_string },
{ XtNreplyToHost, XtCReplyToHost, 
      XtRString, sizeof(char*),
      XtOffset(AppResources*, mail_host), 
      XtRImmediate, (void*) default_string },
{ XtNmyMailName, XtCMyMailName, 
      XtRString, sizeof(char*),
      XtOffset(AppResources*, mail_name), 
      XtRImmediate, (void*) default_string },
{ XtNpersonalName, XtCPersonalName, 
      XtRString, sizeof(char*),
      XtOffset(AppResources*, personal_name), 
      XtRImmediate, (void*) default_string },
{ XtNdefaultBcc, XtCDefaultBcc, 
      XtRString, sizeof(char*),
      XtOffset(AppResources*, default_bcc), 
      XtRImmediate, (void*) default_string },
{ XtNmailEmpty, XtCMailEmpty, 
      XtRBitmap, sizeof(Pixmap),
      XtOffset(AppResources*, mail_empty), 
      XtRString, "mailempty"},
{ XtNmailFull, XtCMailFull, 
      XtRBitmap, sizeof(Pixmap),
      XtOffset(AppResources*, mail_full), 
      XtRString, "mailfull"},
{ XtNhighlightNew, XtCHighlightNew, 
      XtRBoolean, sizeof(Boolean),
      XtOffset(AppResources*, highlight_new), 
      XtRBoolean, (void*) &default_true  },
{ XtNfilterHeaders, XtCFilterHeaders, 
      XtRBoolean, sizeof(Boolean),
      XtOffset(AppResources*, filter_headers), 
      XtRBoolean, (void*) &default_true },
{ XtNinitialView, XtCInitialView, 
      XtRInt, sizeof(int),
      XtOffset(AppResources*, initial_view), 
      XtRString, (void*) default_view },
{ XtNcheckFrequency, XtCCheckFrequency, 
      XtRFloat, sizeof(float),
      XtOffset(AppResources*, check_interval), 
      XtRString, (void*) default_check },
{ XtNdefaultCc, XtCDefaultCc, 
      XtRString, sizeof(char*),
      XtOffset(AppResources*, default_cc), 
      XtRImmediate, (void*) default_string },
{ XtNflagAll, XtCFlagAll, 
      XtRBoolean, sizeof(Boolean),
      XtOffset(AppResources*, flag_all_mailboxes), 
      XtRBoolean, (void*) &default_false },
{ XtNflagNo, XtCFlagNo, 
      XtRBoolean, sizeof(Boolean),
      XtOffset(AppResources*, flag_no_mailboxes), 
      XtRBoolean, (void*) &default_false },
{ XtNringBell, XtCRingBell, 
      XtRBoolean, sizeof(Boolean),
      XtOffset(AppResources*, ring_bell_for_new), 
      XtRBoolean, (void*) &default_true },
{ XtNnumberOfBells, XtCNumberOfBells, 
      XtRInt, sizeof(int),
      XtOffset(AppResources*, number_of_bells), 
      XtRString, (void*) default_rings },
{ XtNmaxLineLength, XtCMaxLineLength, 
      XtRInt, sizeof(int),
      XtOffset(AppResources*, line_length), 
      XtRString, (void*) default_length },
{ XtNindentPrefix, XtCIndentPrefix,
      XtRString, sizeof(char*),
      XtOffset(AppResources*, indentPrefix),
      XtRImmediate, (void*) default_insertPrefix },
{ XtNprintCommand, XtCPrintCommand,
      XtRString, sizeof(char*),
      XtOffset(AppResources*, printCommand),
      XtRImmediate, (void*) default_printCommand },
{ XtNsmtpHost, XtCSmtpHost,
      XtRString, sizeof(char*),
      XtOffset(AppResources*, smtpHost),
      XtRImmediate, (void*) default_string },
{ XtNserverPort, XtCServerPort,
      XtRInt, sizeof(int),
      XtOffset(AppResources*, serverPort),
      XtRString, (void*) default_serverPort},
{ XtNinsertWrapCol, XtCInsertWrapCol,
      XtRInt, sizeof(int),
      XtOffset(AppResources*, insert_wrap_columns),
      XtRString, (void*) default_insertWrapCol },

};

XrmOptionDescRec  ximap_options[] = 
{
    {"-port",           "*serverPort",     XrmoptionSepArg, NULL},
    {"-replyTo",        "*replyToHost",    XrmoptionSepArg, NULL},
    {"-mailName",       "*myMailName",     XrmoptionSepArg, NULL},
    {"-personal",       "*personalName",   XrmoptionSepArg, NULL},
    {"-highlight",      "*highlightNew",   XrmoptionNoArg,  "False" },
    {"+highlight",      "*highlightNew",   XrmoptionNoArg,  "True" },
    {"-check",          "*checkFrequency", XrmoptionSepArg, NULL},
    {"-filter",         "*filterHeaders",  XrmoptionNoArg,  "False" },
    {"+filter",         "*filterHeaders",  XrmoptionNoArg,  "True" },
    {"-view",           "*initialView",    XrmoptionSepArg, NULL},
    {"-flagNo",         "*flagNo",         XrmoptionNoArg,  "True" },
    {"-printCommand",   "*printCommand",   XrmoptionSepArg, NULL},
    {"-smtpHost",       "*smtpHost",       XrmoptionSepArg, NULL},
    {"-bells",          "*numberOfBells",  XrmoptionSepArg, NULL}
};

static void removeSTREAMARRAY(
#ifdef __XB_NEEDPROTOTYPES__
#endif
);

static void appOpen(
#ifdef __XB_NEEDPROTOTYPES__
#endif
);

static void appQuit(
#ifdef __XB_NEEDPROTOTYPES__
#endif
);

static void appQuitAct(
#ifdef __XB_NEEDPROTOTYPES__
#endif
);

static void appOpenAct(
#ifdef __XB_NEEDPROTOTYPES__
#endif
);

static XtActionsRec telemetry_actions[] =
{
  { "app_quit", appQuitAct },
  { "app_open", appOpenAct },
};

ButtonStruct telBtns[] = 
{
  {"Quit", appQuit, NULL,  NEVER},
  {"Open", appOpen, NULL,  NEVER},
  NULL,
};

extern DRIVER imapdriver;

Widget toplevel;
Widget telemetry;
Widget telemetry_debug;

XtAppContext app_context;

/* application resources */
AppResources res;             

/* 
 * open mailboxes and hosts that have been 
 * logged into during this session 
 * 
 */
MAILSESSION id;

StateList *states;
int updateMailbox;

static void SyntaxError( argc, argv )
     int argc;
     char **argv;
{
    int i;
    static int errs = False;

    for( i = 1; i < argc; i++ )
    {
	if(!errs++)
	    fprintf(stderr, "ximap: unknown command line option:\n");
	
	fprintf(stderr, "option: %s\n", argv[i]);
    }

    fprintf(stderr, "ximap understands all of the standard Xt \ncommand line options.\n");

    fprintf(stderr, "Additional options are as follows:\n");
    fprintf(stderr, "Option Name              Valid Range\n");
    fprintf(stderr, "-port                IMAP server port number\n");
    fprintf(stderr, "-replyTo             Hostname\n");
    fprintf(stderr, "-mailName            Your login name\n");
    fprintf(stderr, "-personal            Your name\n");
    fprintf(stderr, "-highlight           Turns off highlightNew (no arg necessary)\n");
    fprintf(stderr, "+highlight           Turns on highlightNew (no arg necessary)\n");
    fprintf(stderr, "-check               Time between checks, in decimal minutes\n");
    fprintf(stderr, "-filter              Turns off filterHeaders (no arg)\n");
    fprintf(stderr, "+filter              Turns on filterHeaders (no arg)\n");
    fprintf(stderr, "-view                Max number of messages visible in scroll window\n");
    fprintf(stderr, "-flagNo              Turns off new mail flag (no args)\n");
    fprintf(stderr, "-printCommand        Command used to print messages\n");
    fprintf(stderr, "-smtpHost            Hostname\n");
    fprintf(stderr, "-bells               Number of bells to ring for new messages\n");
}

int where;

void XimapMainLoop(test)
     int *test;
{
    XEvent event;
    while(!test || !*test) 
    {
	XtAppNextEvent(app_context,&event);
	XtDispatchEvent(&event);
	
	/* If there's been an unsolicited message from the server... */
	if(updateMailbox)
	    checkMailboxes();
    }
}

#define MIN_TO_MILLS 60000
#define ONE_MIN 1

int main(argc, argv)
     int  argc;
     char **argv;
{            
    Arg warg[ARGLISTSIZE];
    int n = 0;
    char *mailcapfilepath = NULL;

    struct passwd *paswd;

    void createTelemetry(/* Widget */);

    updateMailbox = 0;

    XtSetArg(warg[n], XtNallowShellResize, TRUE); n++;
    toplevel = XtAppInitialize(&app_context,
			       "XImap",
			       ximap_options, XtNumber(ximap_options),
			       &argc, argv,
			       NULL,                /* fallback resources */
			       warg, n); n = 0; /* app. shell resources */

    if( argc > 1 )
	SyntaxError( argc, argv );

    XtGetApplicationResources(toplevel, (char*) &res,
			      ximap_resources, XtNumber(ximap_resources),
			      NULL, 0);

    /*
     * Fix check interval to a reasonable value */
    if ((int)(res.check_interval * MIN_TO_MILLS) < MIN_TO_MILLS)
      res.check_interval = (float)ONE_MIN;

    mailcapfilepath = getenv("MAILCAPS");
    if(mailcapfilepath == NULL)
	mailcapfilepath = cpystr(MAILCAPPATH);

    setHandlers(parseMailcap(mailcapfilepath));

    if( res.flag_no_mailboxes == TRUE )
	res.flag_all_mailboxes = FALSE;

    XtAppAddActions(app_context, 
		    telemetry_actions, 
		    XtNumber(telemetry_actions));

    addMailboxActions(app_context);
    addReadWindowActions(app_context);
    addSearchActions(app_context);
    addComposeActions(app_context);
    addConfirmActions(app_context);
    addFbActions(app_context);

    setHomeDir((char* ) getenv("HOME"));
    setActiveLogin((MailBox*) NULL);
    setCurhost(NULL);

    id.localhost = XtMalloc(1024);
    gethostname(id.localhost, 1024);

    paswd = getpwuid(getuid());
    id.username = cpystr(paswd->pw_name);
    if(res.mail_name[0] == '\0'
       && id.username)
	res.mail_name = cpystr(id.username);

    createTelemetry(toplevel);	
    
    mail_link(&imapdriver);

    XtRealizeWidget(toplevel);
    XtSetKeyboardFocus( toplevel, telemetry );
    /* ouvrir le fichier d'information */
    open_ximap_logfile();

    /* NULL makes this loop forever... */
    XimapMainLoop(NULL);

    /* We'll never get here.... We always exit from a callback */
}

FILE *fichier= NULL;

#ifdef XIMAPDEBUG
int
open_ximap_logfile()
{
  fichier = fopen("/tmp/ximap.log", "w");
}
#else
open_ximap_logfile()
{
  return;
}
#endif

/*
 *
 * Creates the actual widgets for the telemetry window...
 * This window displays all messages, and is the window 
 * from which you open new mailboxes or quit the application 
 *
 *
 */
static Widget telemetry_label;

void createTelemetry(parent)
     Widget parent;
{
    Arg warg[16];
    int n = 0;
    Widget  TelemetryPanes,TelemetryButtonBox;
    
    TelemetryPanes = XtCreateManagedWidget("telemetry_panes", panedWidgetClass,
					   parent, warg, n); n = 0;

    XtSetArg(warg[n], XtNskipAdjust, TRUE); n++;
    XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
    TelemetryButtonBox = XtCreateManagedWidget("telemetry_buttons",
					       boxWidgetClass,
					       TelemetryPanes,
					       warg, n); n = 0;

    states = createButtons(TelemetryButtonBox, 
		  NULL,
		  telBtns);
    XtSetArg(warg[n], XtNborderWidth, 0); n++;
    XtSetArg(warg[n], XtNjustify, XtJustifyCenter); n++;
    XtSetArg(warg[n], XtNlabel , ""); n++;
    
    /* A special label for user notifications in the telemetry box */
    telemetry_label = XtCreateManagedWidget("tel_label", 
					    labelWidgetClass,
					    TelemetryButtonBox,
					    warg, n); n = 0;

    XtSetArg(warg[n], XtNeditType, XawtextAppend); n++;
    XtSetArg(warg[n], XtNscrollVertical, XawtextScrollWhenNeeded); n++;
    XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
    XtSetArg(warg[n], XtNallowResize, TRUE); n++;
    telemetry = XtCreateManagedWidget("telemetry_text",asciiTextWidgetClass,
				      TelemetryPanes, warg, n); n = 0;

    XtSetArg(warg[n], XtNscrollVertical, XawtextScrollWhenNeeded); n++;
    XtSetArg(warg[n], XtNeditType, XawtextAppend); n++;
    XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
    XtSetArg(warg[n], XtNallowResize, TRUE); n++;
    telemetry_debug = XtCreateWidget("telemetry_debug",
				     asciiTextWidgetClass,
				     TelemetryPanes, warg, n); n = 0;
}
/*
 * This function will clear the telemetry widget. 
 * It is an application timeout */
void update_telemetry_label();

static void
clear_telemetry_label(data, id)
     caddr_t data;
     XtIntervalId *id;
{
  update_telemetry_label("", FALSE);
}
     
/*
 * This function updates the telemetry label */
#define CLEARTIMEOUT 2000		/* milliseconds */
void
update_telemetry_label(txt, clear)
     char *txt;
     Boolean clear;
{
  Arg warg[16];
  int n= 0;

  XtSetArg(warg[n], XtNlabel, txt); n++;
  XtSetValues(telemetry_label, warg, n);

  if (clear) {
    XtAppAddTimeOut(app_context,
		    CLEARTIMEOUT,
		    clear_telemetry_label, 
		    NULL);
  }
}


/*
 *
 * Callback for checking mailbox at specific intervals -- needs
 * to reset itself each time it's used.  The timeout resource is 
 * in minutes, while the period of the timeout is in milliseconds,
 * hence the factor of 60000.
 *
 *
 */
void timedCheck(ms, id)
     MailBox*       ms;
     XtIntervalId *id;
{
    if (ISINACTIVE(ms->status))
    {
        ximap_log("\n!! TIMED-CHECK !!");

	ms->status |= TIMEDCHECK;
	checkMailBox(NULL, ms, NULL);
	ms->status &= ~TIMEDCHECK;

        ximap_log("\n!! TIMED-CHECK DONE !!");
    }

    ms->timeout = XtAppAddTimeOut(app_context,
				  (int)(res.check_interval * MIN_TO_MILLS),
				  timedCheck, 
				  ms);
}


/*
 *
 * getStatus() -- 
 *   returns the MAILSTATUS* associated with 
 *   a particular MAILSTREAM*.  MAILSTREAM is a structure
 *   set up and used by the c-Client.
 * 
 * arguments -- MAILSTREAM* 
 *
 * returns -- MAILSTATUS*
 */
MailBox* getStatus(stream)
     MAILSTREAM *stream;
{
    register STREAMARRAY *temp = id.mailboxes;
    MailBox*  stat = NULL;
    
    int number = 0;
    
    while (number++ < id.no_boxes        
	   && temp                       
	   && temp->stream != stream )
    {
	temp = temp->next;
    }
    
    if ( temp == NULL )
    {
	if ((stat = getActiveLogin()) == NULL)
	    mm_log("getStatus -- stream not found", ERROR);      
    }
    else
    {
	stat = temp->status;
    }

    return (stat);
}

/*
 *  putStream() -- assigns a given MAILSTREAM* to the
 *    stream member of a STREAMARRAY structure, based on the 
 *    MAILSTATUS* passed to the function.
 *
 *  arguments -- MAILSTATUS* - the status member of a STREAMARRAY.  Should
 *                     already be there.
 *               MAILSTREAM* - stream member of STREAMARRAY. To be added.
 * 
 *  returns -- nothing
 */
void putStream(stream, status)
     MAILSTREAM *stream;
     MailBox*     status;
{
    register STREAMARRAY *temp = id.mailboxes;
    
    while ( temp && temp->status != status )
    {
	temp = temp->next;
    }
    
    if (  temp->status == status )
    {
	temp->stream = stream;
	temp->login = FALSE;
    }
    else
    {
	mm_log("putStream -- status not found", ERROR);
    }
}

/************************************************************
*************************************************************

  Functions for dealing with more than one stream


*************************************************************
************************************************************/
MailBox *ximap_newstatus( )
{
    STREAMARRAY  *temp = NULL;
    MailBox      *status = NULL;
    
    if (( temp = (STREAMARRAY*)XtMalloc(sizeof(STREAMARRAY))) != NULL )
    {
	temp->next = temp->prev = NULL;

	if (( status = (MailBox*)XtMalloc(sizeof(MailBox))) != NULL )
	{
	    status->mailbox = (MAILSTREAM*) NULL;
	    status->mailboxName = (char*) NULL;
	    status->status = 0;
	    status->host = NULL;
	    status->messages = (MessageNode**) NULL;
	    status->last_message = (MessageNode**) NULL;
	    status->list_size = 0;
	    status->nmsgs = 0;
	    status->new = 0;
	    status->replyto = (MessageNode*) NULL;
	    status->mailWindow = (Widget) NULL;
	    status->browserList = NULL;
	    status->rwinList = NULL;
	    status->compList = NULL;
	    status->flagShell = (Widget) NULL;
	    status->mailboxFlag = (Widget) NULL;
	    status->keywords = (Widget) NULL;
	    status->curdisplay = NULL;
	    status->timeout = 0;
	    status->background_proc = 0;
	    status->lastcopy = (char*) NULL;
	    status->virtual_size = 0;
	    status->actual_size = 0;
	    status->first_header = BAD_MESSAGE;
	    status->view = 0;
	    status->start_highlight = BAD_MESSAGE;
	    status->num_found = 0;
	    status->found = NULL;
	    status->filter_heads = TRUE;
	    status->zoom = FALSE;
	    status->changing_buttons = NULL;
	    status->empty_box_buttons = NULL;
	    status->defaultMapping = NULL;
	    status->zoomMapping = NULL;
	    status->changedSize = 0;

	    temp->status = status;
	}
	else
	{
	    mm_log("newStatus -- malloc failed", ERROR);
	    XtFree(temp);
	    return(NULL);
	}
    }
    else
    {
	mm_log("newStatus -- malloc failed", ERROR);
	return (NULL);
    }
    
    if( id.mailboxes )
    {
	temp->next = id.mailboxes;
	id.mailboxes->prev = temp;
    }

    id.mailboxes = temp;
    temp->stream = NULL;
    ++id.no_boxes;
    
    return(status);
}

void killStatus(status)
     MailBox* status;
{
    STREAMARRAY *temp = id.mailboxes;
    int num = 0;
    
    while (num++ < id.no_boxes && temp->status != status)
	temp = temp->next;
    
    if(status->browserList)
    {
	BrowserList temp = status->browserList->next;
	ximap_freeBrowserEntry(status->browserList);
	status->browserList = temp;

	while(temp)
	{
	    temp = status->browserList->next;
	    status->browserList = temp;
	}
    }
    
    if (  temp->status == status )
    {
	removeSTREAMARRAY(temp);
    }
}

static void removeSTREAMARRAY(sa)
     STREAMARRAY *sa;
{
    if (sa)
    {
	if ( sa->prev )
	    sa->prev->next = sa->next;
	else
	    id.mailboxes = sa->next;
	
	if ( sa->next )
	    sa->next->prev = sa->prev;
	
	XtFree((char*)sa->status);
	XtFree((char*)sa);
	--id.no_boxes;
	if ( id.no_boxes <= 0 )
	    id.mailboxes = NULL;
    }
}

void closeAll()
{
    if ( id.mailboxes != NULL )
    {
	while ( id.mailboxes )
	{
	    closeMailBox((Widget)NULL, id.mailboxes->status, (XEvent*)NULL);
	    if (id.mailboxes)
		id.mailboxes = id.mailboxes->next;
	}
    }
}

/* 
 *
 * set and get various variables and structures -- members
 * of the MAILSESSION structure.
 *
 */

void setHandlers( h )
     HandlerArray h;
{
    id.handlers = h;
}


HandlerArray getHandlers()
{
    return( id.handlers );
}

char* getUsername()
{
    return(cpystr(id.username));
}

void setCurhost(host)
     register char *host;
{
    if(host)
    {
	char *temp = (char*) XtMalloc ((strlen(host)+1)*sizeof(char));
	register char *hostname = temp;

	if (temp)
	    while ( *temp++ = *host++)
		; /* VOID LOOP */
    
	id.lasthost = hostname;
    }
    else
    {
	id.lasthost = NULL;
    }
}

char* getCurhost()
{
    char *n = NULL;
    char *temp2 = id.lasthost;

    if(temp2)
    {
	char *temp = (char*) XtMalloc ((strlen(id.lasthost)+1)*sizeof(char));

	n = temp;
	if (temp)
	    while ( *temp++ = *temp2++)
		; /* VOID LOOP */
    }
    return (n); 
}

OPENHOSTS* newHost()
{
    OPENHOSTS *newhost = (OPENHOSTS*)XtMalloc(sizeof(OPENHOSTS));
    if ( newhost != NULL )
    {
	newhost->next = id.hosts;
	newhost->name = NULL;
	newhost->password = NULL;
	newhost->host = NULL;
	newhost->loggedIn = FALSE;
	id.hosts = newhost;
    }
    return(newhost);
}

OPENHOSTS* findHost(hnm)
     char *hnm;
{
    OPENHOSTS *retval = NULL;
    OPENHOSTS *temp = id.hosts;
    
    while ( temp != NULL )
    {
	if (strcmp( temp->host, hnm ))
	{
	    temp = temp->next;
	}
	else 
	{
	    retval = temp;
	    temp = NULL;
	}
    }
    return(retval);
}

void setName(host, name)
     char *host;
     char *name;
{
    char *temp = NULL;
    char *n = NULL;
    OPENHOSTS *hosttemp = NULL;
    
    if ((hosttemp = findHost( host )) != NULL)
    {
	n = temp = (char*) XtMalloc ((strlen(name)+1)*sizeof(char));
	
	if (temp)
	    while ( *temp++ = *name++)
		; /* VOID LOOP */
	
	if (hosttemp->name != NULL)
	    XtFree((char*) hosttemp->name);
	
	hosttemp->name = n;
    }
}

char* getName(hnm)
     char *hnm;
{
    char *temp = NULL;
    char *n = NULL;
    char *temp2 = NULL;
    
    OPENHOSTS *hosttemp = NULL;
    
    if ((hosttemp = findHost(hnm)) != NULL)
    {
	n = temp = (char*) XtMalloc (strlen(hosttemp->name)+1);
	
	temp2 = hosttemp->name;
	
	if (temp)
	    while ( *temp++ = *temp2++)
		; /* VOID LOOP */
    }
    return (n); 
}

void setHomeDir(c)
     char *c;
{
    char *temp = (char*) XtMalloc ((strlen(c)+1)*sizeof(char));
    char *n = temp;

    if (temp)
	while ( *temp++ = *c++)
	    ; /* VOID LOOP */
    
    id.home_dir = n;
}

char* getHomeDir( )
{
    char *n = NULL;
    register char *temp = NULL;
    register char *temp2 = NULL;
    
    if ( id.home_dir )
    {
	temp2 = id.home_dir;
	temp = (char*) XtMalloc ((strlen(id.home_dir)+1)*sizeof(char));
    }
    
    if (temp)
    {
	n = temp;
	while ( *temp++ = *temp2++)
	    ; /* VOID LOOP */
    }
    return (n); 
}

/* Mindless, but it's *something* */
#define PASSWORD_PATTERN 0x59
void setPassword(h, c)
     char *h;
     char *c;
{
    char *temp = NULL;
    char *p = NULL;
    OPENHOSTS *hosttemp = NULL;
    
    if ((hosttemp = findHost( h )) != NULL)
    {
	p = temp = (char*) XtMalloc ((strlen(c)+1)*sizeof(char));
	
	if (temp)
	{
	    while ( *c)
	    {
		*temp = *c ^ PASSWORD_PATTERN;
		temp++; c++;
	    }
	    *temp = '\0';
	}
	
	if (hosttemp->password != NULL)
	{
	    /* Zero it out so it can't be read in memory. */
	    memset(hosttemp->password, '\0', strlen(hosttemp->password));
	    XtFree((char*)hosttemp->password);
	}
	
	hosttemp->password = p;
    }
}

char* getPassword(hnm)
     char *hnm;
{
    char *temp = NULL;
    char *p = NULL;
    char *temp2 = NULL;
    
    OPENHOSTS *hosttemp = NULL;
    
    if ((hosttemp = findHost(hnm)) != NULL)
    {
	p = temp = (char*) XtMalloc ((strlen(hosttemp->password)+1));
	temp2 = hosttemp->password;
	
	if (temp)
	{
	    while (*temp2)
	    {
		*temp = *temp2 ^ PASSWORD_PATTERN;
		temp++; temp2++;
	    }
	    *temp = '\0';
	}
    }
    return (p);
}

/*
 * 
 * Checks to see if the mailhost is in the list of hosts opened this session.
 * If it is, it returns without doing anything.  If not it adds it.
 *
 * 
 */
void putHost(mailhost)
     char *mailhost;
{
    OPENHOSTS *hosttemp = NULL;

    if (mailhost)
    {
	if ( findHost(mailhost) == NULL )
	{
	    hosttemp = newHost();
	    if ( hosttemp != NULL )
	    {
		hosttemp->host = XtMalloc(strlen(mailhost) + 1);
		strcpy(hosttemp->host, mailhost);
	    }
	}
    }
}

void setActiveLogin(ms)
     MailBox* ms;
{
    id.activeLogin = ms;
}

MailBox* getActiveLogin()
{
    return( (MailBox*) id.activeLogin );
}

int loggedIn(hnm)
     char *hnm;
{
    OPENHOSTS *htmp = findHost( hnm );
    
    if ( htmp != NULL )
    {
	return ( ( htmp->loggedIn == TRUE ) ? 1 : 0 );
    }
    else 
    {
	return(0);
    }
}

/*
 *
 * Mark a specific host as having been successfully logged in to
 *
 *
 */
void setLoggedIn(mailhost)
     char *mailhost;
{
    OPENHOSTS *hosttemp = findHost( mailhost );
    
    if ( hosttemp != NULL )
	hosttemp->loggedIn = TRUE;
}

/***********************************************************

  
  Actions


***********************************************************/

static void appQuitAct( w, e )
     Widget w;
     XEvent *e;
{
    appQuit( w, NULL, e );
}

static void appOpenAct( w, e )
     Widget w;
     XEvent *e;
{
    appOpen( w, NULL, e );
}


/***********************************************************

  
  Callbacks


***********************************************************/

/*
 * Callback from session window
 * Open a mailbox:
 *      get the mailbox name
 *      log onto server ( once per session per server )
 *      create browser
 *      
 */
static void appOpen(w, empty, event)
     Widget w;
     void   *empty;
     XEvent *event;
{
    MailBox*    ms;
    XEvent e;
 
    static int busy;
    if( busy != 1 )
    {
	ms = ximap_newstatus();
	ms->status |= LOGGINGIN;
	
	/* create mailbox chooser */
	mailboxList( toplevel, ms, event );
	
	busy = 1;
	while (!(ms->mailboxName))
	{
	    XtAppNextEvent(app_context,&e);
	    XtDispatchEvent(&e);
	}
	busy = 0;

	if( !*ms->mailboxName )
	{
	    killStatus(ms);
	    return;
	}

	setActiveLogin(ms);
	setCurhost(NULL);
	if (ms->mailbox = mail_open (ms->mailbox,
				     ms->mailboxName,
				     0))
	{
	    char *host = NULL;
	    char *mailbox = NULL;
	    
	    putStream( ms->mailbox, ms );

	    if(!getCurhost())
	    {
		parseMailboxName(ms->mailboxName, &host, &mailbox);
		setCurhost(host);
		setName(host, getUsername());
	    }

	    setLoggedIn(getCurhost());
	    ms->host = getCurhost();
	    ms->nmsgs = ms->mailbox->nmsgs;	
	    ms->filter_heads = res.filter_headers;
	    
	    if(createMailbox(ms))
	    {
		/* 
		  the timeout resource should be specified in
		  MINUTES not SECONDS (I mean, really, how many people
		  want their mailbox checked at intervals of less 
		  than a minute?)
		  */
		
		ms->timeout = XtAppAddTimeOut(app_context,
					      (int)(res.check_interval * 
						    MIN_TO_MILLS),
					      timedCheck, 
					      ms);
		toggleDebug( toplevel, ms, (XEvent*) NULL );
		updateMailboxList( ms );
		ms->status &= ~LOGGINGIN;
		setActiveLogin((MailBox*) NULL);
		return;
	    }
	    else
	    {
		mail_close(ms->mailbox);
	    }
	    
	}
	setActiveLogin((MailBox*) NULL);
	killStatus(ms);
    }
}

/*
 * Callback from session window
 * Close all mailboxes and quit application
 *
 */
static void appQuit(w, empty, event)
     Widget w;
     void   *empty;
     XEvent *event;
{
    Widget temp = NULL;
    extern void closeAll();
    
    if (XbConfirm(toplevel,
		  "Do you really want to quit?",
		  FALSE) == TRUE)
    {
	closeAll();
	XtDestroyWidget(toplevel);
	free_statelist(states);
	exit(0);
    }
    else
    {
	temp = XtNameToWidget( toplevel, "confirm_shell" );
	XtDestroyWidget( temp );
    }
}



/***********************************************************

  
  Stuff to find out where you are.  Actions just pass the ID 
  of the widget they occurred in, so you need to translate the
  widget ID into the proper structure...


***********************************************************/

MailBox* getStatusFromBrowser(brw)
     Widget brw;
{
    register STREAMARRAY *temp = id.mailboxes;
    BrowserList btemp = NULL;
    
    int number = 0;
    
    while (number++ < id.no_boxes        
	   && temp)
    {
	btemp = temp->status->browserList;
	while(btemp)
	{
	    if( btemp->browser == brw )
		return( temp->status );

	    btemp = btemp->next;
	}
	temp = temp->next;
    }
    
    mm_log("getStatusFromBrowser -- stream not found", ERROR);      
    return ( NULL );
}

BrowserEntry* getBEFromBrowser(brw)
     Widget brw;
{
    register STREAMARRAY *temp = id.mailboxes;
    BrowserList btemp = NULL;
    
    int number = 0;
    
    while (number++ < id.no_boxes        
	   && temp)
    {
	btemp = temp->status->browserList;
	while(btemp)
	{
	    if( btemp->browser == brw )
		return( btemp );

	    btemp = btemp->next;
	}
	temp = temp->next;
    }
    
    mm_log("getBEFromBrowser -- stream not found", ERROR);      
    return ( NULL );
}

ComposeWindow* getCompWinFromWidget(w)
     Widget w;
{
    register STREAMARRAY *temp = id.mailboxes;
    ComposeWindow *cmpwin = NULL;
    int number = 0;

    Widget sh = getShell( w );

    while (number++ < id.no_boxes        
	   && temp)
    {
	cmpwin = temp->status->compList;
	while(cmpwin)
	{
	    if( cmpwin->shell == sh )
		return( cmpwin );

	    cmpwin = cmpwin->next;
	}
	temp = temp->next;
    }
    return ( NULL );
}

SEARCHWINDOW* getSearchWinFromWidget(w)
     Widget w;
{
    register STREAMARRAY *temp = id.mailboxes;
    BrowserList           btemp = NULL;
    int                   number = 0;

    Widget sh = getShell( w );

    while (number++ < id.no_boxes        
	   && temp)
    {
	btemp = temp->status->browserList;
	while(btemp)
	{
	    if(   btemp->searchWin
	       && btemp->searchWin->shell == sh )
	    {
		return ( btemp->searchWin );
	    }

	    btemp = btemp->next;
	}
	temp = temp->next;
    }
    return ( NULL );
}

