/*
 * Copyright (C) 2001 Thomas Roessler <roessler@does-not-exist.org>
 * 
 *     This program is free software; you can redistribute it
 *     and/or modify it under the terms of the GNU General Public
 *     License as published by the Free Software Foundation; either
 *     version 2 of the License, or (at your option) any later
 *     version.
 * 
 *     This program is distributed in the hope that it will be
 *     useful, but WITHOUT ANY WARRANTY; without even the implied
 *     warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *     PURPOSE.  See the GNU General Public License for more
 *     details.
 * 
 *     You should have received a copy of the GNU General Public
 *     License along with this program; if not, write to the Free
 *     Software Foundation, Inc., 59 Temple Place - Suite 330,
 *     Boston, MA  02111, USA.
 */ 

/* 
 * Much of this code was originally borrowed from pilot-mail.c,
 * which is Copyright (c) 1997, Kenneth Albanowski, and licensed
 * under the GNU Public License V2. 
 */

#include "global.h"

void pm_setup_context (const char *port)
{
  struct pi_sockaddr psa;
  char buff[BUFFLEN];
  
  memset (&Context, 0, sizeof (CONTEXT));

  Context.app_info  = safe_calloc (1, sizeof (struct MailAppInfo));
  Context.user 	    = safe_calloc (1, sizeof (struct PilotUser));
  
  
  if ((Context.sd = pi_socket (PI_AF_SLP, PI_SOCK_STREAM, PI_PF_PADP)) == 0)
  {
    pm_perror ("pi_socket");
    pm_exit (1);
  }
  
  
  psa.pi_family = PI_AF_SLP;
  strfcpy (psa.pi_device, port, sizeof (psa.pi_device));
  
  if (pi_bind (Context.sd, (struct sockaddr *) &psa, sizeof (psa)) == -1)
  {
    pm_perror ("pi_bind");
    pm_exit (1);
  }

  printf ("Waiting for connection on %s... ", port);
  fflush (stdout);
  
  if (pi_listen (Context.sd, 1) == -1)
  {
    pm_perror ("pi_listen");
    pm_exit (1);
  }
  
  if ((Context.sd = pi_accept (Context.sd, 0, 0)) == -1)
  {
    pm_perror ("pi_accept");
    pm_exit (1);
  }

  Context.sd_valid = 1;
  
  puts ("Connected.");
  
  dlp_ReadUserInfo (Context.sd, Context.user);
  dlp_OpenConduit (Context.sd);
  
  if (dlp_OpenDB (Context.sd, 0, 0x80|0x40, "MailDB", &Context.db) < 0)
  {
    pm_msg ("Can't open MailDB.");
    pm_exit (1);
  }
  
  Context.db_valid = 1;
  
  dlp_ReadAppBlock (Context.sd, Context.db, 0, buff, sizeof (buff));
  unpack_MailAppInfo (Context.app_info, buff, sizeof (buff));
  
  Context.sync_pref = safe_calloc (1, sizeof (struct MailSyncPref));
  Context.sync_pref->truncate = 8 * 1024;
  
  if (pi_version (Context.sd > 0x0100))
  {
    if (dlp_ReadAppPreference (Context.sd, makelong ("mail"), 1, 1, sizeof (buff), buff, 0, 0) >= 0)
    {
      pm_msg ("Got local backup mail preferences.\n");
      unpack_MailSyncPref (Context.sync_pref, buff, sizeof (buff));
    }
    else 
      pm_msg ("Could not get preferences.\n");
    
    if (dlp_ReadAppPreference (Context.sd, makelong ("mail"), 3, 1, sizeof (buff), buff, 0, 0) > 0)
    {
      Context.signature = safe_calloc (1, sizeof (struct MailSignaturePref));
      unpack_MailSignaturePref (Context.signature, buff, sizeof (buff));
    }
  }
}

void pm_destroy_context (void)
{
  if (Context.sync_pref)
  {
    free_MailSyncPref (Context.sync_pref);
    safe_free ((void **) &Context.sync_pref);
  }
  if (Context.signature)
  {
    free_MailSignaturePref (Context.signature);
    safe_free ((void **) &Context.signature);
  }
  if (Context.app_info)
  {
    free_MailAppInfo (Context.app_info);
    safe_free ((void **) &Context.app_info);
  }
  if (Context.sd_valid)
  {
    dlp_ResetLastSyncPC (Context.sd);
    if (Context.db_valid)
      dlp_CloseDB (Context.sd, Context.db);
    pi_close (Context.sd);
    
    Context.sd_valid = Context.db_valid = 0;
  }
  
  if (Context.user)
    safe_free ((void **) &Context.user);
}

