/*
 * Copyright (C) 2001 Thomas Roessler <roessler@does-not-exist.org>
 * 
 *     This program is free software; you can redistribute it
 *     and/or modify it under the terms of the GNU General Public
 *     License as published by the Free Software Foundation; either
 *     version 2 of the License, or (at your option) any later
 *     version.
 * 
 *     This program is distributed in the hope that it will be
 *     useful, but WITHOUT ANY WARRANTY; without even the implied
 *     warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *     PURPOSE.  See the GNU General Public License for more
 *     details.
 * 
 *     You should have received a copy of the GNU General Public
 *     License along with this program; if not, write to the Free
 *     Software Foundation, Inc., 59 Temple Place - Suite 330,
 *     Boston, MA  02111, USA.
 */ 

#include "global.h"

static int check_8bit (char *);

static void purge_lf (char *);
static void write_header (FILE *fp, const char *, char *);
static void write_rfc2047 (FILE *fp, const char *, char *, int);
static void write_qp (FILE *fp, const char *);

/* send pending messages */

void pm_send_mail (void)
{
  struct Mail mail;
  recordid_t mail_id;

  int attr, size, len;
  char buff[BUFFLEN];

  FILE *fp;

  int count = 0;
  
  while ((len = dlp_ReadNextRecInCategory (Context.sd, Context.db, 1, buff, 
					   &mail_id, 0, &size, &attr)) >= 0)
  {
    printf ("Sending message %d: ", ++count); fflush (stdout);
    if (attr & (dlpRecAttrDeleted|dlpRecAttrArchived))
    {
      puts ("skipping.");
      continue;
    }

    unpack_Mail (&mail, buff, len);
    
    /* clean up newlines in headers */
    purge_lf (mail.to);
    purge_lf (mail.cc);
    purge_lf (mail.bcc);
    purge_lf (mail.replyTo);
    purge_lf (mail.subject);
    
    /* start sendmail */
    if ((fp = popen (Opt.sendmail, "w")) == NULL)
    {
      pm_perror ("popen");
      pm_exit (1);
    }
    
    /* write out the header */
    write_header (fp, "From", Opt.from);
    write_header (fp, "To", mail.to);
    write_header (fp, "Cc", mail.cc);
    write_header (fp, "Bcc", mail.bcc);
    write_header (fp, "Reply-To", mail.replyTo);
    if (check_8bit (mail.subject) || strstr (mail.subject, "=?"))
      write_rfc2047 (fp, "Subject", mail.subject, 1);
    else
      write_header (fp, "Subject", mail.subject);
    
    /* write MIME headers */
    if (check_8bit (mail.body) || (Context.signature && 
				   check_8bit (Context.signature->signature)))
    {
      fputs ("MIME-Version: 1.0\n", fp);
      fputs ("Content-Transfer-Encoding: quoted-printable\n", fp);
      fprintf (fp, "Content-Type: text/plain; charset=\"%s\"\n\n", 
	       Opt.charset);
      write_qp (fp, mail.body);
      if (Context.signature)
      {
	fputs ("-- \n", fp);
	write_qp (fp, Context.signature->signature);
      }
    }
    else
    {
      fprintf (fp, "\n%s\n", mail.body);
      if (Context.signature)
	fprintf (fp, "-- \n%s\n", Context.signature->signature);
    }
    
    if (pclose (fp) == -1)
    {
      int oe = errno;
      puts ("failed.");
      errno = oe;
      pm_perror ("pclose");
      pm_exit (1);
    }
    
    puts ("done.");
    
    switch (Opt.dispose)
    {
      case DISP_DELETE:
        dlp_DeleteRecord (Context.sd, Context.db, 0, mail_id);
        break;
      case DISP_FILE:
        dlp_WriteRecord (Context.sd, Context.db, attr, mail_id, 3, buff, size, 0);
        break;
      case DISP_KEEP:
        /* do nothing */
    }
    
    free_Mail (&mail);
  }
  
  if (!count)
    pm_msg ("No new messages.\n");
  
}
  
/* Replace any line feed or carriage returns by spaces */

static void purge_lf (char *s)
{
  for (;s && *s; s++)
    if (*s == '\n' || *s == '\r')
      *s = ' ';
}

/* check whether a string contains non-usascii characters */

static int check_8bit (char *s)
{
  for (; s && *s; s++)
    if (*s & 0x80)
      return 1;
  
  return 0;
}

/* write an individual header */

static void write_header (FILE *fp, const char *tag, char *value)
{
  if (value)
    fprintf (fp, "%s: %s\n", tag, value);
}

/* write an individual header, but RFC2047-encode the value */

/* 
 * note that we don't care about line length limitations here -
 * it's extremely unlikely that anyone on the palm would produce
 * subjects of concern anyway. 
 */

static void write_rfc2047 (FILE *fp, const char *tag, char *value, 
			   int fix_subject)
{
  const char *charset;
  
  if (!value) return;

  if (check_8bit (value))
    charset = Opt.charset;
  else
    charset = "us-ascii";
  
  fprintf (fp, "%s: ", tag);
  
  if (fix_subject && 
      (!strcasecmp (value, "Re:") || !strcasecmp (value, "Antw:")))
  {
    fputs ("Re: ", fp);
    while (*value++ != ':') 
      ;
    SKIPWS (value);
  }
  
  fprintf (fp, "=?%s?Q?", charset);

  for (; *value; value++)
  {
    if (*value == ' ')
      fputc ('_', fp);
    else if (ISSPACE (*value) || (*value & 0x80) ||
	     *value == '?' || *value == '=')
      fprintf (fp, "=%02X", ((unsigned int) *value) & 0xff);
    else
      fputc ((unsigned int) *value, fp);
  }

  fputs ("?=\n", fp);
}

/* write out the message body */

static void write_qp (FILE *fp, const char *string)
{
  char last = 0;
  const char *t;
  int col = 0;
  
  for (t = string; *t; t++)
  {
    if (col > 70)
    {
      fputs ("=\n", fp);
      col = 0;
    }
    if ((*t & 0x80) || (ISSPACE (*t) && t[1] == '\n'))
    {
      fprintf (fp, "=%02X", ((unsigned int) *t) & 0xff);
      col += 3;
    }
    else
    {
      fputc (*t, fp);
      col++;
    }
    last = *t;
  }
  
  if (last != '\n')
    fputs ("\n", fp);
}

