/*
 *	Copyright 1988 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 */

/* LINTLIBRARY */

#include "mailer.h"
#ifdef	USE_GDBM
#include <gdbm.h>
#include <fcntl.h>
#include <sys/file.h>
#include "search.h"
#include "io.h"

extern int gdbm_errno;

/*
 * Search an GDBM format database for a key pair.
 */

struct conscell *
search_gdbm(sip)
	struct search_info *sip;
{
	GDBM_FILE db;
	datum val, key;
	struct conscell *tmp;
	struct spblk *spl;
	int retry, symid;
	extern int deferit;

	if (sip->file == NULL)
		return NULL;
	retry = 0;
	symid = symbol((u_char *)sip->file);
	spl = sp_lookup(symid, spt_files);
	if (spl == NULL || (db = (GDBM_FILE)spl->data) == NULL) {
reopen:
		db = gdbm_open(sip->file, 0, GDBM_READER, 0, NULL);
		if (db == NULL) {
			++deferit;
			v_set(DEFER, DEFER_IO_ERROR);
			fprintf(stderr, "search_gdbm: cannot open %s!\n",
					sip->file);
			return NULL;
		}
		if (spl == NULL)
			sp_install(symid, (u_char *)db, O_RDONLY, spt_files);
		else
			spl->data = (u_char *)db;
	}
	key.dptr = (char *)(sip->key);
	key.dsize = strlen((char *)(sip->key)) + 1;
	val = gdbm_fetch(db, key);
	gdbm_close(db);
	if (val.dptr == NULL) {
		if (!retry && gdbm_errno) {
			++retry;
			goto reopen;
		}
		return NULL;
	}
	return newstring((u_char *)strnsave(val.dptr, val.dsize));
}

/*
 * Flush buffered information from this database, close any file descriptors.
 */

void
close_gdbm(sip)
	struct search_info *sip;
{
	GDBM_FILE db;
	struct spblk *spl;
	int symid;

	if (sip->file == NULL)
		return;
	symid = symbol((u_char *)sip->file);
	spl = sp_lookup(symid, spt_modcheck);
	if (spl != NULL)
		sp_delete(spl, spt_modcheck);
	spl = sp_lookup(symid, spt_files);
	if (spl == NULL || (db = (GDBM_FILE)spl->data) == NULL)
		return;
	gdbm_close(db);
	(void) sp_install(symid, (u_char *)NULL, 0, spt_files);
}


GDBM_FILE
open_gdbm(sip, flag, comment)
	struct search_info *sip;
	int flag;
	char *comment;
{
	GDBM_FILE db;
	struct spblk *spl;
	int symid;
	extern int deferit;

	if (sip->file == NULL)
		return NULL;
	symid = symbol((u_char *)sip->file);
	spl = sp_lookup(symid, spt_files);
	if (spl != NULL && flag == O_RDWR && spl->mark != O_RDWR)
		close_gdbm(sip);
	if (flag == O_RDWR)	flag = GDBM_WRITER;
	else			flag = GDBM_READER;
	if (spl == NULL || (db = (GDBM_FILE)spl->data) == NULL) {
		db = gdbm_open(sip->file, flag, 0);
		if (db == NULL) {
			++deferit;
			v_set(DEFER, DEFER_IO_ERROR);
			fprintf(stderr, "%s: cannot open %s!\n",
					comment, sip->file);
			return NULL;
		}
		if (spl == NULL)
			sp_install(symid, (u_char *)db, flag, spt_files);
		else
			spl->data = (u_char *)db;
	}
	return db;
}

/*
 * Add the indicated key/value pair to the database.
 */

int
add_gdbm(sip, value)
	struct search_info *sip;
	char *value;
{
	GDBM_FILE db;
	datum val, key;
	extern int deferit;

	if ((db = open_gdbm(sip, O_RDWR, "add_gdbm")) == NULL)
		return EOF;
	key.dptr = (char *)(sip->key);
	key.dsize = strlen((char *)(sip->key)) + 1;
	val.dptr = value;
	val.dsize = strlen(value)+1;
	if (gdbm_store(db, key, val, GDBM_REPLACE) < 0) {
		++deferit;
		v_set(DEFER, DEFER_IO_ERROR);
		fprintf(stderr, "add_gdbm: cannot store (\"%s\",\"%s\")\n",
				sip->key, value);
		return EOF;
	}
	return NULL;
}

/*
 * Remove the indicated key from the database.
 */

int
remove_gdbm(sip)
	struct search_info *sip;
{
	GDBM_FILE db;
	datum key;
	extern int deferit;

	if ((db = open_gdbm(sip, O_RDWR, "remove_gdbm")) == NULL)
		return EOF;
	key.dptr = (char *)(sip->key);
	key.dsize = strlen((char *)(sip->key)) + 1;
	if (gdbm_delete(db, key) < 0) {
		++deferit;
		v_set(DEFER, DEFER_IO_ERROR);
		fprintf(stderr, "remove_gdbm: cannot remove \"%s\"\n",
				sip->key);
		return EOF;
	}
	return NULL;
}

/*
 * Print the database.
 */

void
print_gdbm(sip, outfp)
	struct search_info *sip;
	FILE *outfp;
{
	GDBM_FILE db;
	datum key, val;

	if ((db = open_gdbm(sip, O_RDONLY, "print_gdbm")) == NULL)
		return;
	for (key = gdbm_firstkey(db); key.dptr != NULL; key = gdbm_nextkey(db)) {
		val = gdbm_fetch(db, key);
		if (val.dptr == NULL)
			continue;
		if (gdbm_errno)
			break;
		if (*val.dptr == '\0')
			fprintf(outfp, "%s\n", key.dptr);
		else
			fprintf(outfp, "%s\t%s\n", key.dptr, val.dptr);
	}
	fflush(outfp);
}

/*
 * Print the uid of the owner of the database.  Note that for gdbm-style
 * databases there are several files involved so picking one of them for
 * security purposes is very dubious.
 */

void
owner_gdbm(sip, outfp)
	struct search_info *sip;
	FILE *outfp;
{
	GDBM_FILE db;
	struct gdbmfoo {	/* extract from gdbmdefs.h - gdbm sources */
		char *name;	/* Valid for GDBM 0.9beta! */
		int read_write;
		void (*fatal_err)();
		int desc;	/* open() handle for file */
	} gdbmfoo;	
	int	fno;
	struct stat stbuf;

	if ((db = open_gdbm(sip, O_RDONLY, "owner_gdbm")) == NULL)
		return;
	fno = ((struct gdbmfoo *) db)->desc;
	if (fstat(fno, &stbuf) < 0) {
		fprintf(stderr, "owner_gdbm: cannot fstat(\"%s\")!\n",
				sip->file);
		return;
	}
	fprintf(outfp, "%d\n", stbuf.st_uid);
	fflush(outfp);
}

int
modp_gdbm(sip)
	struct search_info *sip;
{
	GDBM_FILE db;
	struct gdbmfoo {	/* extract from gdbmdefs.h - gdbm sources */
		char *name;	/* Valid for GDBM 0.9beta! */
		int read_write;
		void (*fatal_err)();
		int desc;	/* open() handle for file */
	} gdbmfoo;	
	int	fno;
	struct stat stbuf;
	struct spblk *spl;
	int symid, rval;

	if (sip->file == NULL
	    || (db = open_gdbm(sip, O_RDONLY, "modp_gdbm")) == NULL)
		return 0;
	fno = ((struct gdbmfoo *) db)->desc;
	if (fstat(fno, &stbuf) < 0) {
		fprintf(stderr, "modp_gdbm: cannot fstat(\"%s\")!\n",
				sip->file);
		return 0;
	}

	symid = symbol((u_char *)sip->file);
	spl = sp_lookup(symid, spt_modcheck);
	if (spl != NULL) {
		rval = stbuf.st_mtime != (time_t)spl->data
			|| stbuf.st_nlink != (int)spl->mark;
	} else
		rval = 0;
	sp_install(symid, (u_char *)stbuf.st_mtime,
			  stbuf.st_nlink, spt_modcheck);
	return rval;
}
#endif	/* USE_GDBM */
