/*
 *	Copyright 1988 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 */

/*
 * ZMailer router, main and miscellany routines.
 */

#include "mailer.h"
#include <grp.h>
#include <pwd.h>
#include <signal.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/file.h>
#include <mail.h>
#ifdef	USE_SYSLOG
#include <syslog.h>
#endif	/* USE_SYSLOG */
#include "interpret.h"

extern char *logdir, *cf_suffix;
extern int D_final;

char	*progname;
char	*mailshare;
char	*myhostname = 0;
char	*postoffice = NULL;
char	*pidfile = PID_ROUTER;
char	*log;
time_t	now;
int	stickymem = MEM_TEMP;
int	mustexit = 0;
int	canexit = 0;
int	deferit;
int	origoptind;
int	savefile = 0;
char	*zshopts = "-O";

int
main(argc, argv)
	int	argc;
	char	*argv[];
{
	int c, errflg, daemonflg, killflg, interactiveflg, tac;
	int version;
	long offout, offerr;
	char *config, *cp, *tav[20], *av[3];
#ifdef	XMEM
	FILE *fp;
#endif	/* XMEM */
	extern int optind;
	extern int zshtoplevel();
	extern char *optarg, *Version, *run_trace();
	extern u_char *rfc822date();
	extern void tfree(), trapexit(), detach(), killprevious();
	extern void getnobody(), prversion(), initialize(), run_daemon();
	extern int s_apply();
	extern time_t time();
	extern SIGNAL_TYPE sig_hup();
	extern int loginit();/* should be SIGNAL_TYPE but we're playing games */

	if ((progname = strrchr(argv[0], '/')) == NULL)
		progname = argv[0];
	else
		++progname;
	origoptind = optind;	/* needed for reuse of getopt() */
	log = config = NULL;
	errflg = daemonflg = killflg = interactiveflg = version = 0;
	tac = 0;
	while ((c = getopt(argc, argv, "m:dikf:o:t:L:P:V")) != EOF)
		switch (c) {
		case 'd':	/* become a daemon */
			daemonflg = 1;
			break;
		case 'm':
#ifdef	XMEM
			if ((fp = fopen(optarg, "w+")) != NULL) {
				dup2(fileno(fp), 30);
				close(fileno(fp));
				fileno(fp) = 30;
				mal_setstatsfile(fp);
				mal_trace(1);
			}
#endif	/* XMEM */
			break;
		case 'o':
			zshopts = optarg;
			break;
		case 't':
			if (tac < (sizeof tav)/(sizeof tav[0]))
				tav[++tac] = optarg;
			else {
				fprintf(stderr, "Too many trace options!\n");
				fprintf(stderr, "Ignoring '%s'\n", optarg);
			}
			break;
		case 'f':	/* override default config file */
			config = optarg;
			break;
		case 'i':	/* first read config file, then read from tty */
			interactiveflg = 1;
			break;
		case 'k':	/* kill the previous daemon upon startup */
			killflg = 1;
			break;
		case 'L':	/* override default log file */
			log = optarg;
			break;
		case 'P':	/* override default postoffice */
			postoffice = optarg;
			break;
		case 'V':
			version = 1;
			break;
		case '?':
		default:
			errflg++;
			break;
		}
	if (errflg || (interactiveflg && daemonflg)) {
		(void) fprintf(stderr,
			"Usage: %s [ -dikV -t traceflag -f configfile -L logfile -P postoffice]\n",
			progname);
		exit(128+errflg);
	}
	now = time((long *)0);
	if ((mailshare = getzenv("MAILSHARE")) == NULL)
		mailshare = MAILSHARE;
	if (config == NULL) {
		config = cf_suffix;
		/* we don't need to remember this for long */
		config = smalloc(MEM_TEMP, 3 + (u_int)(strlen(mailshare)
					     + strlen(progname)
					     + strlen(cf_suffix)));
		(void) sprintf(config, "%s/%s.%s",
			       mailshare, progname, cf_suffix);
	}
	if (postoffice == NULL
	    && (postoffice = getzenv("POSTOFFICE")) == NULL)
		postoffice = POSTOFFICE;

	getnobody();

	c = optind;	/* save optind since builtins can interfere with it */

	if (daemonflg && log == NULL) {
		if ((cp = getzenv("LOGDIR")) != NULL)
			logdir = cp;
		log = smalloc(MEM_PERM, 2 + (u_int)(strlen(logdir)
					  + strlen(progname)));
		sprintf(log, "%s/%s", logdir, progname);
	}
	(void) setvbuf(stdout, (char *)NULL, _IOLBF, 0);
	(void) setvbuf(stderr, (char *)NULL, _IOLBF, 0);
	if (log != NULL) {
		/* loginit is a signal handler, so can't pass log */
		if (loginit() < 0)	/* do setlinebuf() there */
			die(1, "log initialization failure");
		(void) signal(SIGHUP, sig_hup); /* close and reopen log files */
	} else
		(void) signal(SIGHUP, SIG_IGN); /* no surprises please */

	if (version || interactiveflg || tac > 0) {
		prversion("router");
		if (version)
			exit(0);
		putc('\n', stderr);
	}
	if (tac > 0) {			/* turn on some trace/debug flags */
		tav[0] = "debug";
		/* lax, no NULL guard on end of tav */
		(void) run_trace(++tac, tav);
	}
	stickymem = MEM_PERM;

	initialize(config, argc - c, &argv[c]);

	stickymem = MEM_TEMP;	/* this is the default allocation type */
	offout = ftell(stdout);
	offerr = ftell(stderr);

#ifdef	MALLOC_TRACE
	mal_leaktrace(1);
#endif	/* MALLOC_TRACE */

	if (daemonflg) {
		if (chdir(postoffice) < 0 || chdir(ROUTERDIR) < 0)
			fprintf(stderr, "%s: cannot chdir.\n", progname);
		/* XX: check if another daemon is running already */
		if (offout < ftell(stdout) || offerr < ftell(stderr)) {
			fprintf(stderr, "%d %d %d %d\n", offout, ftell(stdout), offerr, ftell(stderr));
			fprintf(stderr, "%s: daemon not started.\n", progname);
			die(1, "errors during startup");
		}
		if (tac == 0)		/* leave worldy matters behind */
			detach();
		(void) printf("%s: router daemon (%s)\n\tstarted at %s\n",
				progname, Version, (char *)rfc822date(&now));
		if (killflg)
			killprevious(SIGTERM, pidfile);
#ifdef	LOG_PID
#ifdef	LOG_MAIL
		openlog("router", LOG_PID, LOG_MAIL);
#else	/* !LOG_MAIL */
		openlog("router", LOG_PID);
#endif	/* LOG_MAIL */
#endif	/* LOG_PID */
	}
	if (c < argc) {
		savefile = 1;
		/*
		 * we need to use a local variable (c) because optind is global
		 * and can (and will) be modified by the funcall()'s we do.
		 */
		do {
			av[0] = "process";
			av[1] = argv[c];
			av[2] = NULL;
#ifdef	XMEM
(void) write(30, "\n", 1);
#endif	/* XMEM */
			(void) s_apply(2, &av[0]);
		} while (++c < argc);
	} else if (daemonflg) {
		av[0] = "daemon";
		av[1] = NULL;
		run_daemon(1, &av[0]);
		/* NOTREACHED */
	} else if (interactiveflg) {
#ifdef	MALLOC_TRACE
		(void) zshtoplevel((char *)NULL);
#else	/* !MALLOC_TRACE */
		trapexit(zshtoplevel((char *)NULL));
#endif	/* MALLOC_TRACE */
		/* NOTREACHED */
	}
#ifdef	MALLOC_TRACE
	dbfree(); zshfree();
#endif	/* MALLOC_TRACE */
	if (mustexit)
		die(0, "signal");
#ifdef	MALLOC_TRACE
	die(0, "malloc trace");
#endif	/* MALLOC_TRACE */
	trapexit(0);
	/* NOTREACHED */
	return 0;
}

/* Run around and gather the necessary information for starting operation */

void
initialize(configfile, argc, argv)
	char *configfile;
	int argc;
	char *argv[];
{
	struct group *grp;
	struct sptree_init *sptip;
	int ac;
	char **cpp, **av;
	extern char *default_trusted[], *files_group;
	extern int files_gid;
	extern struct sptree *spt_goodguys, *icdbspltree();
	extern struct sptree_init splaytrees[];
	extern int sh_include(), run_relation();
	extern void zshinit(), setfreefd(), init_header();
	extern struct group *getgrnam();

	av = (char **)emalloc((5+argc)*(sizeof (char *)));
	/* initialize shell */
	ac = 0;
	av[ac++] = progname;
	av[ac++] = "-s";
	av[ac++] = zshopts;
	while (argc-- > 0)
		av[ac++] = *argv++;
	av[ac] = NULL;
	zshinit(ac, av);

	/* add builtin router functions to list of builtin shell functions */
	{
		register struct shCmd *shcmdp;
		extern struct shCmd fnctns[];
		extern struct sptree *spt_builtins;

		for (shcmdp = &fnctns[0]; shcmdp->name != NULL; ++shcmdp)
			sp_install(symbol((u_char *)shcmdp->name),
				   (u_char *)shcmdp, 0, spt_builtins);
	}

	/* initialize splay trees in router */
	av[0] = "relation";
	av[1] = "-t";
	av[4] = NULL;

	for (sptip = &splaytrees[0]; sptip->spta != NULL; ++sptip) {
		if (sptip->incore_name != NULL) {
			extern struct sptree *spt_headers;

			if (sptip->spta == &spt_headers)
				av[2] = "header";
			else
				av[2] = "incore";
			av[3] = sptip->incore_name;
			if (run_relation(4, av) == 0)
				*(sptip->spta) =
					icdbspltree(sptip->incore_name);
		} else
			*(sptip->spta) = sp_init();
	}

	init_header();

	/* trusted users */
	for (cpp = default_trusted; cpp != NULL && *cpp != NULL; ++cpp)
		sp_install(symbol((u_char *)*cpp), (char *)NULL, 0, spt_goodguys);

	if (files_group != NULL) {
		if ((grp = getgrnam(files_group)) == NULL)
			files_gid = -1;
		else
			files_gid = grp->gr_gid;
	}

	/* source the router config file */
	ac = 0;
	av[ac++] = ".";
	if (strchr(configfile, '/') == NULL) {
		av[ac] = emalloc(strlen(configfile)+sizeof "./"+1);
		(void) sprintf(av[ac++], "./%s", configfile);
	} else
		av[ac++] = configfile;
	av[ac] = NULL;
	setfreefd();
	(void) sh_include(ac, av);
	if (av[1] != configfile)
		(void) free(av[1]);
	(void) free((char *)av);
}

short
login_to_uid(name)
	u_char	*name;
{
	struct passwd *pw;
	short uid;
	char buf[BUFSIZ];
	u_char *cp, *fn;
	struct spblk *spl;
	extern struct sptree *spt_uidmap, *spt_loginmap, *spt_fullnamemap;
	extern short nobody;
	extern struct passwd *getpwnam();

	spl = sp_lookup(symbol(name), spt_loginmap);
	if (spl == NULL) {
		int oval = stickymem;

		stickymem = MEM_MALLOC;
		if ((pw = getpwnam((char *)name)) == NULL) {
			uid = nobody;
		} else {
			uid = pw->pw_uid;
			cp = (u_char *)strnsave(pw->pw_name,
						strlen(pw->pw_name));
			(void) sp_install((u_int)uid, cp, 0, spt_uidmap);
			(void)fullname(pw->pw_gecos,buf,sizeof buf,pw->pw_name);
			fn = (u_char *)strnsave(buf, strlen(buf));
			(void) sp_install(symbol(cp), fn, 0, spt_fullnamemap);
		}
		(void) sp_install(symbol(name), (u_char *)uid, 0, spt_loginmap);
		stickymem = oval;
	} else
		uid = (int)spl->data;
	return uid;
}

u_char *
uidpwnam(uid)
	int	uid;
{
	struct passwd *pw;
	register u_char *cp;
	u_char *fn;
	struct spblk *spl;
	char buf[BUFSIZ];
	extern struct sptree *spt_uidmap, *spt_loginmap, *spt_fullnamemap;
	extern struct passwd *getpwuid();

	spl = sp_lookup((u_int)uid, spt_uidmap);
	if (spl == NULL) {
		if ((pw = getpwuid(uid)) == NULL) {
			(void) sprintf(buf, "uid#%d", uid);
			cp = (u_char *)strnsave(buf, strlen(buf));
		} else {
			int oval = stickymem;

			stickymem = MEM_MALLOC;
			cp = (u_char *)strnsave(pw->pw_name,
						strlen(pw->pw_name));
			(void) sp_install(symbol(cp),
						(u_char *)pw->pw_uid, 0,
						spt_loginmap);
			(void)fullname(pw->pw_gecos,buf,sizeof buf,pw->pw_name);
			fn = (u_char *)strnsave(buf, strlen(buf));
			(void) sp_install(symbol(cp), fn, 0, spt_fullnamemap);
			(void) sp_install((u_int)uid, cp, 0, spt_uidmap);
			stickymem = oval;
		}
	} else
		cp = spl->data;
	return cp;
}


/* Can we trust this person? */

int
isgoodguy(uid)
	int	uid;
{
	extern struct sptree *spt_goodguys;
	struct spblk *spl;

	/*
	 * If you're wondering about this comparison... I had to store
	 * *something* in the splay tree; I'm really using it as a boolean.
	 */
	spl = sp_lookup(symbol((u_char *)uidpwnam(uid)), spt_goodguys);
	return spl != NULL;
}

#define	MAXSAFESIZE	700 /* syslog dumps core if we have much over this */

void
logmessage(e)
	struct envelope *e;
{
	int n, len;
	char *from, *to, *cp;
	char buf[MAXSAFESIZE];
	struct header *h;
	struct addr *p;
	struct address *ap;
	extern char *saveAddress();
	extern void logit();

	from = NULL;
	for (h = e->e_eHeaders; h != NULL; h = h->h_next) {
		if (h->h_descriptor->class == eFrom
		    && h->h_contents.a != NULL) {
			p = h->h_contents.a->a_tokens;
			if (p != NULL) {
				from = saveAddress(p);
				break;
			}
		}
	}
	if (from == NULL)
		from = "?from?";
	n = 0;
	to = buf;
	for (h = e->e_eHeaders; h != NULL; h = h->h_next) {
		if (h->h_descriptor->class == eTo && h->h_contents.a != NULL) {
			for (ap = h->h_contents.a; ap != NULL; ap=ap->a_next) {
				cp = saveAddress(ap->a_tokens);
				len = strlen(cp);
				if (to + len + 2 >= buf + sizeof buf) {
					/* print what we've got so far */
					if (to != buf) {
						*to = '\0';
						logit(e->e_file,
						      e->e_messageid,
						      from, buf);
						to = buf;
					}
					if (to + len + 2 >= buf + sizeof buf)
						continue;
				} else {
					if (n) *to++ = ',';
					*to++ = ' ';
				}
				strncpy(to, cp, len);
				to += len;
				++n;
			}
		}
	}
	if (to != buf) {
		*to = '\0';
		logit(e->e_file, e->e_messageid, from, buf);
	}
}

/*
 * All the strangeness in the logit() routine is because syslog() is
 * broken; it can only handle a certain size buffer before it'll dump core.
 * We do the same processing for stdout so that stdout can be fed into
 * logger without having to fix or customize a vendor program.  Sigh.
 */

void
logit(file, id, from, to)
	char *file, *id, *from, *to;
{
	int flen, baselen;
	char c;

	if (id == NULL)
		id = file;
	baselen = strlen(file) + strlen(id) + 4;
	c = '\0';
	while (baselen + strlen(from) > MAXSAFESIZE) {
		/* Wonderful software we're dealing with here... */
		c = *(from+MAXSAFESIZE-baselen);
		*(from+MAXSAFESIZE-baselen) = '\0';
		printf("%s: file: %s %s...\n", id, file, from);
#ifdef	LOG_INFO
		syslog(LOG_INFO, "%s: file: %s %s...", id, file, from);
#endif	/* LOG_INFO */
		from += MAXSAFESIZE-baselen;
		*from = c;
		*--from = '.';
		*--from = '.';
		*--from = '.';
	}
	flen = strlen(from);
	while (baselen + flen + strlen(to) > MAXSAFESIZE) {
		c = *(to+MAXSAFESIZE-baselen-flen);
		*(to+MAXSAFESIZE-baselen-flen) = '\0';
		if (flen > 0)
			printf("%s: file: %s %s =>%s\n", id, file, from, to);
		else
			printf("%s: file: %s %s\n", id, file, to);
#ifdef	LOG_INFO
		if (flen > 0)
			syslog(LOG_INFO, "%s: file: %s %s =>%s",
					 id, file, from, to);
		else
			syslog(LOG_INFO, "%s: file: %s %s", id, file, to);
#endif	/* LOG_INFO */
		to += MAXSAFESIZE-baselen-flen;
		*to = c;
		*--to = '.';
		*--to = '.';
		*--to = '.';
		flen = 0;
	}
	if (flen > 0) {
		printf("%s: file: %s %s =>%s\n", id, file, from, to);
#ifdef	LOG_INFO
		syslog(LOG_INFO, "%s: file: %s %s =>%s", id, file, from, to);
#endif	/* LOG_INFO */
	}
}

char *
mail_host()
{
	extern char *myhostname;

	return myhostname;
}
