/*
 *	Copyright 1988 by Rayan Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 *	This file is mostly the work of Dennis Ferguson.  Thanks Dennis.
 */

/*
 * settrusteduser, runastrusteduser - if running as root, setuid to
 *	a nonroot but trusted (by mail) user, if not running as root
 *	ignore any effect of a setuid bit.
 */

#include "hostenv.h"
#include "mailer.h"
#include <ctype.h>
#include <pwd.h>

#ifdef	USE_SETUIDX	/* IBM AIXism */
# include <sys/id.h>
# ifdef  USE_SETREUID
#  define setreuid(x,y) setuidx(ID_REAL|ID_EFFECTIVE, y)
# else
#  define setuid(y) setuidx(ID_REAL|ID_EFFECTIVE, y)
# endif
#endif

#define	DEFTRUSTEDUSER	"daemon"	/* default trusted user if no other */

static int trusteduid = -1;	/* set to trusted uid if we find it */

/*
 * As far as I know, BSD systems are the only ones which allow you
 * to undo the effects of a setuid to a nonroot user.  This stores
 * the original real uid mostly for this purpose.
 */
static int origruid;

/*
 * settrusteduser - find the trusted uid if we can
 */
void
settrusteduser()
{
	char *trusteduser;
	struct passwd *pw;
	extern struct passwd *getpwnam();
	extern char *getzenv();

	if ((trusteduser = getzenv("TRUSTEDUSER")) == NULL)
		trusteduser = DEFTRUSTEDUSER;
	if (isascii(*trusteduser) && isdigit(*trusteduser)) {
		int n = atoi(trusteduser);
		if (n > 0) {
			trusteduid = n;
			return;
		}
	}
	if ((pw = getpwnam(trusteduser)) == NULL) {
		trusteduid = 0;		/* can't do anything, run as root */
		return;
	}
	trusteduid = pw->pw_uid;
}

/*
 * runastrusteduser - setuid us to the user ID with minimal loss of security
 *	(i.e. invoking user if not root, trusted user if root).  This should
 *	be done reversibly if possible.  This routine should never be called
 *	twice without an intervening call to runasrootuser().
 */
int
runastrusteduser()
{
	int uid;

	uid = geteuid();
	origruid = getuid();	/* this may be wrong if called 2nd time */
	if (origruid != 0 && origruid != uid)
		trusteduid = origruid;
	else if (uid != 0)
		return uid;		/* forget it.  No need, no way */
	if (trusteduid == -1)
		settrusteduser();
	if (trusteduid == 0)
		return 0;		/* trusted uid not found */
#ifdef	USE_SETREUID
	setreuid(0, -1);		/* set real uid to root */
	setreuid(-1, trusteduid);	/* set euid to trusted */
#else	/* !USE_SETREUID */
#ifdef	USE_SETEUID
	seteuid(trusteduid);
#else	/* !USE_SETEUID */
	setuid(trusteduid);
#endif	/* !USE_SETEUID */
#endif	/* USE_SETREUID */
	return trusteduid;
}

/*
 * runasrootuser - undo what was done by runastrusteduser()
 */
void
runasrootuser()
{
	if (trusteduid <= 0)
		return;			/* nothing done before */
#ifdef	USE_SETREUID
	setreuid(-1, 0);		/* make us effectively root */
	setreuid(origruid, -1);		/* reset to old real uid */
#else	/* !USE_SETREUID */
#ifdef	USE_SETEUID
	seteuid(0);
#else	/* !USE_SETEUID */
	setuid(0);
#endif	/* !USE_SETEUID */
#endif	/* USE_SETREUID */
}
