/* Small pieces for scanning forward on a buffer of RFC-821/822 compliant
   addresses */

/* (c) Matti Aarnio 1993 <mea@nic.funet.fi> */

/* All these routines scan over the lexical elements they are after, and
   if successfull, return pointer just AFTER such element.
   If they fail, they return their input -- thus no scan forward.. */

#ifndef __STDC__
#define const
#endif

const char *rfc821_error = 0;		/* Error text */
const char *rfc821_error_ptr = 0;	/* scan position of the error */

extern char *rfc821_domain();		/* Entry point */
extern char *rfc821_path();		/* Entry point */
extern char *rfc821_path2();		/* Entry point */
extern char *rfc821_adl();		/* Entry point */

static char *rfc821_localpart();	/* static forward definition */

static const char *premature_end  = "Premature end of input";
static const char *no_input       = "No input";

/* ================================================================ */


#define CHAR_ALPHA 0x0001
#define CHAR_ALNUM 0x0002
#define CHAR_SPECL 0x0004
#define CHAR_DIGIT 0x0008
#define CHAR_C     0x0010
#define CHAR_X	   0x0020
#define CHAR_Q	   0x0040

#define CHR_ENT(a,b,c,d,e,f,g,h,i)				\
	(a?CHAR_ALPHA:0)|(b?CHAR_ALNUM:0)|(c?CHAR_SPECL:0)|	\
	(d?CHAR_DIGIT:0)|(e?CHAR_C:0)|(f?CHAR_X:0)|(g?CHAR_Q:0)

static int char_array[] = {
		/*        /----------------- CHAR_ALPHA
			  | /--------------- CHAR_ALNUM
			  | | /------------- CHAR_SPECL
			  | | | /----------- CHAR_DIGIT
			  | | | | /--------- CHAR_C
			  | | | | | /------- CHAR_X
			  | | | | | | /----- CHAR_Q
			  | | | | | | | V V  Unused	*/
/*   0: `^@'   */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/*   1: `^A'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*   2: `^B'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*   3: `^C'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*   4: `^D'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*   5: `^E'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*   6: `^F'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*   7: `^G'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*   8: `^H'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*   9: `^I'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  10: `^J'   */ CHR_ENT(0,0,1,0,0,1,0,0,0),
/*  11: `^K'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  12: `^L'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  13: `^M'   */ CHR_ENT(0,0,1,0,0,1,0,0,0),
/*  14: `^N'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  15: `^O'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  16: `^P'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  17: `^Q'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  18: `^R'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  19: `^S'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  20: `^T'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  21: `^U'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  22: `^V'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  23: `^W'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  24: `^X'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  25: `^Y'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  26: `^Z'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  27: `^['   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  28: `^\'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  29: `^]'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  30: `^^'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  31: `^_'   */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  32: ` '    */ CHR_ENT(0,0,0,0,0,1,1,0,0),
/*  33: `!'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  34: `"'    */ CHR_ENT(0,0,1,0,0,1,0,0,0),
/*  35: `#'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  36: `$'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  37: `%'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  38: `&'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  39: `''    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  40: `('    */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  41: `)'    */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  42: `*'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  43: `+'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  44: `,'    */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  45: `-'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  46: `.'    */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  47: `/'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  48: `0'    */ CHR_ENT(0,1,0,1,1,1,1,0,0),
/*  49: `1'    */ CHR_ENT(0,1,0,1,1,1,1,0,0),
/*  50: `2'    */ CHR_ENT(0,1,0,1,1,1,1,0,0),
/*  51: `3'    */ CHR_ENT(0,1,0,1,1,1,1,0,0),
/*  52: `4'    */ CHR_ENT(0,1,0,1,1,1,1,0,0),
/*  53: `5'    */ CHR_ENT(0,1,0,1,1,1,1,0,0),
/*  54: `6'    */ CHR_ENT(0,1,0,1,1,1,1,0,0),
/*  55: `7'    */ CHR_ENT(0,1,0,1,1,1,1,0,0),
/*  56: `8'    */ CHR_ENT(0,1,0,1,1,1,1,0,0),
/*  57: `9'    */ CHR_ENT(0,1,0,1,1,1,1,0,0),
/*  58: `:'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  59: `;'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  60: `<'    */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  61: `='    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  62: `>'    */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  63: `?'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  64: `@'    */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  65: `A'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  66: `B'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  67: `C'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  68: `D'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  69: `E'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  70: `F'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  71: `G'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  72: `H'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  73: `I'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  74: `J'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  75: `K'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  76: `L'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  77: `M'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  78: `N'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  79: `O'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  80: `P'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  81: `Q'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  82: `R'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  83: `S'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  84: `T'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  85: `U'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  86: `V'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  87: `W'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  88: `X'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  89: `Y'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  90: `Z'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  91: `['    */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  92: `\'    */ CHR_ENT(0,0,1,0,0,1,0,0,0),
/*  93: `]'    */ CHR_ENT(0,0,1,0,0,1,1,0,0),
/*  94: `^'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  95: `_'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  96: ``'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/*  97: `a'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  98: `b'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/*  99: `c'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 100: `d'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 101: `e'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 102: `f'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 103: `g'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 104: `h'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 105: `i'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 106: `j'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 107: `k'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 108: `l'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 109: `m'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 110: `n'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 111: `o'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 112: `p'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 113: `q'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 114: `r'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 115: `s'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 116: `t'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 117: `u'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 118: `v'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 119: `w'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 120: `x'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 121: `y'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 122: `z'    */ CHR_ENT(1,1,0,0,1,1,1,0,0),
/* 123: `{'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/* 124: `|'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/* 125: `}'    */ CHR_ENT(0,0,0,0,1,1,1,0,0),
/* 126: `~'    */ CHR_ENT(0,0,0,0,0,1,1,0,0),
/* 127: `DEL'  */ CHR_ENT(0,0,0,0,0,1,1,0,0),
/* 128: `\200' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 129: `\201' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 130: `\202' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 131: `\203' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 132: `\204' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 133: `\205' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 134: `\206' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 135: `\207' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 136: `\210' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 137: `\211' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 138: `\212' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 139: `\213' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 140: `\214' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 141: `\215' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 142: `\216' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 143: `\217' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 144: `\220' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 145: `\221' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 146: `\222' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 147: `\223' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 148: `\224' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 149: `\225' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 150: `\226' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 151: `\227' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 152: `\230' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 153: `\231' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 154: `\232' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 155: `\233' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 156: `\234' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 157: `\235' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 158: `\236' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 159: `\237' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 160: `\240' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 161: `\241' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 162: `\242' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 163: `\243' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 164: `\244' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 165: `\245' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 166: `\246' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 167: `\247' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 168: `\250' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 169: `\251' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 170: `\252' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 171: `\253' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 172: `\254' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 173: `\255' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 174: `\256' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 175: `\257' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 176: `\260' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 177: `\261' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 178: `\262' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 179: `\263' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 180: `\264' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 181: `\265' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 182: `\266' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 183: `\267' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 184: `\270' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 185: `\271' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 186: `\272' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 187: `\273' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 188: `\274' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 189: `\275' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 190: `\276' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 191: `\277' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 192: `\300' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 193: `\301' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 194: `\302' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 195: `\303' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 196: `\304' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 197: `\305' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 198: `\306' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 199: `\307' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 200: `\310' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 201: `\311' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 202: `\312' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 203: `\313' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 204: `\314' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 205: `\315' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 206: `\316' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 207: `\317' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 208: `\320' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 209: `\321' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 210: `\322' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 211: `\323' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 212: `\324' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 213: `\325' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 214: `\326' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 215: `\327' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 216: `\330' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 217: `\331' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 218: `\332' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 219: `\333' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 220: `\334' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 221: `\335' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 222: `\336' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 223: `\337' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 224: `\340' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 225: `\341' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 226: `\342' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 227: `\343' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 228: `\344' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 229: `\345' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 230: `\346' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 231: `\347' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 232: `\350' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 233: `\351' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 234: `\352' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 235: `\353' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 236: `\354' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 237: `\355' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 238: `\356' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 239: `\357' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 240: `\360' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 241: `\361' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 242: `\362' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 243: `\363' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 244: `\364' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 245: `\365' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 246: `\366' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 247: `\367' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 248: `\370' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 249: `\371' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 250: `\372' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 251: `\373' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 252: `\374' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 253: `\375' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 254: `\376' */ CHR_ENT(0,0,0,0,0,0,0,0,0),
/* 255: `\377' */ CHR_ENT(0,0,0,0,0,0,0,0,0)
};

int is_821_alpha(chr) unsigned int chr;
{
  return (char_array[chr] & CHAR_ALPHA);
}

int is_821_alnum(chr) unsigned int chr;
{
  return (char_array[chr] & CHAR_ALNUM);
}

int is_821_specl(chr) unsigned int chr;
{
  return (char_array[chr] & CHAR_SPECL);
}

int is_821_digit(chr) unsigned int chr;
{
  return (char_array[chr] & CHAR_DIGIT);
}

int is_821_C(chr) unsigned int chr;
{
  return (char_array[chr] & CHAR_C);
}

int is_821_X(chr) unsigned int chr;
{
  return (char_array[chr] & CHAR_X);
}

int is_821_Q(chr) unsigned int chr;
{
  return (char_array[chr] & CHAR_Q);
}

/* ================================================================ */

static
char *rfc821_at_domain(s,strict)	/* "@" <domain> */
char *s;
int strict;
{
	char *p;

	if (!s || !*s) return s; /* Pathological termination */

	if (*s != '@') {
	  rfc821_error = "<at-domain> missing initial \"@\"";
	  rfc821_error_ptr = s;
	  return s;
	}

	p = rfc821_domain(s+1,strict);
	if (!p || p == s+1) {
	  rfc821_error_ptr = s+1;
	  rfc821_error = "missing domain entry";
	  return s;
	}
	return p;
	  
}

char *rfc821_adl(s,strict)	/* <at-domain> | <at-domain> "," <a-d-l> */
char *s;
int strict;
{
	char *p = s, *q;

	if (!s || !*s) return s; /* Pathological termination */
	while ((q = rfc821_at_domain(p,strict)) && (q > p)) {
	  /* Scanned over an "@"+<domain> */
	  p = q;
	  if (*p != ',') return p;
	  ++p;
	}
	return s;
}

static
char *rfc821_dotnum(s)		/* <snum> "." <snum> "." <snum> "." <snum> */
char *s;
{
	int i,val,cnt;
	char *p = s-1;

	for (i=0; i<4; ++i) {
	  val = cnt = 0;
	  ++p;
	  while (*p && is_821_digit(*p)) {
	    val *= 10;
	    val += (*p - '0');
	    ++cnt;
	    ++p;
	  }
	  if (val > 255 || cnt < 1 || cnt > 3) {
	    rfc821_error_ptr = p;
	    rfc821_error = "Bad syntax of <dotnum> value";
	    return s;
	  }
	  if (i < 3 && (!*p || *p != '.')) {
	    rfc821_error_ptr = p;
	    rfc821_error = "Bad syntax of <dotnum>, missing '.'";
	    return s;
	  }
	}
	return p;
}

static
char *rfc821_name(s,strict,allnump)
char *s;
int strict;
int *allnump;	/* Return a flag about all-numeric field.. */
{
	char c;
	char *p = s;
	char *bad_name = "RFC821 <name> bad syntax";
	int has_alpha = 0;

	if (!s || !*s) return s; /* Pathological termination */

	/* The first test should be  is_821_alpha(), as per RFC821
	   we should accept only A-Z,a-Z in the begining, but
	   3COM spoiled that... Grr...				*/

	if (!is_821_alnum(*p)) {
	  rfc821_error_ptr = p;
	  rfc821_error = bad_name;
	  return s; /* Don't advace, leave! */
	}
	has_alpha |= is_821_alpha(*p);
	c = *(p++);
	while (is_821_alnum(*p) || (*p == '-') || (*p == '_')) {
	  has_alpha |= is_821_alpha(*p);
	  c = *(p++);
	}

	*allnump &= !has_alpha;

	if (!is_821_alnum(c) || (strict && !has_alpha)) {
	  rfc821_error_ptr = p-1;
	  rfc821_error = bad_name;
	  return s; /* Bad <name>, don't advance */
	}
	return p;	/* Name ok, return advanced pointer */
}

char *rfc821_domain(s,strict)		/* "#" <number> | "[" <dotnum> "]" | <name> | <name> "." <domain> */
char *s;
{
	char *p = s, *q;
	int allnum;	/* If all fields are numeric, it isn't domain.. */

	if (!s || !*s) return s; /* Pathological termination */

	if (*p == '[') {
	  q = rfc821_dotnum(p+1,strict);
/* printf("    dotnum: p='%s', q='%s'\n",p,q); */
	  if (q == p) return s;
	  if (*q != ']') {
	    rfc821_error_ptr = q-1;
	    rfc821_error = "RFC821 <dotnum> element missing terminating \"]\"";
	    return s;
	  }
	  return q+1;
	}
	if (*p == '#') {
	  ++p;
	  while (*p >= '0' && *p <= '9') ++p;
	  if (p > s+1)	return p;
	  rfc821_error_ptr = s;
	  rfc821_error = "RFC821 Domain \"#numbers\" has inadequate count of numbers";
	  return s;		/* Failure, don't advance */
	}

	allnum = 1; /* Collect info about all fields being numeric..
		       To accept  "1302.watstar.waterloo.edu" et.al.
		       but not something which looks like all numbers.. */
	q = rfc821_name(p,strict,&allnum);

	while (p && q > p && *q == '.') {
	  p = q+1;
	  q  = rfc821_name(p,strict,&allnum);
	}
	if (allnum) {
	  rfc821_error = "Should this be of <dotnum> format ? ( [nn.nn.nn.nn] )";
	  return s;
	}
	/*rfc821_error = "bad syntax on domain";*/
	if (!q || p == q) return s;	/* Report whatever  <name>  reports */
	return q;	/* Ok */
}

static
char *rfc821_mailbox(s,strict)	/* <local-part> "@" <domain> */
char *s;		/* Report error */
{
	char *p = s, *q;

	if (!s || !*s) {
	  /* rfc821_error_ptr = s;
	     rfc821_error = no_input; */
	  return s; /* Pathological termination */
	}
	p = rfc821_localpart(p,strict);
	if (p == s) {
	  /*rfc821_error_ptr = s;*/
	  /*rfc821_error = "No mailbox definition";*/
	  return s;
	}
	if (!strict) {
	  if (*p == 0 || *p == '>') /* If it terminates here, it is
				       only the <local-part> */
	    return p;
	}

	if (*p != '@') {
	  rfc821_error_ptr = p;
	  rfc821_error = "Missing \"@\" from mailbox definition";
	  return s;
	}
	++p;
	q = rfc821_domain(p,strict);
	if (q == p) {
	  rfc821_error_ptr = s;
	  rfc821_error = "Missing domain from mailbox definition";
	  return s;
	}
	return q;
}

static
char *rfc821_char(s)
char *s;
{
	if (!s || !*s) return s;
	if (*s == '\\') {
	  if (!is_821_X(*(s+1))) return s;
	  return s+2;
	}
	if (is_821_C(*s)) return s+1;
	return s;
}

static
char *rfc821_string(s,strict)
char *s;
{
	char *p = s, *q;

	if (!s || !*s) return s;
	while ((q = rfc821_char(p)) && (q > p)) {
	  p = q;
	}
	if ((unsigned char)*q > 127) {
	  rfc821_error_ptr = q;
	  rfc821_error     = "Improper 8-bit character in string";
	  return s;
	}
	if (q == s) {
	  rfc821_error_ptr = s;
	  rfc821_error     = "Expected an rfc821_string";
	}
	return q;		/* Advanced or not.. */
}

static
char *rfc821_dot_string(s,strict)
char *s;
{
	char *p = s, *q;

	if (!s || !*s) return s; /* Pathological termination */

	q = rfc821_string(p,strict);
	if (q == p) return s;	/* Missing string */
	while (*q == '.') {	/* Well, intermediate dot.. */
	  p = q+1;
	  q = rfc821_string(p,strict);
	  if (q == p) return s;	/* Missing string */
	}
	return q;
}

static
char *rfc821_qtext(s,strict)
char *s;
{
	char *p = s;
	int fail = 0;

	if (!s || !*s) return s;

	while (*p && !fail) {
	  if (*p == '\\') {
	    ++p;
	    if (is_821_X(*p)) ++p;
	    else	  fail = 1;
	  } else if (is_821_Q(*p))
	    ++p;
	  else
	    break;
	}
	if ((unsigned char)*p > 127) {
	  rfc821_error_ptr = p;
	  rfc821_error     = "Improper 8-bit character in qtext";
	  return s;
	}
	if (fail || p == s) {
	  rfc821_error_ptr = p;
	  rfc821_error     = "RFC821 qtext data failure";
	  return s;
	}
	return p;
}

static
char *rfc821_quoted_string(s,strict)
char *s;
{
	char *p = s, *q;

	if (!s || !*s) return s; /* Pathological termination */
	if (*p != '"') {
	  rfc821_error_ptr = p;
	  rfc821_error     = "Quoted string w/o initial quote (\")";
	  return s;
	}
	++p;
	q = rfc821_qtext(p,strict);
	if (p == q) {
	  /* rfc821_error_ptr = q;
	     rfc821_error     = "Quoted string of 0 length :-("; */
	  return s;
	}
	if (*q != '"') {
	  rfc821_error_ptr = q;
	  rfc821_error     = "Quoted string w/o final quote (\")";
	  return s;
	}
	return q+1;
}

static
char *rfc821_localpart(s,strict)/* <dot-string> | <quoted-string> */
char *s;			/* Stretched RFC821 a bit here.. !%-hacks */
{
	char *p = s, *q;

	if (!s || !*s) return s; /* Pathological termination */
	while (*p) {
	  if (*p == '"') {
	    q = rfc821_quoted_string(p,strict);
	    if (q == p) return s;	/* Uh... */
	    if (*q == '%' || *q == '!') {
	      p = q+1;
	      continue;
	    }
	    return q;
	  }
	  q = rfc821_dot_string(p,strict);
	  if (q == p) return s;	/* Uh... */
	  if (*q == '%' || *q == '!') {
	    p = q+1;
	    continue;
	  }
	  return q;
	}
	return p; /* Run to end of string.. */
}

char *rfc821_path2(s,strict)	/*  [ <a-d-l> ":" ] <mailbox>  */
char *s;
{
	char *p = s, *q;

	if (!s || !*s) {
	  rfc821_error_ptr = s;
	  rfc821_error = no_input;
	  return s; /* Pathological termination */
	}
	if (*p == '@' && (q = rfc821_adl(p,strict)) && (q > p)) {
	  p = q;
	  if (*p != ':') {
	    rfc821_error_ptr = p;
	    rfc821_error = "Missing colon (:) from <@xxx:yyy@zzz>";
	    return s;
	  }
	  ++p;
	}
	q = rfc821_mailbox(p,strict);
	if (q == p) {
	  /* Report whatever mailbox() reports as an error */
	  return s;
	}
	return q;
}


char *rfc821_path(s,strict)	/*  "<" [ <a-d-l> ":" ] <mailbox> ">"  */
char *s;
{
	char *p = s, *q;

	if (!s || !*s) {
	  rfc821_error_ptr = s;
	  rfc821_error = no_input;
	  return s; /* Pathological termination */
	}
	if (*p != '<') {
	  rfc821_error = "Missing \"<\" from begining";
	  rfc821_error_ptr = p;
	  return s;
	}
	++p;
	if (!*p) {
	  rfc821_error = premature_end;
	  rfc821_error_ptr = p;
	  return s;
	}
	if (*p == '>') {
	  return p+1; /* Termination ok */
	}
	q = rfc821_path2(p,strict);
	if (q == p) {
	  /* Report whatever part2() reports as an error */
	  return s;
	}
	if (*q != '>') {
	  rfc821_error = "Missing \">\" at end";
	  if (*q)
	    rfc821_error = "Extra garbage before terminating \">\"";
	  rfc821_error_ptr = q;
	  return s;
	}
	return q+1;
}
