/*
 *	Aviation navigation program
 *	getairways.c [1.2] from /preflight/src/nav/SCCS/src/s.getairways.c
 *		Retrieved 16:26:59 88/07/07; latest mod 16:03:08 88/07/07
 *	Alan M. Marcum		marcum@nescorna.Sun.COM
 *	Robert J. Evans		tolerant!procase!rje
 */

#ifndef	LINT
static char *SCCSid = "@(#)getairways.c\tRevision 1.2\t88/07/07";
#endif

/***************************************************************************
 *
 * Navigation Flight Plan Program
 *
 * This program is public domain software.  No claim is made for the
 * accurracy of either the program or the databases used by the program.
 * Always check the flight plan provided by the program for reasonable
 * results.  The pilot in command of an aircraft is directly responsible
 * for, and is the final authority as to, the operation of that aircraft.
 * (FAR 91.3)  Each pilot in command shall, before beginning a flight,
 * familiarize himself with all available information concerning that
 * flight. (FAR 91.5)
 *
 ***************************************************************************/

#include <stdio.h>
#include <ctype.h>
#ifdef	SysV
#include	<string.h>
#else	!SysV
#include	<strings.h>
#endif	SysV
#include <Nav.h>
#include <NavGlobals.h>

void	uppit();

/*
 * Interpret Victor Airways and Jet Routes in VIA list.  Modifies altitude,
 * wind, and speed arrays and counts to compensate for additional VORs
 * inserted during airways interpretation.  Also modifies the VOR names in
 * the VIA list, surrounding those which are on the airway segment with
 * AIRWAY_SEPARATOR, and following the second AIRWAY_SEPARATOR with the
 * airway name.  Returns whether all airways were found or not.
 *
 * 07Oct86 amm
 */
int
GetRoutes(ViaArr, PVia_C, AltArr, PAlt_C, WindArr, PWind_C, SpeedArr, PSpeed_C,
	  FuelArr, PFuel_C, fp)
char	ViaArr[MAXVIA][VIALEN];
int	AltArr[],
        WindArr[],
	SpeedArr[];
float	FuelArr[];
int	*PVia_C,
	*PAlt_C,
	*PWind_C,
	*PSpeed_C,
	*PFuel_C;
FILE	*fp[MAX_DBs];

{ /* GetRoutes() */
int	all_found = 1;	/* Assume eventual success */
int	i,
        j;
char	*NViaArr[MAXVIA],		/* New VIA array */
	*route[MAXVIA], 		/* Airway fixes */
	airway[VIALEN];			/* Name of the current airway */
int	NAltArr[MAXVIA],		/* New altitude array */
        NWindArr[MAXVIA*2],		/* New wind array (2 els/component) */
	NSpeedArr[MAXVIA];		/* New speed array */
float	NFuelArr[MAXVIA];		/* New fuel array */
int	Via_C = 0,			/* How many VIAs in revised list? */
	Alt_C = 0,			/* ...and altitudes */
	Wind_C = 0,			/* ...and wind specs */
	Speed_C = 0,			/* ...and speeds */
	Fuel_C = 0,			/* ...and fuel flows */
	routeC,				/* Number of fixes on the airway */
	via_skip;			/* Essentially, how many airways
					   we've found.  This is to skip the
					   exit point in ViaArr, without
					   skipping things in the Speed,
					   Wind, or Altitude arrays */

for (i = 0; i < MAXVIA; i++) {	/* Initialize all these */
	if (NULL == (NViaArr[i] = malloc(VIALEN)) ||
	    NULL == (route[i] = malloc(IDLEN))) {
	        (void)perror("nav: out of space");
		(void)exit(1);
	}
	NViaArr[i][0] = '\0';
	route[i][0] = '\0';
	NAltArr[i] = 0;
	NWindArr[i*2] = 0;
	NWindArr[(i*2)+1] = 0;
	NSpeedArr[i] = 0;
	NFuelArr[i] = 0;
}

/* 
 * Go through each element in the VIA array (ViaArr), checking whether it's
 * an airway or not.  (Note the assumption that if an element begins with a v
 * or a j [upper or lower case] and the second character is a number, it's an
 * airway designator).  If it's not an airway, copy the fix and its
 * associated information (speed, etc.) into the new arrays we're building.
 * If it IS an airway, determine the fixes on the airway (the elements
 * surrounding the airway designator indicate the entry and exit points of
 * the airway), copy the airway fixes to NViaArr, and copy the associated
 * information, also.  Note that the entry fix is explicitly copied, and is
 * not included in what's returned by the airway finding routine; the exit
 * fix IS returned by that airway finding routine, and it is skipped in
 * ViaArr, though the associated data must be handled properly.
 */

for (i = 0, via_skip = 0; i+via_skip < *PVia_C && all_found; i++) {
	/* Ensure initial letter is upper case to simplify V and J comparison*/
	uppit(ViaArr[i + via_skip]);
        if ((ViaArr[i+via_skip][0] == 'V' || ViaArr[i+via_skip][0] == 'J') &&
	    strlen(ViaArr[i+via_skip]) > 1 && isdigit(ViaArr[i+via_skip][1])) {
				/* An airway!*/
		(void)strcpy(airway, ViaArr[i + via_skip]);
		switch (routeC = FindAirway(ViaArr, i+via_skip, route, fp)) {
		case FindAirway_Error:
		        (void)fprintf(stderr, "Error in finding airway %s.\n",
			       ViaArr[i+via_skip]);
			all_found = FALSE;
			break;
		case FindAirway_NotFound:
			(void)fprintf(stderr, "Airway %s not found.\n",
			        ViaArr[i+via_skip]);
			all_found = FALSE;
			break;
		case FindAirway_NoEntry:
		        (void)fprintf(stderr,
			        "Entry %s to airway %s not found.\n",
			        ViaArr[(i-1)+via_skip], ViaArr[i+via_skip]);
			all_found = FALSE;
			break;
		case FindAirway_NoExit:
			(void)fprintf(stderr,
			        "Exit %s to airway %s not found.\n",
			        ViaArr[i+1+via_skip], ViaArr[i+via_skip]);
			all_found = FALSE;
			break;
		default:	/* We found it, and the entry&exit points */
			if (routeC + Via_C > MAXVIA) {	/* Already full? */
				(void)fprintf(stderr,
				    "Too many VIAs (including airway %s).\n",
			            ViaArr[i+via_skip]);
				all_found = FALSE;
			}

			for(j = 0; j < routeC; j++, Via_C++) { /* Copy fixes */
				NViaArr[Via_C][0] = AIRWAY_SEPARATOR;
				(void)strncpy(NViaArr[Via_C]+1,
				        route[j], IDLEN);
				(void)strcat(NViaArr[Via_C],
				        AIRWAY_SEPARATOR_STG);
				(void)strcat(NViaArr[Via_C], airway);
			}
			if (*PAlt_C > 1)		/* Duplicate altitude */
				Alt_C += extend_int(AltArr, i, NAltArr,
						   Alt_C, 1, routeC);
			if (*PWind_C > 1)	/* and wind */
				Wind_C += extend_int(WindArr, i, NWindArr,
						    Wind_C, 2, routeC);
			if (*PSpeed_C > 1)	/* and speed */
				Speed_C += extend_int(SpeedArr, i, NSpeedArr,
						     Speed_C, 1, routeC);
			if (*PFuel_C > 1)	/* and fuel */
				Fuel_C += extend_flt(FuelArr, i, NFuelArr,
						     Fuel_C, 1, routeC);
			via_skip++;	/* FindAirway() includes the exit
					   fix, so skip it in the VIA array */
			break;
		}
	} else {	/* Not an airway; simply copy stuff old to new */
		(void)strncpy(NViaArr[Via_C++], ViaArr[i+via_skip], VIALEN);
		/* Note assumption that AltArr, etc., are fully allocated */
		NAltArr[Alt_C++] = AltArr[i];
		NWindArr[Wind_C++] = WindArr[2*i];	/* Need 2: two ... */
		NWindArr[Wind_C++] = WindArr[(2*i)+1];	/* ...pieces to wind */
		NSpeedArr[Speed_C++] = SpeedArr[i];
		NFuelArr[Fuel_C++] = FuelArr[i];
	}
}

/* Copy remainders of each array to the new one. */
for (j = i+via_skip; j < *PVia_C; j++) NViaArr[Via_C++] = ViaArr[j];
Alt_C += extend_int(AltArr, i, NAltArr, Alt_C, 1, (*PAlt_C)-i);
Wind_C += extend_int(WindArr, i, NWindArr, Wind_C, 2, (*PWind_C)-(2*i));
Speed_C += extend_int(SpeedArr, i, NSpeedArr, Speed_C, 1, (*PSpeed_C)-i);
Fuel_C += extend_flt(FuelArr, i, NFuelArr, Fuel_C, 1, (*PFuel_C)-i);

/* Set the output parameters. */
*PVia_C = copy_arr_c(ViaArr, NViaArr, Via_C);
*PAlt_C = copy_arr_i(AltArr, NAltArr, Alt_C);
*PWind_C = copy_arr_i(WindArr, NWindArr, Wind_C);
*PSpeed_C = copy_arr_i(SpeedArr, NSpeedArr, Speed_C);
*PFuel_C = copy_arr_f(FuelArr, NFuelArr, Fuel_C);

return(all_found);
} /* GetRoutes() */

/*
 * FindAirway translates Victor Airways and Jet Routes to a series of
 * VORs (and intersections).  ViaArr[i] specifies the name of the
 * airway, ViaArr[i-1] is the point of entry to the airway, and ViaArr[i+1]
 * is the exit point.  The result is placed in Route; the airways database
 * is found using fp.  FindAirway returns the number of VORs on the
 * airway between entry and exit , or the following error codes:
 *	0	FindAirway_Error	Non-specific error
 *	-1	FindAirway_NotFound	Airway not found in airways file
 *	-2	FindAirway_NoEntry	Entry fix not on designated airway
 *	-3	FindAirway_NoExit	Exit fix not on designated airway
 *
 * Note that as of 07Oct86, the search algorithm is inefficient: linear
 * search through the entire airways database for each airway...amm.
 *
 * 07Oct86 amm
 */
int
FindAirway(ViaArr, i, Route, fp)
char	ViaArr[MAXVIA][VIALEN],
	*Route[MAXVOR];
int	i;
FILE	*fp[MAX_DBs];

{ /* FindAirway() */
int	found = FALSE;		/* Assume eventual success */
int	j,
	k,
	l,
	airway_entry,		/* The index of the airway entry point */
	airway_exit,		/* and that of the exit point */
	route_vorsC;		/* How many fixes on the airway? */
char	route_vors[MAXVOR][IDLEN],	/* The IDs of the fixes */
        buf[BUFSIZ],
	tmp[BUFSIZ],
	*route_id();

#ifdef	Apollo
/* Ensure things properly initialized, to preclude any nasties later */
for (j = 0; j < BUFSIZ; j++) {
	buf[j] = '\0';
	tmp[j] = '\0';
}
#endif	Apollo

for (j = 0; j < MAXVOR; j++)
	route_vors[j][0] = '\0';

for (l = 0; l < MAX_DBs && !found; l++)	 /* Go through all available files */
   if (fp[l] != NULL) {
      rewind(fp[l]);		/* Let's start at the beginning... */
      while (fgets(buf, BUFSIZ, fp[l]) != NULL) { /* Go thru entire file */
	if (buf[0] == '#') continue;	/* comment line */
	(void)strncpy(tmp, buf, BUFSIZ);
	if (0 == strcmp(route_id(tmp), ViaArr[i])) {	/* Found it! */
		found = TRUE;
		break;
#ifdef	Apollo
	/* Apollo seems to require clearing buf on each loop. */
	} else {
		for (j = 0; j < BUFSIZ; j++) {
			buf[j] = '\0';
		} 
#else	!Apollo
	}
#endif	Apollo
      }
      if (found) {	/* Get vors copied to Route */
	/* Get all VORs into array */
	if (0 == (route_vorsC = ParseAirway(buf, route_vors, MAXVOR)))
		return(FindAirway_Error);
	if (route_vorsC > MAXVOR) {	/* Serious internal error */
		(void)fprintf(stderr,
			"Internal error: route_vorsC [%d] > MAXVOR[%d].\n",
			route_vorsC, MAXVOR);
		(void)exit(1);
	}
	if (route_vorsC == MAXVOR)	/* Warn them of possible errors */
		(void)fprintf(stderr,
			"Possible lost segments on %s; max %d segments.\n",
			ViaArr[i], MAXVOR);
	/* Find airway entry point */
	if (-1 == (airway_entry =
			ArrInx(route_vors, ViaArr[i-1], route_vorsC)))
		return(FindAirway_NoEntry);
	/* ...and exit point */
	if (-1 == (airway_exit = ArrInx(route_vors, ViaArr[i+1], route_vorsC)))
		return(FindAirway_NoExit);
	/* Copy VORs from entry through exit */
	if (airway_entry < airway_exit) {	/* Forward copy */
		for (j = 0, k = airway_entry; k < airway_exit; j++, k++) {
			(void)strncpy(Route[j], route_vors[k], IDLEN);
		}
	} else {		/* Reverse copy */
		for (j = 0,  k = airway_entry-1; k >= airway_exit; j++,  k--) {
		        (void)strncpy(Route[j], route_vors[k-1], IDLEN);
		}
	}
      }
   }
if (found)
	return(abs(airway_entry-airway_exit));
else	return(FindAirway_NotFound);
} /* FindAirway() */


/*
 * ParseAirway turns a string [buf] with spaces separating words (VORs)
 * into an array [arr] of those words (VORs).  It returns the number of
 * words it found,  and 0 if errors.  lim is the maximum number of VORs
 * which will fit in arr.
 *
 * Note we assume that the first token is the airway name, to be discarded.
 */
int
ParseAirway(buf, arr, lim)
char	*buf,
	arr[MAXVOR][IDLEN];
int	lim;
{ /* ParseAirway() */
int	i;
char	*tmp;

tmp = strtok(buf, AWY_DELIMS);	/* Initialize token search */
for (i = 0; i < lim; i++)
	if (NULL != (tmp = strtok((char *)NULL, AWY_DELIMS)))
		(void)strncpy(arr[i], tmp, IDLEN);
	else	break;

return(i);
} /* ParseAirway() */

/*
 * Returns the first token (airway name) in the string buf.
 */
char *
route_id(buf)
char	buf[BUFSIZ];
{ /* route_id() */
char	*id;
switch ((id = strtok(buf, AWY_DELIMS))[0]) {
case 'v':
        id[0] = 'V';
        break;
case 'j':
        id[0] = 'J';
        break;
default:
        break;
}

return(id);
} /* route_id() */


/*
 * Find a character string (str) in an array of strings (arr).  There
 * are lim elements in arr.
 */
int
ArrInx(arr, str, lim)
char	arr[MAXVOR][IDLEN],
	str[IDLEN];
int	lim;
{ /* ArrInx() */
	uppit(str);
        for (;lim > 0 ;lim--) {
		uppit(arr[lim-1]);
	        if (0 == strncmp(arr[lim-1], str, IDLEN)) return(lim);
	}
return(-1);
} /* ArrInx() */



/*
 * extend_int copies the INXth entry in FROM N times to TO; ELS_PER
 * specifies how many physical elements per logical element.  It returns N.
 */
int
extend_int(from, finx, to, tinx, els_per, count)
int	from[],
	to[];
int	finx,
	tinx,
	els_per,
	count;
{ /* extend_int() */
int	from_i,
        to_i,
	iter;

for (iter = 0, to_i = 0; iter < count; iter++)
        for (from_i = 0; to_i < els_per; from_i++, to_i++)
	        to[tinx+to_i] = from[(els_per*finx)+from_i];
return(count*els_per);
} /* extend_int() */


/*
 * extend_flt copies the INXth entry in FROM N times to TO; ELS_PER
 * specifies how many physical elements per logical element.  It returns N.
 */
int
extend_flt(from, finx, to, tinx, els_per, count)
float	from[],
	to[];
int	finx,
	tinx,
	els_per,
	count;
{ /* extend_flt() */
int	from_i,
        to_i,
	iter;

for (iter = 0, to_i = 0; iter < count; iter++)
        for (from_i = 0; to_i < els_per; from_i++, to_i++)
	        to[tinx+to_i] = from[(els_per*finx)+from_i];
return(count*els_per);
} /* extend_flt() */


/* Copy the contents of one string array into another */
int
copy_arr_c(target, source, limit)
char	target[MAXVIA][VIALEN];
char	*source[MAXVIA];
int	limit;
{ /* copy_arr_c() */
	int i;
	for (i = 0; i < limit; i++) {
		(void)strncpy(target[i], source[i], VIALEN);
	}
	return(limit);
} /* copy_arr_c() */


/* Copy the contents of one integer array into another. */
int
copy_arr_i(target, source, limit)
int	target[],
	source[],
	limit;
{ /* copy_arr_i() */
	int i;
        for (i = 0; i < limit; i++) target[i] = source[i];
	return(limit);
} /* copy_arr_i() */

/* Copy the contents of one float array into another. */
int
copy_arr_f(target, source, limit)
float	target[],
	source[];
int	limit;
{ /* copy_arr_f() */
	int i;
        for (i = 0; i < limit; i++) target[i] = source[i];
	return(limit);
} /* copy_arr_f() */
