/*
 *	Aviation navigation program
 *	getnavaids.c [1.2] from /preflight/src/nav/SCCS/src/s.getnavaids.c
 *		Retrieved 16:26:58 88/07/07; latest mod 16:03:02 88/07/07
 *	Alan M. Marcum		marcum@nescorna.Sun.COM
 *	Robert J. Evans		tolerant!procase!rje
 */

#ifndef	LINT
static char *SCCSid = "@(#)getnavaids.c\tRevision 1.2\t88/07/07";
#endif

/***************************************************************************
 *
 * Navigation Flight Plan Program
 *
 * This program is public domain software.  No claim is made for the
 * accurracy of either the program or the databases used by the program.
 * Always check the flight plan provided by the program for reasonable
 * results.  The pilot in command of an aircraft is directly responsible
 * for, and is the final authority as to, the operation of that aircraft.
 * (FAR 91.3)  Each pilot in command shall, before beginning a flight,
 * familiarize himself with all available information concerning that
 * flight. (FAR 91.5)
 *
 ***************************************************************************/

#include <stdio.h>
#include <ctype.h>
#ifdef	SysV
#include	<string.h>
#else	!SysV
#include	<strings.h>
#endif	SysV
#include <Nav.h>
#include <NavGlobals.h>

#define max(x, y)	((x) > (y) ? (x) : (y))

#define IS_TO	0
#define IS_FROM	1

struct vor_status {
	int	is_rnavable;
	int	on_airway;
	double	rad,
		dme;
	char	vor_name[IDLEN];
	char	awy_name[AWYLEN];
};

double	var_v;
int	max_t = 0,
	max_f = 0;

char	*binary_search();
double	atof();
void	uppit();
struct vor	*LineToVor();

static	void	NearestVORs(),
		IsNearest();
static	int	FindNavAid(),
		ParseVORs(),
		Setup_ROV(),
		Setup_AWY(),
#ifdef	SPECIFY_NAVAID_TYPE
		Setup_NAV(),
#endif	SPECIFY_NAVAID_TYPE
		Setup_REG(),
		not_yet_found();

/**************************************************************************
 * GetVors scans the vors database file looking for id fields that match
 *         those passed in in the ViaIDs array. Matching records are
 *         passed back in via_array.
 *  Returns: number of records found.
 *  Parameters: via_array - set of vor records returned.
 *              via_c - number of IDs in ViaIDs and number of vor records
 *			to be returned.
 *              ViaIDs - list of vor IDs to be found.
 *              scratch_vor - scratch vor record to pass to LineToVor.
 *              fp - file pointers to vor database files.
 *              From_Airport - origin airport record.
 *              To_Airport - destination airport record
 *              nearest_vors - vor records for two vors nearest to airports
 *              distance_from, distance_to - distance from vor to airport
 *              radial_from, radial_to - radial from vor to airport
 *  Side Effects: stores vor records into via_array, scratch_vor;
 *                puts NULL into ViaIDs[n][0] if a match is found;
 *                stores distance and radial information into dist_x,rad_x;
 *                stores vor records into nearest_vors.
 **************************************************************************/
int
GetVors(via_array, via_c, ViaIDs, scratch_vor, fp,
        From_Airport, To_Airport, from_flg, to_flg, find_nearest,
        nearest_vors)
struct vor	via_array[],
		*scratch_vor,
		nearest_vors[];
struct apt	*From_Airport,
		*To_Airport;
char	ViaIDs[MAXVIA][VIALEN];
int	from_flg,
	to_flg,
	via_c,
	find_nearest;
FILE	*fp[MAX_DBs];

{ /* GetVors() */
	int	found,
		i,
		j;
	extern int	nearestall;
	struct vor_status status[MAXVIA];

	found = 0;

	if (!ParseVORs(ViaIDs, status, via_c)) {
		return(0);
	}

	for (i = 0; i < MAX_DBs && found < via_c; i++) {
		if (fp[i] != NULL) {
			for (j = 0; j < via_c; j++) { /* Find each navaid */
				found += FindNavAid(via_array, fp[i], i,
						    status[j], ViaIDs[j], j);
			}
		}
	}
	if (find_nearest) {
		NearestVORs(From_Airport, To_Airport,
				fp, scratch_vor, nearest_vors,
				from_flg, to_flg);
	}
	return(found);
} /* GetVors() */


/* Parse all the navaids in ViaIDs, filling in the status array.
 * Returns TRUE if all worked OK, FALSE if an error occurred.
 */
static int
ParseVORs(ViaIDs, status, via_c)
char	ViaIDs[MAXVIA][VIALEN];
struct	vor_status status[];
int	via_c;

{ /* ParseVORs() */

	int	cnt,		/* Loop counter */
		error_flag = FALSE;	/* Has an error occurred? */

	for (cnt = 0; cnt < via_c; cnt++) {
		uppit(ViaIDs[cnt]);
		switch (ViaIDs[cnt][0]) {
		case RNAV_OVER_VORTAC:	/* RNAV, w/waypoint forced at VORTAC */
			error_flag = Setup_ROV(ViaIDs, status, cnt);
			break;
		case AIRWAY_SEPARATOR:	/* This waypoint is on an airway */
			error_flag = Setup_AWY(ViaIDs, status, cnt);
			break;
#ifdef	SPECIFY_NAVAID_TYPE
		case NAVAID_DESIGNATOR:	/* Command line specifies which
					 * navaid type to use */
			error_flag = Setup_NAV(ViaIDs, status, cnt);
			break;
#endif	SPECIFY_NAVAID_TYPE
		default:
			error_flag = Setup_REG(ViaIDs, status, cnt);
			break;
		}
	}
	return(!error_flag);
} /* ParseVORs() */

static int
Setup_ROV(ViaIDs, status, cnt)
char	ViaIDs[MAXVIA][VIALEN];
struct	vor_status  status[];
int	cnt;
{ /* Setup_ROV() */
	int j;

	status[cnt].is_rnavable = FALSE;
	status[cnt].rad = status[cnt].dme = 0.0; /* 0 out */
	status[cnt].on_airway = FALSE;

	for (j = 0; j < IDLEN; j++) {
		status[cnt].vor_name[j] = ViaIDs[cnt][j+1];
	}

	status[cnt].vor_name[j] = '\0';	/* Termination */
	status[cnt].awy_name[0] = '\0';
	return(FALSE);
} /* Setup_ROV() */


static int
Setup_AWY(ViaIDs, status, cnt)
char	ViaIDs[MAXVIA][VIALEN];
struct	vor_status  status[];
int	cnt;
{ /* Setup_AWY() */

	int	j;	/* Temp for loop count */

	status[cnt].is_rnavable = TRUE;
	status[cnt].on_airway = TRUE;
	status[cnt].rad = status[cnt].dme = 0.0; /* 0 out */

	for (j = 1;
	     j < IDLEN && ViaIDs[cnt][j] != AIRWAY_SEPARATOR;
	     j++) {
		status[cnt].vor_name[j-1] = ViaIDs[cnt][j];
	}

	status[cnt].vor_name[j-1] = '\0';
	(void)strncpy(status[cnt].awy_name,
		      ViaIDs[cnt]+j+1, AWYLEN);
	status[cnt].awy_name[AWYLEN] = '\0';
	return(FALSE);
} /* Setup_AWY() */

#ifdef	SPECIFY_NAVAID_TYPE
static int
Setup_NAV(ViaIDs, status, cnt)
char	ViaIDs[MAXVIA][VIALEN];
struct	vor_status  status[];
int	cnt;
{ /* Setup_NAV() */

	int error_flag = FALSE;

	switch (ViaIDs[cnt][1]) {
	case NAVAID_VOR:
		break;
	case NAVAID_VORTAC:
		break;
	case NAVAID_TACAN:
		break;
	case NAVAID_NDB:
		break;
	case NAVAID_INTERSECTION:
		break;
	case NAVAID_AIRPORT:
		status[cnt].is_rnavable = FALSE;
		status[cnt].on_airway = FALSE;
		status[cnt].rad = status[cnt].dme = 0.0;
		status[cnt].awy_name[0] = '\0';
		status[cnt].type = NAVAID_AIRPORT;
		(void)strncpy(status[cnt].vor_name,
			      ViaIDs[cnt]+2);
		break;
	default:
		(void)fprintf(stderr,
			      "Unknown navaid designator %c in %s.\n",
			      ViaIDs[cnt][1], ViaIDs[cnt]);
		error_flag = TRUE;
		break;
	}
	return(error_flag);
} /* Setup_NAV() */
#endif	SPECIFY_NAVAID_TYPE

static int
Setup_REG(ViaIDs, status, cnt)
char	ViaIDs[MAXVIA][VIALEN];
struct	vor_status  status[];
int	cnt;
{ /* Setup_REG() */

	int	error_flag = FALSE;

	char	*rad,		/* Radial in user wpt specification */
		*dme;		/* And distance */

	/*
	 * Check for, and process if present, RAD/DME specification.
	 * Format is VOR/RAD/DME (eg. SFO/123/10.3)
	 */
	if (NULL != (rad = index(ViaIDs[cnt], RADSEP))) {
		/*
		 * Delay terminating VOR string until after
		 * any necessary message printing
		 */
		rad++;
		if (NULL != (dme = index(rad, DMESEP))) {
			dme[0] = '\0';
			dme++;
		} else {
			(void)fprintf(stderr,
		    "Could not parse VOR%cRAD%cDME in %s.\n",
				      RADSEP, DMESEP,
				      ViaIDs[cnt]);
			error_flag = TRUE;
		}
		(rad - 1)[0] = '\0';	/* Safe to terminate */
		status[cnt].is_rnavable = FALSE;
		status[cnt].rad = atof(rad);
		/* Want 0 <= radial < 360 */
		while (360.0 <= status[cnt].rad) {
			status[cnt].rad -= 360.0;
		}
		while (0.0 > status[cnt].rad) {
			status[cnt].rad += 360.0;
		}
		status[cnt].dme = atof(dme);
		if (status[cnt].dme < 0) {
			(void)fprintf(stderr,
				"VOR%cRAD%cDME DME < 0 in %s.\n",
				RADSEP, DMESEP, dme);
			error_flag = TRUE;
		}
	} else {
		status[cnt].is_rnavable = TRUE;
		status[cnt].rad = status[cnt].dme = 0.0;
	}
	status[cnt].on_airway = FALSE;
	(void)strncpy(status[cnt].vor_name, ViaIDs[cnt], IDLEN);
	status[cnt].vor_name[IDLEN] = '\0'; /* String termination */
	status[cnt].awy_name[0] = '\0';
	return(error_flag);
} /* Setup_REG() */

static void
NearestVORs(From_Airport, To_Airport,
		fp, scratch_vor,
		nearest_vors, from_flg, to_flg)
struct apt	*From_Airport,
		*To_Airport;
int	from_flg,
	to_flg;
FILE	*fp[MAX_DBs];
struct	vor	*scratch_vor,
		nearest_vors[];
{ /* NearestVORs() */

	int	max_apt_file,	/* Biggest index of file descriptors where
				 * airports were found */
		i;		/* Ubiquitous loop counter */

	char	*line;

	double	lat_f,
		lon_f,
		lat_t,
		lon_t,
		lat_v,
		lon_v;

	if (from_flg) {
		lat_f = From_Airport->loc.lat.rad;
		lon_f = From_Airport->loc.lon.rad;
	}
	if (to_flg) {
		lat_t = To_Airport->loc.lat.rad;
		lon_t = To_Airport->loc.lon.rad;
	}

	dist_t[0] = 99999.9; dist_f[0] = 99999.9;
	dist_t[1] = 99999.9; dist_f[1] = 99999.9;

	max_apt_file = max(From_Airport->file_found,
			   To_Airport->file_found);

	for (i = 0; i < MAX_DBs; i++) {
		if (fp[i] != NULL) {
			rewind(fp[i]);
		}
	}
	if (NULL == (line = malloc(BUFSIZ))) {
		(void)perror("nav: GetVors could not allocate space");
		(void)exit(1);
	}

	/*
	 * In finding the nearst navaids, we only look through the
	 * databases as far as necessary, stopping the search as early as
	 * possible (to save time with very large public databases that
	 * aren't really referenced).  Yes, this may lead to inaccuracies;
	 * the user can specify the -nearestall option to forestall this
	 * behavior.
	 */
	for (i = 0; nearestall || i <= max_apt_file; i++) {
		while ((fgets(line, BUFSIZ, fp[i]) != NULL)) {
			if (line[0] == '#') {
				continue;		/* Comment line */
			}
			scratch_vor = LineToVor(line);
			if (scratch_vor == NULL) {
				(void)fprintf(stderr,
					      "LineToVor returned NULL:\n%s\n",
					      line);
				return;
			}
			lat_v = scratch_vor->loc.lat.rad;
			lon_v = scratch_vor->loc.lon.rad;
			var_v = scratch_vor->var;
			if (to_flg) {
				IsNearest(lat_v, lon_v, lat_t, lon_t,
					  scratch_vor, nearest_vors,
					  IS_TO);
			}
			if (from_flg) {
				IsNearest(lat_v, lon_v, lat_f, lon_f,
					  scratch_vor, nearest_vors,
					  IS_FROM);
			}
		}
	}
} /* NearestVORs() */


static int
FindNavAid(via_array, fp, file_inx, status, ViaID, wp_inx)
struct vor	via_array[];
struct vor_status status;
FILE	*fp;
char	ViaID[VIALEN];
int	file_inx,
	wp_inx;
{ /* FindNavAid() */

	struct	fix	*rad_dme2lat_lon();
	char	*line;
	int	found = 0;	/* Number found */
	struct	vor	*scratch_vor;

	if (via_array[wp_inx].vor_type == TYPE_UNDEFINED) { /* Not yet found */
		line = binary_search(fp, status.vor_name, RECLEN_VORS);
		if (NULL != line) {
			scratch_vor = LineToVor(line);
			if (scratch_vor == NULL) {
				(void)fprintf(stderr,
					     "LineToVor returned NULL:\n%s\n",
					     line);
				return(0);
			}
			found++;
			ViaID[0] = '\0';
			(void)free(line);
			via_array[wp_inx] = *scratch_vor;
			via_array[wp_inx].file_found = file_inx;
			via_array[wp_inx].rnavable &= status.is_rnavable;
			via_array[wp_inx].on_airway = status.on_airway;
			if (via_array[wp_inx].on_airway) {
				(void)strncpy(via_array[wp_inx].waypoint,
					      status.awy_name,
					      AWYLEN);
			}
			if (0.0 != status.dme) {
				via_array[wp_inx].vor_type = TYPE_USERWAYPOINT;
				via_array[wp_inx].loc = *rad_dme2lat_lon(
						via_array[wp_inx].loc.lat.deg,
						via_array[wp_inx].loc.lon.deg,
						status.rad,
						status.dme,
						via_array[wp_inx].var);
				(void)sprintf(via_array[wp_inx].waypoint,
					      "WP%-2.2d", wp_inx+1);
			}
		}
	} /* if not yet found */
	return(found);
} /* FindNavAid() */


static void
IsNearest(lat_v, lon_v, lat, lon, scratch_vor, nearest_vors, which)
double	lat_v,
	lon_v,
	lat,
	lon;
struct	vor	nearest_vors[],
		*scratch_vor;
int	which;
{ /* IsNearest() */

	int	orig,
		max_selector;
	double	distance,
		*dist,		/* Really dist[] */
		*rad;		/* Really rad[] */

	switch (which) {
	case IS_TO:
		max_selector = max_t;
		orig = TO_VORS_ORIG;
		rad = rad_t;
		dist = dist_t;
		break;
	case IS_FROM:
		max_selector = max_f;
		orig = FROM_VORS_ORIG;
		rad = rad_f;
		dist = dist_f;
		break;
	default:
		(void)fprintf(stderr, "%s: IsNearest() internal error.\n",
			      pgm_name);
		(void)fprintf(stderr, "Value of WHICH (%d) out of range.\n",
			      which);
		exit(1);
		break;
	}

	distance = Distance(lat_v, lon_v, lat, lon);
	if (distance < dist[max_selector] &&
	    TYPE_INTERSECTION != scratch_vor->vor_type &&
	    not_yet_found(nearest_vors, orig, scratch_vor)) {
		nearest_vors[(max_selector + orig)] = *scratch_vor;
		dist[max_selector] = distance;
		rad[max_selector] = Bearing(lat_v, lon_v,
					     lat, lon,
					     var_v);
		rad[max_selector] = Magnetic(rad[max_selector], var_v);
		switch (which) {
		case IS_TO:
			max_t = (dist[0] < dist[1] ? 1 : 0);
			break;
		case IS_FROM:
			max_f = (dist[0] < dist[1] ? 1 : 0);
			break;
		default:
			(void)fprintf(stderr,
				      "%s: IsNearest() internal error.\n",
				      pgm_name);
			(void)fprintf(stderr,
				      "Value of WHICH (%d) out of range.\n",
				      which);
		exit(1);
		break;
		}
	}
} /* IsNearest() */

static int
not_yet_found(nearest, which_one, scratch)
struct vor	nearest[],
		*scratch;
int	which_one;
{ /* not_yet_found() */

	return(0 != strncmp(nearest[which_one].id, scratch->id, IDLEN) &&
	       0 != strncmp(nearest[which_one+1].id, scratch->id, IDLEN));
} /* not_yet_found() */
