/*
 *	Aviation navigation program
 *	nav.c [2.11] from /preflight/src/nav/SCCS/src/s.nav.c
 *		Retrieved 16:26:42 88/07/07; latest mod 16:02:33 88/07/07
 *	Alan M. Marcum		marcum@nescorna.Sun.COm
 *	Robert J. Evans		tolerant!procase!rje
 */

#ifndef LINT
static char *SCCSid = "@(#)nav.c\tRevision 2.11\t88/07/07";
#endif	LINT
/* *********************************************************************** */
/*                                                                         */
/* Navigation Flight Plan Program                     NAV.C                */
/*                                                                         */
/* This program is public domain software.  No claim is made for the       */
/* accurracy of either the program or the databases used by the program.   */
/* Always check the flight plan provided by the program for reasonable     */
/* results.  The pilot in command of an aircraft is directly responsible   */
/* for, and is the final authority as to, the operation of that aircraft.  */
/* (FAR 91.3)  Each pilot in command shall, before beginning a flight,     */
/* familiarize himself with all available information concerning that      */
/* flight. (FAR 91.5)                                                      */
/*                                                                         */
/* *********************************************************************** */
#include <math.h>
#include <stdio.h>
#include <Nav.h>
#include <NavGlobals.h>

/* *********************************************************************** */
/* Radians converts an angle given in degrees into the equivalent in       */
/*         angle expressed in radians.                                     */
/* Returns: angle expressed in radians.                                    */
/* Parameters: degrees - angle expressed in degrees.                       */
/* Side Effects: none.                                                     */
/* *********************************************************************** */
double
Radians(degrees)
double	degrees;
{
        return((degrees/180.0) * pi);
} /* Radians */


/* *********************************************************************** */
/* Degrees converts an angle given in radians into the equivalent in       */
/*         angle expressed in degrees.                                     */
/* Returns: angle expressed in degrees.                                    */
/* Parameters: radians - angle expressed in radians.                       */
/* Side Effects: none.                                                     */
/* *********************************************************************** */
double
Degrees(radians)
double	radians;
{
        return(radians * (180 / pi));
} /* Degrees */

/* *********************************************************************** */
/* Distance computes the distance, in nautical miles, between two points   */
/*          given in terms of latitude and longitude. Latitudes are        */
/*          positive for North, negative for South. Longitudes are         */
/*          positive for West, negative for East.                          */
/*          The algorithm is from the HP-25 Application Programs, page 65. */
/* Returns: distance, in nautical miles.                                   */
/* Parameters: N1, W1 - latitude and longitude expressed in radians for    */
/*                      first point.                                       */
/*             N2, W2 - latitude and longitude expressed in radians for    */
/*                      second point.                                      */
/* Side Effects: none.                                                     */
/* *********************************************************************** */
double
Distance(N1, W1, N2, W2)
double	N1,
	W1,
	N2,
	W2;
{
        double	C,
		t1,
		t2;

	if (N1 == N2 && W1 == W2) {
		return(0.0);
	}
	errno = 0;
        C = atan2 (
            t1 = ( 2. * asin(sin(.5 * (W1 - W2))) ),
            t2 = ( log(tan(pi*.25 + .5*N2)) - log(tan(pi*.25 + .5*N1)) )
                 );
	if (errno != 0) {
		(void)perror("Distance");
		(void)fprintf(stderr, "arg1=%.3f, arg2=%.3f, result=%.3f\n",
			      t1, t2, C);
		(void)fprintf(stderr, "Fix1: %.3f/%.3f, Fix2: %.3f/%.3f\n",
			      Degrees(N1), Degrees(W1),
			      Degrees(N2), Degrees(W2));
	}

	if (N1 == N2) {
		return(60. * fabs(Degrees(2. * asin(sin(.5 * (W1 - W2))))) *
		       cos(N1));
	} else {
		return(60. * Degrees(N2 - N1) / cos(fabs(C)));
	}
} /* Distance */

/* *********************************************************************** */
/* Bearing computes the bearing from true north for a line drawn between   */
/*          two points given in terms of latitude and longitude. Latitudes */
/*          are positive for North, negative for South. Longitudes are     */
/*          positive for West, negative for East.                          */
/*          The algorithm is from the HP-25 Application Programs, page 65. */
/* Returns: bearing,in decimal degrees from true north.                    */
/* Parameters: N1, W1 - latitude and longitude expressed in radians for    */
/*                      first point.                                       */
/*             N2, W2 - latitude and longitude expressed in radians for    */
/*                      second point.                                      */
/* Side Effects: none.                                                     */
/* *********************************************************************** */
double
Bearing(N1, W1, N2, W2, variation)
double	N1,
	W1,
	N2,
	W2,
	variation;
{
        double	C,
		t1,
		t2;

	if (N1 == N2 && W1 == W2) {
		return(-variation);
	}
	errno = 0;
        C = atan2 (
            t1 = ( 2. * asin(sin(.5 * (W1 - W2)))),
            t2 = ( log(tan(pi*.25 + .5*N2)) - log(tan(pi*.25 + .5*N1)) )
                 );
	if (errno != 0) {
		(void)perror("Bearing");
		(void)fprintf(stderr, "arg1=%.3f, arg2=%.3f, result=%.3f\n",
			      t1, t2, C);
		(void)fprintf(stderr, "Fix1: %.3f/%.3f, Fix2: %.3f/%.3f\n",
			      Degrees(N1), Degrees(W1),
			      Degrees(N2), Degrees(W2));
	}
        return( (asin(sin(W1-W2)) >= 0.)
                        ? Degrees(fabs(C))
                        : 360. - Degrees(fabs(C)) );


} /* Bearing */

/* *********************************************************************** */
/* Magnetic computes a magnetic bearing given a true course and the local  */
/*         magnetic variation. East variation is negative. West variation  */
/*         is positive.                                                    */
/*         The algorithm is from the HP-25 Application Programs, page 65.  */
/* Returns: magnetic bearing, in decimal degrees.                          */
/* Parameters: true - true course in decimal degrees.                      */
/*             variation - in decimal degrees.                             */
/* Side Effects: none.                                                     */
/* *********************************************************************** */
double
Magnetic(true,variation)
double	true,
	variation;
{
        double	course;

        course = true + variation;
        if ( course < 0. ) course = 360. + course;
        if ( course >= 360. ) course = course - 360.;
        return(course);
} /* Magnetic */

/* *********************************************************************** */
/* LatIntercept computes the latitude, in radians, of the intercept of     */
/*              the great circle route from (N1,W1) to (N2,W2) at Wi.      */
/*              This procedure will fail if the course from (N1,W1) to     */
/*              (N2,W2) is true north or true south.                       */
/*              Latitude is positive for north, negative for south.        */
/*              Longitude is positive for west, negative for east.         */
/*              The algorithm is from the HP-25 Application Programs,      */
/*              page 65.                                                   */
/* Returns: latitude, in radians, of the intercept point for Wi.           */
/* Parameters: N1, W1 - latitude and longitude of first point.             */
/*             N2, W2 - latitude and longitude of second point.            */
/*             Wi     - longitude to be used to find the intercept point.  */
/* Side Effects: none.                                                     */
/* *********************************************************************** */
double
LatIntercept(N1, W1, N2, W2, Wi)
double	N1,
	W1,
	N2,
	W2,
	Wi;
{
        if (W1 == W2)
		W2 += 1e-15;	/* Avoid breaking with true N/S routes */

        return( atan(
                     (tan(N2)*sin(Wi-W1) - tan(N1)*sin(Wi-W2))
                                /
                     (sin(W2-W1))) );
} /* LatIntercept */

/*
 * Return wind-corrected true heading vector
 *
 * Algorithm (from gLENN iNN's UseNet posting):
 *	You need to use two trig laws:
 *		c^2 = a^2 + b^2 - 2ab*cos(C)	;Law of Cosines
 *		(Sin A)/a == (Sin B)/b 		;Law of Sines
 *	For WCA (wind correction angle):
 *		 Wsp
 *		--->
 *	    TC	^  ^		Sin (-WCA)/(Wsp) = Sin([180+W]-TC)/(Asp)
 *		! / Track	WCA = -Arcsin ([Wsp/Asp]*sin(180+W-TC))
 *		!/
 *	      -WCA		WHERE:
 *				  WCA: wind correction angle
 *				  Wsp: wind speed
 *				    W: wind direction
 *				   TC: true course
 *				  Asp: airspeed
 *
 *				NOTE: (180+W) corrects for wind given as
 *				      FROM direction
 *			True Heading = TC + WCA
 *	For Ground Speed:
 *		Gsp^2 = Asp^2 + Wsp^2 - 2(Asp)(Wsp)*cos(TC-W+WCA)
 */
vector
WindCorrection(AirSpeed, TrueCourse, WindSpeed, TrueWindDirection)
double	AirSpeed,
	TrueCourse,
	WindSpeed,
	TrueWindDirection;
{
double	WindCorrectionAngle,
	TrueHeading,
	GroundSpeed,
	temp1;
vector	Result;

#define sqr(n)	((n) * (n))

	if (TrueCourse >= 360.0) TrueCourse -= 360.0;

	temp1 = sin(Radians(180.0 + TrueWindDirection - TrueCourse)) *
		 (WindSpeed / AirSpeed);
	WindCorrectionAngle = -Degrees(asin(temp1 > 1.0 ? temp1 - 1.0 :
							  temp1));

	if ((cos(Radians(WindCorrectionAngle)) * WindSpeed) >= AirSpeed) {
				/* Big direct headwind */
		(void)fprintf(stderr,
	"\tWinds of %03.0f@%.0f (true), with a true course and speed of ",
			(TrueWindDirection == 0.0 ? 360.0 : TrueWindDirection),
			WindSpeed);
		(void)fprintf(stderr,
	"%03.0f@%.0f means\n\t\tyou are flying backwards!\n",
			(TrueCourse == 0.0 ? 360.0 : TrueCourse),
			AirSpeed);
		(void)exit(1);
	}

	TrueHeading = TrueCourse + WindCorrectionAngle;
	GroundSpeed = sqrt(sqr(AirSpeed) + sqr(WindSpeed) -
		(2.0 * AirSpeed * WindSpeed * cos(Radians(
		      TrueCourse - TrueWindDirection + WindCorrectionAngle))));
	if (TrueHeading < 0.0)
		TrueHeading += 360.0;
	else if (TrueHeading >= 360.0)
		TrueHeading -= 360.0;
	Result.course = TrueHeading;
	Result.speed = GroundSpeed;
	return(Result);
} /* WindCorrection */

/* Following is the old implementation of WindCorrection */
#ifdef	OLDCODE
/* *********************************************************************** */
/* WindCorrection will compute a new course and ground speed from the      */
/*         given true airspeed, course, wind speed and direction, and      */
/*         local magnetic variation. Wind direction is given with respect  */
/*         to TRUE north.                                                  */
/* Returns: void.                                                          */
/* Parameters: tas - true airspeed for the aircraft, in knots.             */
/*             crs - desired magnetic course, in decimal degrees.          */
/*             speed - wind speed, in knots.                               */
/*             direction - wind direction, from true north.                */
/*             var - local magnetic variation.                             */
/* Side Effects: modifies crs and gs. crs becomes the magnetic heading     */
/*             to fly, gs becomes the computed ground speed.               */
/* *********************************************************************** */
WindCorrection (tas, crs, speed, direction, gs, var)
int tas, speed, direction, *gs;
double *crs, var;
{
double B, wca;

        if (*crs == (direction+var)) /* heading is INTO the wind */
        {
            *gs = tas - speed;
        }
        else
        if ((fabs(*crs - (direction+var))) == 180.0) /* heading WITH the wind */
        {
            *gs = tas + speed;
        }
        else /* compute effect of wind on heading and ground speed */
        {
            B = Radians (*crs - (direction+var));
            *gs = sqrt (tas*tas + speed*speed -  2.*tas*speed*cos(B));
            wca = Degrees(asin (speed / (*gs / sin(B))));
            *crs = *crs - wca;
            if (*crs < 0) *crs = *crs + 360.0;
            if (*crs >= 360) *crs = *crs - 360.0;
        }
}
#endif	OLDCODE
