/*
 *	Aviation navigation program
 *	output.c [2.29] from /preflight/src/nav/SCCS/src/s.output.c
 *		Retrieved 16:26:47 88/07/07; latest mod 16:02:23 88/07/07
 *	Alan M. Marcum		marcum@nescorna.Sun.COM
 *	Robert J. Evans		tolerant!procase!rje
 */

#ifndef	LINT
static char *SCCSid = "@(#)output.c\tRevision 2.29\t88/07/07";
#endif	LINT
/************************************************************************
 *
 * Navigation Flight Plan Program
 *
 * This program is public domain software.  No claim is made for the
 * accurracy of either the program or the databases used by the program.
 * Always check the flight plan provided by the program for reasonable
 * results.  The pilot in command of an aircraft is directly responsible
 * for, and is the final authority as to, the operation of that aircraft.
 * (FAR 91.3)  Each pilot in command shall, before beginning a flight,
 * familiarize himself with all available information concerning that
 * flight. (FAR 91.5)
 *
 ************************************************************************/

#include <stdio.h>
#include <math.h>

#ifdef	SysV
#include	<string.h>
#else	/* !SysV */
#include	<strings.h>
#endif	/* SysV */

#if	defined(Sun) || defined(SysV) || defined(Xenix386)
#include <values.h>

#else	/* ! (defined(Sun) || defined(SysV) || defined(Xenix386)) */
#ifdef	MSC
#include <limits.h>
#define MAXINT INT_MAX

#else	/* ! MSC */
#define MAXINT	(~ (1 << ((8 * (int)sizeof(int)) - 1)) )

#endif	/* MSC */
#endif	/* defined(Sun) || defined(SysV) || defined(Xenix386) */

#include <Nav.h>
#include <NavFormat.h>
#include <NavGlobals.h>

double Distance();
double Bearing();
double Magnetic();
double Degrees();
double Radians();

float	tac_to_dme();
void	print_lat_lon();
void	print_vor();
float	get_degrees(),
	get_minutes(),
	get_seconds();
void	unparse_type();
static	void	comment(),
		PrintWp(),
		PrintVor(),
		PrintApt(),
		PrintLine(),
		PrintHead(),
		PrintLeg(),
		PrintNearest(),
		DoPrintNearest();

/*char *index();*/

extern int comflg;
extern int rviaflg;

#if	defined(Sun) || defined(BSD4_3)
#define INTEGER_INFINITY	((int)drem(1,0))
#else
#define INTEGER_INFINITY	MAXINT
double	rint();
#endif

/************************************************************************
 * FlightGuide causes the actual printing of the flight plan.
 * Returns: void.
 * Parameters:
 *     output_format	- select report format
 *     start_airport	- airport record for origin airport.
 *     to_airport  	- airport record for destination airport.
 *     from_flag   	- if > 0, start_airport is valid.
 *     to_flag     	- if > 0, to_airport is valid.
 *     pos_flag	   	- if > 0, print lat-lon of navaids and airports
 *     alt_flag      	- if > 0, Alt_Array is valid.
 *     Alt_Array    	- contains altitude for each leg, if alt_flag > 0.
 *     speed_flag    	- if >0, Speed_Array is valid.
 *     Speed_Array  	- contains speed for each leg, if speed_flag > 0.
 *     wind_flag     	- if > 0, Wind_Array is valid.
 *     Wind_Array   	- contains winds aloft for each leg, if wind_flag > 0.
 *     fuel_flag	- if > 0, Fuel_Array is valid.
 *     cost_flag	- if > 0, "fuel" data are actually cost
 *     Fuel_Array	- contains fuel flows for each leg, if fuel_flag > 0.
 *     via_flag		- if > 0, Via_Array is valid.
 *     lrvia_flag	- if > 0, Via_Array records contain waypoint info.
 *     Via_C		- number of entries in Via_Array.
 *     Via_Array	- contains vor records, if via_flag > 0.
 *
 * Side Effects: prints entire flight plan.
 ************************************************************************/
void
FlightGuide(output_format, Start_Airport, To_Airport,
            from_flag, to_flag, pos_flag,
            alt_flag, Alt_Array,
            speed_flag, Speed_Array,
            wind_flag, Wind_Array,
	    fuel_flag, cost_flag, Fuel_Array,
            via_flag, lrvia_flag, Via_C, Via_Array,
            print_nearest_vors, nearest_navaids, dist_orig, rad_orig, dist_dest, rad_dest)
struct apt *Start_Airport, *To_Airport;
int	output_format, from_flag, to_flag, pos_flag,
	alt_flag, Alt_Array[],
	speed_flag, Speed_Array[],
	wind_flag, Wind_Array[],
	fuel_flag, cost_flag,
	print_nearest_vors, via_flag, lrvia_flag, Via_C;
struct vor	Via_Array[], nearest_navaids[];
float	Fuel_Array[];
double	dist_orig[], rad_orig[],
	dist_dest[], rad_dest[];

{ /* FlightGuide */
int ALT,                      /* altitude, in feet */
    DIR,                      /* wind direction, with respect to true north */
    ELAPSE,                   /* elapsed time */
    GS,                       /* ground speed */
    last_alt,                 /* altitude, in feet */
    last_type,		      /* type (see nav.h) */
    TAS,                      /* true air speed, in knots */
    WIND,                     /* wind speed, in knots */
    to_alt,                   /* altitude, in feet, of to navigation point */
    to_type,		      /* type (see nav.h) */
    leg,                      /* indicate which leg is being printed */
    scrat,                        /* scratch variable */
    i;			      /* ditto */

double Course,                /* projected course, usually true */
       CRS,                   /* projected course, usually magnetic */
       Dist,                  /* distance for this leg, in nautical miles */
       HDG,                   /* heading to fly, magnetic */
       last_var,              /* magnetic variation of lat navigation fix */
       last_freq,             /* frequency of last airport or nav aid */
       Mag,                   /* projected course, usually magnetic */
       NM,                    /* distance, in nauatical miles */
       Radial,                /* radial from vortac for RNAV waypoint */
       TotalDist,             /* total distance for flight */
       to_freq;               /* frequency of next airport or navaid */

float	FuelFlow,	      /* Fuel flow for this leg */
	FuelLeg,	      /* Fuel used for this leg */
	FuelTotal;	      /* Total fuel used */
char last_name[NAMLEN],       /* name of last navigation fix */
     to_name[NAMLEN],         /* name of next navigation fix */
     last_id[IDLEN],          /* id of last navigation fix */
     to_id[IDLEN],            /* id of next navigation fix */
     to_comments[COMLEN],     /* comments for next navigation fix */
     last_comments[COMLEN],   /* comments for last navigation fix */
     to_wpid[WPLEN];          /* way point id */

vector	heading_vector;		/* Wind corrected heading vector */
vector	WindCorrection();

struct fix last_fix;          /* latitude, longitude, radial and distance */

extern char	*RELEASE;

        scrat = 0;
        (void)printf(Title,
		"Flight Plan - Release",
	        RELEASE);
        if (from_flag != 0)
        {
           (void)strncpy(last_id, Start_Airport->id, IDLEN);
           (void)strncpy(last_name, Start_Airport->name, NAMLEN);
           (void)strncpy(last_comments, Start_Airport->comments, COMLEN);
           last_alt = Start_Airport->alt;
           last_fix = Start_Airport->loc;
           last_var = Start_Airport->var;
           last_freq = Start_Airport->freq;
	   last_type = TYPE_AIRPORT;
	   PrintHead(output_format, cost_flag, wind_flag, alt_flag);
        }
        else
        {
           if (via_flag == 0) {
                (void)fprintf(stderr,
		        "nav: internal error - no origin, no vors given.");
		(void)exit(-1);
	   }
           (void)strncpy(last_id, Via_Array[scrat].id, IDLEN);
           (void)strncpy(last_name, Via_Array[scrat].name, NAMLEN);
           (void)strncpy(last_comments, Via_Array[scrat].comments, COMLEN);
           last_alt = Via_Array[scrat].alt;
           last_fix = Via_Array[scrat].loc;
           last_var = Via_Array[scrat].var;
           last_freq = Via_Array[scrat].freq;
	   last_type = Via_Array[scrat].vor_type;
           PrintHead(output_format, cost_flag, wind_flag, alt_flag);
           scrat++;
        }

        Dist = TotalDist = Course = Mag = 0.0;
	FuelLeg = FuelTotal = 0.0;
        TAS = 0; ELAPSE = 0; GS = 0;
        for (; scrat <= Via_C; scrat++) {
		if (scrat < Via_C) {
/*
 * Following is removed, pending a decision of proper output format
 * for multiple-page flight logs...29Dec86 amm
 */
#ifdef	MULTI_PAGE
			if ((scrat % 10) == 0)
				PrintHead(output_format,
					  cost_flag, wind_flag, alt_flag);
#endif	MULTI_PAGE
			Dist = Distance(last_fix.lat.rad, last_fix.lon.rad,
					Via_Array[scrat].loc.lat.rad,
					Via_Array[scrat].loc.lon.rad);
			Course = Bearing(last_fix.lat.rad, last_fix.lon.rad,
					 Via_Array[scrat].loc.lat.rad,
					 Via_Array[scrat].loc.lon.rad,
					 last_var);
			(void)strncpy(to_name, Via_Array[scrat].name, NAMLEN);
			(void)strncpy(to_id, Via_Array[scrat].id, IDLEN);
			(void)strncpy(to_comments, Via_Array[scrat].comments,
				      COMLEN);
			(void)strncpy(to_wpid,
				      Via_Array[scrat].waypoint,
				      WPLEN);
			to_freq = Via_Array[scrat].freq;
			to_alt = Via_Array[scrat].alt;
			to_type = Via_Array[scrat].vor_type;
		}
		else if (to_flag != 0) {
/*
 * Following is removed, pending a decision of proper output format
 * for multiple-page flight logs...29Dec86 amm
 */
#ifdef	MULTI_PAGE
			if ((scrat % 10) == 0)
				PrintHead(output_format,
					  cost_flag, wind_flag, alt_flag);
#endif	MULTI_PAGE
				Dist = Distance(last_fix.lat.rad,
						last_fix.lon.rad,
						To_Airport->loc.lat.rad,
						To_Airport->loc.lon.rad);
				Course = Bearing(last_fix.lat.rad,
						 last_fix.lon.rad,
						 To_Airport->loc.lat.rad,
						 To_Airport->loc.lon.rad,
						 last_var);
				(void)strncpy(to_name, To_Airport->name,
					      NAMLEN);
				(void)strncpy(to_id, To_Airport->id, IDLEN);
				(void)strncpy(to_comments,
					      To_Airport->comments, COMLEN);
				(void)strncpy(to_wpid, "    ", WPLEN);
				to_freq = To_Airport->freq;
				to_alt = To_Airport->alt;
				to_type = TYPE_AIRPORT;
		}
		if ((scrat < Via_C) || ((scrat == Via_C) && (to_flag != 0))) {
			TotalDist += Dist;
			if (lrvia_flag || (Via_Array[scrat].loc.nm != 0.0)) {
				Radial = Via_Array[scrat].loc.radial;
				NM = Via_Array[scrat].loc.nm;
			} else {
				Radial = 0.0;
				NM = 0.0;
			}
			TAS = speed_flag ? Speed_Array[scrat] : 0;
			ALT = alt_flag ? Alt_Array[scrat] : 0;
			if (wind_flag) {
				WIND = Wind_Array[(scrat*2)+1] ;
				DIR = Wind_Array[(scrat*2)] ;
			} else {
				WIND = 0;
				DIR = 0;
			}
			FuelFlow = fuel_flag ? Fuel_Array[scrat] : 0.0;
			GS = 0;
			Mag = CRS = Magnetic(Course, last_var);
			if ((wind_flag != 0) && (speed_flag != 0)) {
				heading_vector =
					WindCorrection((double)TAS,
						       Mag - last_var,
						       (double)WIND,
						       (double)DIR);
				HDG = Magnetic(heading_vector.course,
					       last_var);
				GS = (int)rint(heading_vector.speed);
			} else	HDG = CRS;    /* Heading = Course if no wind */
			if (2 == Via_C + to_flag + from_flag)
				leg = 3; /* only one leg for this flight */
			else if (((scrat == 0) && (from_flag != 0)) ||
				 ((scrat == 1) && (from_flag == 0)))
				leg = 1; /* first leg to print */
			else if (((scrat == Via_C) && (Via_C > 0) &&
				 (to_flag != 0)) ||
				 ((scrat == (Via_C - 1)) && (to_flag == 0)))
				leg = 2; /* last leg to print */
			else
				leg = 0; /* any leg but first or last */
			if (fuel_flag) {	/* Guaranteed that TAS != 0 */
				FuelLeg = FuelFlow *
						(Dist / (GS == 0 ? (float)TAS :
								   (float)GS));
				FuelTotal += FuelLeg;
			}
			PrintLeg(output_format, CRS, HDG,
				 Dist, TotalDist,
				 Radial, NM,
				 TAS, ALT,
				 WIND, DIR, wind_flag,
				 GS, &ELAPSE, speed_flag,
				 FuelFlow, FuelLeg, FuelTotal, cost_flag,
				 last_name, last_id, last_freq,
				 last_alt, last_type, last_comments, from_flag,
				 to_name, to_id, to_freq, to_alt, to_type,
				 to_comments, to_flag, to_wpid,
				 leg, lrvia_flag);
			(void)strncpy(last_name, Via_Array[scrat].name, NAMLEN);
			(void)strncpy(last_id, Via_Array[scrat].id, IDLEN);
			(void)strncpy(last_comments, Via_Array[scrat].comments,
				      COMLEN);
			last_fix = Via_Array[scrat].loc;
			last_alt = Via_Array[scrat].alt;
			last_var = Via_Array[scrat].var;
			last_freq = Via_Array[scrat].freq;
			last_type = Via_Array[scrat].vor_type;
		}
	}

	if (pos_flag > 0 && (from_flag || to_flag || via_flag || lrvia_flag)) {
	        (void)printf(LLHeader);
		if (from_flag)
			print_lat_lon(Start_Airport->name,
			        Start_Airport->id,
				Start_Airport->loc, TYPE_AIRPORT, pos_flag);
		if (via_flag)
			for (i = 0; i < Via_C; i++)
			        print_lat_lon(
				      ((lrvia_flag && Via_Array[i].rnavable) ||
				        (Via_Array[i].loc.nm != 0.0)) ?
							Via_Array[i].waypoint :
							Via_Array[i].name,
				        Via_Array[i].id,
					Via_Array[i].loc,
					Via_Array[i].vor_type, pos_flag);
		if (to_flag)
			print_lat_lon(To_Airport->name, To_Airport->id,
			              To_Airport->loc, TYPE_AIRPORT, pos_flag);
	}

        if (((from_flag) || (to_flag)) && print_nearest_vors) {
		PrintNearest(from_flag, to_flag, Start_Airport,
			     To_Airport, nearest_navaids, rad_orig, rad_dest,
			     dist_orig, dist_dest);
	}
        if (comflg != 0) {
		(void)printf("\n\n");
		if (from_flag != 0) {
			PrintApt(Start_Airport);
		}
		for (scrat = 0; scrat < Via_C; v++) {
			PrintVor(&Via_Array[scrat]);
		}
		if (to_flag !=0) {
			PrintApt(To_Airport);
		}
	}
} /* FlightGuide() */


/************************************************************************
 * Comment will display the comment field pointed to by s.
 * Returns: void.
 * Parameters: s - pointer to comment string of vor or airport record.
 * Side Effects: prints comment field.
 ************************************************************************/
static void
comment(s)
char *s;
{
        register char *p;

        for (p=s; *p != 0; p++)
                if (*p == '\n')
                        *p = 0;
        if (comflg && strlen(s))
                (void)printf("Info: %s \n",s);
}


/************************************************************************
 * PrintWp will print the waypoint field of a vor record.
 * Returns: void.
 * Parameters: vortac - a vor record.
 * Side Effects: prints the vor record waypoint field.
 ************************************************************************/
static void
PrintWp(vortac)
struct vor *vortac;
{
        (void)printf("WayPoint: %s NM\n", vortac->waypoint);
}


/************************************************************************
 * PrintVor will print a formatted version of the vor record.
 * Returns: void.
 * Parameters: vortac - a vor record.
 * Side Effects: prints the vor record.
 ************************************************************************/
static void
PrintVor(vortac)
struct vor *vortac;
{
        if (rviaflg)
        {
           (void)printf("\nRvor: %s\t\tFreq: %.1f\n", vortac->name,vortac->freq);
           (void)printf("ID: %s,\tAltitude: %d feet\n",
                  vortac->id,vortac->alt);
           PrintWp (vortac);
        }
        else
        {
           (void)printf("\nvor %s\t\tfreq: %.1f\n", vortac->name,vortac->freq);
           (void)printf("ID: %s,\tAltitude: %d feet\n",
                   vortac->id,vortac->alt);
        }
        comment(vortac->comments);

}
/************************************************************************
 * PrintApt will print a formatted version of the airport record
 * Returns: void.
 * Parameters: airport - an airport record.
 * Side Effects: prints the airport record.
 ************************************************************************/
static void
PrintApt(airport)
struct apt *airport;
{
        (void)printf("\nAirport: %s,\tCity: %s\n", airport->name, airport->city);
        (void)printf("ID: %s,\tAltitude: %d feet\n",
                airport->id,airport->alt);
        (void)printf("Lat %5.1f, Lon %5.1f Degrees\n",
                airport->loc.lat.deg,airport->loc.lon.deg);
        comment(airport->comments);
}


/************************************************************************
 * PrintLine prints a separator line of dashes as needed by the flight plan
 * Returns:  void.
 * Parameters: output_format - selects format of output.
 * Side Effects: prints the separator line.
 ************************************************************************/
static void
PrintLine(output_format)
int	output_format;
{
char dashes[10];
   (void)strcpy(dashes, "--------");
   if (output_format == 1)
   {
     (void)printf("+-%8s%8s+-----+-------+-------+----+-----+-------+%8s%8s--+\n",
            dashes, dashes, dashes, dashes);
   }
   else
   if (output_format == 2)
   {
     (void)printf("+-%8s%8s%8s---+-----+-------+--------+-----+-------+---------+\n",
            dashes, dashes, dashes);
   }
   else
   if (output_format == 3)
   {
    (void)printf("+%8s%8s%8s---+-------------------------------------------------+\n",
            dashes, dashes, dashes);
   }
}


/************************************************************************
 * PrintHead prints the heading required by the flight plan.
 * Returns:  void.
 * Parameters:
 * 	output_format - selects the report format
 * 	cost_flag 	- > 0 if using cost instead of fuel information
 * 	wind_flag 	- > 0 if winds aloft information provided
 * 	alt_flag  	- > 0 if altitude information provided
 * Side Effects: printd the heading information.
 ************************************************************************/
static void
PrintHead(output_format, cost_flag, wind_flag, alt_flag)
int	output_format,
	cost_flag,
	wind_flag,
	alt_flag;
{
	if (output_format == 1) {		/* Old format */
		(void)printf("\n");
		PrintLine(output_format);
		(void)printf(F1Header1, "FROM", "TO");
		(void)printf(F1Header2, "FREQ/ID/ALT", "FREQ/ID/ALT");
		PrintLine(output_format);
	}
	else if (output_format == 2) {		/* New format */
		(void)printf("\n");
		PrintLine(3);
		(void)printf(F2Title1,
			"Origin/Fix/Destination",
			"Navigation", "Log");
		(void)printf(F2Title2,
			"Comments",
			"|");
		(void)printf(F2Header1,
			"Freq  / Alt / Type",
			wind_flag ? "Wind" : "",
			cost_flag ? "Cost" : "Fuel");
		(void)printf(F2Header2,
			alt_flag ? "Alt" : "");
	}
}

/************************************************************************
 * PrintLeg prints the navigation information needed to fly one leg of
 *          the flight plan.
 * Returns: void.
 * Parameters:
 *   output_format - selects the report format
 *   crs     	- the desired magnetic course.
 *   mag     	- the magnetic heading to fly to achieve the desired crs.
 *   dist    	- nautical miles for this leg.
 *   total   	- nautical miles up to the end of this leg.
 *   rad     	- RNAV waypoint radial to use.
 *   nm      	- distance from vortac along rad for RNAV waypoint.
 *   tas     	- true airspeed, in knots
 *   alt     	- altitude, in feet.
 *   wind    	- wind speed, in knots.
 *   dir     	- wind direction, with respect to true north.
 *   wind_flag 	- were winds aloft provided?
 *   fuel_flag 	- were fuel flows provided?
 *   FuelFlow 	- Fuel burn rate, units per hour
 *   FuelLeg 	- Fuel burned on this leg
 *   FuelTotal 	- Fuel burned cumulatively
 *   cost_flag 	- "Fuel..." numbers are really cost
 *   gs      	- projected groundspeed, in knots.
 *   elapse  	- total elapsed time.
 *   speed_flag 	- was speed information supplied?
 *   from_name 	- name of point navigating from.
 *   from_id   	- id of point navigating from.
 *   from_freq 	- vor frequency.
 *   from_alt  	- elevation of point navigating from.
 *   from_type 	- type of departure point
 *   from_comments 	- comments relating to departure point
 *   to_name 	- name of point navigating to.
 *   to_id   	- id of point navigating to.
 *   to_freq 	- vor frequency.
 *   to_alt  	- elevation of point navigating to.
 *   to_type 	- type of destination point
 *   to_comments 	- comments relating to destination
 *   lrvia_flag 	- is this an RNAV flight plan?
 *   leg 	- which leg type is this?
 *
 * Side Effects: causes one complete leg description to be printed.
 ************************************************************************/
static void
PrintLeg(output_format, crs, mag, dist, total, rad, nm, tas, alt,
	 wind, wind_dir, wind_flag,
	 gs, elapse, speed_flag,
	 FuelFlow, FuelLeg, FuelTotal, cost_flag,
	 from_name, from_id, from_freq, from_alt, from_type,
	 from_comments, from_flag,
	 to_name, to_id, to_freq, to_alt, to_type,
	 to_comments, to_flag, to_wpid,
	 leg, lrvia_flag)
int	output_format, tas, alt,
	wind, wind_dir, wind_flag,
	gs, *elapse, speed_flag,
	cost_flag,
	from_alt, from_type, to_alt, to_type,
	leg,
	from_flag, to_flag, lrvia_flag;
double	crs, mag,
	dist, total,
	rad, nm,
	from_freq, to_freq;
float	FuelFlow, FuelLeg, FuelTotal;
char	from_name[], from_id[],
	to_name[], to_id[],
	from_comments[], to_comments[],
	to_wpid[];
{ /* PrintLeg() */
int	t1, t2, t3, t4;		/* Temps */
char	type[TYPELEN],
	altstg[ALTSIZE],	/* Character string version of altitude;
				 * increase length if we want to handle
				 * altitudes greater than 99,999' */
	ete_stg[ETESIZE],	/* Ditto, for Elapsed Time Enroute */
	eta_stg[ETASIZE];	/* and Elapsed Time Aloft */

#define FMTSIZ	200		/* Should be long enough for our needs */
static char	headerfmt[FMTSIZ],	/* Formatting strings */
		legfmt1[FMTSIZ],
		legfmt2a[FMTSIZ],
		legfmt3[FMTSIZ];
char	legfmt2[FMTSIZ];		/* This one might change leg to leg */

static int	firsttime = TRUE,
		InfiniteGS = FALSE;

	if (firsttime) {		/* Set up format strings */
		firsttime = !firsttime;
		switch (output_format) {
		case FORMAT_OLD:
			(void)sprintf(legfmt1,
				F1Leg1,
				wind_flag ? "%.3d@%.2d" : "    %c%c");
			break;
		case FORMAT_NEW:
			(void)sprintf(headerfmt,
				F2LegHeader,
				lrvia_flag ? "  WP# " : "Airway");
			(void)sprintf(legfmt1,
				F2Leg1,
				wind_flag ? "%03d@%02d" : "   %c%c ",
				cost_flag ? "%6.2f" : "%6.1f");
			(void)sprintf(legfmt2a,
				F2Leg2a,
				cost_flag ? "%%6.2f" : "%%6.1f");
			(void)sprintf(legfmt3,
				F2Leg3,
				cost_flag ? "%6.2f" : "%6.1f");
			break;
		default:
			(void)fprintf(stderr,
				"%s: Internal error - unknown FORMAT %d.\n",
				pgm_name, output_format);
			(void)exit(-1);
			break;
		}
	}

	type[0] = '\0';
	if (!wind_flag)
		gs = tas;
	if (!InfiniteGS && gs != 0) {
		t1 = (int) ((dist / gs) * 60);
		*elapse += t1 ;
		t2 = t1 % 60;
		t1 = t1 / 60;
		t4 = *elapse % 60;
		t3 = *elapse / 60;
		(void)sprintf(ete_stg, TimeFmt, t1, t2);
		(void)sprintf(eta_stg, TimeFmt, t3, t4);
	} else if (wind_flag && speed_flag) {
		/* Elapsed time is infinite if wind --> GS == 0 */
		InfiniteGS = TRUE;
		(void)strcpy(ete_stg, "*****");
		(void)strcpy(eta_stg, ete_stg);
		t1 = t2 = t3 = t4 = INTEGER_INFINITY;
	} else {
		(void)strcpy(ete_stg, "     ");
		(void)strcpy(eta_stg, ete_stg);
		t1 = t2 = t3 = t4 = 0;
	}

	alt /= 100;

	(void)sprintf(altstg, "%d", alt >= 180 ? alt : alt*100);

	switch (output_format) {
	case FORMAT_OLD:
		(void)printf(legfmt1,
	                from_name, crs,
			(wind_flag ? (0 == wind_dir ? 360 : wind_dir) : ' '),
			(wind_flag ? wind : ' '), dist,
			tas, rad, total, to_name);
		(void)sprintf(legfmt2, F1Leg2,
			alt >= 180 ? "FL%3s" : "%5s");
		(void)printf(legfmt2,
			(from_type == TYPE_TACAN ?
				tac_to_dme(from_freq) : from_freq),
			from_id, from_alt, mag,
			alt == 0 ? "" : altstg,
			ete_stg, gs, nm, eta_stg,
			(to_type == TYPE_TACAN ?
				tac_to_dme(to_freq) : to_freq),
			to_id, (to_alt < 0 ? 0 : to_alt));
		PrintLine(output_format);
		break;
	case FORMAT_NEW:
		if ((leg == 1) || (leg == 3)) {
			(void)printf(headerfmt, from_name, from_id);
			(void)printf(F2Leg4,
				from_comments);
		}
		unparse_type(type, from_type);
		if (((leg == 1) || (leg == 3)) && (from_flag))
			(void)strcpy(type, "Airport");

		(void)printf(legfmt1,
			(TYPE_TACAN == from_type ?
				tac_to_dme(from_freq) : from_freq),
			(from_alt < 0 && TYPE_AIRPORT != from_type ?
				0 : from_alt),
			type, crs,
			(TYPE_TACAN == to_type ?
				tac_to_dme(to_freq) :
				(TYPE_AIRPORT == to_type ? 0.0 : to_freq)),
			(wind_flag ? (0 == wind_dir ? 360 : wind_dir) : ' '),
			(wind_flag ? wind : ' '),
			tas, dist, total, FuelFlow);

		(void)sprintf(legfmt2, legfmt2a,
			alt >= 180 ? "FL%3s" : "%5s");
		(void)printf(legfmt2,
			mag, rad,
			alt == 0 ? "" : altstg,
			gs, ete_stg, eta_stg, FuelLeg);

		(void)printf(legfmt3,
			to_name, to_id, to_wpid, nm, to_id, FuelTotal);
		(void)printf(F2Leg4,
				to_comments);
		if ((leg == 2) || (leg == 3)) {
			unparse_type(type, from_type);
			if (to_flag != 0)
				(void)strcpy(type, "Airport");
			(void)printf(F2Leg5,
				(TYPE_TACAN == to_type ?
					tac_to_dme(to_freq) : to_freq),
				(to_alt < 0 && TYPE_AIRPORT != to_type ?
					0 : to_alt),
				type);
		}
		break;
	default:
		(void)fprintf(stderr,
			"nav: Internal error - unknown FORMAT %d.\n",
			output_format);
		(void)exit(-1);
		break;
	}
} /* PrintLeg() */


static void
PrintNearest(from_flag, to_flag, Start_Airport, To_Airport,
	     nearest_navaids, rad_orig, rad_dest, dist_orig, dist_dest)
int	from_flag,	/* Have we a valid origin airport? */
	to_flag;	/* Have we a valid destination airport? */
struct	apt	*Start_Airport,	/* Origin airport */
		*To_Airport;	/* Destination airport */
struct	vor	nearest_navaids[];	/* The nearest navaids */
double	rad_orig[],		/* Radials from nearest navaids to ... */
	rad_dest[],		/* ... FROM and TO airports */
	dist_orig[],		/* Ditto, distances */
	dist_dest[];

{ /* PrintNearest() */

	(void)printf(NearestHeader);
	if (from_flag) {
		DoPrintNearest(FROM_VORS_ORIG, Start_Airport,
			       nearest_navaids, rad_orig, dist_orig);
	}
	if (to_flag) {
		DoPrintNearest(TO_VORS_ORIG, To_Airport,
			       nearest_navaids, rad_dest, dist_dest);
	}
} /* PrintNearest() */


static void
DoPrintNearest(orig, airport, navaids, rad, dist)
int	orig;			/* From where are we starting? */
struct	apt	*airport;	/* Airport of interest */
struct	vor	navaids[];	/* Navaids nearest the airport */
double	rad[],			/* Radials from navaids to airport */
	dist[];			/* ...and distances */
{ /* DoPrintNearest() */
	int	i;
	for(i = orig; i < orig + N_NEAREST_VORS; i++) {
		if (i == orig) {
			(void)printf(Nearest1,
				     airport->name,
				     airport->id,
				     navaids[i].name,
				     navaids[i].id,
				     (TYPE_TACAN == navaids[i].vor_type ?
					  tac_to_dme(navaids[i].freq) :
					  navaids[i].freq));
		} else {
			(void)printf(Nearest2,
				     " ", " ",
				     navaids[i].name,
				     navaids[i].id,
				     (TYPE_TACAN == navaids[i].vor_type ?
					  tac_to_dme(navaids[i].freq) :
					  navaids[i].freq));
		}
		(void)printf(Nearest3,
			     rad[i - orig],
			     dist[i - orig]);
	}
	(void)printf("\n");
} /* DoPrintNearest() */
