/*
 * lansort.c
 *
 * Paul Grepps		6/10/91
 *
 * Purpose: To sort the lancard entries by their IP Address.
 *		
 * Exit Codes:
 *		CLEAN		good and clean exit
 *		ERR		bad exit
 *
 * Bugs:
 * Additional Notes:
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>

/* Accell includes */
$include "/usr/db/ACL4/db/file.h"
#include "/usr/db/ACL4/include/unisel.h"
#include "/usr/db/ACL4/db/lancard.h"

/* delimiter for IP addresses*/
#define DOT "."

/* exit codes */
#define CLEAN 1
#define ERR -1

/* booleans */
#define FALSE 0
#define TRUE 1

/* Tab locations for fields */
#define NODE_LOC 0
#define SUBNET_LOC 21
#define IP_LOC 28
#define ETHER_LOC 44
#define MMAC_LOC 62

#define PAGE_OFF 25 /* offset used for centering printer page */

#define MAXREC 1000		/* maximum number of records in lancard */

/* lancard record */
struct lancard_s	
	{
	unsigned int key;		/* used as the sort key */
	char 	tcpip[16],		/* ip address */
			ether[13],		/* ethernet address */
			subnet[4],		/* subnet */
			node_id[21],	/* node id */
			mmac[13];		/* mmac */
	};

strip_end_spaces(char *string)
	{
	char *temp;
	int done;

	temp = string + (strlen(string) - 1); 
	done = FALSE;
	while (!done)
		{
		if (*temp != ' ')
			{
			temp++;
			*temp = '\0';  /* null the string at the space */
			done = TRUE;
			}
		temp--;
		}
	}

get_field_data(int field,char *string)
	{
	int rc;
	
	/* inbuf returns -3 when a carriage return was hit */
	rc = inbuf(field,string);
	if (rc == -3)
		strcpy(string,"*");
	else if (rc == 0)
		strip_end_spaces(string);
	}

get_search_criteria(struct lancard_s *lc)
	{
	int rc;

	clr_crt(1,'a');	
	loadscr("lancard");	/* loads lancard screen form */

	get_field_data(node_sc,lc->node_id);
	get_field_data(sbnet_sc,lc->subnet);
	get_field_data(tcpip_sc,lc->tcpip);
	get_field_data(ether_sc,lc->ether);
	get_field_data(mmac_sc,lc->mmac);
	setcook();
	}

int build_key(char *tcpip_addr)
	{
	unsigned int subnet, host, key;
	char *tempaddr;

	strcpy(tempaddr,tcpip_addr);

	strtok(tempaddr,DOT);
	strtok((char *) NULL,DOT);
	subnet=atoi(strtok((char *) NULL,DOT));
	host=atoi(strtok((char *) NULL,DOT));
	
	/* all IP addresses are 138.39.<subnet>.<host> where <subnet> and <host>
	 * are numbers up to 255.  So by multiplying the subnet number by 256
	 * and adding it to the host we get the subnet in the MSB of the 
	 * number and the host in the LSB
	 */
	key = (subnet * 256) + host; /* subnet gets MSB, host gets LSB */
	return(key);
	}

int keycompare(struct lancard_s *firstnum, struct lancard_s *secondnum)
	{
	if (firstnum->key < secondnum->key)
		return(-1);	/* less than */
	else if (firstnum->key > secondnum->key)
		return(1);	/* greater than */
	else 
		return(0);	/* equal to */
	}

print_screen_heading()
	{
	clr_crt(1,'a'); /* clear screen */
	prmprv(19,0,"Lancard sorted by IP Address");
	prmprv(NODE_LOC,2,"Node ID");
	prmprv(SUBNET_LOC-2,2,"Subnet");
	prmprv(IP_LOC,2,"IP Address");
	prmprv(ETHER_LOC-2,2,"Ethernet Address");
	prmprv(MMAC_LOC+4,2,"MMAC");
	}

int get_output_type()
	{
	int done, softcopy, ch;

	done=FALSE;
	while (!done)
		{
		printf("\n\n\nPrint to (s)creen or (p)rinter?");
		ch=getchar();	
		softcopy=FALSE;
		switch(ch)
			{
			case 's':
				softcopy=TRUE;
				done=TRUE;
				break;
			case 'p':
				softcopy=FALSE;
				done=TRUE;
				break;
			default:
				printf("\nEnter again 's' or 'p'");
				done=FALSE;
			}
		}
		return(softcopy);
	}
	
print_to_screen(struct lancard_s lc_entry[], int size)
	{
	int i, line;

	print_screen_heading();

	line=3;
	for (i=0;i<size;i++)
		{
		if (line>21)
			{
			prtmsg(0,22,"Hit return for next page");
			print_screen_heading();
			line=3;
			}
		prmp(NODE_LOC,line,lc_entry[i].node_id);
		prmp(SUBNET_LOC,line,lc_entry[i].subnet);
		prmp(IP_LOC,line,lc_entry[i].tcpip);
		prmp(ETHER_LOC,line,lc_entry[i].ether);
		prmp(MMAC_LOC,line,lc_entry[i].mmac);
		line++;
		}
	}

print_to_printer(struct lancard_s lc_entry[], int size)
	{
	int i, line, fd;

	/* get file desc. for the printer */
	if ( (fd = oprf()) == -1)
		{
		printf("lansort: pipe to lpr could not be created!\n");
		exit(-1);
		}

	oblank();	/* blank the printer buffer */

	/* print printer heading */
	flush(fd);
	flush(fd);
   prstr("(8U(s0p16.67h8.5v0s0b0T&l0o");
	prstr("X                                   Lancard sorted by IP Address",22+PAGE_OFF);
	flush(fd);
	flush(fd);
	prstr("Node ID",NODE_LOC+1+PAGE_OFF);
	prstr("Subnet",SUBNET_LOC-1+PAGE_OFF);
	prstr("IP Address",IP_LOC+PAGE_OFF);
	prstr("Ethernet Address",ETHER_LOC-1+PAGE_OFF);
	prstr("MMAC",MMAC_LOC+5+PAGE_OFF);
	flush(fd);

	line=3;
	for (i=0;i<size;i++)
		{
		if (line>55)
			{
			write(fd,"\f",1); /* send a form feed to the printer */
			line=3;
			/* print printer heading */
			flush(fd);
			flush(fd);
			prstr("Lancard sorted by IP Address",22+PAGE_OFF);
			flush(fd);
			flush(fd);
			prstr("Node ID",NODE_LOC+1+PAGE_OFF);
			prstr("Subnet",SUBNET_LOC-1+PAGE_OFF);
			prstr("IP Address",IP_LOC+PAGE_OFF);
			prstr("Ethernet Address",ETHER_LOC-1+PAGE_OFF);
			prstr("MMAC",MMAC_LOC+5+PAGE_OFF);
			flush(fd);
			}
		prstr(lc_entry[i].node_id,NODE_LOC+1+PAGE_OFF);
		prstr(lc_entry[i].subnet,SUBNET_LOC+1+PAGE_OFF);
		prstr(lc_entry[i].tcpip,IP_LOC+1+PAGE_OFF);
		prstr(lc_entry[i].ether,ETHER_LOC+1+PAGE_OFF);
		prstr(lc_entry[i].mmac,MMAC_LOC+1+PAGE_OFF);
		flush(fd);
		line++;
		}
	}

main()
	{
	long count;					/* number of records in the selection file */
	int rc;						/* return code from Unify functions */
	int i, size, softcopy;
	int key[MAXREC];			/* sort key for IP addresses */
	char filename[20];		/* name of selection file */
	struct lancard_s lc_info;	/* holds lancard search criteria */
	struct lancard_s lc_entry[MAXREC]; /* holds current records for sorting */
	SELFILE *ss, *opensf();	/* selection file declarations */

	/* get search criteria from user */
	get_search_criteria(&lc_info); 
/* debug 
	printf("Search criteria: |%s|%s|%s|%s|%s|\n",lc_info.node_id,lc_info.subnet,
				lc_info.tcpip,lc_info.ether,lc_info.mmac);
*/
	/* sets up search criteria for lancard table */
	rc=entsitm(ln_ether,lc_info.ether,(char *) 0,EQ); 
	if (rc == -1)
		fprintf(stderr,"lansort: invalid search field\n");
	rc=entsitm(ln_tcpip,lc_info.tcpip,(char *) 0,EQ);
	if (rc == -1)
		fprintf(stderr,"lansort: invalid search field\n");
	rc=entsitm(ln_sbnet,lc_info.subnet,(char *) 0,EQ);
	if (rc == -1)
		fprintf(stderr,"lansort: invalid search field\n");
	rc=entsitm(ln_node,lc_info.node_id,(char *) 0,EQ);
	if (rc == -1)
		fprintf(stderr,"lansort: invalid search field\n");
	rc=entsitm(ln_mmac,lc_info.mmac,(char *) 0,EQ);
	if (rc == -1)
		fprintf(stderr,"lansort: invalid search field\n");

	/* output to printer or screen? */
	softcopy=get_output_type();

	/* use process ID for selection filename */
	sprintf(filename,"/tmp/lansort%05d",getpid());

	rc = unisel (filename,lancard,&count);	/* create selection file */
	switch(rc)
		{
		case -3:
			fprintf(stderr,"lansort: selection file cannot be created\n");
			exit(ERR);
		case -2:
			fprintf(stderr,"lansort: match item has the wrong selection file name\n");
			exit(ERR);
		case -1:
			fprintf(stderr,"lansort: No records found.\n");
			exit(CLEAN);
		}

	ss = opensf(filename);		/* open selection file */
	switch(errno)
		{
		case 101:
			fprintf(stderr,"lansort: selection file could not be opened\n");
			exit(ERR);
		case 102:
			fprintf(stderr,"lansort: selection file in not in correct format\n");
			exit(ERR);
		case 103:
			fprintf(stderr,"lansort: max amount of selections files opened has been reached\n");
			exit(ERR);
		case 104:
			fprintf(stderr,"lansort: an error due to sbrk\n");
			exit(ERR);
		}

	frstsel(ss);									/* make first record current */
	if (rc == -1)
		{
		fprintf(stderr,"lansort: no records found in selection file.\n");
		exit(ERR);
		}

	printf("\n");
	/* get fields from the selected records */
	i=0;
	do
		{
		gfield(ln_ether,lc_entry[i].ether);
		gfield(ln_tcpip,lc_entry[i].tcpip);
		lc_entry[i].key=build_key(lc_entry[i].tcpip);
		gfield(ln_sbnet,lc_entry[i].subnet);
		gfield(ln_node,lc_entry[i].node_id);
		gfield(ln_mmac,lc_entry[i].mmac);
/* debug
		printf("finding: |%s|%s|%s|%s|%s|%d\n",lc_entry[i].node_id,
			lc_entry[i].subnet,lc_entry[i].tcpip, lc_entry[i].ether, 
			lc_entry[i].mmac,lc_entry[i].key);
*/
		i++;
		}
	while (nextsel(ss) == 1);

	size=i;
	/* sort the records according to the IP addresses key that was built */
	qsort((char *)lc_entry,size,sizeof(struct lancard_s),keycompare);

	/* output to the user selected device */
	if (softcopy)
		print_to_screen(lc_entry,size);
	else
		print_to_printer(lc_entry,size);

	printf("\n\n\n");

	closesf(ss);			/* close selection file */
	unlink(filename);		/* delete selection file from /tmp */
	setcook();				/* return to normal terminal mode */
	exit(CLEAN);
	}
