/*************************************************************************
 *
 *	prob.c
 *
 *	This program reads the records in the problem table of the
 *      database with a pr_stat of 'O'.  The escalation levels of these
 *      problems are monitored and possible changed.
 *
 *	Return codes:
 *	   1:  Error reading or opening database, btree, or index files.
 *	   2:  Error updating fields.
 *
 *	Original Author:  Bill Tsai
 *	Date:             January 1990
 *
 *      Updated by:       John Stewart
 *                        Changed the program so that the escalation
 *                        level is not changed, but instead, the
 *                        priority is changed.
 ************************************************************************
 *
 *     Bugs/To Be Completed:
 *         + Bugs:
 *              - the strange segmentation error has to go.     :[done?]
 *              - Smile face in trouble ticket can go.          :[done]
 *         + Clarify:
 *		- that thing about esc date counting from open
 *                date.                                         :[done]
 *              - when to print the report - NEW and CRITICAL.  :[done]
 *         + Error handling for when:
 *              - error handling for when pfield is not successful.
 *                (Especially in the case when pfield doesn't
 *                 work and the program will simply keep
 *                 printing.                                    :[done]
 *              - printer not found.
 *         + Additions:
 *              - Serial number for trouble tickets and logs in case
 *                they have to be traced.
 *
 ************************************************************************
 * Modifications:
 *
 *      6/29/90 - Convert to UNIX
 *                Change file destinations
 *                Change so write to file then system to lp
 *      7/90    - Added signal trapping for Segmentation and Bus errors
 *      2/7/91  - Changed so it does not print a banner page
 *                Changed so it only prints the trouble tickets for
 *                   northern resp on the trouble printer. 
 *      3/23/91 - Major logic change
 *                Original comments (above) were changed to reflect the
 *                   current algorithm that the program follows
 ************************************************************************/

$include "/usr/db/ACL4/db/file.h"
#include "/usr/db/ACL4/include/btincl.h"
#include <stdio.h>
#include <time.h>
#include <sys/types.h>
#include <signal.h>

#define PIDFILE			"/usr/db/ACL4/src/prob/prob.pidfile"
#define CRITICAL_LEVEL	9
#define DEFAULTPRINTER	"ncctic"

#define NONE				-2
#define SEG					1
#define BUS					2

#define MAIL				"mail -s 'prob Terminated'"
#define RECEIVERS			"john grepps boyles yearsley"

#define SEGV_MESSAGE		"\nThe signal that caused the termination was:\n\n     SIGSEGV\n\nThis should NOT be a result of reconfiguring the database.\n"
#define BUS_MESSAGE		"\nThe signal that caused the termination was:\n\n     SIGBUS\n\nThis should NOT be a result of reconfiguring the database.\n"
#define NONE_MESSAGE		"\nThe reason why prob stopped was because of an error returned by a UNIFY library function\n"

#define PROGLOG			"/usr/local/logs/problog"
#define RPTTMP				"/usr/local/logs/RPTTMP"

#define BOLD				printf(printer,"E");
#define UNBOLD				printf(printer,"F");

#define ERROR				1
#define OKAY				0

#define TRUE				1
#define FALSE				0

#define SLEEPING			0
#define AWAKE				1

#define PAST				0
#define NOTYET				1

#define SEC_IN_YEAR		(365 * 24 * 60 * 60)
#define SEC_IN_DAY		(24 * 60 * 60)
#define SEC_IN_HOUR		(60 * 60)
#define SEC_IN_MINUTE	(60)

#define COURIER			fprintf(printer, "\(10U\(s0p10.00h12.0v0s0b3T\$l0o");

#define UNCOURIER			'(8U(s0p16.67h8.5v0s0b0T&l0o' */

/************************************************************************
 *	Global variables
 ************************************************************************/

int		btree,
			rc,
			status,
			sltime;
char		*btptrs[2];
char		xcode;
BTIDS		*probbts,
			*fldidxd();
int		count=1;
int		total_printers=0;
struct	tm *t;
FILE		*printer;
int		process_status;
FILE		*proglog;
FILE		*pidfile;

/************************************************************************
 *   Array for escalation levels (0-CRITICAL_LEVEL)
 ************************************************************************/

struct esc_level_struc
	{
	int esc_days;
	int esc_hours;
	} esc_level_array[20];

struct ptr_struct
	{
	char trouble_prt[9];
	char prt_name[22];
	} prt_array[30];

/************************************************************************
 *   Program flags
 ************************************************************************/

int CRITICAL=0,    /* is prob critical? */
    NEW=0,         /* is new prob? */
    CHANGED=0,     /* is tprt/pec changed? */
    BUMPED=0;      /* bumped */

/************************************************************************
 *   Variables for fields in prob table
 ************************************************************************/

char problem_status;

int	prob_num=0,				/* problem number										*/
		esc_level=0,			/* escaluation level									*/
		saved_level=0,			/* saved esc level									*/
		prio_level=0,			/* priority level										*/
		saved_prio=0,			/* save prio level									*/
		PRINTED=0;				/* printed on superv's prt?						*/

char	trouble_ptr[9],		/* trouble printer									*/
		saved_prt[9];			/* saved trouble prt									*/

int	next_esc_year,			/* next esc year										*/
		next_esc_month,		/* next esc month										*/
		next_esc_day,			/* next esc day										*/
		next_esc_hour,			/* next esc time										*/
		next_esc_minute;		/* next esc minute									*/

int	opened_year,			/* opened year											*/
		opened_month,			/* opened month										*/
		opened_day,				/* opened day											*/
		opened_hour,			/* opened time											*/
		opened_minute;			/* opened minute										*/

/************************************************************************
 * The following variables are for printing reports.  They need not be
 * defined globally, I couldn't get them to work otherwise.
 ***********************************************************************/

char	prob_node[27],
		prob_contact[23],
		prob_site[10],
		prob_id[6],
		prob_model[18],
		prob_vendor[17],
		prob_vphone[18],
		prob_phone[19],
		prob_serialno[27],
		prob_location[33],
		prob_comment1[80],
		prob_comment2[80],
		prob_comment3[80],
		prob_comment4[80];

char 	pres_res1[80],
		pres_res2[80],
		pres_res3[80],
		pres_res4[80],
		pres_res5[80],
		pres_tech[10],
		pres_ticket[22];

short	numres;

/************************************************************************
 *   Function prototypes
 ***********************************************************************/

int  all_space(char *);
void beep(void);
int  bump_esc(void);
void calc_esc_dt(void);
int  catcher(void);
void chk_modify(void);
int  chk_time(void);
void draw_a_line(void);
int  found_prob(void);
void open_esc_table(void);
void open_files(void);
void open_printer(char *);
void open_prt_table(void);
void pcol(char *);
void print_date(char [], int, int, int, int, int);
void print_dt(long);
void print_res(int);
void print_report(void);
void print_stat(void);
void print_sup_report(void);
void print_tp_report(void);
void process_date(int, int *, int *, int *);
int  record_printed(void);
int  save_fld(void);
void sendmail(int);
int  update_next_esc_dt(void);
void wait_key(void);
void write_log(char *);

/*************************************************************************
 *
 *	main()
 *
 ************************************************************************/

int main(argc, argv)
int argc;
char *argv[];
	{
	#ifdef DEBUG
	printf("Starting main program block!\n");
	#endif

	signal (SIGBUS, catcher);			/* bus error								*/
	signal (SIGSEGV, catcher);			/* segmentation fault					*/

	t = (struct tm *) malloc (sizeof (struct tm));

	open_esc_table();
	open_prt_table();

	sltime = 60;					/* default wait_key time						*/
	if (argc > 1)
		{
      sltime = atoi(argv[1]);
		}
   if (sltime == 0)
		sltime = 60;
   
   while (TRUE)
		{
		open_files();

		xcode = 'O';				/* find 'O'pen problems							*/
		btptrs[0] = (char *) &xcode;
		btptrs[1] = (char *) 0;

		rc=btsrch(btree,btptrs);			/* search a b-tree index			*/

		/*******************************************************************
		 * btsrch returns the following values:
		 * 0 or 1 if found
		 * -6 if there is no records selected
		 * anything other then those is error
		 ******************************************************************/

		if ((rc == -6) || (rc >= 0))		/* no error								*/
			{
			if(rc >= 0) 				/* record found								*/
				{
				do
					{
					found_prob();
					if (problem_status == 'O')
						{
						#ifdef DEBUG
						printf("Problem found\n");
						#endif
						wait_key();
						#ifdef DEBUG
						printf("-----------------------------------------------\n");
						printf("count = %d\n", count++);
						#endif

						/* print_sup_report(); */

						if (!NEW)			/* old problems							*/
							{
							print_stat();
							#ifdef DEBUG
							printf("This is a old problem.\n");
							#endif

							chk_modify();

							if (CHANGED)
								{
								#ifdef DEBUG
								printf("Level/prt is changed.\n");
								printf("Now I should jump to A1.\n");
								printf("Changed = %d NEW = %d.\n", CHANGED, NEW);
								#endif

								wait_key();
								}
							else				/* not changed								*/
								{
								#ifdef DEBUG
								printf("Level/prt not changed.\n");
								printf("Checking if esc date/time past now.\n");
								#endif

								if ((chk_time() == NOTYET) || (CRITICAL))
									{				/* No need to update esc lev		*/
									#ifdef DEBUG
									printf("Before esc time or already critical.\n");
									printf("Now should go to entry point A2.\n");
									#endif
									}
								else			/* update esc level						*/
									{
									if (bump_esc() == ERROR)
										{
										#ifdef DEBUG
										printf("Error updating esc level.\n");
										#endif
										}

									print_sup_report();
									#ifdef DEBUG
									printf("Also calling save_fld.\n");
									#endif
									if (save_fld() == ERROR)
										{
										#ifdef DEBUG
										printf("save_fld's pfield error.\n");
										#endif

										if ((BUMPED) && (!CRITICAL))
											{
											#ifdef DEBUG
											printf("Finding time to next esc.\n");

											printf("Time to next escalation is ");
											printf("%d days and %d hours.\n",
												esc_level_array[esc_level].esc_days,
												esc_level_array[esc_level].esc_hours);

											printf("Adding (time to next esc) to cur time.\n");
											#endif

											calc_esc_dt();		/* calc nxt esc d/t */

											print_date("Calculated next esc date = ",
												next_esc_year, next_esc_month, next_esc_day,
												next_esc_hour, next_esc_minute);

											if (update_next_esc_dt() == ERROR)
												{
												#ifdef DEBUG
												printf("Error updating esc date 1.\n");
												#endif
												}
											} /* if ((BUMPED) && (!CRITICAL)) */
										}
									}  /* else */
								}

		/* The following line is the A1 entry point in
		   float chart #ISNOPFCN1939051289 */

							#ifdef DEBUG
							printf("this is entry point A1.  How're you?\n");
							#endif
							wait_key();

							if ((NEW) ||				/* not OLD - NEW */
								((!NEW) && (CHANGED)) ||		/* Reentry 4 changed */
								((BUMPED) && (CRITICAL)))		/* bumped to critical */
								{					/* problems */
								print_stat();
								#ifdef DEBUG
								printf("This is a new, or modified, problem.\n");
								#endif

								if (CRITICAL)
									{
									#ifdef DEBUG
									printf("This is a critical problem.\n");

									printf("Checking PRINTED = %d.\n", PRINTED);
									#endif
									wait_key();

									if (PRINTED)
										{
										#ifdef DEBUG
										printf("The ticket has been printed on supervisor's");
										printf(" printer so won't print again.\n");
										#endif
										}
									else			/* hasn't print in sup */
										{
										/* here I try to record that report had	*/
										/* been printed before do the actual		*/
										/* printing because if pfield in				*/
										/* record_printed() isn't successful the	*/
										/* report will be printed out again and	*/
										/* again.											*/

										if (record_printed() != ERROR)
											print_sup_report();
										}

									#ifdef DEBUG
									printf("Printing a report on trouble printer.\n");
									#endif

									if (record_printed() != ERROR)
										{
										#ifdef DEBUG
										printf("print_tp_report (I) ??\n");
										#endif
										print_tp_report();
										#ifdef DEBUG
										printf("Does it get this far (I) ??\n");
										#endif
										}
									}
								else				/* not CRITICAL */
									{
									#ifdef DEBUG
									printf("This is not a critical problem.\n");

									printf("Finding time to next esc level.\n");
									printf("Time to next escalation is ");
									printf("%d days and %d hours.\n",
										esc_level_array[esc_level].esc_days,
										esc_level_array[esc_level].esc_hours);
									#endif

									wait_key();

									if (CHANGED)
										{
										#ifdef DEBUG
										printf("The level/prt is changed.\n");
										#endif

										/* To handle the situation where a probem	*/
										/* is changed, in Bruce's flow chart he	*/
										/* wanted to add time to next esc to the	*/
										/* opened d/t.  It doesn't make sense		*/
										/* that way, and he doesn't remember way	*/
										/* he thought of the first way.				*/

										/* For the time being, this CHANGED part	*/
										/* is implimented exactly as !CHANGED, as	*/
										/* it add time to next esc to CURRENT d/t */

										#ifdef DEBUG
										printf("Adding (time to next esc) to\n");
										printf("pr_open_date, time.\n");
										printf("Update pr_esc_date, time w/ new time.\n");
										#endif

										calc_esc_dt();			/* calc nxt esc d/t */

										print_date("Calculated next esc date = ",
											next_esc_year, next_esc_month, next_esc_day,
											next_esc_hour, next_esc_minute);

										if (update_next_esc_dt() == ERROR)
											{
											#ifdef DEBUG
											printf("Error updating esc date 2.\n");
											#endif
											}
										}
									else			/* !CHANGED - brand new prob */
										{
										#ifdef DEBUG
										printf("The level/prt is not changed.\n");
										printf("Adding (time to next esc) to cur time.\n");
										#endif

										calc_esc_dt();			/* calc nxt esc d/t */

										print_date("Calculated next esc date = ",
											next_esc_year, next_esc_month, next_esc_day,
											next_esc_hour, next_esc_minute);

										if (update_next_esc_dt() != ERROR)
											{
											#ifdef DEBUG
											printf("print_tp_report (II) ??\n");
											#endif
											print_tp_report();
											#ifdef DEBUG
											printf("Does it get this far (II) ??\n");
											#endif
											}
										else
											{
											#ifdef DEBUG
											printf("Error updating esc date 3.\n");
											#endif
											}
										}
									}
								}

							/* following statements are used to save pr_tp	*/
							/* and pr_pec and print trouble ticket on			*/
							/* trouble printer.  It will be executed if one	*/
							/* of the following conditions are met:			*/
							/*																*/
							/* 1.  (NEW) - It's a new problem, need to		*/
							/*					print out ticket						*/
							/* 2.  (CHANGED) && (!NEW) - It's an old			*/
							/*					problem, but pec or tp is			*/
							/*					changed manually by user, so we	*/
							/*					need to print out a trouble		*/
							/*					ticket									*/
							/* 3.  (BUMPED) + if esc level is bumped up		*/
							/*					one, it will either reach			*/
							/*					CRITICAL or not
							/*					a.	(CRITICAL) - Will go through	*/
							/*						as a NEW problem with flags	*/
							/*						(BUMPED) && (CRITICAL) set.	*/
							/*						It needs a trouble ticket		*/
							/*						printed out.
							/*					b.	(!CRITICAL) - Will update		*/
							/*						esc date and time, then			*/
							/*						should print out new trouble	*/
							/*						ticket in trouble prt.			*/
							/*																*/
							/* Note: This part is in the very bottom middle	*/
							/*			of the float chart #ISNOPFCN1939051289 */

							#ifdef DEBUG
							printf("Does it get this far (III) ??\n");
							#endif

							if ((NEW) || ((!NEW) && (CHANGED)) || (BUMPED))
								{
								#ifdef DEBUG
								printf("=== Ready to update.\n");
								#endif
								wait_key();

								#ifdef DEBUG
								printf("Updating -\n");
								printf("   pr_code = pr_pec\n");
								printf("   pr_prt = pr_tp\n");
								#endif

								if (save_fld() == ERROR)
									{
									#ifdef DEBUG
									printf("save_fld's pfield error.\n");
									#endif

									/* Originally following this comment	*/
									/* there will be statements to print	*/
									/* out a report on trouble printer,		*/
									/* but Bruce decided that there will	*/
									/* only be a trouble ticket when the	*/
									/* problem esc level == 7					*/
									}
  								}
							}
						while (btnext(btree) == 0);
						}
					else					/* rc = -6 */
						{					/* no problems open */
						}
  					}
				else					/* error reading btree */
					{
					fprintf(stderr,"Btsrch failed");
					fprintf(stderr,"rc = %d\n",rc);
					setcook();
					write_log("Btsrch failed on or around line 503");
					sendmail(NONE);
					exit(1);
					}

				if (rc = closbt(btree))			/* try to close btree */
					{
					fprintf(stderr,"closbt failed");
					fprintf(stderr,"rc = %d\n",rc);
					setcook();
					write_log("Closebt failed on or around line 513");
					sendmail(NONE);
					exit(1);
					}
				#ifdef DEBUG
				printf("sleeping\n");
				#endif
				sleep(sltime);				/* wait for a minute */
				#ifdef DEBUG
				printf("-------------------\n");
				#endif
				count = 1;
				}
			}

/*************************************************************************
 *
 *	all_space(char *)
 *
 *	This function returns:
 *	   - TRUE  if the parameter string is all spaces, or
 *	   - FALSE if string is not all spaces.
 *	This function is called by print_report to check whether
 *	or not a comment is to be printed (if a comment is full
 *	of spaces, it shouldn't be printed as it will only waste
 *	space).
 *
 ************************************************************************/

int all_space(string)
char *string;
	{
	int i;

	for (i=0; i<strlen(string); i++)
		{
		if (string[i] != ' ')
			return(FALSE);
		}

	return (TRUE);
	}

/*************************************************************************
 *
 *	beep()
 *
 *************************************************************************/

void beep()
	{
	wait_key();
	#ifdef DEBUG
	printf("\7pfield called.\n");
	#endif
	}

/*************************************************************************
 *
 *	int bump_esc()
 *
 *	This function bumps the escalation level of current record
 *	up one, including update the record in file.db.
 *
 *************************************************************************/

int bump_esc()
	{
	#ifdef DEBUG
   printf("Now I am going to bump the esc level up one.\n");
   printf("As I understand, this problem has a esc level ");
   printf("of %d now.\nI am going to make it %d.\n", esc_level, esc_level+1);
	#endif

	if (++prio_level == CRITICAL_LEVEL)		/* reset CRITICAL flag */
		CRITICAL = 1;				/* after adjustment */
	else
		CRITICAL = 0;

	BUMPED = 1;

	beep();
	if (pfield (pr_prio, &prio_level) != 0)
		{
		#ifdef DEBUG
		printf("bump_esc: pfield fucked up.\n");
		#endif
      return(ERROR);
		}
	else
		{
		#ifdef DEBUG
		printf("bump_esc: pfield succeed - priority level successfully updated to %d.\n", prio_level);
		#endif
		return(OKAY);
		}
	}

/*************************************************************************
 *
 *	calc_esc_dt()
 *
 *	This procedure calculates the next escaluation date/time
 *	based on:
 *	   o  Escaluation level
 *	   o  Current date/time
 *
 *************************************************************************/

void calc_esc_dt()
	{
   long	clock,					/* time in sec */
			delay,					/* time to next esc in sec */
			curr_clock;
   int	NOT_DURING_WORK=0,			/* flag if not 8 to 5 */
			WEEKEND=0;				/* esc d fall on weekend */
   int	curr_hour,
			curr_minute,
			curr_wday;				/* sunday = 0 */
   int	add_hour,
			add_minute;
   int	curr_wday_score,
			wday_score;

	#ifdef DEBUG
	printf("This is calc_esc_dt... the ultimate procedure!\n");
	#endif

	clock = time((long *) 0);
	t = localtime(&clock);			/* t = current d/t */

	curr_clock = clock;				/* save curr d/t */
	curr_hour = t->tm_hour;
	curr_minute = t->tm_min;
	curr_wday = t->tm_wday;

	delay = esc_level_array[esc_level].esc_days * SEC_IN_DAY +
		esc_level_array[esc_level].esc_hours * SEC_IN_HOUR;

	clock += delay;				/* get 1st estimation */

	print_dt(clock);

	t = localtime(&clock);			/* conv to indiv quantity */

	#ifdef DEBUG
	printf("  which is %d/%d/%d %.2d:%.2d:00\n\n",
		t->tm_year, t->tm_mon+1, t->tm_mday, t->tm_hour, t->tm_min);
	#endif

	if ((t->tm_hour >= 17) || (t->tm_hour < 8))
		NOT_DURING_WORK = 1;
	else
		NOT_DURING_WORK = 0;

   /* following statements check to see if the problem is opened		*/
	/* during non-working hours. (Open off-hours) If so, the problem	*/
	/* should count as being opened first thing next morning.  If		*/
	/* problem is opened on weekend, it counts as opened on monday		*/

	if ((curr_hour >= 17) || (curr_hour < 8) || (curr_wday == 6) ||
		(curr_wday == 0))
		{
		#ifdef DEBUG
		printf("*** curr_hour = %d:%d\n", curr_hour, curr_minute);
		printf("*** Problem opened during off-hours.\n");
		printf("Therefore the it counts as being opened at first thing ");
		printf("in the morning (or monday morning if wkends.)\n");
		#endif

		if ((curr_hour <= 23) && (curr_hour >= 17))	/* open late at night */
			{
			add_hour = 24 - curr_hour;
			add_minute  = 0 - curr_minute;
			if (add_minute < 0)
				{
				add_minute += 60;
				add_hour--;
				}
			add_hour += 8;

			clock = curr_clock + add_hour * SEC_IN_HOUR + add_minute * SEC_IN_MINUTE;

			print_dt(clock);

			clock += esc_level_array[esc_level].esc_days * SEC_IN_DAY +
				esc_level_array[esc_level].esc_hours * SEC_IN_HOUR;

			print_dt(clock);
			}
		else						/* open early in morn */
			{
			add_hour = 8 - curr_hour;
			add_minute  = 0 - curr_minute;
			if (add_minute < 0)
				{
				add_minute += 60;
				add_hour--;
				}

			clock = curr_clock + add_hour * SEC_IN_HOUR + add_minute * SEC_IN_MINUTE;

			print_dt(clock);
			clock += esc_level_array[esc_level].esc_days * SEC_IN_DAY +
				esc_level_array[esc_level].esc_hours * SEC_IN_HOUR;

			print_dt(clock);
			}

		/* following two lines add a day for saturday for weekend.  In	*/
		/* case of sunday, above part will add it to next day, which	*/
		/* is monday, so no need to add another day.							*/

		if (curr_wday == 6)
			clock += SEC_IN_DAY;
		}
	/* The follwoing lines are for problems that are opened during		*/
	/* work time but the calculated esc d/t falls on nights.  These	*/
	/* problems had to be wrapped to morning; ie, the esc d/t should	*/
	/* stop at 5:00pm and starts again at 8:00am (5:00pm to 8:00am		*/
	/* should be skipped.)  Just add 15 hours (5 - 8) will solve this	*/
	/* problem.																			*/

	else if (NOT_DURING_WORK)
	{
		#ifdef DEBUG
		printf("*** curr_hour = %.2d:%.2d\n", curr_hour, curr_minute);
      printf("*** This esc d/t falls at night time.\n");
		#endif

		clock += 15 * SEC_IN_HOUR;		/* add 15 hours to wrap */
	}

	print_dt(clock);

   /* the following line take the value of clock (= esc d/t) and check	*/
	/* and check to see if it falls on a weekend (saturday or sunday).	*/
	/* If so, add two or one day(s) accordingly.									*/

	t = localtime(&clock);

	if ((t->tm_wday == 0) || (t->tm_wday == 6))
	{
		#ifdef DEBUG
		printf("Esc date falls on a weekend.\nBring it to monday.\n");
		#endif
		if (t->tm_wday == 0)			/* sunday */
			clock += SEC_IN_DAY * 2;			/* add a day */
		else					/* saturday */
			clock += SEC_IN_DAY * 2;		/* add two days */
		#ifdef DEBUG
		printf("^^^");
		#endif
		print_dt(clock);
		WEEKEND = 1;
	}

   /* the following section with curr_* stuff is to see if the esc
      delay time cross between friday and monday, ie, if the esc d/t
      and open d/t is on the same week.  For example,
      if a esc level 0 problem is opened on Friday during work,
      this will create a 3 day esc time.  Adding 3 days to Friday
      makes Monday, which is a legitimate esc date, but the actual
      esc date should be on wednesday (skipping weekend.)
      The task of detecting this situation is accomplished by giving
      each day of the week a number (score) and scores look like
      the this:
		 wday: Sun Mon Tue Wed Thu Fri Sat
		score:  1   2   3   4   5   6   0  (<- desired)
	   t->tm_wday:  0   1   2   3   4   5   6  (<- given)

      The program first calculate the score based on the t->tm_wday
      value for the est._esc_d/t and open_d/t.  If the score
      for esc wday is less then (<) open wday score, there is a weekend
      in between.
      It works because if Friday has the highest score (6).  Once
      it reaches Saturday the score returns to 0.  If open date is
      Friday (score 6) and esc date is Monday (score 1), 1 < 6,
      a weekend is between day 1 (friday) and day 2 (monday).  An
      counter example: If problem is opened on Tuesday and esc date
      is friday, 6 !< 3, so the two days are on the same week. */
      /* Thanks to Dennis for the idea */

	else
		{
		curr_wday_score = curr_wday + 1;
		if (curr_wday_score == 7)
			curr_wday_score = 0;

		wday_score = t->tm_wday + 1;
		if (wday_score == 7)
			wday_score = 0;

		if (wday_score < curr_wday_score)
			{
			#ifdef DEBUG
			printf("A weekend is in between.\n");
			#endif
         clock += SEC_IN_DAY * 2;
			}
		}

	print_dt(clock);

	t = localtime(&clock);			/* conv time in sec to */
						/* individual quantities */
	next_esc_year  = t->tm_year;
	next_esc_month = t->tm_mon+1;
	next_esc_day   = t->tm_mday;
	next_esc_hour  = t->tm_hour;
	next_esc_minute= t->tm_min;
	}

/*************************************************************************
 *
 *	catcher
 *
 *	This function is called when an exception occurs during
 *	the execution of this program.  It is set up to catch
 *	Bus error and Segmentation Fault by the first two lines
 *	of main procedure.
 *
 *	signal (SIGBUS, catcher);		(* bus error *)
 *	signal (SIGSEGV, catcher);		(* segmentation fault *)
 *
 *************************************************************************/

int catcher(sig)
int sig;
	{
	switch (sig)
		{
		case SIGSEGV:
			write_log("Signal Caught!\nAbnormal program termination due to sigmation fault or bus error.\nPossible if database reconfigured while running.");
			sendmail(SEG);
			exit(1);
			break;
		case SIGBUS:
			write_log("Signal Caught!\nAbnormal program termination due to sigmation fault or bus error.\nPossible if database reconfigured while running.");
			sendmail(BUS);
			exit(2);
			break;
		}
	}

/*************************************************************************
 *
 *	chk_modify()
 *
 *	This function checks to see if either printer or esc level
 *	is manually changed by the user.
 *
 *************************************************************************/

void chk_modify()
	{
	#ifdef DEBUG
	printf("Checking to see if pr_tp or pr_pec is changed by users.\n");
	#endif

	if ((prio_level == saved_prio) && (esc_level == saved_level) &&
		(strncmp(trouble_ptr, saved_prt, 7) == 0))
		{
		CHANGED = FALSE;
		#ifdef DEBUG
		printf("Not Changed!!\n");
		#endif
		}
	else
		{
		CHANGED = TRUE;
		#ifdef DEBUG
		printf("Changed!!\n");
		#endif
		}
	wait_key();
	}

/*************************************************************************
 *
 *	int chk_time()
 *
 *	This function check the 
 *
 *************************************************************************/

int chk_time()
	{
	long clock;
	char str[30], str2[30];

	clock = time((long *) 0);
	t = localtime(&clock);

	sprintf(str, "%.2d%.2d%.2d%.2d%.2d%.2d",
		t->tm_year, t->tm_mon+1, t->tm_mday,
		t->tm_hour, t->tm_min,   t->tm_sec);

	#ifdef DEBUG
	printf("\nCurrent Time: %s\n", str);
	#endif

	sprintf(str2, "%.2d%.2d%.2d%.2d%.2d%.2d%",
		next_esc_year, next_esc_month, next_esc_day,
		next_esc_hour, next_esc_minute, 0);

	#ifdef DEBUG
	printf(  "Next Esc T:   %s\n", str2);
	#endif

	wait_key();

	if (strcmp(str, str2) < 0)
		{
		#ifdef DEBUG
		printf("Not yet.!\n");
		#endif
		return(NOTYET);				/* Not esc d/t yet */
		}
	else
		{
		#ifdef DEBUG
		printf("Already past!  Time to update!\n");
		#endif
		return(PAST);				/* Already past esc d/t */
		}
	}

/*************************************************************************
 *
 *	draw_a_line
 *
 *************************************************************************/

void draw_a_line()
	{
	int i;

	for (i=0; i<80; i++)
		fprintf(printer, "-");
	fprintf(printer, "\n");
	}

/*************************************************************************
 *
 *	int found_prob()
 *
 *	This function is called by main() when an open problem
 *	is found in the btree.  It does:
 *
 *	   o get fields into global variables
 *	   o call process_date(..) to turn raw date into real date
 *	   o set CRITICAL flag
 *	   o set NEW flag
 *	   o reset BUMPED flag
 *	   o return
 *
 *************************************************************************/

int found_prob()
	{
   short	next_esc_date_buf=0,			/* date - HAS to be short */
			opened_date_buf=0;

	gfield(pr_stat, &problem_status);
	gfield(pr_num,  &prob_num);
	gfield(pr_pec,  &esc_level);
	gfield(pr_prio, &prio_level);
	gfield(pr_tp,    trouble_ptr);

	gfield(pr_ed,   &next_esc_date_buf);
	gfield(pr_et,   &next_esc_hour);

	gfield(pr_date, &opened_date_buf);
	gfield(pr_time, &opened_hour);

	gfield(pr_code, &saved_level);
	gfield(pr_old_p,&saved_prio);
	gfield(pr_prt,   saved_prt);
	gfield(pr_chg,  &PRINTED);

	next_esc_minute = next_esc_hour - ((int) next_esc_hour / 100) * 100;
	next_esc_hour = (int) next_esc_hour / 100;

	opened_minute = opened_hour - ((int) opened_hour / 100) * 100;
	opened_hour = (int) opened_hour / 100;

	process_date(next_esc_date_buf, &next_esc_month, &next_esc_day,
		&next_esc_year);

	process_date(opened_date_buf, &opened_month, &opened_day, &opened_year);

	BUMPED = 0;

	if (prio_level == CRITICAL_LEVEL)
		CRITICAL = TRUE;
	else
		CRITICAL = FALSE;

	if ((next_esc_date_buf != -32768) || (PRINTED))
		NEW = FALSE;
	else
		NEW = TRUE;
	}

/*************************************************************************
 *
 *	open_esc_table()
 *
 *	This procedure opens and reads the escalation table
 *	into an array "esc_level_array"
 *
 *************************************************************************/

void open_esc_table()
	{
   int total_rec = 0;
   short count;

	if (seqacc(ecode, first))				/* p 3-32 */
		{
		fprintf(stderr,"seqacc failed for ecode.\n");
		write_log("seqacc failed for ecode on or around line 966");
		sendmail(NONE);
		exit(1);
		}
	do							/* read into array */
		{						/* of structures */
		gfield(ec_num, &count);
		gfield(ec_days, &esc_level_array[count].esc_days);
		gfield(ec_hours, &esc_level_array[count].esc_hours);
		total_rec++;
		}
	while (!seqacc (ecode, next));
	}

/*************************************************************************
 *
 *	open_files()
 *
 *	this procedure opens all necessary files (btree, data file,
 *	etc) exit on error accordingly.
 *
 *************************************************************************/

void open_files()
	{
	if ((probbts = fldidxd(pr_stat,&status)) == BTIDNUL)
		{
		if (status == IFNBT_R)
			{
			printf("Problem Status is not indexed\n");
			setcook();				/* set term to cooked mode */
			write_log("Problem Status is not indexed on or around line 1002");
			sendmail(NONE);
			exit(1);
			}
		else
			{
			printf("Internal error %d returned by fldidxd.\n", status);
			setcook();				/* set term to cooked mode */
			write_log("Internal error returned by fldidxd on or around line 1010");
			sendmail(NONE);
			exit(1);
			}
		}

	if (probbts[0].btfrank != 0)
		{
		printf("pr_stat must be the first element of the index.");
		setcook();
		write_log("pr_stat must be the first element of the index on or around line 1024");
		sendmail(NONE);
		exit(1);
		}

	btree = probbts[0].btidnum;
	
	if (rc = opnbts(btree))			/* return code = open btree */
		{
		printf("Open btree failed");
		printf("Return Code = %d\n",rc);
		setcook();				/* set term to cooked mode */
		write_log("Open btree failed on or around line 1036");
		sendmail(NONE);
		exit(1);					/* error code */
		}
	}

/*************************************************************************
 *
 *	open_printer(printer_name)
 *
 *	This procedure gets the printer name from initial
 *	(eg. NCC -> ncctic)
 *
 *************************************************************************/

void open_printer(name)
char *name;
	{
	FILE *ptr;					/* printer */
	int i;
	char logstr[100],				/* str that goes into log */
	dest[50],				/* print destinations */
	lpcommand[150];				/* command for 'system()' */

	for (i=0; i<total_printers; i++)
		{
		#ifdef DEBUG
		printf("trouble_prt = \"%s\"\n", prt_array[i].trouble_prt);
      printf("printer nme = \"%s\"\n", prt_array[i].prt_name);
		#endif
		}

	wait_key();

	#ifdef DEBUG
	printf("This is open_printer.  The printer's name is \"%s\"\n", name);
   printf("total_printers = %d\n", total_printers);
	#endif

	for (i=0; ((strncmp(name, prt_array[i].trouble_prt,7) != 0) &&
		(i<total_printers)); i++)
		{
		#ifdef DEBUG
		printf("Comparing: %s & %s -> %d\n", name, prt_array[i].trouble_prt,
			strncmp(name, prt_array[i].trouble_prt,7));
		#endif
		}

	strcpy(dest, prt_array[i].prt_name);

	if (i == total_printers)
		strcpy(dest, DEFAULTPRINTER);		/* ncctic by default */

	#ifdef DEBUG
	printf("After searching... the number is %d\n", i);
   printf("Destination: %s\n", dest);
	#endif

	wait_key();

	if ((printer = fopen(RPTTMP, "w")) == NULL)	/* open temp file */
		{
		printf("Error opening temp report %s\n", RPTTMP);
		write_log("Error temp report on or around line 1091");
		sendmail(NONE);
		exit(1);
		}

	#ifdef DEBUG
	printf("\7Calling print_report!\n");
	#endif

	print_report();

	fclose(printer);

	/* lp options:																			*/
	/*		-o nobanner  - so it doesn't print a banner before every			*/
	/*							ticket														*/
	/*		-s           - silence - so it doesn't say 'ncctic-234...'		*/
	/*		-c           - copy - because we need to delete file				*/
	/*		-d dest      - destination string										*/
	/*		RPTTMP       - the ticket file											*/

	sprintf(lpcommand, "lp -o b -s -c -d %s %s", dest, RPTTMP);
	system(lpcommand);

	#ifdef DEBUG
	printf("report printed.\n");
	#endif

	unlink(RPTTMP);				/* delete file */

	sprintf(logstr, "Trouble Ticket for prob %d is printed to %s (%s).",
	prob_num, name, dest);
	#ifdef DEBUG
	printf("logstr = %s\n", logstr);
	#endif

	write_log(logstr);
	#ifdef DEBUG
	printf("Report printed.\n");
	printf("It's okay here...  right before returning\n");
	#endif
	}

/*************************************************************************
 *
 *	open_prt_table()
 *
 *	This procedure opens and reads the printer id (eg "NCC",
 *	"JKS") and store in "prt_array" that id and the trouble
 *	printer's name (eg "ncclaser", "netopsup", etc.)
 *
 *************************************************************************/

void open_prt_table()
	{
	if (seqacc(tprt, first))				/* p 3-32 */
		{
		fprintf(stder,"seqacc failed for tprt.\n");
		write_log("seqacc failed for tprt on or around line 1134");
		sendmail(NONE);
		exit(1);
		}

	do							/* read into array */
		{							/* of structures */
		gfield(tp_name, prt_array[total_printers].trouble_prt);
		gfield(tp_tty, prt_array[total_printers].prt_name);
		total_printers++;
		}
	while (!seqacc (tprt, next));
	}

/*************************************************************************
 *
 *	pcol(int)
 *
 *	This procedure sends to HPLJ printer a string to set the
 *	horizontal position (column.)
 *
 ************************************************************************/

void pcol(x)
char *x;
	{
	fprintf (printer, "\&a%sC", x);
	}

/*************************************************************************
 *
 *	print_date(char*, int, int, int, int, int)
 *
 *	This procedure takes three integers and prints them out
 *	with a heading.  Mainly used for testing purposes.
 *
 *************************************************************************/

void print_date(mesg, month, day, year, hour, minute)
char mesg[];
int month, day, year;
	{
	#ifdef DEBUG
	printf("%s%d/%d/%d %.2d:%.2d\n", mesg, month, day, year, hour, minute);
	#endif
	}

/*************************************************************************
 *
 *	print_dt()
 *
 *************************************************************************/

void print_dt(clock)
long clock;
	{
	#ifdef DEBUG
	printf("!!! Estimation: %s", ctime(&clock));
	#endif
	}

/*************************************************************************
 *
 *	print_res(int)
 *
 *	This procedure prints out the bottom part of a trouble ticket,
 *	which is the resolution to a problem.  It access resolution
 *	table by opening a b-tree.
 *
 ************************************************************************/

void print_res(prob_num_srch)
int prob_num_srch;
	{
	int btree, rc, status, probnum;
	char *btptrs[2];
	short xcode;
	BTIDS *resbts, *fldidxd();

	int pres_num;

	#ifdef DEBUG
	printf("in proc print_res, arg = %d\n", prob_num_srch);
	#endif

	if ((resbts = fldidxd(pt_num, &status)) == BTIDNUL) 
		{
		if (status == IFNBT_R)
			{
			fprintf(stderr,"Resolution key is not indexed.\n");
			write_log("Resolution key is not indexed for print_res on or around line 1219");
			sendmail(NONE);
			exit(1);
			}
		else
			{
			fprintf(stderr,"Internal error %d returned by fldidxd in print_res.\n", status);
			write_log("Internal error returned by fldidxd in print_res on or around line 1226");
			sendmail(NONE);
			exit(1);
			}
		}

	if (resbts[0].btfrank != 0)
		{
		fprintf(stderr,"pt_key must be the first element of the index in print_res.");
		write_log("pt_key must be the first element of the index on or around line 1235");
		sendmail(NONE);
		exit(1);
		}

	btree = resbts[0].btidnum;
	#ifdef DEBUG
	printf("btree, or resbts[0].btidnum, = %d\n", btree);
	printf("rank, or  resbts[0].btfrank, = %d\n", resbts[0].btfrank);
	#endif
	
	if (rc = opnbts(btree))
		{
		fprintf(stderr,"Open btree failed for print_res.");
		fprintf(stderr,"rc = %d\n",rc);
		write_log("Open btree failed for print_res on or around line 1249");
		sendmail(NONE);
		exit(1);
		}

	xcode = prob_num_srch;
	btptrs[0] = (char *) &xcode;
	btptrs[1] = (char *) 0;

	#ifdef DEBUG
	printf("xcode = %d\n", xcode);
	#endif

	if ((rc=btsrch(btree,btptrs)) < 0)
		{
		fprintf(stderr,"Btsrch failed for print_res.");
		fprintf(stderr,"rc = %d\n",rc);
		write_log("Btsrch failed for print_res on or around line 1264");
		sendmail(NONE);
		exit(1);
		}

	gfield(pr_nr, &numres);
	#ifdef DEBUG
	printf("num of resolutions numres     = %d\n", numres);
	#endif

	if (numres != 0)
		{
		do
			{
			gfield(pt_num, &probnum);
			if (probnum == prob_num_srch)
				{
				#ifdef DEBUG
				printf("Prob num = %d\n", probnum);
				#endif
				numres--;

				gfield(pt_num, &pres_num);
				gfield(pt_pc1,  pres_res1);
				gfield(pt_pc2,  pres_res2);
				gfield(pt_pc3,  pres_res3);
				gfield(pt_pc4,  pres_res4);
				gfield(pt_pc5,  pres_res5);
				gfield(pt_tech, pres_tech);
				gfield(pt_tt,   pres_ticket);

				#ifdef DEBUG
				printf("About to display trouble ticket\n");
				#endif
				wait_key();

				fprintf(printer, "\n          Trouble Ticket:    ");
				fprintf(printer, "%s\n", pres_ticket);

				fprintf(printer, "              Technician:    ");
				fprintf(printer, "%s\n\n", pres_tech);

				if (!all_space(pres_res1))
					fprintf(printer, "%s\n", pres_res1);
				if (!all_space(pres_res2))
					fprintf(printer, "%s\n", pres_res2);
				if (!all_space(pres_res3))
					fprintf(printer, "%s\n", pres_res3);
				if (!all_space(pres_res4))
					fprintf(printer, "%s\n", pres_res4);
				if (!all_space(pres_res5))
					fprintf(printer, "%s\n", pres_res5);

				fprintf(printer, "\n");
				} /* if */
			}
		while ((btnext(btree) == 0) && (numres != 0));
	} /* if */

	if (rc = closbt(btree))
		{
		fprintf(stderr,"closbt failed for print_res.");
		fprintf(stderr,"rc = %d\n",rc);
		write_log("closebt failed for print_res on or around line 1314");
		sendmail(NONE);
		exit(1);
		}
	}

/*************************************************************************
 *
 *	print_report
 *
 *	This prints out a trouble ticket in trouble printer or
 *	John Scoggin's printer.
 *
 ************************************************************************/

void print_report()
{
long clock;
char prob_resp;

	#ifdef DEBUG
	printf("\7\7Printing Report...\n");
	#endif

	clock = time((long *) 0);			/* get current time */
	t = localtime(&clock);

	gfield(pr_node, prob_node);			/* fields that are necessary */
	gfield(pr_ct, prob_contact);			/* only for report, so read */
	gfield(pr_id, prob_id);			/* them in */
	gfield(pr_model, prob_model);
	gfield(pr_phone, prob_phone);
	gfield(pr_sn, prob_serialno);
	gfield(pr_loc, prob_location);
	gfield(pr_site, prob_site);
	gfield(pr_vend, prob_vendor);
	gfield(pr_pc1, prob_comment1);
	gfield(pr_pc2, prob_comment2);
	gfield(pr_pc3, prob_comment3);
	gfield(pr_pc4, prob_comment4);
	gfield(pr_vp, prob_vphone);
	gfield(pr_resp, &prob_resp);

	fprintf(printer, "%d/%d/%d   %.2d:%.2d",  t->tm_mon+1, t->tm_mday,
		t->tm_year, t->tm_hour, t->tm_min);
	fprintf(printer, "              TROUBLE TICKET\n");

	draw_a_line();

	fprintf(printer, "\n\n              Problem No:    %d\n",prob_num);
	fprintf(printer, "               Opened On:    %.2d/%.2d/%.2d\n                             %.2d:%.2d\n", opened_month, opened_day, opened_year, opened_hour, opened_minute);

	fprintf(printer, "             Responsible:    ");
	switch (prob_resp)
		{
		case 'N':
			fprintf(printer, "North\n\n");
			break;
		case 'S':
			fprintf(printer, "South\n\n");
			break;
		case 'V':
			fprintf(printer, "Vendor\n\n");
			break;
		}

	fprintf(printer, "        Escalation Level:    %d\n", esc_level);
	fprintf(printer, "          Priority Level:    %d\n", prio_level);
	fprintf(printer, "                 Node ID:    %s\n", prob_node);
	fprintf(printer, "                   Model:    %s\n", prob_model);
	fprintf(printer, "               Serial No:    %s\n\n", prob_serialno);
	fprintf(printer, "                  Vendor:    %s\n", prob_vendor);
	fprintf(printer, "            Vendor Phone:    %s\n\n", prob_vphone);
	fprintf(printer, "                    Site:    %s\n", prob_site);
	fprintf(printer, "                Location:    %s\n", prob_location);
	fprintf(printer, "                 Contact:    %s\n", prob_contact);
	fprintf(printer, "           Contact Phone:    %s\n\n", prob_phone);
	fprintf(printer, "            Problem Code:    %s\n\n", prob_id);
	fprintf(printer, "        Trouble Printers:    %s\n\n", trouble_ptr);
	fprintf(printer, "                Comments:\n\n");
	#ifdef DEBUG
	printf("About to display comments.\n");
	#endif
	wait_key();

	#ifdef DEBUG
	printf("prob_comment1 = %s\n", prob_comment1);
	printf("prob_comment2 = %s\n", prob_comment2);
	printf("prob_comment3 = %s\n", prob_comment3);
	printf("prob_comment4 = %s\n", prob_comment4);
	#endif DEBUG

	if (!all_space(prob_comment1))
		fprintf(printer, "%s\n", prob_comment1);
	if (!all_space(prob_comment2))
		fprintf(printer, "%s\n", prob_comment2);
	if (!all_space(prob_comment3))
		fprintf(printer, "%s\n", prob_comment3);
	if (!all_space(prob_comment4))
		fprintf(printer, "%s\n", prob_comment4);

	#ifdef DEBUG
	printf("Displayed all the comments.\n");
	#endif

	wait_key();

	fprintf(printer,"\n\n");

	draw_a_line();

	fprintf(printer, "\nResolutions:\n");

	print_res(prob_num);

	fprintf(printer, "\n\n\r<End of Report>\n");
	}

/*************************************************************************
 *
 *	print_stat()
 *
 *	This procedure prints out several fields in the record.
 *	Mainly used for testing purposes.
 *
 *************************************************************************/

void print_stat()
	{
	#ifdef DEBUG
	printf("problem number =         %d\n", prob_num);
	printf("escalation level =       %d\n", esc_level);
	printf("trouble printer =        \"%s\"\n", trouble_ptr);
	printf("saved esc level =        %d\n", saved_level);
	printf("saved prt =              \"%s\"\n", saved_prt);
	printf("PRINTED =                %d\n", PRINTED);
	printf("opened h:m =             %.2d:%.2d\n", opened_hour, opened_minute);
	printf("next esc h:m =           %.2d:%.2d\n", next_esc_hour, next_esc_minute);
	#endif
	}

/*************************************************************************
 *
 *	print_sup_report()
 *
 *	This procedure prints a report on John Scoggin's printer
 *
 *************************************************************************/

void print_sup_report()
 	{
	#ifdef DEBUG
	printf("Printing ticket on Scoggin's printer.\n");
	#endif
	open_printer("NOS    ");
	}

/*************************************************************************
 *
 *	print_tp_report();
 *
 *	This procedure prints a report on trouble printer.
 *
 *************************************************************************/

void print_tp_report()
	{
	char prob_resp;

	gfield(pr_resp,&prob_resp);
	if (prob_resp == 'N')  /* only print if Northern Netops is responsible */
		{
		#ifdef DEBUG
		printf("Printing ticket on trouble printer.\n");
		printf("Printing Northern trouble tickets.\n");
		#endif
		open_printer(trouble_ptr);
		}
	}

/*************************************************************************
 *
 *	process_date(int, int, int)
 *
 *	This procedure takes a raw date (buf) and calculates the
 *	real date (month, day, year) based on that raw date
 *	using kdate function of Unify (page 12-78).
 *
 *************************************************************************/

void process_date(buf, month, day, year)
int buf, *month, *day, *year;
	{
	int date_array[3];

	kdate(buf, date_array);

	*month = date_array[0];
	*day   = date_array[1];
	*year  = date_array[2];
	}

/*************************************************************************
 *
 *	int record_printed()
 *
 *	This function put 1 in the field pr_chg to record that a
 *	report had been printed on supervisor's printer to
 *	ensure it won't get printed again
 *
 *************************************************************************/

int record_printed()
	{
	int prted = 1;

	#ifdef DEBUG
	printf("This is record_printed.  I am changing pr_chg field.\n");
	printf("Here I go...\n");
	#endif

	wait_key();

	beep();
	if (pfield (pr_chg, &prted) != 0)
		{
		#ifdef DEBUG
		printf("record_printed: pfield fucked up.\n");
		#endif
		return(ERROR);
		}
	else
		{
		#ifdef DEBUG
		printf("record_printed: pfield succeed - pr_chg changed to 1.\n");
		#endif
		return(OKAY);
		}
	}

/*************************************************************************
 *
 *	save_fld()
 *
 *	This procedure saves:
 *	o   pr_pec to pr_code
 *	o   pr_prt to pr_tp
 *
 *************************************************************************/

int save_fld()
	{
	int i;

	#ifdef DEBUG
	printf("esc_level   = %d\n", esc_level);
	printf("prio_level  = %d\n", prio_level);
	printf("trouble_ptr = %s\n", trouble_ptr);
	#endif

	beep();
	if ((pfield(pr_code, &esc_level)) != 0)
		{
		#ifdef DEBUG
		printf("save_fld: pfield fucked up.\n");
		#endif
		return(ERROR);
		}
	else
		{
		#ifdef DEBUG
		printf("save_fld: pfield succeed.\n");
		#endif
		}

	if ((pfield(pr_old_p, &prio_level)) != 0)
		{
		#ifdef DEBUG
		printf("save_fld: pfield fucked up.\n");
		#endif
		return(ERROR);
		}
	else
		{
		#ifdef DEBUG
		printf("save_fld: pfield succeed.\n");
		#endif
		}

	beep();
	if (pfield (pr_prt, trouble_ptr) != 0)
		{
		#ifdef DEBUG
		printf("pfield fucked up.\n");
		#endif
		return(ERROR);
		}
	else
		{
		#ifdef DEBUG
		printf("save_fld: pfield succeed.\n");
		#endif
		}

	return(OKAY);
	}

/*************************************************************************
 *
 *	sendmail()
 *
 *	This procedure is called when the program is about to exit.
 *	It sends mail to appropriate person(s) informing them
 *	about it.
 *
 *************************************************************************/

void sendmail(int sig)
	{
	char mailcommand[150];

	switch (sig)
		{
		case NONE:
			sprintf(mailcommand, "%s %s << EOT_MAIL\n%s\nEOT_MAIL", MAIL, RECEIVERS, NONE_MESSAGE);
			break;
		case SEG:
			sprintf(mailcommand, "%s %s << EOT_MAIL\n%s\nEOT_MAIL", MAIL, RECEIVERS, SEGV_MESSAGE);
			break;
		case BUS:
			sprintf(mailcommand, "%s %s << EOT_MAIL\n%s\nEOT_MAIL", MAIL, RECEIVERS, BUS_MESSAGE);
			break;
		}

	system(mailcommand);
	}

/*************************************************************************
 *
 *	update_next_esc_dt();
 *
 *	This procedure actually updates the following fields in the db:
 *	   o  pr_esc_date
 *	   o  pr_esc_time
 *
 *	Date and time obtained from procedure calc_esc_dt().
 *
 *************************************************************************/

int update_next_esc_dt()
	{
	int date_array[3];
	short julian;				/* julian esc date - internal */
	short next_esc_time;				/* internal time HHMM */
	int rc;

	#ifdef DEBUG
	printf("Now I am supposed to update pr_esc_date and pr_esc_time.\n");
	printf("Okay, wish me luck.\n");
	#endif

	date_array[0] = next_esc_month;
	date_array[1] = next_esc_day;
	date_array[2] = next_esc_year;

	julian = (short) kday(date_array);

	next_esc_time = next_esc_hour * 100 + next_esc_minute;

	#ifdef DEBUG
	printf("julian = %d. next_esc_time = %d\n", julian, next_esc_time);
	#endif

	beep();
	if ((rc = pfield (pr_ed, &julian)) != 0)
		{
		#ifdef DEBUG
		printf("update_next_esc_dt: pfield fucked up.  Return code = %d\n", rc);
		#endif
		if (rc == -5)
			{
			#ifdef DEBUG
			printf("The record is currently busy.\n");
			#endif
			}
		return(ERROR);
		}
	else
		{
		#ifdef DEBUG
		printf("update_next_esc_dt: pfield succeed - esc date successfully updated.\n");
		#endif
		}

	beep();
	if ((rc = pfield (pr_et, &next_esc_time)) != 0)
		{
		#ifdef DEBUG
		printf("update_next_esc_dt: pfield fucked up.  Return code = %d\n", rc);
		#endif
		if (rc == -5)
			{
			#ifdef DEBUG
			printf("The record is currently busy.\n");
			#endif
			}
		return(ERROR);
		}
	else
		{
		#ifdef DEBUG
		printf("update_next_esc_dt: pfield succeed - esc time successfully updated.\n");
		#endif
		}

	#ifdef DEBUG
	printf("okay, done w/updating d/t.\n");
	#endif
	return(OKAY);
	}

/*************************************************************************
 *
 *	wait_key()
 *
 *************************************************************************/

void wait_key()
	{
	#ifdef DEBUG
	printf("Pausing: Press any key to continue...");
	getchar();
	#endif
	}

/*************************************************************************
 *
 *	write_log
 *
 *	This procedure is called before every exit condition.  It will
 *	write to a file defined by PROGLOG a error message and date and
 *	time of the error.
 *	This procedure is also called before printing a trouble ticket.
 *
 ************************************************************************/

void write_log(message)
char *message;
	{
	long clock;

	if ((proglog = fopen(PROGLOG, "a")) == NULL)
		{
		#ifdef DEBUG
		printf("Error opening the log file.\n");
		#endif
		}

	clock = time((long *) 0);

	fprintf(proglog, "Log date/time: %s", ctime(&clock));
	fprintf(proglog, "     %s\n", message);

	fclose(proglog);
	}
