/*************************************************************************
*
*  statmon.c
*
*	This program reads the problems in the 'problem' database
*	with a pr_stat of 'O' and lists a summary of these
*	every DEFSLTIME or argv[1] seconds.
*
*       Original Author: Bruce Yearsley
*
*       Revision History:
*            7/5/90 John Stewart
*                   The list of problems is now sorted by escalation
*                   level and problem number within escalation level
*                   in an array of ARRAYSIZE before printed to the
*                   screen.
*                   Exit codes:
*                        1 - Not indexed
*                        2 - Internal error
*                        3 - pr_stat not first element
*                        4 - open tree failed
*                        5 - search failed
*                        6 - close treee failed
*                        7 - Array size is not large enough, increase
*                            ARRAYSIZE constant
************************************************************************/

$include "/usr/db/ACL4/db/file.h"
#include "/usr/db/ACL4/include/btincl.h"

#define NULL '\0'
#define DEFSLTIME 10		/* Default screen update time     */
#define ARRAYSIZE 50            /* Max number of records that can */
				/*      be sorted before printing */
#define LINESPERSCREEN 16       /* Max number of records that can */
				/*      be printed on one screen  */

main(argc, argv)
   int argc;
   char *argv[];
{
   int btree,
       rc,
       status,
       sltime;
   char *btptrs[2];
   char xcode;
   BTIDS *probbts, *fldidxd();

   if (argc > 1)
   {
      if ((atoi(argv[1]))>1)
         sltime = atoi(argv[1]);
      else
         sltime=DEFSLTIME;
   }
   else
      sltime=DEFSLTIME;

   while(1)
   {
      if ((probbts = fldidxd(pr_stat,&status)) == BTIDNUL) 
         /* A return of BTIDNUL means that the table could not be */
         /* indexed.  This is a fatal error with the database.    */
      {
         if (status == IFNBT_R)
            /* IFNBT_R signifies that Problem Status does not exist */
         {
            printhead();
            printf("Problem Status is not indexed\n");
            setcook();
            exit(1);
         }
	 else   /* For all other errors */
	 {
            printhead();
	    printf("Internal error %d returned by fldidxd.\n", status);
	    setcook();
	    exit(2);
	 }
      }

      if (probbts[0].btfrank != 0)
      {
          printhead();
	  printf("pr_stat must be the first element of the index.");
	  setcook();
	  exit(3);
      }

      btree = probbts[0].btidnum;
	
      if (rc = opnbts(btree))
      {
          printhead();
	  prtmsg(0,24,"Open btree failed");
	  printf("rc = %d\n",rc);
	  setcook();
	  exit(4);
      }

      xcode = 'O';
      btptrs[0] = (char *) &xcode;
      btptrs[1] = (char *) 0;
      rc=btsrch(btree,btptrs);     /* search for Open problems */

      if((rc == -6) || (rc >= 0))
         /* -6 and 0 are status codes which respresent no problems */
      {
         if(rc >= 0)         /* 0 means that there are records */
            prtstat(sltime);
         else                /* -6 means there are no records  */
         {
            printhead();
            prmprv(1,4, "Amazing, there are no open problems!");
         }
      }
      else	     /* If there was a problem in the search */
      {
         printhead();
	 prtmsg (0,24,"Btsrch failed");
	 printf("rc = %d\n",rc);
	 setcook();
	 exit(5);
      }

      if (rc = closbt(btree))     /* When all records have been */
				  /* printed for this cycle,    */
				  /* attempt to close tree.     */
      {
         printhead();
	 prtmsg(0,24,"closbt failed");
	 printf("rc = %d\n",rc);
	 setcook();
	 exit(6);
      }
   }
}

prtstat(sltime)
   int sltime;
{
   BTIDS *probbts, *fldidxd();
   char statuscode;
   int status,
       btree,
       l, j,
       month, day, year,
       hour, minute,
       count;   /* Number of structures read in */
   struct item
   {
      long num;
      long unsigned int key;  /* This field is present so that the sort
                                 can be performed on one field, but it
                                 actually accomplishes a two field sort */
      short date, time, prio;
      char nde[21], ste[9], id[4], vend[9], tp[4];
   };
   struct item table[ARRAYSIZE];
   struct item temp;  /* needed for bubble sort */

   probbts=fldidxd(pr_stat,&status);
   btree=probbts[0].btidnum;
   count=-1;

   printhead();

/* Read open records into structure array until  */
/* all are read in OR the array is full          */

   do
   {
      gfield(pr_stat,&statuscode);
      if (statuscode=='O') 
      {
         count++;
         gfield(pr_num,&table[count].num);
         gfield(pr_date,&table[count].date);
         gfield(pr_time,&table[count].time);
         gfield(pr_node,table[count].nde);
         table[count].nde[20]='\0';   /* needs to be done manually,   */
                                      /* gfield() doesn't assign null */
         gfield(pr_site,table[count].ste);
         table[count].ste[8]='\0';
         gfield(pr_id,table[count].id);
         table[count].id[3]='\0';
         gfield(pr_vend,table[count].vend);
         table[count].vend[8]='\0';
         gfield(pr_tp,table[count].tp);
         table[count].tp[3]='\0';
         gfield(pr_prio,&table[count].prio);
         table[count].key=((((unsigned long int)table[count].prio)^15)<<24) +
                          (unsigned long int)table[count].num;
         /* key is built in the following manner:
                  1) key=priority
                  2) key=key EXCLUSIVE-OR 7 (binay 111) so that the number
                        is reversed at the binary level.
                  3) shift key 24 bits to the left leaving the most sig.
                        byte as the reversed value of escalation code
                  4) add the problem number to this value
                        the result of the above is a field, which when used as
                        the key sort field, sorts the records, first in
                        descending order of escalation level, and second in
                        ascending order of problem number */
      }
   } while ((btnext(btree)==0) && (count<ARRAYSIZE));

   count+=1;

   if (count>ARRAYSIZE)
   {
      printf("\n\nERROR -- More than 50 problems encountered.\n");
      printf("/usr/db/ACL4/src/statmon/statmon.c needs reprogramming.");
      exit(7);
   }
 
   printnum(count);

   /* Bubble sort records read into array */

   for (l=0; l<count; l++)
   {
      for (j=l+1; j<count; j++)
      {
         if (table[l].key > table[j].key)
         {
            temp=table[l];
            table[l]=table[j];
            table[j]=temp;
         }
      }
   }

   l=0;  /* counts the number of lines printed per screen */

   for (j=0; j<count; j++)
   {
      l++;
      printf("[0m");
      switch(table[j].prio)  /* different escalation codes are given */
			    /* on-screen attributes                 */
      {
         case 7:
         {
            printf("[1;4;5m");		 /* flash, underline */
	    break;
         }
         case 6:
         {
            printf("[0;4m");
            break;
         }
         case 5:
         {
            printf("[1;1m");
            break;
         }
         default:
         printf("[0m");			/* normal attribute */
      };

      prcs_date(table[j].date, &month, &day, &year);
      prcs_time(table[j].time,&hour, &minute);
      printf(" %4d %.2d/%.2d/%.2d %.2d:%.2d %20s %8s %3s %8s  %3s     %d\n[2;2m",
         table[j].num,month,day,year,hour,minute,table[j].nde, table[j].ste,
         table[j].id,table[j].vend,table[j].tp,table[j].prio);
      if (j==(count-1))
      {
           prmp(1,21,"");
           printf("Last page.");
      }
      if ((l==LINESPERSCREEN) && (j<count))
      {
           if (j<count)
                printf("\n More problems on the next page.\n");
           else
                printf("\n Last Page.\n");
           waitfor(sltime);
           printhead();
           printnum(count);
           l=0;
      }
   }
   if (l!=0)   /* Waits at last screen */
      waitfor(sltime);
}

printnum(num)
int num;
{
     prmp(1,2,"Number of Record(s)=");
     printf("%d",num);
     prmprv(0,4,"");
}

printhead()
{
   clr_crt(1,'a');
   prmprv(25,1," Open Problem Summary ");
   prmprv(1,3,"#    Opened on      Node ID              Site     Pcd Vendor    Prt  Priority\n");
}

waitfor(sltime)
int sltime;
{
     int i;
     for (i=0; i<sltime; i++)
     {
          prmp(1,22,"Next update in ");
          printf("%2d seconds\n",sltime-i);
          sleep(1);
      }
}

prcs_date(buf, month, day, year)
int buf, *month, *day, *year;
{
   int date_array[3];

   kdate(buf, date_array);

   *month = date_array[0];
   *day   = date_array[1];
   *year  = date_array[2];
}

prcs_time(buf, hour, minute)
int buf, *hour, *minute;
{
   *hour=buf/100;
   *minute=buf-((int)(buf/100)*100);
}
