/**********************************************************************
 *
 *  vg.c
 *
 *  Original Author: John Stewart
 *
 *  Revision History:
 *
 **********************************************************************

1. Purpose:  This program reads in a vendor, and then accesses the
             problem database.  Data is then presented to the user on
             the average time to correct problems for the given vendor
             for the current month as well as the previous 11 months.
             Problems are not considered if they are still open or if
             they took longer than 7 days to correct.
2. John Stewart, January 1991
3. Inputs:  User is prompted for a vendor if he/she did not specify
            the vendor name on the command line.
4. Outpus:  A graph representing the average time to fix non-perpetual
            (less than or equal to seven days) problems for the given
            vendor for the current month and the past 11 months.  Also,
            the total number of problems and the number of problems that
            took more than seven days to correct.
5. Procedures/Functions called:
      strcpy(), fldidxd(), setcook(), opnbts(), btsrch(), get_data(),
      print_data_to_screen(), print_data_to_printer(), putenv(),
      printf(), exit()
6. Procedures/Functions calling: none
7. Local Variales:
      vend_bts:
           the B-Tree index for the pr_vend fieldname
      status:
           the return status of the call to fldidxd()
      btree:
           the B-Tree I.D. number for the pr_vend fieldname
      rc:
           the return status of the call to opnbts()
      btptrs:
           the search array for the B-Tree
8. Global Variables:
      vendr:
           the name of the vendor for which to do report
      month0, month1, ..., month11:
           the running total for the number of days a vendor has
           taken to correct problems for the corresponding month
      month0_count, month1_count, ..., month11_count:
           the running total for the number of problems a vendor
           has had in the corresponding month
      perpetual:
           the number of problems which took longer than seven
           days to solve
      total_probs:
           the total number of NON-PERPETUAL problems
      average0, average1, ..., average11:
           the result of monthX/monthX_count (where X is one
           of {0..11})
      print:
           a BOOLEAN variable specifying whether the report should
           be sent to the printer or the screen
9. Global Constants:
      TRUE, FALSE: self explanatory
      MONTH, DAY, YEAR: indexes for accessing the UNIFY date data
                        structure after converting from Julian
10. Exit codes:
       4 = Couldn't delete working file
       3 = Couldn't print report
       2 = Couldn't create file to print report
       1 = Incorrect number of command-line arguments
       0 = Everything was fine
      -1 = Not indexed
      -2 = Internal error
      -3 = Not first element
      -4 = opnbts (Open B-Tree) failed
      -5 = btsrch (B-Tree Search) failed
      -6 = closbt (Close B-Tree) failed
11. Bugs: none known
**********************************************************************/

$include "/usr/db/ACL4/db/file.h"
#include "/usr/db/ACL4/include/btincl.h"
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>

#define TRUE 1
#define FALSE 0
#define MONTH 0
#define DAY 1
#define YEAR 2

char vendr[16];
int month0=0, month0_count=0, month1=0, month1_count=0, month2=0,
   month2_count=0, month3=0, month3_count=0, month4=0, month4_count=0,
   month5=0, month5_count=0, month6=0, month6_count=0, month7=0,
   month7_count=0, month8=0, month8_count=0, month9=0, month9_count=0,
   month10=0, month10_count=0, month11=0, month11_count=0, perpetual=0,
   total_probs, print=FALSE, subtract_month();
float average0, average1, average2, average3, average4, average5, average6,
   average7, average8, average9, average10, average11;
struct tm *date;
time_t clock;

main(argc,argv)
int argc;
char *argv[];
   {
   BTIDS *vend_bts, *fldidxd();
   int status, btree, rc;
   char *btptrs[2];

   putenv("DBPATH=/usr/db/ACL4/db");
   putenv("UNIFY=/usr/db/ACL4/lib");
   if (argc>3)
      {
      printf("\nUSAGE:  vg [-p] [vendor_name]\n\n");
      printf("    The vendor name should be enclosed in quotes\n");
      printf("    if it contains spaces or special characters.\n");
      exit(1);
      }
   if (argc==3)
      {
      strcpy(vendr,argv[2]);
      print=TRUE;
      }
   if (argc<2)
      {
      printf("\nPlease enter the name of the vendor for which you\n");
      printf("would like a graph.  (Make sure to use ALL CAPS.)\n");
      scanf("%s",vendr);
      }
   if ((argc==2) && (*(argv[1])!='-'))
      strcpy(vendr,argv[1]);

   if ((argc==2) && (*(argv[1])=='-'))
      {
      print=TRUE;
      printf("\nPlease enter the name of the vendor for which you\n");
      printf("would like a graph.  (Make sure to use ALL CAPS.)\n");
      scanf("%s",vendr);
      }

   if ((vend_bts=fldidxd(pr_vend,&status)) == BTIDNUL)
      {
      if (status == IFNBT_R)
         {
         printf("The field 'pr_vend' is not indexed.\n");
         setcook();
         exit(-1);
         }
      else
         {
         printf("Internal error %d returned by 'fldidx' function.\n",status);
         setcook();
         exit(-2);
         }
      }

   if (vend_bts[0].btfrank != 0)
      {
      printf("'pr_vend' is not the first element of the index.\n");
      setcook();
      exit(-3);
      }

   btree=vend_bts[0].btidnum;

   if (rc=(opnbts(btree)))
      {
      printf("'opnbts' failed.\n");
      printf("'rc'=%d.\n",rc);
      setcook();
      exit(-4);
      }

   btptrs[0]=vendr;
   btptrs[1]=(char *) 0;
   rc=(btsrch(btree,btptrs));
   if ((rc==-6) || (rc>=0))
      {
      if (rc>=0)
      {
	 get_data(btree);
         if (print==FALSE)
            print_data_to_screen();
         else
            print_data_to_printer();
         exit(0);
      }
      else
	 printf("\nI didn't find any records.\n");
      }
   else
      {
      printf("\n'btsrch' failed.\n");
      printf("\n'rc'=%d\n",rc);
      setcook();
      exit(-5);
      }

   if (rc=(closbt(btree)))
      {
      printf("\n'closbt' failed.\n");
      printf("'rc'=%d.\n",rc);
      setcook();
      exit(-6);
      }
}

/**********************************************************************
 *                          procedure get_data()                      *
 **********************************************************************
 1. Purpose: get_data() reads the data from the database and calculates
             items such as the average, the total problems, number of
             perpetual problems, etc...
 2. John Stewart, January 1991
 3. Inputs:
       btree:
            the B-Tree I.D. number of the pr_vend field
 4. Returned Values: none
 5. Procedures/Functions Called:
       time(), localtime(), gfield(), strip_whites(), strcmp(),
       kdate(), subtract_month(), btnext(), clr_crt(),
       prmp()
 6. Procedures/Functions Calling:
       main()
 7. Local Variables:
       temp_vendr:
          buffer to hold the vendor name gotten from the active record
          in the database; this is necessary because when you search in
          a B-Tree for a value, say, MEMOREX, the data that will be found
          is MEMOREX and ALL VALUES COMING AFTER MEMOREX IN THE ALPHABET
       open_date, close_date:
          the Julian date gotten from the database from the active record
       temp:
          the value of close_date-open_date
       month, day, year:
          the current month, day and year gotten from time() and localtime()
       already:
          BOOLEAN variable specifying whether all occurences of the given
          vendor have been read in (see temp_vendr comment)
       open_array[], close_array[]:
          the return value of the UNIFY date conversion function
       tmp:
          the value of open_date[MONTH]-month
       date:
          the return value of localtime()
       clock:
          the return value of time()
 8. Global Variables Used (and how modified):
       vendr:
          used to compare against the vendor gotten from the active record
 **********************************************************************/

get_data(btree)
int btree;
   {
   char temp_vendr[15];
   short open_date;
   short close_date;
   int temp=0, month, day, year, already=FALSE, open_array[3], close_array[3], tmp=0;

   time(&clock);
   date=localtime(&clock);
   month=(date->tm_mon)+1;
   year=date->tm_year;
   day=date->tm_mday;

   clr_crt(1,'a');

   do
      {
      prmp(32,9,"Please Hold\n");
      prmp(32,7,"Reading...   \n");
      gfield(pr_vend,temp_vendr);
      prmp(32,7,"Processing...\n");
      strip_whites(temp_vendr);
      if ((strcmp(vendr,temp_vendr))==0)
         {
         gfield(pr_date,&open_date);
	 kdate(open_date,open_array);
	 if (open_array[YEAR]>=(year-1))
	    {
            gfield(pr_cldat,&close_date);
            kdate(close_date,close_array);
            temp=(int)close_date-open_date;
	    if (temp>7)
	       perpetual++;
            else if (temp<0) /* if the problem is still open, the */
               ;             /* close date has a value of -32767  */
            else if ((open_array[YEAR]==year) || ((open_array[YEAR]==(year-1)) && (open_array[MONTH]>month)))
	       {
               tmp=subtract_month(month,open_array[MONTH]);
               switch (tmp)
                  {
                  case 0: month0=month0 + temp;
                          month0_count=month0_count + 1;
                          break;
                  case 1: month1=month1 + temp;
                          month1_count=month1_count + 1;
                          break;
                  case 2: month2=month2 + temp;
                          month2_count=month2_count + 1;
                          break;
                  case 3: month3=month3 + temp;
                          month3_count=month3_count + 1;
                          break;
                  case 4: month4=month4 + temp;
                          month4_count=month4_count + 1;
                          break;
                  case 5: month5=month5 + temp;
                          month5_count=month5_count + 1;
                          break;
                  case 6: month6=month6 + temp;
                          month6_count=month6_count + 1;
                          break;
                  case 7: month7=month7 + temp;
                          month7_count=month7_count + 1;
                          break;
                  case 8: month8=month8 + temp;
                          month8_count=month8_count + 1;
                          break;
                  case 9: month9=month9 + temp;
                          month9_count=month9_count + 1;
                          break;
                  case 10: month10=month10 + temp;
                           month10_count=month10_count + 1;
                           break;
                  case 11: month11=month11 + temp;
                           month11_count=month11_count + 1;
                           break;
                  } /* switch */
               } /* else */
            } /* if */
	 }
      else
         already=TRUE;
      }
   while ((btnext(btree)==0)&&(!already));

   prmp(32,7,"Calculcating...\n");

   if (month0_count>0)
      average0=(float) month0/month0_count;
   else
      average0=0.0;
   if (month1_count>0)
      average1=(float) month1/month1_count;
   else
      average1=0.0;
   if (month2_count>0)
      average2=(float) month2/month2_count;
   else
      average2=0.0;
   if (month3_count>0)
      average3=(float) month3/month3_count;
   else
      average3=0.0;
   if (month4_count>0)
      average4=(float) month4/month4_count;
   else
      average4=0.0;
   if (month5_count>0)
      average5=(float) month5/month5_count;
   else
      average5=0.0;
   if (month6_count>0)
      average6=(float) month6/month6_count;
   else
      average6=0.0;
   if (month7_count>0)
      average7=(float) month7/month7_count;
   else
      average7=0.0;
   if (month8_count>0)
      average8=(float) month8/month8_count;
   else
      average8=0.0;
   if (month9_count>0)
      average9=(float) month9/month9_count;
   else
      average9=0.0;
   if (month10_count>0)
      average10=(float) month10/month10_count;
   else
      average10=0.0;
   if (month11_count>0)
      average11=(float) month11/month11_count;
   else
      average11=0.0;
   total_probs=month0_count + month1_count + month2_count + month3_count +
      month4_count + month5_count + month6_count + month7_count +
      month8_count + month9_count + month10_count + month11_count;
   }

/**********************************************************************
 *                      function subtract_month()                     *
 **********************************************************************
 1. Purpose: subtract_month() will calculate the difference between two
             months so that the program will know which month the
             active record applies to
 2. John Stewart, Januray 1991
 3. Inputs:
       month1, month2:
            the two months begin subtracted
 4. Returned Values:
       month1-month2
 5. Procedures/Functions Called: none
 6. Procedures/Functions Calling:
       get_data()
 7. Local Variables:
       diff:
            holds the value of a temporary subtraction
 8. Global Variables Used (and how modified): none
 *********************************************************************/

int subtract_month(month1,month2)
int month1, month2;
   {
   int diff;

   diff=month1-month2;
   if (diff >= 0) 
      return(diff);
   else
      return(diff + 12);
   }

/**********************************************************************
 *                      procedure strip_whites()                      *
 **********************************************************************
 1. Purpose: strip_whites() looks at a string and puts a NULL character
             after the last non-alphanumreic character
 2. John Stewart, Januray 1991
 3. Inputs:
       str:
          the string array being stripped of white
 4. Returned Values:
       str is a pass-by-reference parameter
 5. Procedures/Functions Called: none
 6. Procedures/Functions Calling:
       get_data()
 7. Local Variables:
       i:
            counter for for loop
       already:
            a BOOLEAN variable indicating if the NULL character has
               already been inserted
 8. Global Variables Used (and how modified): none
 *********************************************************************/

strip_whites(str)
char str[16];
   {
   int i, already=FALSE;

   for (i=0;i<16;i++)
      {
      if ((str[i]==' ')&&(!already)&&(str[i+1]==' '))
	 {
	 str[i]='\0';
	 already=TRUE;
	 }
      }
   str[15]='\0';
   str[16]='\0';
   }

/*********************************************************************
 *                 procedure print_data_to_screen()                  *
 *********************************************************************
 1. Purpose: print_data() presents the user with an on-screen graph of
             the data collected in get_data()
 2. John Stewart, January 1991
 3. Inputs: none
 4. Returned Values: none
 5. Procedures/Functions Called:
       clr_crt(), prmprv(), prmp(), printf(), date_print()
 6. Procedures/Functions Calling: main()
 7. Local Variables:
       i:
            counter for the for loops
 8. Global Variables Used (and how modified): none
 *********************************************************************/

print_data_to_screen()
   {
   int i;

   clr_crt(1,'a');
   prmprv(0,0," Vendor Name:");
   printf("        %s\n",vendr);
   prmprv(0,1," Total Problems:");
   printf("     %.3d",total_probs+perpetual);
   printf("              Perpetual problems are not\n");
   prmprv(4,2," Non-perpetual:");
   printf("  %.3d",total_probs);
   printf("                 included in average.\n");
   prmprv(4,3," Perpetual:");
   printf("      %.3d\n\n",perpetual);
   printf("================================================================================\n");
   prmp(72,6,"# Probs\n");
   date_print(0,'s',(FILE *)NULL);
   for (i=0;i<(int)(9*average0);i++)
      printf("*");
   printf("%1.1f",average0);
   prmp(72,7,"");
   printf("%6d\n",month0_count);
   date_print(1,'s',(FILE *)NULL);
   for (i=0;i<(int)(9*average1);i++)
      printf("*");
   printf("%1.1f",average1);
   prmp(72,8,"");
   printf("%6d\n",month1_count);
   date_print(2,'s',(FILE *)NULL);
   for (i=0;i<(int)(9*average2);i++)
      printf("*");
   printf("%1.1f",average2);
   prmp(72,9,"");
   printf("%6d\n",month2_count);
   date_print(3,'s',(FILE *)NULL);
   for (i=0;i<(int)(9*average3);i++)
      printf("*");
   printf("%1.1f",average3);
   prmp(72,10,"");
   printf("%6d\n",month3_count);
   date_print(4,'s',(FILE *)NULL);
   for (i=0;i<(int)(9*average4);i++)
      printf("*");
   printf("%1.1f",average4);
   prmp(72,11,"");
   printf("%6d\n",month4_count);
   date_print(5,'s',(FILE *)NULL);
   for (i=0;i<(int)(9*average5);i++)
      printf("*");
   printf("%1.1f",average5);
   prmp(72,12,"");
   printf("%6d\n",month5_count);
   date_print(6,'s',(FILE *)NULL);
   for (i=0;i<(int)(9*average6);i++)
      printf("*");
   printf("%1.1f",average6);
   prmp(72,13,"");
   printf("%6d\n",month6_count);
   date_print(7,'s',(FILE *)NULL);
   for (i=0;i<(int)(9*average7);i++)
      printf("*");
   printf("%1.1f",average7);
   prmp(72,14,"");
   printf("%6d\n",month7_count);
   date_print(8,'s',(FILE *)NULL);
   for (i=0;i<(int)(9*average8);i++)
      printf("*");
   printf("%1.1f",average8);
   prmp(72,15,"");
   printf("%6d\n",month8_count);
   date_print(9,'s',(FILE *)NULL);
   for (i=0;i<(int)(9*average9);i++)
      printf("*");
   printf("%1.1f",average9);
   prmp(72,16,"");
   printf("%6d\n",month9_count);
   date_print(10,'s',(FILE *)NULL);
   for (i=0;i<(int)(9*average10);i++)
      printf("*");
   printf("%1.1f",average10);
   prmp(72,17,"");
   printf("%6d\n",month10_count);
   date_print(11,'s',(FILE *)NULL);
   for (i=0;i<(int)(9*average11);i++)
      printf("*");
   printf("%1.1f",average11);
   prmp(72,18,"");
   printf("%6d\n",month11_count);
   printf("        |--------+--------+--------+--------+--------+--------+--------+\n");
   printf("        0        1        2        3        4        5        6        7\n\n");
   printf("                         Average Number of Days to Correct\n");
   }

/**********************************************************************
 *                  procedure print_data_to_printer()                 *
 **********************************************************************
 1. Purpose: print_data_to_printer() will take the data collected in
             get_data() and print a summary of that data to a file
             (called .vgXfile) in the current working directory, that
             file will then be sent to the printer via the system()
             function call
 2. John Stewart, Januray 1991
 3. Inputs: none
 4. Returned Values: none
 5. Procedures/Functions Called:
       fopen(),fclose(),printf(),fprintf(),date_print(),system()
 6. Procedures/Functions Calling:
       main()
 7. Local Variables:
       file_buffer:
            a pointer to a working file where the data is sent; after
            all of the data is there, a system call is made and the
            file is printed
       i,j:
            counters needed for the for loops
 8. Global Variables Used (and how modified): none
 *********************************************************************/

print_data_to_printer()
   {
   FILE *file_buffer;
   int i, j;

   file_buffer=fopen(".vgXfile","w");
   if (file_buffer==NULL)
      {
      printf("\nvg:Error creating file.\n");
      fclose(file_buffer);
      exit(2);
      }

   fprintf(file_buffer,"Vendor Name:");
   fprintf(file_buffer,"        %s\n",vendr);
   fprintf(file_buffer,"Total Problems:");
   fprintf(file_buffer,"     %.3d\n",total_probs+perpetual);
   fprintf(file_buffer,"     Non-perpetual:");
   fprintf(file_buffer," %.3d\n",total_probs);
   fprintf(file_buffer,"     Perpetual:");
   fprintf(file_buffer,"     %.3d\n\n",perpetual);
   fprintf(file_buffer,"======================================================================================\n\n");
   fprintf(file_buffer,"                                                                      Number of Probs\n");

   date_print(0,'p',file_buffer);
   for (i=0;i<(int)(9*average0);i++)
      fprintf(file_buffer,"*");
   fprintf(file_buffer,"%1.1f",average0);
   for (j=i;j<64;j++)
      fprintf(file_buffer," ");
   fprintf(file_buffer,"%d\n",month0_count);

   date_print(1,'p',file_buffer);
   for (i=0;i<(int)(9*average1);i++)
      fprintf(file_buffer,"*");
   fprintf(file_buffer,"%1.1f",average1);
   for (j=i;j<64;j++)
      fprintf(file_buffer," ");
   fprintf(file_buffer,"%d\n",month1_count);

   date_print(2,'p',file_buffer);
   for (i=0;i<(int)(9*average2);i++)
      fprintf(file_buffer,"*");
   fprintf(file_buffer,"%1.1f",average2);
   for (j=i;j<64;j++)
      fprintf(file_buffer," ");
   fprintf(file_buffer,"%d\n",month2_count);

   date_print(3,'p',file_buffer);
   for (i=0;i<(int)(9*average3);i++)
      fprintf(file_buffer,"*");
   fprintf(file_buffer,"%1.1f",average3);
   for (j=i;j<64;j++)
      fprintf(file_buffer," ");
   fprintf(file_buffer,"%d\n",month3_count);

   date_print(4,'p',file_buffer);
   for (i=0;i<(int)(9*average4);i++)
      fprintf(file_buffer,"*");
   fprintf(file_buffer,"%1.1f",average4);
   for (j=i;j<64;j++)
      fprintf(file_buffer," ");
   fprintf(file_buffer,"%d\n",month4_count);

   date_print(5,'p',file_buffer);
   for (i=0;i<(int)(9*average5);i++)
      fprintf(file_buffer,"*");
   fprintf(file_buffer,"%1.1f",average5);
   for (j=i;j<64;j++)
      fprintf(file_buffer," ");
   fprintf(file_buffer,"%d\n",month5_count);

   date_print(6,'p',file_buffer);
   for (i=0;i<(int)(9*average6);i++)
      fprintf(file_buffer,"*");
   fprintf(file_buffer,"%1.1f",average6);
   for (j=i;j<64;j++)
      fprintf(file_buffer," ");
   fprintf(file_buffer,"%d\n",month6_count);

   date_print(7,'p',file_buffer);
   for (i=0;i<(int)(9*average7);i++)
      fprintf(file_buffer,"*");
   fprintf(file_buffer,"%1.1f",average7);
   for (j=i;j<64;j++)
      fprintf(file_buffer," ");
   fprintf(file_buffer,"%d\n",month7_count);

   date_print(8,'p',file_buffer);
   for (i=0;i<(int)(9*average8);i++)
      fprintf(file_buffer,"*");
   fprintf(file_buffer,"%1.1f",average8);
   for (j=i;j<64;j++)
      fprintf(file_buffer," ");
   fprintf(file_buffer,"%d\n",month8_count);

   date_print(9,'p',file_buffer);
   for (i=0;i<(int)(9*average9);i++)
      fprintf(file_buffer,"*");
   fprintf(file_buffer,"%1.1f",average9);
   for (j=i;j<64;j++)
      fprintf(file_buffer," ");
   fprintf(file_buffer,"%d\n",month9_count);

   date_print(10,'p',file_buffer);
   for (i=0;i<(int)(9*average10);i++)
      fprintf(file_buffer,"*");
   fprintf(file_buffer,"%1.1f",average10);
   for (j=i;j<64;j++)
      fprintf(file_buffer," ");
   fprintf(file_buffer,"%d\n",month10_count);

   date_print(11,'p',file_buffer);
   for (i=0;i<(int)(9*average11);i++)
      fprintf(file_buffer,"*");
   fprintf(file_buffer,"%1.1f",average11);
   for (j=i;j<64;j++)
      fprintf(file_buffer," ");
   fprintf(file_buffer,"%d\n",month11_count);
   fprintf(file_buffer,"        |--------+--------+--------+--------+--------+--------+--------+\n");
   fprintf(file_buffer,"        0        1        2        3        4        5        6        7\n\n");
   fprintf(file_buffer,"                        Average Number of Days to Correct\n");
   fprintf(file_buffer,"\nNOTE:\n");
   fprintf(file_buffer,"     Perpetual problems (more than seven\n");
   fprintf(file_buffer,"     days) are not included in the average.");

   fclose(file_buffer);

   if ((system("$SPOOLER .vgXfile"))!=0)
      {
      printf("\nCould not print report to printer.\n");
      printf("Check for existence of .vgXfile in current\n");
      printf("directory and delete it if it exists.\n");
      exit(3);
      }
   if ((system("rm .vgXfile"))!=0)
      {
      printf("\nCould not delete working file after printing.\n");
      printf("Check for existence of .vgXfile in current\n");
      printf("directory and delete it if it exists.\n");
      exit(4);
      }

   prmp(0,10,"");
   printf("\n\nPrinting complete.\n");
   }

/**********************************************************************
 *                       procedure date_print()                       *
 **********************************************************************
 1. Purpose: date_print() will take three arguments and print the text
             of the appropriate month either to the printer or the
             screen
 2. John Stewart, Januray 1991
 3. Inputs:
       diff:
            the difference between the current month and the month
            that is on the active record
       dest:
            a single character (either 's' or 'p') that specifies
            whether the output should go to the screen or the printer
       file:
            if the output is supposed to go to the printer, this is
            the file pointer where the output should go; if the output
            is supposed to go to the screen, a NULL file pointer is
            sent in
 4. Returned Values: none
 5. Procedures/Functions Called:
       printf()
 6. Procedures/Functions Calling:
       print_data_to_screen(), print_data_to_printer()
 7. Local Variables: none
 8. Global Variables Used (and how modified): none
 *********************************************************************/
date_print(diff,dest,file)
int diff;
char dest;
FILE *file;
   {
   switch (dest)
      {
      case 's': switch ((date->tm_mon+1)-diff)
                   {
                   case 0: printf(" Dec%d  |",(date->tm_year)-1);
                           break;
                   case 1: printf(" Jan%d  |",date->tm_year);
                           break;
                   case 2: printf(" Feb%d  |",date->tm_year);
	                   break;
                   case 3: printf(" Mar%d  |",date->tm_year);
	                   break;
                   case 4: printf(" Apr%d  |",date->tm_year);
	                   break;
                   case 5: printf(" May%d  |",date->tm_year);
	                   break;
                   case 6: printf(" Jun%d  |",date->tm_year);
	                   break;
                   case 7: printf(" Jul%d  |",date->tm_year);
	                   break;
                   case 8: printf(" Aug%d  |",date->tm_year);
	                   break;
                   case 9: printf(" Sep%d  |",date->tm_year);
	                   break;
                   case 10: printf(" Oct%d  |",date->tm_year);
	                   break;
                   case 11: printf(" Nov%d  |",date->tm_year);
                            break;
                   case 12: printf(" Dec%d  |",date->tm_year);
                            break;
                   case -1: printf(" Nov%d  |",(date->tm_year)-1);
	                    break;
                   case -2: printf(" Oct%d  |",(date->tm_year)-1);
	                    break;
                   case -3: printf(" Sep%d  |",(date->tm_year)-1);
	                    break;
                   case -4: printf(" Aug%d  |",(date->tm_year)-1);
	                    break;
                   case -5: printf(" Jul%d  |",(date->tm_year)-1);
	                    break;
                   case -6: printf(" Jun%d  |",(date->tm_year)-1);
	                    break;
                   case -7: printf(" May%d  |",(date->tm_year)-1);
	                    break;
                   case -8: printf(" Apr%d  |",(date->tm_year)-1);
	                    break;
                   case -9: printf(" Mar%d  |",(date->tm_year)-1);
	                    break;
                   case -10: printf(" Feb%d  |",(date->tm_year)-1);
	                    break;
                   case -11: printf(" Jan%d  |",(date->tm_year)-1);
	                    break;
                   }
                break;
      case 'p': switch ((date->tm_mon+1)-diff)
                   {
                   case 0: fprintf(file," Dec%d  |",(date->tm_year)-1);
                           break;
                   case 1: fprintf(file," Jan%d  |",date->tm_year);
                           break;
                   case 2: fprintf(file," Feb%d  |",date->tm_year);
	                   break;
                   case 3: fprintf(file," Mar%d  |",date->tm_year);
	                   break;
                   case 4: fprintf(file," Apr%d  |",date->tm_year);
	                   break;
                   case 5: fprintf(file," May%d  |",date->tm_year);
	                   break;
                   case 6: fprintf(file," Jun%d  |",date->tm_year);
	                   break;
                   case 7: fprintf(file," Jul%d  |",date->tm_year);
	                   break;
                   case 8: fprintf(file," Aug%d  |",date->tm_year);
	                   break;
                   case 9: fprintf(file," Sep%d  |",date->tm_year);
	                   break;
                   case 10: fprintf(file," Oct%d  |",date->tm_year);
	                   break;
                   case 11: fprintf(file," Nov%d  |",date->tm_year);
                            break;
                   case 12: fprintf(file," Dec%d  |",date->tm_year);
                            break;
                   case -1: fprintf(file," Nov%d  |",(date->tm_year)-1);
	                    break;
                   case -2: fprintf(file," Oct%d  |",(date->tm_year)-1);
	                    break;
                   case -3: fprintf(file," Sep%d  |",(date->tm_year)-1);
	                    break;
                   case -4: fprintf(file," Aug%d  |",(date->tm_year)-1);
	                    break;
                   case -5: fprintf(file," Jul%d  |",(date->tm_year)-1);
	                    break;
                   case -6: fprintf(file," Jun%d  |",(date->tm_year)-1);
	                    break;
                   case -7: fprintf(file," May%d  |",(date->tm_year)-1);
	                    break;
                   case -8: fprintf(file," Apr%d  |",(date->tm_year)-1);
	                    break;
                   case -9: fprintf(file," Mar%d  |",(date->tm_year)-1);
	                    break;
                   case -10: fprintf(file," Feb%d  |",(date->tm_year)-1);
	                    break;
                   case -11: fprintf(file," Jan%d  |",(date->tm_year)-1);
	                    break;
                   }
                break;
                }
   }
