#!/usr/local/bin/perl

use Config;
use File::Basename qw(&basename &dirname);

# List explicitly here the variables you want Configure to
# generate.  Metaconfig only looks for shell variables, so you
# have to mention them as if they were shell variables, not
# %Config entries.  Thus you write
#  $startperl
# to ensure Configure will look for $Config{startperl}.

# This forces PL files to create target in same directory as PL file.
# This is so that make depend always knows where to find PL derivatives.
chdir(dirname($0));
($file = basename($0)) =~ s/\.PL$//;
$file =~ s/\.pl$//
	if ($Config{'osname'} eq 'VMS' or
	    $Config{'osname'} eq 'OS2');  # "case-forgiving"
&readconfig('../config.sh') if -e '../config.sh'; # SFgate config
&readconfig('config.sh') if -e 'config.sh'; # SFgate config
open OUT,">$file" or die "Can't create $file: $!";

print "Extracting SFproxy/$file (with variable substitutions)\n";

# In this section, perl variables will be expanded during extraction.
# You can use $Config{...} to use Configure variables.

print OUT <<"!GROK!THIS!";
$Config{'startperl'}
!GROK!THIS!

# In the following, perl variables are not expanded during extraction.

print OUT <<'!NO!SUBS!';
eval 'exec perl -S $0 "$@"'
    if 0;

########################## -*- Mode: Perl -*- ##########################
##
## File             : SFproxy
##
## Description      : SFproxy an indexing http proxy server
##
#
# Copyright (C) 1995 Ulrich Pfeifer, Kai Grossjohann
#
# This file is part of SFgate.
#
# SFgate is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# SFgate is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with SFgate; see the file COPYING.  If not, write to
# the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
#
##
## Author           : Kai Grossjohann
## Created On       : Fri Jul  7 14:16:45 1995
##
## Last Modified By : Ulrich Pfeifer
## Last Modified On : Wed Mar  6 15:03:03 1996
##
## $State: Exp $
##
## $Id: SFproxy.PL,v 5.1 1996/11/05 16:52:47 goevert Exp goevert $
##
## $Log: SFproxy.PL,v $
## Revision 5.1  1996/11/05 16:52:47  goevert
## *** empty log message ***
##
## Revision 4.0.1.4  1995/12/04  15:42:55  pfeifer
## Small bug fixes.
## Support for request bodies added.  Needs Content-Length header, though.
##
## Revision 4.4  1995/11/13 14:29:03  grossjoh
## Small bug fixes.
##
## Revision 4.3  1995/11/13 14:26:39  grossjoh
## Support for request bodies added.  Needs Content-Length header,
## though.
##
## Revision 4.2  1995/10/10 09:13:42  grossjoh
## Print response first, then index, when in daemon mode.
##
## Revision 4.1  1995/10/09  14:33:28  grossjoh
## Closing connection to client after output of response.
##
## Revision 4.0  1995/09/29  17:54:22  pfeifer
## *** empty log message ***
##
## Revision 1.5  1995/09/07  13:55:00  grossjoh
## changed $http_proxy from www to fbi-www
## added user option --addurl
##
## Revision 1.5  1995/09/07  13:55:00  grossjoh
## changed $http_proxy from www to fbi-www
## added user option --addurl
##
## Revision 1.4  1995/08/30  10:45:40  grossjoh
## Used to convert everything to lower case.
## This does not seem to be necessary, and fouls up URLs beyond all
## repair.
##
## Revision 1.3  1995/08/15  11:48:52  goevert
## SFgate 4.0
##
## Revision 1.2  1995/07/21  14:00:39  goevert
## SFgate 4.0pre
##
## Revision 1.1  1995/07/21  13:59:20  goevert
## Initial revision
##
########################################################################


## #################################################################
## Configuration Variables
## #################################################################

## No of seconds to wait before retrying to access a locked file.  5
## means wait 5 seconds, -5 means wait (5 + ($$ % 10)) seconds.
$lockwait = -5;

## If a lock is older than this number of days it will be broken anyway.
$lockexpire = 1/48;

## The name of the WAIS database.
$database = "SFproxy-db";

## Do logging iff this is true.  See -log option.
$log = 1;

## ($indexprefix . $database . "." . $$) is the name of the files to
## be indexed with the waisindex program.
## $indexprefix must end with slash.
$indexprefix = "/tmp/"; 

## The name (possibly with path) of the WAISINDEX program.
$waisindex = '/usr/local/ls6/wais/bin/waisindex';
$waisindex = 'waisindex' unless -x $waisindex;
## No proxy is used if host matches this regexp .
$http_no_proxy = "informatik\\.uni-dortmund\\.de\$";

## Number of seconds to wait between checks for resource usage.
$resource_wait = 5;

## No more than this number of this resource may be used if the nice
## option is given.
$maxresource = 0.8;

## Maximum number of children to spawn.  -1 means no limit.
$maxchildren = -1;

## Maximum number of children to spawn if --maxchildren option is
## given without a number.
$maxchildren_default = 10;

## Maximum number of retries if something fails.
#$max_retries = 5;


## #################################################################
## Other Variables
## #################################################################

$debug = 0;                     ## produce debugging output if true
$ddebug = 0;                    ## produce more debugging output
$debug_fh = STDERR;             ## where to print debugging output

$http_port = 'http';            ## http port is often 80
chop($myhostname = `hostname`); ## name of local host
$logname = getlogin || (getpwuid($>))[0] || $ENV{'USER'};
                                ## login name
$sockaddr = 'S n a4 x8';        ## for sockets (I don't understand
                                ## this one)

!NO!SUBS!
use Socket qw(SOCK_STREAM PF_INET);

printf OUT "\$pf_inet          = %d;\n", PF_INET;
printf OUT "\$sock_stream      = %d;\n", SOCK_STREAM;
my($name, $aliases, $proto) = getprotobyname( 'tcp' );
printf OUT "\$tcp_proto        = %d;\n", $proto;
printf OUT "## The name of the http proxy or 0 if you don't want to use one.\n";
printf OUT "\$http_proxy       = '%s';\n", $CONFIG{'HTTPPROXY'};
printf OUT "## The port to use for proxy server.\n";
printf OUT "\$http_proxy_port  = '%s' || 'http';\n", $CONFIG{'HTTPPROXYPORT'};
printf OUT "## The directory where the WAIS database goes.  Must end with slash.\n";
printf OUT "\$dir              = '%s/';\n", $CONFIG{'DBDIR'};
printf OUT "## The name of the logfile.\n";
printf OUT "\$logfile          = '%s/SFproxy.log';\n", $CONFIG{'LOGDIR'};
print OUT <<'!NO!SUBS!';
$serverfh = '';                 ## file handle for server connection
$clientfh = '';                 ## file handle for client connection

%URLDB = ();                    ## a dbm file of indexed URLs
                                ## the name of the dbm file, usually
$urldbfilename = $database . "-url";;                 
                                ## $datbase . "-url"

$SIG{'INT'} = 'dokill';         ## signal handler for SIGKILL
$SIG{'ALRM'} = 'doalrm';        ## signal handler for SIGALRM

$child = 0;                     ## is ppid if this is a child process
$numchildren = 0;               ## number of currently spawned children
%children = ();                 ## associative array giving children
                                ## process status


## #################################################################
## Preliminary Stuff
## #################################################################

use Getopt::Long;


## #################################################################
## dokill -- signal handler for SIGKILL
## #################################################################
## This needs $urldbfilename
sub dokill
{
    ## local variables
    local( $cpid );

    print $debug_fh "Caught SIGKILL\n" if $debug;
    &close_url_db();
    &release_lock($urldbfilename);
    kill 9,$child if $child;
    if (!$child) {
        foreach $cpid ( keys(%children) ) {
            kill($cpid) if ( $children{$cpid} != 0 );
        }
    }
    shutdown(SERVERFH,2);
    close(SERVERFH);
    shutdown(CLIENTFH,2);
    close(CLIENTFH);
    shutdown(MYSERVERFH,2);
    close(MYSERVERFH);
}


## #################################################################
## doalrm -- signal handler for SIGALRM
## #################################################################
sub doalrm
{
    ## do nothing
    return 1;
}


## #################################################################
## set_variables
## #################################################################
## Mostly, this function reads the command line options.

sub set_variables
{
    ## local variables
    local( $port, $protocol, $path );

    $opt_debug = $opt_lockwait = $opt_lockexpire = -1;
    $opt_nice = $opt_nicewait = $opt_ddebug = -1;
    $opt_database = $opt_dir = $opt_indexprefix = -1;
    $opt_waisindex = $opt_proxy = $opt_maxchildren = -1;
    $opt_list = $opt_re = $opt_reindex = $opt_momspider = -1;
    $opt_mosaichotlist = $opt_server = $opt_proxyport = -1;
    $opt_daemon = $opt_urlfile = $opt_printurls = -1;
    $opt_noproxy = $opt_recreate = $opt_netscapehotlist = -1;
    $opt_addurl = -1;
    $opt_log = $opt_logfile = -1;
    $opt_fix = undef;
    &GetOptions('debug!',
                'ddebug!',
                'log',
                'fix',
                'logfile=s',
                'lockwait=i',
                'lockexpire=i',
                'nice:i',
                'nicewait=i',
                'dir=s',
                'database=s',
                'urlfile=s',
                'indexprefix=s',
                'waisindex=s',
                'proxy=s',
                'proxyport=s',
                'noproxy:s',
                'maxchildren:i',
                'addurl=s',
                'list=s',
                're=s',
                'reindex=n',
                'mosaichotlist=s',
                'netscapehotlist=s',
                'momspider=s',
                'daemon',
                'server=i',
                'printurls',
                'recreate') || die "Usage $0 ....\n";

    ## Do a sanity check on these parameters.
    die "set_variables($$): Value for option --database may not contain\n"
        . "any slashes. Use --dir option instead.\n"
            if (($opt_database != -1) && ($database =~ m:/:o));
    die "set_variables($$): Value for option --indexprefix must end with\n"
        . "slash.\n"
        if (($opt_indexprefix != -1) && ($opt_indexprefix !~ m:/$:o));

    ## Some options are mutually exclusive.
    if ( ($opt_list!=-1)
        + ($opt_addurl!=-1)
        + ($opt_mosaichotlist!=-1)
        + ($opt_momspider!=-1)
        + ($opt_netscapehotlist!=-1)
        + ($opt_server!=-1)
        + ($opt_daemon!=-1)
        + ($opt_printurls!=-1)
        + ($opt_recreate!=-1) > 1) {
        die "set_variables($$): Use at most one of --list, --mosaichotlist,\n"
            . "--netscapehotlist, --momspider, --server, --daemon,\n"
                . "--printurls, --recreate, --addurl.\n";
    }
    $waisindex = $opt_waisindex if $opt_waisindex && -x $opt_waisindex;
    $debug = 1 if ($opt_debug!=-1);
    $ddebug = 1 if ($opt_ddebug!=-1);
    $lockwait = $opt_lockwait if ($opt_lockwait != -1);
    $lockexpire = $opt_lockexpire if ($opt_lockexpire != -1);
    $maxresource = ($opt_nice/100)
        if (($opt_nice != -1) && ($opt_nice != 0));
    $resource_wait = $opt_nicewait if ($opt_nicewait != -1);
    $dir = $opt_dir if ($opt_dir != -1);
    ## Add trailing slash if missing.
    $dir = $dir . "/" if ($dir !~ m:/$:o);
    $database = $opt_database if ($opt_database != -1);
    ## Default name for file of URLs.
    $urldbfilename = $database . "-url";
    ## Default may be overridden with --urlfile option.
    $urldbfilename = $opt_urlfile if ($opt_urlfile != -1);
    $indexprefix = $opt_indexprefix if ($opt_indexprefix != -1);
    $http_proxy = $opt_proxy if ($opt_proxy != -1);
    $http_proxy = 0 if ($opt_proxy eq "");
    $http_proxy_port = $opt_proxyport if ($opt_proxyport != -1);
    $http_no_proxy = $opt_noproxy if ($opt_noproxy != -1);
    ## If no string is given, use no proxy at all.
    $http_proxy = 0 if ($opt_noproxy eq "");
    $maxchildren = $opt_maxchildren if ($opt_maxchildren != -1);
    ## Use default number of max children if no number is given.
    $maxchildren = $maxchildren_default if ($maxchildren == 0);
    $logfile = $opt_logfile if ($opt_logfile != -1);
    $log = 1 if ($opt_log != -1);
    if ( $opt_mosaichotlist != -1 ) {
        $opt_list = $opt_mosaichotlist;
        $opt_re = "^([^ \t]+) .*";
        $opt_reindex = 0;
    }
    if ($opt_netscapehotlist != -1 ) {
        $opt_list = $opt_netscapehotlist;
        $opt_re = '^    <DT><A HREF=.([^"]+)';
        $opt_reindex = 0;
    }
    if ( $opt_momspider != -1 ) {
        $opt_list = $opt_momspider;
        $opt_re = "^Testing ([^ \t]+) \.\.\. 200\$";
        $opt_reindex = 0;
    }
    if ($opt_fix) {
        chdir($dir);
        &fix_seen;
    }
    ## If $http_proxy is http://HOST:PORT, we take it apart and set
    ## $http_proxy and $http_proxy_port correctly.
    if ( $http_proxy =~ m:^http:o ) {
        ( $protocol, $http_proxy, $port, $path ) =
            &parse_url($http_proxy);
        $port = $protocol if ! $port;
        $port = $http_proxy_port if ! $port;
        $http_proxy_port = $port;
    }

}

## #################################################################
## #################################################################
## Entry point functions
## #################################################################
## #################################################################


## #################################################################
## do_add_url($url)
## #################################################################
## Given the $url, it is added to the WAIS database $database.  The
## actual indexing is done in a child process.  The number of
## concurrent child processes can be controlled two ways:
##   - The -maxchildren option tells SFproxy not to fork more than
##     this number of children concurrently.
##   - The -nice option tells SFproxy not to use more than the
##     specified percentage of system resources, as given by
##     pstat.

sub do_add_url
{
    local( $url ) = @_;
    ## local variables
    local( $serverprotocol, $serverhost, $serverport, $serverurl );
    local( $childpid, $serverfh );
    local( $old0 );

    $old0 = $0;
    $0 = "SFproxy::do_add_url $url";
    ( $serverprotocol, $serverhost, $serverport, $serverurl ) =
        &parse_url($url);
    if ( $http_proxy && $serverhost !~ m:$http_no_proxy: ) {
        $serverprotocol = "http";
        $serverhost = $http_proxy;
        $serverport = $http_proxy_port;
        $serverurl = $url;
    }

    ## Wait for a child to die if max number of children is exceeded.
    while ( ($maxchildren != -1) && ($numchildren >= $maxchildren) ) {
        print($debug_fh "do_add_url($$): Number of children exceeded,"
              . " waiting for child to die.\n")
            if $debug;
        $childpid = wait();
        die "do_add_url($$): wait(): no more children -- $?\n"
            if ($childpid == -1);
        print($debug_fh "do_add_url($$): Child died, continuing.\n") if $debug;
        $numchildren--;
        $children{$childpid} = 0 if ($childpid != -1);
    }

    ## Wait for sufficiently low resource usage if appropriate.
    &wait_low_resources() if ($opt_nice != -1);

    ## Fork now to have child do the indexing.
    $child = fork();
    if ( ! defined($child) ) {
        die "do_add_url($$): Couldn't fork.\n";
    } elsif ($child != 0) {
        ## parent process
        $numchildren++;
        $children{$child} = 1;  ## running
        print($debug_fh "do_add_url($$): parent process returning\n")
            if $ddebug;
        $0 = $old0;
        return 0;
    } else {
        ## child process
        $serverfh = &open_server($serverhost, $serverport);
        &put_request($serverfh, "GET $serverurl HTTP/1.0\r\n\r\n");
        ( $response_header, $response_body, $status_code,
         $reason_phrase, $content_type, $http_version )
            = &get_response($serverfh);
        close($serverfh);
        &index_maybe($response_body, $url, $status_code, $content_type);
        print($debug_fh "do_add_url($$): child process exiting\n")
            if $debug;
        exit();
    }
    die "do_add_url($$): THIS CANNOT HAPPEN!!!!!\n";
}


## #################################################################
## do_list($listfn, $re, $reindex)
## #################################################################
## Iterate over a list of URLs, indexing each of them.  A regexp and
## an integer are taken into account as follows.
##
## $listfn: Name of file containing list of URLs.  `-' means stdin.
## $re: Regular expression.  Is applied to each line of the input
##      file.  If a line matches this regexp, the matching part is
##      considered to be the URL, but see below.
## $reindex: If this is set, a pair of parentheses is taken into
##      account, ie only that part of the line that matches
##      subexpression between the parentheses is considered to be the
##      URL.  0 means first pair of parentheses.
##
## For processing each URL, the function do_add_url is used.

sub do_list
{
    local( $listfn, $re, $reindex ) = @_;
    ## return value
    local( $result, $old0 );

    $old0 = $0;
    $0 = "SFproxy::do_list";

    $re = -1 unless defined($re);
    $reindex = -1 unless defined($reindex);

    if ( $listfn ne "-" ) {
        open(STDIN, $listfn) || die "could not open $listfn: $!\n";
    }

    print($debug_fh "do_list($$): Reading URLs from $listfn\n") if $debug;

    $result = 0;
    while ( ($url = <STDIN>) && (! $result) ) {
        chop($url);
        if ( $re != -1 ) {
            if ( $reindex != -1 ) {
                $url = ($url =~ m/$re/)[$reindex];
            } else {
                $url = ($url =~ m/$re/);
            }
        }
        if ( $url =~ m/[a-zA-Z]/ ) {
            print($debug_fh "do_list($$): Adding URL <$url>\n") if $debug;
            $result = &do_add_url($url);
        }
    }
    print($debug_fh "do_list($$): ENDING\n") if ($ddebug);
    $0 = $old0;
    return $result;
}

    
## #################################################################
## do_server($port)
## #################################################################
## Start a server, accepting connections on the given port.  Each
## connection is a request/response pair.  If the request is a GET and
## the response contains text/html data, the document together with
## the URL are indexed.
##
## Mostly, this is snarfed from the Perl man page, section a sample
## tcp server.
##
## For processing a request a child is forked.  The number of children
## running concurrently can, at the moment, not be limited in any way,
## though for do_list this can be done.  Ie the -maxchildren and -nice
## options do not take effect in this function.

sub do_server
{
    local( $port ) = @_;
    ## local variables
    local( $name, $aliases, $this, $af, $inetaddr, $addr );
    local( $oldfh );
    local( $req, $method, $url );
    local( $serverport, $path );
    local( $old0 );

    $old0 = $0;
    $0 = "SFproxy::do_server";

    #($name, $aliases, $proto) = getprotobyname('tcp');
    ($name, $aliases, $port) = getservbyname($port, 'tcp')
        unless $port =~ /^\d+$/;

    $this = pack($sockaddr, $pf_inet, $port, "\0\0\0\0");

    select(CLIENTFH); $| = 1; select(STDOUT);

    socket(MYSERVERFH, $pf_inet, $sock_stream, $tcp_proto)
        || die "do_server($$): socket: $!";
    bind(MYSERVERFH, $this) || die "do_server($$): bind: $!";
    listen(MYSERVERFH, 5) || die "do_server($$): connect: $!";

    $oldfh = select; select(MYSERVERFH); $| = 1; select($oldfh);

    for (;;) {
        print $debug_fh "do_server($$): Listening again\n" if $debug;
        ($addr = accept(CLIENTFH, MYSERVERFH)) || die $!;
        print $debug_fh "do_server($$): accept ok\n" if $debug;

        ($af,$port,$inetaddr) = unpack($sockaddr,$addr);
        @inetaddr = unpack('C4',$inetaddr);
        print $debug_fh "do_server($$): $af $port @inetaddr\n" if $debug;

        $child = fork();
        if ( ! defined($child) ) {
            die "do_server($$): Coudn't fork.";
        } elsif ( $child != 0 ) {
            ## parent
            print $debug_fh "do_server($$): parent, awaiting next request\n"
                if $debug;
            ## do nothing, serve next request
        } else {
            ## child
            print $debug_fh "do_server($$): child, calling do_daemon\n"
                if $debug;
            &do_daemon(CLIENTFH, CLIENTFH);
            print $debug_fh "do_server($$): child, exiting\n"
                if $debug;
            exit()
        }
    }
    
}


## #################################################################
## do_daemon($clientfh_in, $clientfh_out)
## #################################################################
## From the filehandle $clientfh, an HTTP request is read and passed
## on to the real server.  If the request was a GET and the server
## responds with a document of type text/html, the document is indexed
## via waisindex.  In all cases the response from the server is passed
## back to the client.
##
## SFproxy can talk to another proxy, if desired.  If the variable
## $http_proxy is set, the request is forwarded to the host
## $http_proxy on port $http_proxy_port.  However, if the host in the
## URL of the request matches the regexp $http_no_proxy, the proxy is
## not used.
##
## After one request and the corresponding response have been
## processed, the function terminates.

sub do_daemon
{
    local( $clientfh_in, $clientfh_out ) = @_;
    ## local variables
    local( $req, $method, $url, $protocol, $serverhost, $serverport,
          $serverurl, $serverfh );
    local( $response_header, $response_body, $status_code,
          $reason_phrase, $content_type, $http_version );
    local( $old0 );

    $old0 = $0;
    $0 = "SFproxy::do_daemon";

    ($req, $method, $url) = &get_request($clientfh_in);
    print($debug_fh "do_daemon($$): Received request $method $url\n")
        if $debug;
    print $debug_fh "do_daemon($$): Request follows:\n***BEGIN**\n"
        . "$req\n**END**\n" if $ddebug;
    ( $serverprotocol, $serverhost, $serverport, $serverurl ) =
        &parse_url($url);
    if ( $http_proxy && $serverhost !~ m:$http_no_proxy: ) {
        print $debug_fh "do_daemon($$): Contacting proxy\n"
            if $debug;
        $serverprotocol = "http";
        $serverhost = $http_proxy;
        $serverport = $http_proxy_port;
        $serverurl = $url;
    }
    print($debug_fh
          "do_daemon($$): Contacting host $serverhost on port $serverport.\n")
        if $debug;
    $serverfh = &open_server($serverhost, $serverport);
    &put_request($serverfh, $req);
    ( $response_header, $response_body, $status_code,
            $reason_phrase, $content_type, $http_version )
        = &get_response($serverfh,$clientfh_out);
    print $debug_fh "do_daemon($$): Received response $status_code"
        . " $reason_phrase $content_type (version $http_version)\n" if $debug;
    print $debug_fh "do_daemon($$): Response follows.\n**BEGIN**\n"
        . "$response_header\n$response_body\n**END**\n" if $ddebug;
    &put_response($clientfh_out); #$response_header . $response_body);
    &index_maybe($response_body, $url, $status_code, $content_type);
    $0 = $old0;
}


## #################################################################
## do_recreate
## #################################################################
## If there is a database containing a number of URLs already, this
## function can be used to reindex all of them.  WAIS doesn't seem to
## have a way to update single documents, so the entire database needs
## to be recreated from time to time.

sub do_recreate
{
    local( $dir, $database, $urldbfilename ) = @_;
    ## local variables
    local( %NURLDB );
    local( $xurl, $old0 );

    $old0 = $0;
    $0 = "SFproxy::do_recreate";

    &open_url_db($urldbfilename);
    %NURLDB = %URLDB;
    &close_url_db();
    ## CCC: Do we need to unlink more files here?
    unlink "$urldbfilename.dir";
    unlink "$urldbfilename.pag";
    unlink "$urldbfilename";
    unlink "$database.doc";
    %URLDB = ();
    foreach $xurl (keys %NURLDB) {
        print $debug_fh "do_recreate:($$): Adding URL $xurl\n" if $debug;
        &do_add_url($xurl);
    }
    $0 = $old0;
}    


## #################################################################
## do_print_urls($urlfile)
## #################################################################
## Prints the URLs saved in the database belonging to the URL file
## given.

sub do_print_urls
{
    local( $urlfile ) = @_;
    ## local variables
    local( $key, $value );
    local( $old0 );

    $old0 = $0;
    $0 = "SFproxy::do_print_urls";
    &open_url_db($urlfile);
    while (($key,$value) = each %URLDB) {
        print "$key = $value\n";
    }
    &close_url_db();
    $0 = $old0;
}


## #################################################################
## #################################################################
## Utility Functions
## #################################################################
## #################################################################

## #################################################################
## create_fmt($fn)
## #################################################################
## This is the default format file used for indexing the HTML
## documents encountered.  The format is written to the file $fn.fmt.

sub create_fmt {
    local( $fn ) = @_;

    open(FMT, ">$fn.fmt");
    print FMT <<'EOFMT'
<record-end> //

<layout>
<headline> /<title>/ /<\/title>/ 
        80 /<title>/
<end>

<field> /<title>/
ti "Title" stemming TEXT BOTH
<end> /<\/title>/

<field> /<h[0-9]>/
hl "Headline" stemming TEXT BOTH
<end> /<\/h[0-9]>/

<field> /<dt>/
dt "Description Term" stemming TEXT BOTH
<end> /<d[dt]>/

<field> /<address>/
ad "Description Term" stemming TEXT BOTH
<end> /<\/address>/

<field> /./
stemming TEXT GLOBAL
<end> /$/

EOFMT
    ;
    close(FMT);
}


## #################################################################
## get_request($clientfh)
## #################################################################
## Get an HTTP request from the client, via the file handle $clientfh.
## Return values are $req, $method, $url.  SFproxy doesn't really
## understand about any other requests than HTTP/1.0 requests.
##
## $req: This is the complete text of the request, as sent from the
##      client.
## $method: This is the request method, eg `GET' or `POST'.  In
##      SFproxy, only GET requests are handled specially.
## $url: This is the URL mentioned in the request.

sub get_request
{
    local( $clientfh ) = @_;
    ## return values
    local( $req, $method, $url );
    ## local variables
    local( $req_line, $line );
    local( $http_version, $full_request );
    local( $old0 );
    local( %req_headers );
    local( $req_header_name, $req_header_val );
    local( $content_length );

    $old0 = $0;
    $0 = "SFproxy::get_request";

    $req = "";
    $method = "";
    %req_headers = ();

    $req_line = <$clientfh>;
    $req = $req . $req_line;

    ## Check for HTTP/0.9 vs HTTP/1.0 requests
    $full_request =
        ( $req_line !~ m:^GET\s+\S+\s*\r$:o );
    if ( $full_request ) {
        ## If we think this is a full request, we make a sanity check.
        if ( $req_line =~ m:HTTP/[0-9.]+\s*\r$:o ) {
            die "get_request($$): Unknown protocol version in request: $req_line"
                if ( $req_line !~ m:HTTP/1.0\s*\r$:o );
        }
        ## Fill in method and url
        ($method, $url) =
            ($req_line =~ m:^(\S+)\s+(\S+)\s+HTTP/1.0\s*\r$:o);
    } else {
    ## An HTTP/0.9 request is finished after the first line.
        $url = ( $req_line =~
              m:^GET\s+(\S+)\s*\r$:o )[0];
        $0 = $old0;
        return ($req, "GET", $url);   ## method is always "GET" for old style
    }
    ## Read rest of HTTP/1.0 request header
    while ( $line = <$clientfh> ) {
        $req = $req . $line;
        ## A CRLF alone is the end of the request
        last if ( $line =~ m/^\r$/o );
        ($req_header_name, $req_header_val)
            = ( $line =~ m/(^[^:]+):\s*(.*)$/ );
        $req_header_name =~ tr/A-Z/a-z/;
        chop($req_header_val) if ( $req_header_val =~ m/\r$/ );
        # This doesn't work for several lines with the same header name!
        $req_headers{$req_header_name} = $req_header_val;
        #print $debug_fh $req_header_name . "::"
        #    . $req_headers{$req_header_name} . "::" . $req_header_val . "\n"
        #        if $debug;
    }

    ## Read rest of HTTP/1.0 request body, if appropriate
    if ( defined($req_headers{'content-length'}) ) {
        #print $debug_fh "Detected Content-Length header\n" if $debug;
        $content_length = $req_headers{'content-length'} + 0;
        ( $content_length == read($clientfh, $line, $content_length) )
            || die "get_request: couldn't read $content_length bytes"
                . " of req body";
#        $bytes = 0;
#        while ( $bytes < $content_length ) {
#            $line = <$clientfh>;
            $req .= $line;
#            $bytes += length($line);
#        }
    }

    $0 = $old0;
    return ($req, $method, $url);
}


## #################################################################
## put_request($serverfh, $req)
## #################################################################
## Sends the request $req to the HTTP server, vie the file handle
## $serverfh.

sub put_request
{
    local( $serverfh, $req ) = @_;
    ## local variables
    local( $old0 );

    $old0 = $0;
    $0 = "SFproxy::put_request $url";
    print($serverfh $req);
    $0 = $old0;
}


## #################################################################
## get_response($serverfh, $clientfh)
## #################################################################
## Gets a response from the HTTP server (via $serverfh).  HTTP/1.0
## responses are understood.  If the response is HTTP/1.0, the return
## values are $response_header, $response_body, $status_code,
## $reason_phrase and $content_type.
##
## $response_header: The header of the response, as defined in
##      HTTP/1.0.  (Ie the part before the first empty line.)
## $response_body: The body of the response, as defined in HTTP/1.0.
##      (Ie the part after the first empty line.)
## $status_code: The first line of the response contains a status code
##      and a reason phrase, as defined in HTTP/1.0.  This is the
##      status code part of that line.
## $reason_phrase: This is the reason phrase of that line.
## $content_type: The value of the Content-Type header field.

sub get_response
{
    local( $serverfh, $clientfh ) = @_;
    ## return value
    local( $response_header, $response_body,
       $status_code, $reason_phrase, $content_type );
    ## local variables
    local( $line, $in_header, $first_line );
    local( $header_field, $header_value, $http_version );
    local( $full_response, $old0 );

    $old0 = $0;
    $0 = "SFproxy::get_response $url";
    $response_header = "";
    $response_body = "";
    ## At the beginning of the response, we're reading a header
    ## information.
    $in_header=1;               ## true
    $first_line=1;              ## true
    ## We assume we're talking HTTP 1.0.
    $http_version = "1.0";
    ## We assume we'll get a full response, as defined by HTTP 1.0.
    $full_response = 1;
    while ( $line = <$serverfh> ) {
        if ($clientfh) {
            print ($clientfh $line);
        }
        ## The first line gets special processing.  It decides between
        ## old (0.9) and new (1.0) HTTP.
        if ( $first_line ) {
            print $debug_fh "get_response($$): Reading first line, follows\n"
                . "$line" if $debug;
            $first_line = 0;
            ## Weed out unknown protocol versions first.
            ($http_version, $status_code, $reason_phrase) =
#                ($line =~ m:^HTTP/([0-9.]+)[ \t]+([0-9]+)[ \t]+(.*)\r:o);
                ($line =~ m:HTTP/([0-9.]*) ([0-9]*) (.*):o);
            print $debug_fh "get_response($$): Version $http_version\n";
            print $debug_fh "get_response($$): Status code $status_code\n"
                if $debug;
            print $debug_fh "get_response($$): Reason phrase $reason_phrase\n"
                if $debug;
            if ( $http_version ne "1.0" ) {
                die "get_response($$): Unsupported protocol version $http_version\n";
            }
            if ( $http_version eq "1.0" ) {
                $full_response = 1;
            }
        }
        $in_header=0 if ( $line =~ m/^(\r)?$/o );
        $in_header=0 if ( ! $full_response );
        if ( $in_header ) {
            ($header_field, $header_value) =
                ( $line =~ m/^([-a-z]+): (.*)$/io );
            $header_value =~ s/\r$//o;
            $header_field =~ tr/A-Z/a-z/;
            $content_type = $header_value 
                if ( $header_field eq "content-type" );
        }
        if ( $in_header ) {
            $response_header = $response_header . $line;
        } else {
            $response_body = $response_body . $line;
        }
    }

    $0 = $old0;
    return ( $response_header, $response_body, $status_code,
            $reason_phrase, $content_type, $http_version );
}


## #################################################################
## put_response($clientfh, $response)
## #################################################################
## This sends the response from the server to the client, via
## $clientfh.

sub put_response
{
    local( $clientfh, $response ) = @_;
    ## local variables
    local( $old0 );

    $old0 = $0;
    $0 = "SFproxy::put_response";
    print( $clientfh $response ) if length($response);
    close($clientfh);
    $0 = $old0;
}


sub decode
{
    local($_) = @_;
    
    s/\+/ /g;
    s/%([\dA-F]{2})/pack('C',hex($1))/eig;
    s/\s+/ /g;
        
    $_;
}

sub fix_seen {
    local($file) = "$database.fn";
    local($/) = "\000\000\000\000URL\000";
    local(%SEEN);

    &get_lock($file);
    open(IN, "<$file")
        || warn "Could not open $file: $!\n";
    do {} until &get_lock($urldbfilename);
    &open_url_db($urldbfilename);
    %URLDB = ();
    while (<IN>) {
        chomp;
        s/\000//g;
        print STDERR "$_\n";
	$URLDB{$_}++;
    }
    &close_url_db();
    close IN;
    &release_lock($file);
    &release_lock($urldbfilename);
}

## #################################################################
## index_response($response, $url)
## #################################################################
## This indexes a response from an HTTP server.  Please note that
## $response is the body part of the response only.  $url is the URL
## in the request that led to this response.
##
## This function is called by index_maybe.

sub index_response
{
    local( $response, $url ) = @_;
    ## local variables
    local( $indexfn );
    local( $add, $error, $errline, $old0 );

    $old0 = $0;
    $0 = "SFproxy::index_response $url";
    ## Normalize the HTML code.
    ## Convert all <TAG>s to <tag>s.
    $response =~ s:<(/?)(\w+)>:<$1\L$2\E>:g;
    ## Remove newlines and whitespace around <TITLE>...</TITLE>
    $response =~ s:<title>(\r\n\w)*:\n<title>:go;
    $response =~ s:(\r\n\w)*</title>:</title>\n:go;
    ##-This is nonsense?
    ##-## Convert whole body to lower case
    ##-$response =~ tr/A-Z/a-z/;

    ## Write the response to a file, ready for indexing.
    $indexfn = $indexprefix . $database . "." . $$;
    open(INDEXFH, ">$indexfn") || die "index_response($$): Cannot open index file $indexfn.\n";
    print($debug_fh $response) if $ddebug;
    print INDEXFH $response;
    close INDEXFH;

    ## Normalize the URL by converting to lower case and replacing all
    ## strange characters with something else.
    $url =~ s#http://([^/]+)/#http://\L$1\E/#;
    $url =~ tr/ \t/X/;
    $url = &decode($url);
    ## Call WAISINDEX program.
    $add = "";
    $add = "-a" if -e "$database.doc";
    unless ( -e "$database.fmt" ) {
        print $debug_fh "index_response($$): creating fmt\n"
            if $debug;
        &create_fmt($database);
    }
    do {} until &get_lock($database.'-waisindex');
    open(OUTFH, "$waisindex $add -t URL $indexfn "
         . "$url -t fields -d $database $indexfn 2>&1|")
        || die "index_response($$): Could not start $waisindex.\n";
    
    ## Check for error messages of WAISINDEX.
    while ( $errline = <OUTFH> ) {
        $error ++ if ( $errline =~ m/error/i );
    }
    close(OUTFH);
    release_lock($database.'-waisindex');
    unlink($indexfn);
    print($debug_fh "index_response($$): exiting\n") if $debug;
    $0 = $old0;
    return ( $error || $? );
}


## #################################################################
## index_maybe($response_body, $url, $status_code, $content_type)
## #################################################################
## If necessary, the $response_body is indexed by waisindex and stored
## together with the $url given.  To do the actual indexing, the
## function index_response is called.  This function only determines
## if indexing is necessary.
##
## A response is indexed if and only if
##   - the status code indicates successful service of the request,
##   - the content type is text/html,
##   - the response body is not empty,
##   - the $url hasn't been seen before,
##   - the $url isn't `strange', ie doesn't seem to be a database request.

sub index_maybe
{
    local( $response_body, $url, $status_code, $content_type ) = @_;
    ## local variables
    local( $old0 );

    $old0 = $0;
    $0 = "SFproxy::index_maybe $url";
    print $debug_fh "index_maybe($$): URL $url\n"
        if $debug;
    ## Nothing to do if status code doesn't indicate success.
    $status_code = $status_code . "";
    if ( $status_code !~ m/^20[0123]/o ) {
        print $debug_fh "index_maybe($$): nothing to do because"
            . " of status code $status_code\n" if $debug;
        $0 = $old0;
        return;
    }

    ## Nothing to do if content type isn't text/html.
    if ( $content_type ne "text/html" ) {
        print $debug_fh "index_maybe($$): nothing to do because"
            . " of content type $content_type\n" if $debug;
        $0 = $old0;
        return;
    }

    ## Nothing to do if response body seems to be empty.
    if ( $response_body !~ m:[A-Za-z]:o ) {
        print $debug_fh "index_maybe($$): nothing to do because"
            . " of emtpy response body\n" if $debug;
        $0 = $old0;
        return;
    }

    ## Nothing to do if URL has already been seen.
    if ( &seen_url($url, $urldbfilename) ) {
        print $debug_fh "index_maybe($$): nothing to do because"
            ." already seen\n" if $debug;
        $0 = $old0;
        return;
    }

    ## Nothing to do if URL is strange.
    if ( $url =~ m:[=?]:o ) {
        print $debug_fh "index_maybe($$): nothing to do because of"
            . " strange URL $url\n" if $debug;
        $0 = $old0;
        return;
    }

    ## Do the indexing.
    $0 = "SFproxy::index_maybe(5) $url";
    &note_url($url, $urldbfilename);
    print($debug_fh "index_maybe($$): Indexing $url\n") if $debug;
    print $log_fh "$$ indexing URL $url\n" if $log;
    print STDERR "$$ indexing URL $url\n" if $opt_list;
    &index_response($response_body, $url);

    print($debug_fh "index_maybe($$): exiting\n") if $debug;
    $0 = $old0;
}


## #################################################################
## parse_url($url)
## #################################################################
## The given $url is parsed, and its component parts are returned.  As
## an example, the component parts of the URL
## http://www.w3.org:80/home.html are:
##   - $protocol: is ftp, gopher, or http.  Here: http
##   - $host: the host name.  Here: www.w3.org
##   - $port: the port.  Defaults to 80 if not given.  Here: 80
##   - $path: the rest.  Here: /home.html

sub parse_url
{
    local( $url ) = @_;
    ## return values
    local( $protocol, $host, $port, $path );

    ( ($protocol, $host, $port, $path) =
     ( $url =~ m@([a-zA-Z]+)://([^:/]+)(:[0-9]+)?(.*)@go ) )
        || die "parse_url($$): Wrong format URL $url\n";
    $port =~ s/^://;
    $port = $port+0;
    $port = $http_port if (!$port);

    return ( $protocol, $host, $port, $path );
}


## #################################################################
## get_lock($fn)
## #################################################################
## Gets a lock on the file $fn.  Locks older than $lockexpire are
## discarded.  If a file is locked, we wait $lockwait seconds.  (But
## see the documentation of $lockwait.)

sub get_lock
{
    local( $fn ) = @_;
    ## local variables
    local( $locked, $i, $old0 );

    $old0 = $0; $0 = "SFproxy::get_lock $fn";
    print $debug_fh "get_lock($$): Creating lock file $fn.LOCK$$\n"
        if $debug;
    open(LOCK, ">$fn.LOCK$$") || die "get_lock($$): Can't create lock file\n";
    print(LOCK "$logname $myhostname $$\n");
    close(LOCK);

    for ($i=0;;$i++) {
        #$0 = "SFproxy::get_lock[$i] $fn";
        $locked = link("$fn.LOCK$$", "$fn.LOCK");
        print $debug_fh "get_lock($$): trying to get lock: $locked\n"
            if $debug;
        if ( (! $locked)
            && ( -M "$fn.LOCK" > $lockexpire ) ) {
            print $debug_fh "get_lock($$): expiring lock $fn.LOCK\n"
                if $debug;
            unlink "$fn.LOCK";
            $locked = link("$fn.LOCK$$", "$fn.LOCK");
        }
        last if $locked;
        sleep( ($lockwait>0) ? ($lockwait) : (-$lockwait + $$ % 10) );
    }
    print $debug_fh "get_lock($$): Got lock.\n" if $debug;
    return( $locked );
}


## #################################################################
## release_lock($fn)
## #################################################################
## A lock on the file $fn obtained with get_lock is released.

sub release_lock
{
    local( $fn ) = @_;

    print $debug_fh "release_lock($$): file $fn\n"
        if $debug;
    unlink("$fn.LOCK$$");
    unlink("$fn.LOCK");
}


## #################################################################
## open_url_db($urldbfilename)
## #################################################################
## URLs in the WAIS database are stored in a dbm file.  This function
## opens that file and associates it with the global variable %URLDB.

sub open_url_db
{
    local( $urldbfilename ) = @_;

    dbmopen(%URLDB, $urldbfilename, 0644) 
        || die "open_url_db $urldbfilename: $!\n";
}


## #################################################################
## close_url_db()
## #################################################################
## An URL dbm file opened with open_url_db is closed.

sub close_url_db
{
    dbmclose(%URLDB);
}


## #################################################################
## note_url($url, $urldbfilename)
## #################################################################
## We have seen the $url and take note of this fact in the dbm file
## $urldbfilename.  The dbm file is locked using get_lock.

sub note_url
{
    local( $url, $urldbfilename ) = @_;
    ## local variables
    local( $old0 );

    $0 = $old0;
    $0 = "SFproxy::note_url $url";
    ## Normalize URL first.
    $url =~ s#http://([^/]+)/#http://\L$1\E/#;
    $url = &decode($url);
    print $debug_fh "note_url($$): noting $url\n" if $debug;

    &get_lock($urldbfilename);
    &open_url_db($urldbfilename);
    $URLDB{$url} = 1;
    &close_url_db();
    &release_lock($urldbfilename);
    print $debug_fh "note_url($$): exiting\n" if $debug;
    $0 = $old0;
}


## #################################################################
## seen_url($url)
## #################################################################
## Returns true iff the $url is in the %URLDB (associative array
## associated with $urldbfilename).

sub seen_url
{
    local( $url, $urldbfilename ) = @_;
    ## local variables
    local( $old0 );

    $old0 = $0;
    $0 = "SFproxy::seen_url $url";
    ## Normalize URL first.
    $url =~ s#http://([^/]+)/#http://\L$1\E/#

    &open_url_db($urldbfilename);
    return ($URLDB{$url});
    &close_url_db();
    $0 = $old0;
}


## #################################################################
## open_server($serverhostname, $port)
## #################################################################
## Open a TCP connection on the $port given on the host
## $serverhostname.  Returns a file handle corresponding to this TCP
## connection.

sub open_server
{
    local( $serverhostname, $port ) = @_;
    ## return value
    local( $serverfh );
    ## local variables
    local( $path, $name, $aliases, $type, $len,
       $thisaddr, $thataddr);
    local( $i );

    print $debug_fh "open_server($$): Connecting to host"
        . " $serverhostname on port $port\n" if $debug;
    #($name, $aliases, $proto) = getprotobyname('tcp');
    ($name, $aliases, $port) = getservbyname($port, 'tcp')
        unless $port =~ /^\d+$/;
    ($name, $aliases, $type, $len, $thisaddr) =
        gethostbyname($myhostname);
    ($name, $aliases, $type, $len, $thataddr) =
        gethostbyname($serverhostname);

    $this = pack($sockaddr, $pf_inet, 0, $thisaddr);
    $that = pack($sockaddr, $pf_inet, $port, $thataddr);

    socket(SERVERFH, $pf_inet, $sock_stream, $tcp_proto)
        || die "open_server($$): socket: $!";
    bind(SERVERFH, $this)
        || die "open_server($$): bind: $!";
    connect(SERVERFH, $that)
        || die "open_server($$): connect failed to $serverhostname/$port: $!";

    $oldfh = select;
    select(SERVERFH);
    $| = 1;
    select($oldfh);

    return ( SERVERFH );
}


## #################################################################
## wait_low_resources($just_check)
## #################################################################
## Waits until resource usage is `low enough', then returns.  If
## $just_check is true, returns immediately instead, returning true
## iff the resourse usage is at the moment low enough.

sub wait_low_resources
{
    local( $just_check ) = @_;
    ## local variables
    local( $low_enough );
    local( $used, $max, $category );

    $low_enough = 0;

    do {
        $low_enough = 1;
        open(PSTAT, "pstat -T|");
        while (($pstatline = <PSTAT>) && ($low_enough)) {
            ($used, $max, $category) =
                ($pstatline =~ m:^\s*(\d+)/\s*(\d+)\s+(\w+)\s*$:o);
            if ($ddebug) {
                print($debug_fh "used/max: $used/$max, category $category");
                if (($max > 0) && ($used/$max > $maxresource)) {
                    print($debug_fh " FAILED\n");
                } else {
                    print($debug_fh " OK\n");
                }
            }
            $low_enough = 0 
                if (($max > 0) && ($used/$max > $maxresource));
        }
        close(PSTAT);
        sleep($resource_wait) unless ($low_enough);
    } until ($low_enough || $just_check);
    print($debug_fh "wait_low_resources: returning $low_enough\n")
        if $ddebug;
    return($low_enough);
}


## #################################################################
## print_settings
## #################################################################

sub print_settings
{
    print($debug_fh "Parameters:\n---------\n");
    print($debug_fh "debug: $debug\n");
    print($debug_fh "ddebug: $ddebug\n");
    print($debug_fh "lockwait: $lockwait\n");
    print($debug_fh "lockexpire: $lockexpire\n");
    if ( $opt_nice != -1 ) {
        print($debug_fh "nice: $maxresource\n");
    } else {
        print($debug_fh "nice: no\n");
    }
    print($debug_fh "nice wait: $resource_wait\n");
    print($debug_fh "dir: $dir\n");
    print($debug_fh "database: $database\n");
    print($debug_fh "urlfile: $urldbfilename\n");
    print($debug_fh "indexprefix: $indexprefix\n");
    print($debug_fh "waisindex: $waisindex\n");
    print($debug_fh "http_proxy: $http_proxy\n");
    print($debug_fh "http_proxy_port: $http_proxy_port\n");
    print($debug_fh "http_no_proxy: $http_no_proxy\n");
    print($debug_fh "maxchildren: $maxchildren\n");
    print($debug_fh "list: $opt_list\n");
    print($debug_fh "re: $opt_re\n");
    print($debug_fh "reindex: $opt_reindex\n");
    print($debug_fh "logfile: $logfile\n");
}



## #################################################################
## Main Program
## #################################################################

&set_variables();
chdir($dir);
if ( $log ) {
    open(LOG, ">>$logfile");
    $oldfh = select;
    select(LOG);
    $debug_fh = select;
    $log_fh = select;
    select($oldfh);
}
$oldfh = select;
select($debug_fh); $| = 1;
select($oldfh);

&print_settings() if $debug;
print $debug_fh "chdir: $dir\n" if $debug;


if ( $opt_nice != -1 ) {
    die "main($$): Resource usage too high already.\n"
        if (!&wait_low_resources(1));
}

if ( $opt_addurl != -1) {
    print($debug_fh "Adding a single URL.\n") if $debug;
    &do_add_url($opt_addurl);
}
if ( $opt_list != -1 ) {
    print($debug_fh "Running in process-list mode.\n") if $debug;
    &do_list($opt_list, $opt_re, $opt_reindex);
}
if ( $opt_server != -1 ) {
    print($debug_fh "Running in server mode.\n") if $debug;
    &do_server($opt_server);
}
if ( $opt_daemon != -1 ) {
    print($debug_fh "Running in daemon mode.\n") if $debug;
    &do_daemon(STDIN, STDOUT);
}
if ( $opt_printurls != -1 ) {
    print($debug_fh "Running in printurls mode.\n") if $debug;
    &do_print_urls($urldbfilename);
}
if ( $opt_recreate != -1 ) {
    print $debug_fh "Running in recreate mode.\n" if $debug;
    &do_recreate($dir, $database, $urldbfilename);
}

print $debug_fh "SFproxy($$): waiting for all children to die.\n"
    if $debug;
while ( wait != -1 ) {};
print $debug_fh "SFproxy($$): all children dead.\n" if $debug;

close($debug_fh) if ($logfile);

## Local Variables:
## comment-start: "## "
## End:
!NO!SUBS!
close OUT or die "Can't close $file: $!";
chmod 0755, $file or die "Can't reset permissions for $file: $!\n";
exec("$Config{'eunicefix'} $file") if $Config{'eunicefix'} ne ':';

sub readconfig {
    my $file = shift;
    open(IN, "<$file") || die "Could not open $file: $!";
    while (<IN>) {
        if (/^([\w_]+)=\'(.*)\'$/) {
            my $left   = $1;
            my $right  = $2;
            if ($right =~ s/\$\{([\w_]+)\}/\$CONFIG\{\'$1\'}/) {
                eval "\$right = \"$right\"";
            }
            $CONFIG{$left} = $right;
            print "CONFIG{$left} = $right\n" if $debug;
        }
    }
}
