/**
 ** Routines that deal with the batchfile.
 **
 ** $Id: batchfile.c,v 1.12 1993/06/04 19:29:36 alden Exp $
 **
 ** $Log: batchfile.c,v $
 ** Revision 1.12  1993/06/04  19:29:36  alden
 ** Modified to only check return status of closedir() if ! VOID_CLOSEDIR
 **
 ** Revision 1.11  1993/06/04  19:23:07  alden
 ** Fixed logic error in update_batchfile()
 **
 ** Revision 1.10  1993/05/17  15:03:52  root
 ** Modified parse_entry() call to include type of entry
 **
 ** Revision 1.9  1993/05/12  13:05:47  alden
 ** Added () inside an if
 **
 ** Revision 1.8  1993/05/11  18:23:53  alden
 ** Fixed loop in update_batchfile() whereby if Reset_signaled is TRUE and
 **     check_sleep() is called, then check_sleep would call update_batchfile()
 **     which would call check_sleep(), etc.
 **
 ** Revision 1.7  1993/05/05  18:40:17  root
 ** Modified update_batchfile() so it removes 0 length batchfiles before
 **     return()'ing
 **
 ** Revision 1.6  1993/05/04  23:37:14  alden
 ** Cleaned up tabs
 **
 ** Revision 1.5  1993/05/04  20:41:02	alden
 ** Modified update_batchfile() to rewrite the batchfile even if we are in
 **	the One_shot mode
 **
 ** Revision 1.4  1993/04/19  21:41:50	alden
 ** Changed the wording on a couple of dlog() messages
 **
 ** Revision 1.3  1993/04/17  23:03:31	alden
 ** Changed log() messages to include filename
 ** Changed check_batchfile() to look for <batchfile>.<pid> and send it if
 **	found
 **
 ** Revision 1.2  1993/04/16  14:44:09	alden
 ** Changed update_batchfile() to call write_link_datafile() so the datafile
 **	will contain the updated information.
 **
 ** Revision 1.1  1993/03/30  13:19:13	alden
 ** Initial revision
 **
 **
 **/
#include "conf.h"
#include "readline.h"
#include "nntplink.h"
#include "strfuns.h"

#ifdef _AIX		/* AIX needs this to use native mode compiler */
#include <sys/select.h>
#endif

#include <sys/stat.h>

extern char *E_closedir;
extern char *E_fopen;
extern char *E_fseek;
extern char *E_fstat;
extern char *E_open;
extern char *E_read;
extern char *E_rename;
extern char *E_unlink;
extern char *E_write;

extern Boolean Abort_signaled;
extern Boolean Debug;
extern long Dtablesize;
extern long Idle_time;
extern int Input_from;
extern Boolean One_shot;
extern pid_t Prog_pid;
extern Boolean Queue_backlog;
extern FILE *Queue_fp;
extern Boolean Reset_signaled;

extern void close_article();
extern void check_sleep();
extern void fail();
extern void log();
extern Boolean parse_entry();

Boolean open_batchfile();

int No_articles = 0;

void
  check_batchfile()
{
  static char *fname = "check_batchfile: ";
  struct stat statb;
  char *bf;
  int bf_len, pid;
  DIR *bd;
  struct dirent *bdp;
  
  Batchfile.use = FALSE;
  
  if (!One_shot) {
    
    if (open_batchfile(Batchfile.nname, FALSE)) {
      Batchfile.nntp_in_use = TRUE;
      return;
    }
    
    if ((bd = opendir(Batchfile.dir)) != NULL) {
      
      bf = basename(Batchfile.name);
      bf_len = strlen(bf);
      
      for (bdp = readdir(bd); bdp != NULL; bdp = readdir(bd)) {
	
	if ((strncmp(bdp->d_name, bf, bf_len) != MATCH) ||
	    (bdp->d_name[bf_len] != '.') ||
	    ((pid = atoi(&bdp->d_name[bf_len + 1])) < 1))
	  continue;
	
	/**
	 ** If the pid is one, then we assume the user made the file
	 ** since a pid of 1 is usually reserved for <init>, otherwise
	 ** we check to see if the process actually exists before we
	 ** try to process the batchfile.
	 **/
	
	if ((pid == 1) ||
	    ((kill(pid, 0) == FAIL) && (errno != EPERM))) {
	  
	  bf = str3save(Batchfile.dir, "/", bdp->d_name);
	  
	  dlog(LOG_DEBUG, fname, "%sFound %s - renaming to %s\n",
	       bdp->d_name, basename(Batchfile.nname));
	  
	  /**
	   ** Rename <batchfile.oldpid> to <batchfile.mypid>. If it
	   ** fails and the error is ENOENT, then that probably
	   ** means some other nntplink process already grabbed it.
	   **/
	  
	  if (rename(bf, Batchfile.nname) != FAIL) {
	    free(bf);
	    break;
	  } else if (errno != ENOENT)
	    fail(fname, E_rename, Host.name, bf, Batchfile.nname,
		 errmsg(errno));
	  
	  free(bf);
	}
      }

#ifdef VOID_CLOSEDIR
      closedir(bd);
#else
      if (closedir(bd) == FAIL)
	fail(fname, E_closedir, Host.name, Batchfile.dir, errmsg(errno));
#endif
      
      if (open_batchfile(Batchfile.nname, FALSE)) {
	Batchfile.nntp_in_use = TRUE;
	return;
      }
    }
    
    if (Input_from & FLG_BATCHFILE) {
      
      dlog(LOG_DEBUG, fname, "%swaiting for %s: ", basename(Batchfile.name));
      
      while (stat(Batchfile.name, &statb) == FAIL)
	check_sleep(Batchfile.nap_time, FALSE);
      
      dlog(LOG_DEBUG, "", "%sGot it\n");
    }
  }
  
  if ((Input_from & FLG_BATCHFILE) || !One_shot)
    if (open_batchfile(Batchfile.name, !(Input_from & FLG_BATCHFILE)))
      return;
  
  if (One_shot && (Input_from & FLG_BATCHFILE))
    fail(fname, "%s%s: %s not found on a One-shot\n", Host.name,
	 Batchfile.name);
  
  return;
}


void
  update_batchfile()
{
  static char *fname = "update_batchfile: ";
  char *entry, *filename = NULL, *mesgid = NULL;
  struct stat batch_stat, log_stat;
  Boolean sig_reset;
  int fd;
  
  No_articles = 0;
  sig_reset = Reset_signaled;
  Reset_signaled = FALSE;
  
  if (sig_reset && (Input_from & FLG_LOGFILE))
    log(LOG_INFO, fname, "%s%s: Resetting to use new logfile\n", Host.name);
  
  if (Debug)
    if (Input_from & FLG_LOGFILE)
      log(LOG_DEBUG, fname, "%sUpdating %s from %s\n",
	  basename(Batchfile.nname), basename(Logfile.name));
    else if (Input_from & FLG_STDIN)
      log(LOG_DEBUG, fname, "%sUpdating %s from <stdin>\n",
	  basename(Batchfile.nname));
    else
      log(LOG_DEBUG, fname, "%sUpdating %s from %s\n",
	  basename(Batchfile.nname), basename(Batchfile.name));
  
  if (Queue_backlog) {
    if (Queue_fp == NULL) {
      
      dlog(LOG_DEBUG, fname, "%sBegin queueing the backlog\n");
      if ((rename(Batchfile.nname, Batchfile.tmp) == FAIL) && errno != ENOENT)
	fail(fname, E_rename, Host.name, Batchfile.nname, Batchfile.tmp,
	     errmsg(errno));
      
      if ((Queue_fp = fopen(Batchfile.tmp, "a")) == NULL)
	fail(fname, E_fopen, Host.name, Batchfile.tmp, "a", errmsg(errno));
    }
  } else if ((Batchfile.fbp != NULL) && (sig_reset || Abort_signaled)) {
    
    if ((Queue_fp = fopen(Batchfile.tmp, "w")) == NULL)
      fail(fname, E_fopen, Host.name, Batchfile.tmp, "w", errmsg(errno));
    
    fd = fb_fileno(Batchfile.fbp);
    
    if (Article.filename != NULL) {
      No_articles++;
#ifdef BNEWS
      fprintf(Queue_fp, "%s\t%s\n", Article.filename, Article.mesgid);
#else
      fprintf(Queue_fp, "%s %s\n", Article.filename, Article.mesgid);
#endif
      close_article();
    }
    
    while ((entry = fb_readline(Batchfile.fbp, NULL)) != NULL) {
      No_articles++;
      if (fprintf(Queue_fp, "%s\n", entry) == FAIL)
	fail(fname, E_write, Host.name, Batchfile.tmp, errmsg(errno));
    }
    
    FCLOSE(Queue_fp);
    close(fb_fileno(Batchfile.fbp));
    fb_close(Batchfile.fbp);
    Batchfile.fbp = NULL;
    Batchfile.offset = Batchfile.inode = 0;
    
    if (rename(Batchfile.tmp, Batchfile.nname) == FAIL)
      fail(fname, E_rename, Host.name, Batchfile.tmp, Batchfile.nname,
	   errmsg(errno));
  }
  
  if ((Queue_fp == NULL) && ((Queue_fp = fopen(Batchfile.nname, "a")) == NULL))
    fail(fname, E_fopen, Host.name, Batchfile.nname, "a", errmsg(errno));
  
  if (Article.filename != NULL) {
    No_articles++;
#ifdef BNEWS
    fprintf(Queue_fp, "%s\t%s\n", Article.filename, Article.mesgid);
#else
    fprintf(Queue_fp, "%s %s\n", Article.filename, Article.mesgid);
#endif
    close_article();
  }
  
  if (Input_from & FLG_LOGFILE) {
    
    if (Logfile.fbp != NULL) {
#ifdef LOOKUP_ARTICLE
      Article.filename = strsave("/");
#endif
      while ((entry = fb_readline(Logfile.fbp, NULL)) != NULL)
	if (parse_entry(entry, Logfile_entry)) {
	  No_articles++;
#ifdef BNEWS
	  fprintf(Queue_fp, "%s\t%s\n", Article.filename, Article.mesgid);
#else
	  fprintf(Queue_fp, "%s %s\n", Article.filename, Article.mesgid);
#endif
	}
      
      if (fb_error(Logfile.fbp))
	log(LOG_INFO, fname, E_read, Host.name,
	    Logfile.name, errmsg(errno));
      
#ifdef LOOKUP_ARTICLE
      free(Article.filename);
#endif
      
      Article.filename = Article.mesgid = NULL;
      
      Logfile.offset = fb_tell(Logfile.fbp);
      close(fb_fileno(Logfile.fbp));
      fb_close(Logfile.fbp);
      Logfile.fbp = NULL;
    }
    
    (void) fflush(Queue_fp);
    
    if (ferror(Queue_fp))
      fail(fname, "%s%s: copy to %s failed: %s\n", Host.name,
	   Batchfile.nname, errmsg(errno));
    
    FCLOSE(Queue_fp);
    
    if (!One_shot && !Abort_signaled) {
      while ((fd = open(Logfile.name, O_RDONLY)) == FAIL)
	check_sleep(Logfile.nap_time, FALSE);
      
      Logfile.fbp = fb_fdopen(fd);
      
      if (fstat(fd, &log_stat) == FAIL)
	fail(fname, E_fstat, Host.name, Logfile.name, errmsg(errno));
      else
	if (Logfile.inode == log_stat.st_ino) {
	  if (fb_seek(Logfile.fbp, Logfile.offset, 0) == FAIL)
	    log(LOG_WARNING, fname, E_fseek, Host.name, Logfile.name,
		errmsg(errno));
	} else
	  Logfile.inode = log_stat.st_ino;
    }
    
  } else if (Input_from & FLG_STDIN) {
    
    if ((Article.filename != NULL) && Abort_signaled) {
#ifdef BNEWS
      fprintf(Queue_fp, "%s\t%s\n", Article.filename, Article.mesgid);
#else
      fprintf(Queue_fp, "%s %s\n", Article.filename, Article.mesgid);
#endif
    } else {
      filename = Article.filename;
      mesgid = Article.mesgid;
    }
    
    while((entry = fb_readline(Stdin, NULL)) != NULL)
      if (parse_entry(entry, Stdin_entry)) {
	No_articles++;
#ifdef BNEWS
	fprintf(Queue_fp, "%s\t%s\n", Article.filename, Article.mesgid);
#else
	fprintf(Queue_fp, "%s %s\n", Article.filename, Article.mesgid);
#endif
      }
    
    Article.filename = filename;
    Article.mesgid = mesgid;
    
#ifdef _POSIX_VERSION
    if (fb_error(Stdin) && (errno != EAGAIN))
#else
      if (fb_error(Stdin) && (errno != EWOULDBLOCK))
#endif
	log(LOG_INFO, fname, "%s%s: error reading stdin: %s\n", Host.name,
	    errmsg(errno));
    
    (void) fflush(Queue_fp);
    
    if (ferror(Queue_fp))
      fail(fname, "%s%s: copy to %s failed: %s\n", Host.name,
	   Batchfile.nname, errmsg(errno));
    
    if (Abort_signaled) {
      FCLOSE(Queue_fp);

      if (Queue_backlog) {
	dlog(LOG_DEBUG, fname, "%sEnd queueing the backlog\n");
	if ((rename(Batchfile.tmp, Batchfile.nname) == FAIL) &&
	    errno != ENOENT)
	  fail(fname, E_rename, Host.name, Batchfile.tmp, Batchfile.nname,
	       errmsg(errno));
      }
    }
  }
  
  Batchfile.use = Batchfile.nntp_in_use = FALSE;
    
  if (stat(Batchfile.nname, &batch_stat) != FAIL)
    if (batch_stat.st_size == 0) {
      
      if ((unlink(Batchfile.nname) == FAIL) && errno != ENOENT)
	log(LOG_WARNING, fname, E_unlink, Host.name, Batchfile.nname,
	    errmsg(errno));
      
    } else if (!Queue_backlog && !One_shot)
      if (Batchfile.fbp == NULL) {

	if (open_batchfile(Batchfile.nname, FALSE)) {
	  if (Debug && !Abort_signaled)
	    log(LOG_DEBUG, fname, "%s%s: Switching to %s\n", Host.name,
		basename(Batchfile.name));
	} else
	  fail(fname, E_fopen, Host.name, Batchfile.nname, "r", errmsg(errno));

	Batchfile.use = Batchfile.nntp_in_use = TRUE;
      }
  
  if ((Input_from & FLG_STDIN) && fb_eof(Stdin) && !Abort_signaled) {
    log(LOG_INFO, fname, "%s%s: EOF on stdin, exiting\n", Host.name);
    Abort_signaled = TRUE;
  } else if (sig_reset || Abort_signaled)
    if (No_articles > 1)
      log(LOG_INFO, fname, "%s%s: %s Updated, %d articles queued\n",
	  Host.name, basename(Batchfile.name), No_articles);
    else
      log(LOG_INFO, fname, "%s%s: %s Updated, No articles queued\n",
	  Host.name, basename(Batchfile.name));
  
    if ((Abort_signaled || sig_reset) && !One_shot)
      write_link_datafile(Prog_pid);
  
  return;
}


Boolean
  open_batchfile(bfile, do_rename)
char *bfile;
Boolean do_rename;
{
  static char *fname = "open_batchfile: ";
  struct stat statb;
  int fd;
  
  if (stat(bfile, &statb) == FAIL)
    return FALSE;
  
  dlog(LOG_DEBUG, fname, "%sFound %s - ", basename(bfile));
  
  if (statb.st_size == 0) {
    
    dlog(LOG_DEBUG, "", "%sBut it is empty\n");
    
    if (unlink(bfile) < 0)
      fail(fname, E_unlink, Host.name, bfile, errmsg(errno));
    
    return FALSE;
  } else if (do_rename) {
    
    dlog(LOG_DEBUG, "", "%srenaming to %s\n", basename(Batchfile.nname));
    
    if (rename(bfile, Batchfile.nname) == FAIL)
      fail(fname, E_rename, Host.name, bfile, Batchfile.nname, errmsg(errno));
    
    if (open_batchfile(Batchfile.nname, FALSE)) {
      Batchfile.nntp_in_use = TRUE;
      return TRUE;
    } else
      return FALSE;
  }
  
  dlog(LOG_DEBUG, "", "%sopening\n");
  
  if ((fd = open(bfile, O_RDONLY)) == FAIL)
    fail(fname, E_open, Host.name, bfile, "r", errmsg(errno));
  
  Batchfile.use = TRUE;
  
  Batchfile.fbp = fb_fdopen(fd);
  
  if ((statb.st_ino == Batchfile.inode) &&
      (Batchfile.offset != 0) &&
      (fb_seek(Batchfile.fbp, Batchfile.offset, 0) == FAIL))
    log(LOG_WARNING, fname, E_fseek, Host.name, bfile, errmsg(errno));
  else
    Batchfile.inode = statb.st_ino;
  
  return TRUE;
}
