/**
 ** Routines that deal with the datafile.
 **
 ** $Id: datafile.c,v 1.7 1993/05/12 13:05:08 alden Exp $
 **
 ** $Log: datafile.c,v $
 ** Revision 1.7  1993/05/12  13:05:08  alden
 ** Modified to work on #*@#* NeXTs
 **
 ** Revision 1.6  1993/05/11  12:16:34  alden
 ** Only call close() and fb_close() if link_fd is FAIL
 **
 ** Revision 1.5  1993/05/11  00:06:18  alden
 ** Kill old nntplink if "-k" was specified
 **
 ** Revision 1.4  1993/05/04  23:37:27  alden
 ** Cleaned up tabs
 **
 ** Revision 1.3  1993/04/17  22:28:57  root
 ** Changed log() messages to include filename
 **
 ** Revision 1.2  1993/04/16  14:36:57  alden
 ** Removed a bogus line (which was already commented out)
 **
 ** Revision 1.1  1993/03/30  13:19:14  alden
 ** Initial revision
 **
 **
 **/
#include "conf.h"
#include "readline.h"
#include "nntplink.h"
#include "patchlevel.h"

#include <signal.h>

extern char *E_open;
extern char *E_fseek;
extern char *E_fopen;

extern Boolean Abort_signaled;
extern Boolean Debug;
extern int Input_from;
extern Boolean Kill_oldlink;
extern pid_t Prog_pid;
extern long Report_time;
extern long Success_time;

extern void log();

/*
 * Format of the datafile for nntplink:
 *
 *  Line  Description
 *  --------------------------------------------------------------------------
 *   1    Pid of current nntplink (or -999 if none)
 *   2    Version of the datafile (3.0 currently)
 *   3    Inode of the current batchfile
 *   4    Offset into the current batchfile
 *   5    Inode of the current logfile
 *   6    Offset into the current logfile
 *   7    Last time we had a successful transmission of an article
 *   8    Last time we reported a failure
 */


void
  write_link_datafile(pid)
pid_t pid;
{
  static char *fname = "write_link_datafile: ";
  FILE *link_fp;
  FileBuf *link_fbp;
  int link_fd;
  char *line;
  pid_t tpid;
  
  dlog(LOG_DEBUG, fname, "%swriting current info to %s\n",
       basename(Host.datafile));
  
  if ((link_fd = open(Host.datafile, O_RDONLY)) == FAIL) {
    dlog(LOG_WARNING, fname, E_fopen, Host.name, Host.datafile, "r",
	 errmsg(errno));
    return;
  }
  
  link_fbp = fb_fdopen(link_fd);
  
  if ((line = fb_readline(link_fbp, NULL)) == NULL) {
    log(LOG_WARNING, fname, "%s%s: error reading pid from datafile: %s\n",
	Host.name, errmsg(errno));
  }
  
  tpid = (pid_t)atof(line);
  
  close(link_fd);
  fb_close(link_fbp);
  
  if (Prog_pid != tpid) {
    if (pid != -999) 
      log(LOG_INFO, fname, "%s%s: pid in datafile has changed, exiting\n",
	  Host.name);
    Abort_signaled = TRUE;
  } else {
    if ((link_fp = fopen(Host.datafile, "w")) == NULL) {
      dlog(LOG_WARNING, fname, E_fopen, Host.name, Host.datafile, "w",
	   errmsg(errno));
      return;
    }
    
    fprintf(link_fp, "%d\n", pid);
    fprintf(link_fp, "%.1f\n", DATAFILE_VERSION);
    fprintf(link_fp, "%lu\n", (unsigned long)Batchfile.inode);
    if (Batchfile.fbp == NULL)
      fprintf(link_fp, "0\n");
    else
      fprintf(link_fp, "%ld\n", Batchfile.offset);
    fprintf(link_fp, "%lu\n", (unsigned long)Logfile.inode);
    if (Logfile.fbp == NULL)
      fprintf(link_fp, "0\n");
    else
      fprintf(link_fp, "%ld\n", (long)fb_tell(Logfile.fbp));
    fprintf(link_fp, "%ld\n", Success_time);
    fprintf(link_fp, "%ld\n", Report_time);
    FCLOSE(link_fp);
  }
}


void
  update_link_datafile()
{
  static char *fname = "update_link_datafile: ";
  int link_fd;
  FileBuf *link_fbp;
  char *line;
  float version;
  int errflg = 0;
  pid_t old_pid = 0;
  FILE *link_fp;
  int attempts = 12;
  
  dlog(LOG_DEBUG, fname, "%supdating current info in %s\n",
       basename(Host.datafile));

  for(;;) {
    if ((link_fd = open(Host.datafile, O_RDONLY)) == FAIL) {
      log(LOG_WARNING, fname, E_open, Host.name, Host.datafile, "r",
	  errmsg(errno));
      Success_time = Report_time = time(NULL);
      errflg++;
      break;
    } else {
      link_fbp = fb_fdopen(link_fd);
      
      if ((line = fb_readline(link_fbp, NULL)) == NULL) {
	dlog(LOG_WARNING, fname,
	     "%s%s: error reading pid from datafile: %s\n",
	     Host.name, errmsg(errno));
	errflg++;
	break;
      } else if (((old_pid = (pid_t)atof(line)) != 0) &&
		 (old_pid != Prog_pid) &&
		 (kill(old_pid, 0) == 0)) {
	if (Kill_oldlink) {
	  
	  dlog(LOG_DEBUG, fname, "%skilling old nntplink(%d)\n", old_pid);
	  
	  while((kill(old_pid, SIGTERM) == 0) && (attempts-- > 0))
	    sleep(5);
	  
	  if (attempts == 0)
	    fail(fname, "%s%s: nntplink(%d) already running - won't die\n",
		 Host.name, old_pid);
	  else if (errno != ESRCH)
	    fail(fname,
		 "%s%s: nntplink(%d) already running - kill() returns: %s\n",
		 Host.name, old_pid, errmsg(errno));
	  
	  dlog(LOG_DEBUG, fname, "%sDone\n");

	} else {
	  sleep(5);
	  fail(fname, "%s%s: nntplink(%d) already running\n",
	       Host.name, old_pid);
	}
      } else
	break;

    }
    close(link_fd);
    fb_close(link_fbp);
  }

  if (!errflg)
    if ((line = fb_readline(link_fbp, NULL)) == NULL) {
      log(LOG_WARNING, fname,
	  "%s%s: error reading version from datafile: %s\n",
	  Host.name, errmsg(errno));
      errflg++;
    } else {
      version = (float)(((int)(atof(line) * 10)) / 10.0);
      if (version!= DATAFILE_VERSION) {
	log(LOG_WARNING, fname,
	    "%s%s: datafile version(%.1f) not same as current version(%.1f)\n",
	    Host.name, version, DATAFILE_VERSION);
	errflg++;
      }
    }
  
  if (!errflg)
    if ((line = fb_readline(link_fbp, NULL)) == NULL) {
      log(LOG_WARNING, fname,
	  "%s%s: error reading batchfile inode from datafile: %s\n",
	  Host.name, errmsg(errno));
      errflg++;
    } else
      Batchfile.inode = (ino_t)atol(line);
  
  if (!errflg)
    if (((line = fb_readline(link_fbp, NULL)) == NULL) ||
	((Batchfile.offset = (off_t)atol(line)) < 0)) {
      log(LOG_WARNING, fname,
	  "%s%s: error reading batchfile offset from datafile: %s\n",
	  Host.name, errmsg(errno));
      errflg++;
    }
  
  if (!errflg)
    if ((line = fb_readline(link_fbp, NULL)) == NULL) {
      log(LOG_WARNING, fname,
	  "%s%s: error reading logfile inode from datafile: %s\n",
	  Host.name, errmsg(errno));
      errflg++;
    } else
      Logfile.inode = (ino_t)atol(line);
  
  if (!errflg)
    if (((line = fb_readline(link_fbp, NULL)) == NULL) ||
	((Logfile.offset = (off_t)atol(line)) < 0)) {
      log(LOG_WARNING, fname,
	  "%s%s: error reading logfile offset from datafile: %s\n",
	  Host.name, errmsg(errno));
      errflg++;
    }
  
  if (!errflg)
    if (((line = fb_readline(link_fbp, NULL)) == NULL) ||
	((Success_time = atol(line)) < 0)) {
      log(LOG_WARNING, fname,
	  "%s%s: error reading success time from datafile: %s\n",
	  Host.name, errmsg(errno));
      errflg++;
    }
  
  if (!errflg)
    if (((line = fb_readline(link_fbp, NULL)) == NULL) ||
	((Report_time = atol(line)) < 0)) {
      log(LOG_WARNING, fname,
	  "%s%s: error reading report time from datafile: %s\n",
	  Host.name, errmsg(errno));
      errflg++;
    }
  
  if (link_fd != FAIL) {
    close(link_fd);
    fb_close(link_fbp);
  }
  
  if ((link_fp = fopen(Host.datafile, "w")) == NULL) {
    dlog(LOG_WARNING, fname, E_fopen, Host.name, Host.datafile, "w",
	 errmsg(errno));
  } else {
    fprintf(link_fp, "%d\n", Prog_pid);
    fprintf(link_fp, "%.1f\n", DATAFILE_VERSION);
    fprintf(link_fp, "%lu\n", (unsigned long)Batchfile.inode);
    fprintf(link_fp, "%ld\n", (long)Batchfile.offset);
    fprintf(link_fp, "%lu\n", (unsigned long)Logfile.inode);
    fprintf(link_fp, "%ld\n", (long)Logfile.offset);
    fprintf(link_fp, "%ld\n", Success_time);
    fprintf(link_fp, "%ld\n", Report_time);
    FCLOSE(link_fp);
  }
  
  return;
}
