/**
 ** Routines that read arbitrarily-long lines without stdio.  (Written by
 ** Mark Moraes from the University of Toronto.)
 **
 ** $Id: readline.c,v 1.1 1993/03/30 13:19:25 alden Exp $
 **
 ** $Log: readline.c,v $
 ** Revision 1.1  1993/03/30  13:19:25  alden
 ** Initial revision
 **
 **
 **
 **/
/*
 * fb_readline - read arbitrarily-long lines without stdio
 */
/*
 * Copyright University of Toronto 1991.
 * Written by Mark Moraes
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. The author and the University of Toronto are not responsible 
 *    for the consequences of use of this software, no matter how awful, 
 *    even if they arise from flaws in it.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Since few users ever read sources,
 *    credits must appear in the documentation.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.  Since few users
 *    ever read sources, credits must appear in the documentation.
 *
 * 4. This notice may not be removed or altered.
 */
/*
 * !! performance nit: if fb_readline() reads a last line which lacks a
 * newline, and returns NULL, and readline() is called again, it will
 * first check each character of the last line before returning NULL
 * again.
 */
/* LINTLIBRARY */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include "readline.h"

/* forwards */
static int refill();
static int alloc();

/* imports */
extern char *malloc(), *realloc();
extern int errno;
extern long lseek();	/* !! off_t? */

#ifndef TESTDEBUG
# define PAGESIZE	8192	/* initial value for rbufsz */
#else
# define PAGESIZE	18
#endif

/*
 * initializes a file buffer for a given file descriptor.  returns NULL
 * if it cannot allocate a buffer
 */
FileBuf *
fb_fdopen(fd)
int fd;
{
	FileBuf *fbp = (FileBuf *) malloc(sizeof(FileBuf));

	if (fbp == NULL)
		return NULL;
	fbp->fb_rbufsz = 0;
	fbp->fb_offset = 0;
	fbp->fb_flags = 0;
	fbp->fb_rbuf = fbp->fb_curp = fbp->fb_endp = NULL;
	fbp->fb_fd = fd;
	return fbp;
}

/* frees the file buffer. note:  does not close the file descriptor */
void
fb_close(fbp)
FileBuf *fbp;
{
	if (fbp->fb_rbuf) {
		fbp->fb_rbuf -= fbp->fb_rbufsz;
		free(fbp->fb_rbuf);
		fbp->fb_rbuf = fbp->fb_curp = fbp->fb_endp = NULL;
		fbp->fb_offset = 0;
		fbp->fb_rbufsz = 0;
	}
	free((char *) fbp);
}

/*
 - readline - read line (sans newline), returns NULL when we hit EOF
 *
 * Minor flaw:  will not return a last line which lacks a newline.  Use
 * fb_getc() after this if you want to make really sure.
 */
char *
fb_readline(fbp, lenp)
register FileBuf *fbp;
unsigned int *lenp;	/* !! should be size_t */
{
	register char c, *cp, *ep;
	register unsigned int n;
	
	/*
	 * for the typical case, this loop is executed 001 times, and
	 * dives out through the "return line".  It will keep going till
	 * it finds a line, or till we hit an error in refill, or till
	 * we die of memory exhaustion.
	 */
	for (;;) {
		c = '\0';
		cp = fbp->fb_curp;
		ep = fbp->fb_endp;
		while (cp != ep && (c = *cp++) != '\n')
			;
		if (c == '\n') {
			register char *line = fbp->fb_curp;
			register unsigned int len = cp - line;
			
			*--cp = '\0';
			if (lenp != NULL)
				*lenp = len - 1;
			fbp->fb_curp = ++cp;
			fbp->fb_offset += len;
			return line;
		}
		/*
		 * if we reach here, the buffer needs (re)filling.  first
		 * copy partial line to old buffer or (re)allocate.
		 */
		n = ep - fbp->fb_curp;
		if (fbp->fb_rbufsz > 0 && n <= fbp->fb_rbufsz) {
			/* usually true */
			if (n != 0 && ep != fbp->fb_rbuf)
				memcpy(fbp->fb_rbuf - n, fbp->fb_curp, n);
			fbp->fb_curp = fbp->fb_rbuf - n;
			fbp->fb_endp = fbp->fb_rbuf;
		} else
			if (alloc(fbp) < 0)	/* (m|re)alloc failed */
				return NULL;
		if (refill(fbp) == 0)	/* refill hit EOF */
			return NULL;
	}
	/* NOTREACHED */
}


/* 
 - refill -- refill readline's buffer.  invariant before refill() is
 * that we can read data into the region from rbuf to rbuf + rbufsz.
 * Invariant after a refill is that endp points to the end of data.
 */
static int
refill(fbp)
register FileBuf *fbp;
{
	register int ret;
	struct stat stbuf;

	/*
	 * this fstat has magical effects on NFS, sometimes it takes a long
	 * while to notice a changed file otherwise.  Cheap on non-NFS files,
	 * shouldn't hurt.
	 */
	if (fb_eof(fbp))
		(void) fstat(fbp->fb_fd, &stbuf);

	/* Try ordinary read. */
	ret = read(fbp->fb_fd, fbp->fb_rbuf, fbp->fb_rbufsz);
	if (ret < 0) {
		fbp->fb_flags |= FB__ERR;
		return 0;
	} else if (ret == 0)
		fbp->fb_flags |= FB__EOF;
	else
		fbp->fb_flags &= ~FB__EOF;
	fbp->fb_flags &= ~FB__ERR;
	fbp->fb_endp = fbp->fb_rbuf + ret;
	return ret;
}

/* usually a macro in readline.h */
#undef fb_tell

unsigned long
fb_tell(fbp)
register FileBuf *fbp;
{
	return fbp->fb_offset;
}

/* 
 * Internal routine for allocating the read buffer if it doesn't exist,
 * or growing it if it exists and isn't big enough.
 */
static int
alloc(fbp)
register FileBuf *fbp;
{
	if (fbp->fb_rbufsz == 0){
		/* need to allocate buffer */
		fbp->fb_rbufsz = PAGESIZE;
		fbp->fb_rbuf = malloc(fbp->fb_rbufsz * 2);
		if (fbp->fb_rbuf == NULL) {
			fbp->fb_flags |= FB__ERR;
			return -1;
		}
		fbp->fb_rbuf += fbp->fb_rbufsz;
	} else {			/* need to grow buffer */
		fbp->fb_rbuf = realloc(fbp->fb_rbuf - fbp->fb_rbufsz,
				       fbp->fb_rbufsz * 4);
		if (fbp->fb_rbuf == NULL) {
			fbp->fb_flags |= FB__ERR;
			return -1;
		}
		fbp->fb_rbufsz *= 2;
		fbp->fb_rbuf += fbp->fb_rbufsz;
	}
	fbp->fb_curp = fbp->fb_endp = fbp->fb_rbuf;
	return 0;
}

/* 
 * Internal routine for use by fb_getc() -- refills the buffer and
 * returns the first character from it.
 */
int
fb__refill(fbp)
register FileBuf *fbp;
{
	if (fbp->fb_rbufsz == 0)
		if (alloc(fbp) < 0)	/* (m|re)alloc failed */
			return EOF;

	if (refill(fbp) == 0)	/* refill hit EOF */
		return EOF;
	fbp->fb_offset++;
	return *fbp->fb_curp++;
}

/* usually a macro in readline.h */
#undef fb_getc

int
fb_getc(fbp)
register FileBuf *fbp;
{
	if (fbp->fb_curp != fbp->fb_endp) {
		fbp->fb_offset++;
		return *fbp->fb_curp++;
	}
	return fb__refill(fbp);
}

/* !! include unistd.h for SEEK_*? */
/* returns 0 for success, -1 for failure */
int
fb_seek(fbp, offset, whence)
FileBuf *fbp;
register int whence;
register long offset;
{
	register long newoff;
	struct stat stbuf;

	switch(whence) {
	case 0:		/* SEEK_SET */
		newoff = offset - fbp->fb_offset;
		break;
	case 1:		/* SEEK_CUR */
		newoff = offset;
		break;
	case 2:		/* SEEK_END */
		if (fstat(fbp->fb_fd, &stbuf) < 0) {
			fbp->fb_flags |= FB__ERR;
			return -1;
		}
		newoff = stbuf.st_size + offset - fbp->fb_offset;
		break;
	default:
		errno = EINVAL;
		return -1;
	}
	if (newoff >= 0 && fbp->fb_curp + newoff < fbp->fb_endp) {
		fbp->fb_curp += newoff;
		fbp->fb_offset += newoff;
		return 0;
	}
	if ((newoff = lseek(fbp->fb_fd, offset, whence)) < 0) {
		fbp->fb_flags |= FB__ERR;
		return -1;
	}
	fbp->fb_curp = fbp->fb_endp = fbp->fb_rbuf;
	fbp->fb_offset = newoff;
	fbp->fb_flags &= ~FB__ERR;
	return 0;
}

/* usually a macro in readline.h */
#undef fb_err

int
fb_err(fbp)
FileBuf *fbp;
{
	return (fbp->fb_flags & FB__ERR);
}

/* usually a macro in readline.h */
#undef fb_eof

int
fb_eof(fbp)
FileBuf *fbp;
{
	return (fbp->fb_flags & FB__EOF);
}

#ifdef TESTDEBUG	/* simple test harness */
char *progname = "nreadline";

#ifndef CSRI_MALLOC
#define mal_verify(x)
#endif

void
unprivileged()
{
}

int
main()
{
	char *cp;
	FileBuf *fbp = fb_fdopen(0);
	unsigned int len;

#ifdef TESTOFFSET
	printf("%d: ", fb_tell(fbp));
#endif
	while ((cp = fb_readline(fbp, &len)) != NULL) {
		mal_verify(1);
		fflush(stderr);
#ifdef TESTOFFSET
		printf("%s\n%d: ", cp, fb_tell(fbp));
#else
		puts(cp);
#endif
		fflush(stdout);
		fprintf(stderr, "%d\n", len);
	}
#ifdef TESTOFFSET
	putc('\n', stdout);
#endif
	if (fb_err(fbp))
		error("fb_readline() failed", "");
	fb_close(fbp);
	mal_verify(1);
	return 0;
}
#endif
