/**
 ** Routine that does the initial setup.
 **
 ** $Id: setupproc.c,v 1.8 1993/05/28 12:53:45 alden Exp $
 **
 ** $Log: setupproc.c,v $
 ** Revision 1.8  1993/05/28  12:53:45  alden
 ** Moved Stdin_fd_flag def to before call to update_datafile()
 **
 ** Revision 1.7  1993/05/04  23:38:45  alden
 ** Cleaned up tabs
 **
 ** Revision 1.6  1993/05/04  20:58:40  alden
 ** Modified setup_proc() to not call check_batchfile() if we are going
 **     to Queue_backlog.
 **
 ** Revision 1.5  1993/05/04  20:51:58  alden
 ** Modified Batchfile.nname to always be <bfile.pid>
 **
 ** Revision 1.4  1993/05/04  19:39:49  alden
 ** Modified batchfile.nname to be <bfile.pid> no matter what (used to be
 **     <bfile.nntp> if input came from FLG_BATCHFILE or FLG_LOGFILE).
 ** Modified batchfile.tmp to be <batchfile.nname>.tmp instead of
 **     <batchfile.name>.tmp.
 **
 ** Revision 1.3  1993/04/17  22:28:57  root
 ** Changed log() messages to include filename
 **
 ** Revision 1.2  1993/04/17  19:04:40  alden
 ** Removed errmsg() def -- it's in nntplink.h
 ** Modified batchfile.nname from bfile.nntp to bfile.<pid> if FLG_STDIN
 **
 ** Revision 1.1  1993/03/30  13:19:35  alden
 ** Initial revision
 **
 **
 **
 **/
#include "conf.h"
#include "readline.h"
#include "nntplink.h"
#include "strfuns.h"

#include <signal.h>
#include <sys/stat.h>

#ifndef HAVE_SYS_TWG_CONFIG_H
#include <sys/ioctl.h>
#endif

extern char *E_fcntl;
extern char *E_fstat;
extern char *E_fseek;

extern Boolean Autobackground;
extern Boolean Debug;
extern long Idle_time;
extern int Input_from;
extern Boolean One_shot;
extern pid_t Prog_pid;
extern char *Prog_name;
extern Boolean Queue_backlog;
extern long Success_time;

extern void check_batchfile();
extern void check_sleep();
extern void fail();
extern char *itoa();
extern void log();
extern void to_read_reply();
extern void update_link_datafile();
extern void sighandler();
extern void write_link_datafile();

int Stdin_fd_flags;

void
  setup_proc()
{
  static char *fname = "setup_proc: ";
  int fd;
  struct stat statb;
  pid_t pid;
  
  Success_time = time(NULL);
  Time.begin_real = Time.elapsed = 0;
  
#ifndef HAVE_SELECT
  signal(SIGALRM, to_read_reply);	/* used by read_reply to timeout */
#endif
  
  if (Autobackground) {
    if ((pid = fork()) < 0)
      fail(fname, "%s%s: fork failed\n", Prog_name);
    
    if (pid != 0)			/* Parent exits */
      exit(0);
    
    if (setsid() < 0)		/* break old association */
      fail(fname, "%s: setsid() failed\n");
    
    /*
     * We're going to set the signals no matter what since we are
     * backgrounding the job.
     */
    signal(SIGINT, sighandler);
    signal(SIGHUP, sighandler);
    signal(SIGTERM, sighandler);
    signal(SIGPIPE, sighandler);
#ifdef SIGURG
    signal(SIGURG, sighandler);
#endif
  } else {
    /*
     * We only want to muck with the signal handlers if they've been set
     * to ignore since since we are running as an interactive process
     */
    if (signal(SIGINT, SIG_IGN) != SIG_IGN)
      signal(SIGINT, sighandler);
    if (signal(SIGHUP, SIG_IGN) != SIG_IGN)
      signal(SIGHUP, sighandler);
    if (signal(SIGTERM, SIG_IGN) != SIG_IGN)
      signal(SIGTERM, sighandler);
    if (signal(SIGPIPE, SIG_IGN) != SIG_IGN)
      signal(SIGPIPE, sighandler);
#ifdef SIGURG
    if (signal(SIGURG, SIG_IGN) != SIG_IGN)
      signal(SIGURG, sighandler);
#endif
  }
  
#if defined(LOG_NEWS)
  (void) openlog(Prog_name, LOG_PID, LOG_NEWS);
#else
#if defined(LOG_LOCAL7)
  (void) openlog(Prog_name, LOG_PID, LOG_LOCAL7);
#else
  (void) openlog(Prog_name, LOG_PID);
#endif
#endif
  
  Prog_pid = getpid();
  
  if (One_shot)
    Batchfile.nname = strsave(Batchfile.name);
  else
    Batchfile.nname = str3save(Batchfile.name, ".", itoa(Prog_pid));
  Batchfile.tmp = str2save(Batchfile.nname, ".tmp");
  
  if (Input_from & FLG_STDIN) {
    Stdin = fb_fdopen(fileno(stdin));
    if ((Stdin_fd_flags = fcntl(fb_fileno(Stdin), F_GETFL, 0)) == FAIL)
      fail(fname, E_fcntl, Host.name, "<stdin>", "F_GETFL", errmsg(errno));
  }
    
  if (!One_shot)
    update_link_datafile();
  
  if (Input_from & FLG_LOGFILE) {
    
    dlog(LOG_DEBUG, fname, "%swaiting for %s: ", basename(Logfile.name));
    
    if (One_shot) {
      if ((fd = open(Logfile.name, O_RDONLY)) == FAIL)
	fail(fname, "%s%s: %s not found on a One-shot\n", Host.name,
	     Logfile.name);
    } else
      while ((fd = open(Logfile.name, O_RDONLY)) == FAIL)
	check_sleep(Logfile.nap_time, FALSE);
    
    dlog(LOG_DEBUG, "", "%sGot it\n");
    
    if (fstat(fd, &statb) == FAIL)
      fail(fname, E_fstat, Host.name, Logfile.name, errmsg(errno));
    
    Logfile.fbp = fb_fdopen(fd);
    if ((statb.st_ino == Logfile.inode) && (Logfile.offset != 0) &&
	(fb_seek(Logfile.fbp, Logfile.offset, 0) == FAIL))
      log(LOG_WARNING, fname, E_fseek, Host.name, Logfile.name,
	  errmsg(errno));
    else
      Logfile.inode = statb.st_ino;
    
    if (!One_shot) {
      write_link_datafile(Prog_pid);
      check_batchfile();
    }
  } else if (Input_from & FLG_STDIN) {
    
#ifdef _POSIX_VERSION
    if (fcntl(fb_fileno(Stdin), F_SETFL, Stdin_fd_flags | O_NONBLOCK)
	== FAIL)
#else
#ifdef FNDELAY
      if (fcntl(fb_fileno(Stdin), F_SETFL, Stdin_fd_flags | FNDELAY)
	  == FAIL)
#else
	if (fcntl(fb_fileno(Stdin), F_SETFL, Stdin_fd_flags | O_NDELAY)
	    == FAIL)
#endif
#endif
	  fail(fname, E_fcntl, Host.name, "<stdin>", "F_SETFL", errmsg(errno));
    
    if (!Queue_backlog)
      check_batchfile();
    
  } else
    check_batchfile();
  
  return;
}


#ifndef _POSIX_VERSION
/*
 * The TIOCNOTTY breaks the association with the old controlling terminal,
 * but incompletely on Irix.  The setpgrp seems to be necessary on at least
 * SunOS 4.0 to form the association with the new controlling terminal, and
 * can be either the SysV or 4BSD setpgrp.
 *
 * Note that this was written by Mark Moraes (mark@cs.toronto.edu) - thanks
 * Mark.   :-)
 */
int
  setsid()
{
  register int fd;
#ifndef HAVE_SETPGRP_NOARGS
  pid_t mypgrp = getpid();
#endif
  
  /* disassociate from old controlling terminal */
  fd = open("/dev/tty", O_RDWR);
  if (fd >= 0) {
    /* NB: Irix doesn't zero this process's pgrp */
#ifdef TIOCNOTTY
    (void) ioctl(fd, TIOCNOTTY, (char *)0);
#else
    if (!(Input_from & FLG_STDIN))
      close(0); /* close stdin */
    
    close(1);   /* close stdout */
    
    if (!Debug)
      close(2); /* close stderr */
#endif
    (void) close(fd);
  }
  /* getpgrp() == 0 here generally */
  /* make ourselves leader of new process group */
#ifdef HAVE_SETPGRP_NOARGS
  return(setpgrp());
#else
  if (setpgrp(0, mypgrp) < 0)	/* redundancy */
    return -1;
  return mypgrp;
#endif
}
#endif /* !_POSIX_VERSION */
