/**
 ** Routines that deal with an article.
 ** 
 ** $Id: article.c,v 1.18 1994/01/09 18:43:43 alden Exp $
 **
 ** $Log: article.c,v $
 ** Revision 1.18  1994/01/09  18:43:43  alden
 ** Removed trailing '\n' from logmsg() calls -- syslog() will add them for us
 **
 ** Revision 1.17  1994/01/04  01:17:38  alden
 ** log_stats() is now Boolean, but we (void) it since we're exiting
 **
 ** Revision 1.16  1993/12/22  00:54:51  root
 ** Changed "Save_fails" variable into a "SAVE_ART_FAILS" definition
 **
 ** Revision 1.15  1993/12/20  14:40:22  alden
 ** Replaced "Host.name" with "Host.sysname" in fail() and logmsg() calls
 **
 ** Revision 1.14  1993/12/20  14:13:07  alden
 ** Don't return FALSE if Article.timeid can't be fetched from history
 **
 ** Revision 1.13  1993/11/22  14:20:02  alden
 ** postconf.h undef's _POSIX_VERSION now if sequent is defined
 **
 ** Revision 1.12  1993/11/10  01:42:45  alden
 ** Changed all occurrences of log() to logmsg().
 **
 ** Revision 1.11  1993/10/25  18:26:48  root
 ** Changed paramaters to close_connection()
 **
 ** Revision 1.10  1993/10/21  17:16:34  root
 ** Removed the LOOKUP_ARTICLE dependencies
 **
 ** Revision 1.9  1993/10/13  20:26:05  root
 ** modified to deal with new '-y' option
 **
 ** Revision 1.8  1993/10/12  21:17:03  root
 ** Use BATCH_SEP for the batchfile seperator (dependent on BNEWS/CNEWS/INN)
 **
 ** Revision 1.7  1993/09/22  23:21:13  root
 ** If "sequent" is defined, then assume it isn't _POSIX_VERSION compliant  :-)
 **
 ** Revision 1.6  1993/05/17  15:03:27  root
 ** Modified parse_entry() call to include type of entry
 **
 ** Revision 1.5  1993/05/12  13:06:03  alden
 ** Added some def's to make compilers happier
 **
 ** Revision 1.4  1993/05/10  13:48:34  alden
 ** Fixed bug in get_next_art() which ignored return status of parse_entry()
 **
 ** Revision 1.3  1993/05/05  18:39:50  root
 ** Modified get_next_art() so it NULL's Batchfile.fbp if it closes it
 **
 ** Revision 1.2  1993/05/04  23:37:00  alden
 ** Cleaned up tabs
 **
 ** Revision 1.1  1993/03/30  13:19:12  alden
 ** Initial revision
 **
 **
 **/
#include "conf.h"
#include "readline.h"
#include "nntplink.h"
#include "strfuns.h"

#include <fcntl.h>
#include <sys/stat.h>

extern char *E_fdopen;
extern char *E_open;

extern Boolean Abort_signaled;
extern Boolean Debug;
extern long Delay;
extern long Dtablesize;
extern long Entry_sleep;
extern long Idle_time;
extern int Input_from;
extern Boolean Log_close;
extern Boolean One_shot;

extern char *E_fopen;
extern char *E_unlink;

extern void close_history();
extern char *get_history_entry();

extern void check_sleep();
extern void close_article();
extern void close_connection();
extern void fail();
extern void logmsg();
extern Boolean log_stats();
extern void my_exit();
extern Boolean parse_entry();

Boolean
  get_next_art()
{
  static char *fname = "get_next_art: ";
  FILE *tmpfp;
  struct stat batch_stat;
  char *entry;
  int already = 0;
  
  if (Article.count >= MAX_ART_FAILS) {

#ifdef SAVE_ART_FAILS
    if (Article.err != NULL)
      logmsg(LOG_NOTICE, fname, "%s%s: article %s failed(saved): %s",
	     Host.sysname, Article.mesgid, Article.err);
    else
      logmsg(LOG_NOTICE, fname, "%s%s: article %s failed(saved)",
	     Host.sysname, Article.mesgid);
    
    if ((tmpfp = fopen(Host.failfile, "a")) == NULL)
      logmsg(LOG_WARNING, fname, E_fopen, Host.sysname, Host.failfile, "a",
	     errmsg(errno));
    else {
      if (Article.filename == NULL)
	fprintf(tmpfp, "/%c%s\n", BATCH_SEP, Article.mesgid);
      else
	fprintf(tmpfp, "%s%c%s\n", Article.filename, BATCH_SEP,
		Article.mesgid);
      FCLOSE(tmpfp);
    }
#else
    if (Article.err != NULL)
      logmsg(LOG_NOTICE, fname, "%s%s: article %s failed(junked): %s",
	     Host.sysname, Article.mesgid, Article.err);
    else
      logmsg(LOG_NOTICE, fname, "%s%s: article %s failed(junked)",
	     Host.sysname, Article.mesgid);
#endif
    
    close_article();
  }
  
  if (Article.mesgid != NULL || Article.filename != NULL)
    return TRUE;
  
  if (Batchfile.use)
    for (;;) {
      if (Debug)
	if (Batchfile.nntp_in_use)
	  logmsg(LOG_DEBUG, fname, "%sretrieving article from %s: ",
		 basename(Batchfile.nname));
	else
	  logmsg(LOG_DEBUG, fname, "%sretrieving article from %s: ",
		 basename(Batchfile.name));
      
      Batchfile.offset = fb_tell(Batchfile.fbp);
      
      if (!Batchfile.nntp_in_use) {
	if (!One_shot) {
	  dlogmsg(LOG_DEBUG, "", "%swaiting for entry: ");
	  while (((entry = fb_readline(Batchfile.fbp, NULL)) == NULL)
		 && fb_eof(Batchfile.fbp) && !Batchfile.nntp_in_use)
	    check_sleep(Entry_sleep, FALSE);
	  
	  if (entry == NULL) {
	    if (fb_error(Batchfile.fbp))
	      logmsg(LOG_WARNING, fname, "%s%s: error reading %s: %s",
		     Host.sysname, Batchfile.name, errmsg(errno));
	    return FALSE;
	  }
	} else
	  if ((entry = fb_readline(Batchfile.fbp, NULL)) == NULL) {
	    if (Debug)
	      if (fb_eof(Batchfile.fbp))
		logmsg(LOG_DEBUG, "", "%s%s at EOF - removing and exiting",
		       basename(Batchfile.name));
	      else
		logmsg(LOG_DEBUG, "", "%serror reading %s: %s",
		       basename(Batchfile.name), errmsg(errno));
	    close(fb_fileno(Batchfile.fbp));
	    fb_close(Batchfile.fbp);
	    Batchfile.fbp = NULL;
	    Batchfile.inode = Batchfile.offset = 0;
	    if (unlink(Batchfile.name) == FAIL)
	      logmsg(LOG_WARNING, fname, E_unlink, Host.sysname,
		     Batchfile.name, errmsg(errno));
	    if (Host.connected) {
	      close_connection(DONT_SEND_QUIT_MSG);
	      if (!Log_close)
		(void) log_stats();
	    } else
	      (void) log_stats();
	    
	    my_exit(0);
	  }
      } else {
	
	while(((entry = fb_readline(Batchfile.fbp, NULL)) == NULL) &&
	      fb_eof(Batchfile.fbp)) {
	  if (!(Input_from & FLG_BATCHFILE) ||
	      (stat(Batchfile.name, &batch_stat) != FAIL)) {
	    
	    dlogmsg(LOG_DEBUG, "", "%s%s at EOF - removing",
		    basename(Batchfile.nname));
	    
	    close(fb_fileno(Batchfile.fbp));
	    Batchfile.inode = Batchfile.offset = 0;
	    fb_close(Batchfile.fbp);
	    Batchfile.fbp = NULL;
	    Batchfile.nntp_in_use = FALSE;
	    
	    if (unlink(Batchfile.nname) == FAIL)
	      logmsg(LOG_WARNING, fname, E_unlink, Host.sysname,
		     Batchfile.nname, errmsg(errno));
	    
	    return FALSE;
	  }
	  
	  if (Debug && !already++)
	    logmsg(LOG_DEBUG, "", "%s%s at EOF: ", basename(Batchfile.nname));
	  
	  check_sleep(Entry_sleep, FALSE);
	}
	
	if (fb_error(Batchfile.fbp))
	  fail(fname, "%s%s: error reading %s: %s", Host.sysname,
	       Batchfile.nname, errmsg(errno));
	
      }
      dlogmsg(LOG_DEBUG, "", "%s\nGot it: %s", entry);
      if (parse_entry(entry, FLG_BATCHFILE))
	return TRUE;
      else
	dlogmsg(LOG_DEBUG, "", "%sbad entry - skipping");
    }
  
  if ((Input_from & FLG_FUNNEL) || (Input_from & FLG_LOGFILE)) {
    
    dlogmsg(LOG_DEBUG, fname, "%sretrieving next article from %s",
	    basename(Datafile.name));
    
    if (!One_shot) {
      dlogmsg(LOG_DEBUG, fname, "%swaiting for entry: ");
      for (;;) {
	while (((entry = fb_readline(Datafile.fbp, NULL)) == NULL) &&
	       fb_eof(Datafile.fbp))
	  if (Batchfile.nntp_in_use)
	    return FALSE;
	  else
	    check_sleep(Entry_sleep, FALSE);
	
	if (entry == NULL) {
	  logmsg(LOG_WARNING, fname, "%s%s: error reading %s: %s",
		 Host.sysname, Datafile.name, errmsg(errno));
	  return FALSE;
	} else if (parse_entry(entry, Input_from)) {
	  break;
	}
      }
    } else {
      while (((entry = fb_readline(Datafile.fbp, NULL)) != NULL) &&
	     (!parse_entry(entry, Input_from)))
	;
      
      if (entry == NULL) {
	if (Debug)
	  if (fb_eof(Datafile.fbp))
	    logmsg(LOG_DEBUG, fname, "%s%s at EOF - exiting",
		   basename(Datafile.name));
	  else
	    logmsg(LOG_DEBUG, fname, "%serror reading %s: %s",
		   basename(Datafile.name), errmsg(errno));
	Abort_signaled = TRUE;
	return FALSE;
      }
    }
    
  } else if (Input_from & FLG_STDIN) {
    
    dlogmsg(LOG_DEBUG, fname, "%swaiting for entry: ");
    for (;;) {
      while (((entry = fb_readline(Stdin, NULL)) == NULL) &&
#ifdef _POSIX_VERSION
	     (errno == EAGAIN)) {
#else
	     (errno == EWOULDBLOCK)) {
#endif
	       errno = 0;
	       if (Batchfile.nntp_in_use)
		 return FALSE;
	       else
		 check_sleep(Entry_sleep, FALSE);
	     }
      
      if (entry == NULL) {
	if (fb_eof(Stdin))
	  logmsg(LOG_DEBUG, fname, "%s%s: EOF on stdin, exiting",
		 Host.sysname);
	else
	  logmsg(LOG_DEBUG, fname, "%serror reading stdin: %s",
		 errmsg(errno));
	Abort_signaled = TRUE;
	return FALSE;
      } else if (parse_entry(entry, Input_from)) {
	break;
      }
    }
  } else
    fail(fname, "%s%s: Input_from has invalid value - exiting",
	 Host.sysname);
  
  dlogmsg(LOG_DEBUG, "", "%sGot it");
  return TRUE;
}


void
  close_article()
{
  Article.filename = NULL;
  Article.mesgid = NULL;
  Article.timeid = 0;
  
  if (Article.fbp != NULL) {
    close(fb_fileno(Article.fbp));
    fb_close(Article.fbp);
    Article.fbp = NULL;
  }
  
  if (Article.err != NULL)
    FREE(Article.err);
  
  Article.count = 0;
  
  return;
}


Boolean
  open_article()
{
  static char *fname = "open_article: ";
  int fd;
  
  if (Article.fbp != NULL)
    return TRUE;
  
  if ((Article.filename == NULL) || ((Article.timeid == 0) && (Delay > 0))){
    
    if (Article.filename == NULL)
      Article.filename = get_history_entry(Article.mesgid, WANT_FILENAME,
					   &Article.timeid);
    else			/* Article.timeid == 0 */
      (void) get_history_entry(Article.mesgid, WANT_FILENAME, &Article.timeid);
    
    /*
     * If we failed then we want to close the history file and retry
     * since it's possible that the history file got aged.
     */
    if (Article.timeid == 0) {
      close_history();

      if (Article.filename == NULL)
	Article.filename = get_history_entry(Article.mesgid, WANT_FILENAME,
					     &Article.timeid);
      else
	(void) get_history_entry(Article.mesgid, WANT_FILENAME,
				 &Article.timeid);

      if (Article.timeid == 0) {
	logmsg(LOG_INFO, fname, "%s%s: History lookup failed for %s",
	       Host.sysname, Article.mesgid);
      }
    }
    
    dlogmsg(LOG_DEBUG, fname, "%shistory returns %s", Article.filename);
  }
  
  if ((fd = open(Article.filename, O_RDONLY)) == FAIL) {
    if (errno != ENOENT)
      logmsg(LOG_WARNING, fname, E_open, Host.sysname, Article.filename, "r",
	     errmsg(errno));
    else
      dlogmsg(LOG_DEBUG, fname, "%sArticle(%s) not found",
	      Article.filename);
    
    return FALSE;
  }
  
  Article.fbp = fb_fdopen(fd);
  
  return TRUE;
}
