/**
 ** Routine that logs the statistics.
 **
 ** $Id: logstats.c,v 1.13 1994/01/09 18:45:19 alden Exp $
 **
 ** $Log: logstats.c,v $
 ** Revision 1.13  1994/01/09  18:45:19  alden
 ** Removed trailing '\n' from logmsg() calls -- syslog() will add them for us
 **
 ** Revision 1.12  1994/01/04  01:18:21  alden
 ** log_stats() now returns FALSE if get_next_art() fails, TRUE otherwise
 **
 ** Revision 1.11  1993/12/22  00:55:11  root
 ** Fixed bug which caused Article.count to get reset to 0, instead of being
 **     reset back to its original value (after closing/reopening batchfile)
 **
 ** Revision 1.10  1993/12/20  14:41:05  alden
 ** Replaced "Host.name" with "Host.sysname" in fail() and logmsg() calls
 **
 ** Revision 1.9  1993/11/19  20:44:06  alden
 ** Cleaned up some if() statements  :-)
 **
 ** Revision 1.8  1993/11/12  01:17:33  alden
 ** Fixed code dealing with "-c <n>" so it really does rewrite the batchfile
 **     after <n> articles have been offered to the remote site
 **
 ** Revision 1.7  1993/11/10  01:48:18  alden
 ** Changed all occurrences of log() to logmsg().
 **
 ** Revision 1.6  1993/05/04  23:38:20  alden
 ** Cleaned up tabs
 **
 ** Revision 1.5  1993/05/04  20:42:10  alden
 ** Modified log_stats() to not rename the batchfile if Abort_signaled
 **
 ** Revision 1.4  1993/04/28  14:34:31  alden
 ** Changed logmsg(LOG_NOTICE, ...) to logmsg(STATS_LOGLEVEL, ...)
 **
 ** Revision 1.3  1993/04/17  22:28:57  root
 ** Changed logmsg() messages to include filename
 **
 ** Revision 1.2  1993/04/16  13:06:15  alden
 ** Add <sys/times.h> inclusion
 **
 ** Revision 1.1  1993/03/30  13:19:21  alden
 ** Initial revision
 **
 **
 **/
#include "conf.h"
#include "readline.h"
#include "nntplink.h"

#ifdef HAVE_GETRUSAGE
#include <sys/resource.h>
#else
#include <sys/param.h>
#endif

#include <sys/stat.h>

#ifdef HAVE_SYS_TIME_H
#ifdef HAVE_SYS_TIMES_H
#include <sys/times.h>
#endif
#endif

extern Boolean Abort_signaled;
extern Boolean Debug;
extern int Input_from;
extern Boolean One_shot;
extern pid_t Prog_pid;

extern char *E_rename;

extern void fail();
extern void logmsg();
extern void write_link_datafile();

static int logged_stats = 0;

int Clear_batchfile = CLEAR_BATCHFILE;
Boolean Report_stats = TRUE;


Boolean
  log_stats()
{
  static char *fname = "log_stats: ";
  long elapsed;
  struct stat statb;
  int article_count = -1;
  
#ifdef HAVE_GETRUSAGE
  struct rusage self, kids;
  
  (void) getrusage(RUSAGE_SELF, &self);
  (void) getrusage(RUSAGE_CHILDREN, &kids);
  
  Time.end_user = (double)(self.ru_utime.tv_sec + kids.ru_utime.tv_sec +
			   self.ru_utime.tv_usec/1000000. +
			   kids.ru_utime.tv_usec/1000000.);
  
  Time.end_sys = (double)(self.ru_stime.tv_sec + kids.ru_stime.tv_sec +
			  self.ru_stime.tv_usec/1000000. +
			  kids.ru_stime.tv_usec/1000000.);
#else /* !HAVE_GETRUSAGE */
  struct tms cpu;
  
  (void) times(&cpu);
  
  Time.end_user = (double)(cpu.tms_utime + cpu.tms_cutime) / HZ;
  Time.end_sys  = (double)(cpu.tms_stime + cpu.tms_cstime) / HZ;
#endif	/* HAVE_GETRUSAGE */
  
  Time.end_real = time(NULL);

  if (Report_stats) {
    
    logmsg(STATS_LOGLEVEL, fname,
    "%s%s stats %lu offered %lu accepted %lu rejected %lu failed %lu connects",
	   Host.sysname, Stats.offered, Stats.accepted, Stats.rejected,
	   Stats.failed, Stats.connects);
    
    elapsed = Time.elapsed;
    Time.elapsed = 0;
    /* Only update Time.begin_real if connection is currently open */
    if (Time.begin_real) {
      elapsed += Time.end_real - Time.begin_real;
      Time.begin_real = Time.end_real;
    }
    
    logmsg(STATS_LOGLEVEL, fname,
	   "%s%s xmit user %.1f system %.1f elapsed %ld",
	   Host.sysname, (Time.end_user - Time.begin_user),
	   (Time.end_sys - Time.begin_sys), elapsed);
    
    Time.begin_user = Time.end_user;
    Time.begin_sys = Time.end_sys;
    Stats.offered = Stats.accepted = Stats.rejected =
      Stats.failed = Stats.connects = 0;
  }
  
  if (!One_shot)
    if (Abort_signaled)
      write_link_datafile(-999);
    else {
      write_link_datafile(Prog_pid);

      if ((++logged_stats >= Clear_batchfile) && (Clear_batchfile > 0)){

	if (Batchfile.fbp != NULL) {

	  if (Input_from & FLG_BATCHFILE)
	    if (!Batchfile.nntp_in_use) {
	      /** Ok, we're reading from <batchfile> -- therefore all we can
	       ** do is rename <batchfile> to <batchfile.pid> and hope that
	       ** by the next time around there exists a new <batchfile> so
	       ** we can rewrite <batchfile.pid>.
	       **/
	      Batchfile.nntp_in_use = TRUE;
	      
	      dlogmsg(LOG_DEBUG, fname, "%srenaming %s to %s",
		      basename(Batchfile.name), basename(Batchfile.nname));
	      
	      if (rename(Batchfile.name, Batchfile.nname) == FAIL)
		fail(fname, E_rename, Host.sysname, Batchfile.name,
		     Batchfile.nname, errmsg(errno));

	      return TRUE;
	    } else {
	      /**
	       ** We're reading from the <batchfile.pid> file, therefore we
	       ** need to make sure there is a <batchfile> before we go and
	       ** rewrite the <batchfile.pid> file, otherwise someone else
	       ** may still be writing to <batchfile.pid>.
	       **/
	      if (stat(Batchfile.name, &statb) == FAIL) {
		dlogmsg(LOG_DEBUG, fname,
			"%s%s doesn't exist, can't rewrite %s",
			basename(Batchfile.name), basename(Batchfile.nname));

		return TRUE;
	      }
	    }

	  logged_stats = 0;

	  dlogmsg(LOG_DEBUG, fname, "%srewriting %s",
		  Batchfile.nntp_in_use ? Batchfile.nname : Batchfile.name);

	  /**
	   ** We want to make sure to save the number of times this article
	   ** has failed.  Therefore, if there is currently an article
	   ** defined, then it will be the first article in the new batchfile,
	   ** therefore, when we open the new batchfile and retrieve the first
	   ** article, it will be the "old" article and we can reset the
	   ** count back to what it was.
	   **/

	  if (Article.filename != NULL || Article.mesgid != NULL)
	    article_count = Article.count;

	  rewrite_batchfile(
	      Batchfile.nntp_in_use ? Batchfile.nname : Batchfile.name);
	  if (open_batchfile(
	      Batchfile.nntp_in_use ? Batchfile.nname : Batchfile.name, FALSE)
	      ) {


	    if (!get_next_art())
	      return FALSE;

	    if (article_count != -1)
	      Article.count = article_count;
	  }

	}
      }
    }
  return TRUE;
}
