/**
 ** Routines that deal with nntp communications functions.
 **
 ** $Id: remote.c,v 1.22 1994/02/03 23:38:56 alden Exp $
 **
 ** $Log: remote.c,v $
 ** Revision 1.22  1994/02/03  23:38:56  alden
 ** socket/connect errors are now logged with name==Host.sysname, instead of
 **     name==remote_server (== Host.name)
 **
 ** Revision 1.21  1994/02/03  23:32:36  alden
 ** Replace MAP_PRIVATE in mmap() call with MMAP_ARGS (defined in postconf.h)
 **
 ** Revision 1.20  1994/01/09  18:46:28  alden
 ** Removed trailing '\n' from logmsg() calls -- syslog() will add them for us
 **
 ** Revision 1.19  1994/01/04  01:19:33  alden
 ** log_stats() now Boolean, but we (void) it since we're just closing conn
 **
 ** Revision 1.18  1993/12/21  23:31:27  alden
 ** Moved clearerr(rmt_wr_fp) before call to close_connection() in
 **     send_command()
 ** Added several dlogmsg() if send_connection() if there is an error
 **
 ** Revision 1.17  1993/12/20  14:41:08  alden
 ** Replaced "Host.name" with "Host.sysname" in fail() and logmsg() calls
 **
 ** Revision 1.16  1993/12/02  13:00:37  alden
 ** If EINTR occurs during select() in read_reply(), then close_connection()
 **     and return.
 **
 ** Revision 1.15  1993/11/27  22:09:30  alden
 ** Only update Time.elapsed if Time.begin_real is greater than 0
 **
 ** Revision 1.14  1993/11/22  13:53:39  alden
 ** Only check HAVE_MMAP, not system types -- postconf.h undef's MMAP if needed
 **
 ** Revision 1.13  1993/11/19  20:42:56  alden
 ** Cleaned up an if() statement  :-)
 **
 ** Revision 1.12  1993/11/16  14:08:32  alden
 ** Don't logmsg() if we receive a ECONNRESET error
 **
 ** Revision 1.11  1993/11/10  02:20:18  alden
 ** Removed "#ifdef _AIX..." -- hoping _ALL_SOURCE takes care of it
 **
 ** Revision 1.10  1993/11/10  01:49:27  alden
 ** Changed all occurrences of log() to logmsg().
 ** Modified to handle unexpected connection closings properly
 **
 ** Revision 1.9  1993/10/25  18:30:33  root
 ** Any write()'s that return an error now go ahead and close the connection
 **
 ** Revision 1.8  1993/10/21  17:31:03  root
 ** Small code cleanup
 **
 ** Revision 1.7  1993/10/12  21:20:23  root
 ** If read() returns EINTR, don't logmsg() since most likely user sent signal
 ** Reset Host.connected if send_command senses the connection was closed
 **
 ** Revision 1.6  1993/05/28  12:53:28  alden
 ** Changed "ifdef HAVE_BROKEN_SIGNAL" to "ifndef HAVE_WORKING_SIGNAL"
 **
 ** Revision 1.5  1993/05/12  14:35:38  alden
 ** Don't use mmap() on the NeXT - it doesn't work the way we need it to
 **
 ** Revision 1.4  1993/05/04  23:38:38  alden
 ** Cleaned up tabs
 **
 ** Revision 1.3  1993/04/20  00:33:21  alden
 ** Modified logmsg() messages to be more consistent
 ** Modified close_connection() to increment Stats.failed if told to
 **
 ** Revision 1.2  1993/04/17  22:41:07  root
 ** Added clearerr() in send_command()
 **
 ** Revision 1.1  1993/03/30  13:19:28  alden
 ** Initial revision
 **
 **
 **
 **/

#include "conf.h"
#include "readline.h"
#include "nntplink.h"
#include "nntp.h"
#include "strfuns.h"

#include <ctype.h>
#include <setjmp.h>
#include <signal.h>
#include <sys/socket.h>

#ifdef HAVE_MMAP
#include <sys/stat.h>
#include <sys/mman.h>
#endif

extern Boolean Debug;
extern int Input_from;
extern Boolean Log_close;
extern long Idle_time;
extern long Dtablesize;

extern void fail();
extern void logmsg();
extern Boolean log_stats();

extern char *E_fdopen;

void close_connection();

#ifndef HAVE_SELECT
RETSIGTYPE	to_read_reply();
static	jmp_buf	RRstack;
int ignore_sigalrm = TRUE;
#endif

FileBuf	*rmt_rd_fbp;
FILE	*rmt_wr_fp;

/*
 * send cmd to remote, terminated with a CRLF.
 */
Boolean
  send_command(command, log_error)
char *command;
Boolean log_error;
{
  static char *fname = "send_command: ";
  
  dlogmsg(LOG_DEBUG, "", "%s>>> %s", command);
  
  Idle_time = 0L;
  
  (void) fprintf(rmt_wr_fp, "%s\r\n", command);
  (void) fflush(rmt_wr_fp);
  
  if (ferror(rmt_wr_fp)) {
    if (log_error && (errno != EPIPE))
      logmsg(LOG_WARNING, fname, "%s%s: error sending command: %s",
	     Host.sysname, errmsg(errno));
    else
      dlogmsg(LOG_DEBUG, fname, "%s%s: error sending command: %s",
	      Host.sysname, errmsg(errno));

    clearerr(rmt_wr_fp);
    close_connection(DONT_SEND_QUIT_MSG);
    return FALSE;
  }
  return TRUE;
}


#ifndef HAVE_SELECT
RETSIGTYPE
  to_read_reply()
{
#ifndef HAVE_WORKING_SIGNAL
  signal(SIGALRM, SIG_IGN);
  signal(SIGALRM, to_read_reply);
#endif
  
  if (!ignore_sigalrm)
#ifdef HAVE__LONGJMP
    _longjmp(RRstack, 1);
#else
  longjmp(RRstack, 1);
#endif
  
}
#endif /* !HAVE_SELECT */


/**
 ** read a reply line from the remote server and return the code number
 ** as an integer, and the message in a buffer supplied by the caller.
 ** Returns NULL if something went wrong.
 **/
char *
  read_reply(resp)
int *resp;
{
  static char *fname = "read_reply: ";
  char *cp;
  char *reply;
  unsigned int rep_len;
#ifdef HAVE_SELECT
  int cnt;
  fd_set readfds;
  static struct timeval timeout = {0, 0};
#endif
  
  *resp = FAIL;			/* Assume we're going to fail */
  
#ifdef HAVE_SELECT
  
  timeout.tv_sec = READ_TIMEOUT;
  FD_ZERO (&readfds);
  FD_SET (fb_fileno(rmt_rd_fbp), &readfds);
  if ((cnt = select((int)Dtablesize, &readfds, 0, 0, &timeout)) == 0) {
    logmsg(LOG_WARNING, fname,
	   "%s%s: connection timed out while reading reply", Host.sysname);
    close_connection(DONT_SEND_QUIT_MSG);
    return NULL;
  } else if (cnt < 0) {
    if (Debug || (errno != EINTR))
      logmsg(LOG_WARNING, fname,
	     "%s%s: select() error while reading reply: %s",
	     Host.sysname, errmsg(errno));
    close_connection(DONT_SEND_QUIT_MSG);
    return NULL;
  }
  
  reply = fb_readline(rmt_rd_fbp, &rep_len);

  /**
   ** If an interrupt error or connection reset error occurs, we don't
   ** want to log it -- since they are both "common" errors they are not
   ** "worthy" of being logged (unless we're in debug mode).  :-)
   **/
  if (fb_err(rmt_rd_fbp) && (Debug || (errno != EINTR && errno != ECONNRESET)))
    logmsg(LOG_WARNING, fname, "%s%s: read() error while reading reply: %s",
	   Host.sysname, errmsg(errno));
  
#else
#ifdef HAVE__SETJMP
  if (_setjmp(RRstack)) {
#else
  if (setjmp(RRstack)) {
#endif
    (void) alarm(0);		/* reset alarm clock */
    errno = ETIMEDOUT;		/* connection timed out */
    logmsg(LOG_WARNING, fname,
	   "%s%s: connection timed out while reading reply", Host.sysname);
    close_connection(DONT_SEND_QUIT_MSG);
    return NULL;			/* bad read, remote time out */
  }
  
  ignore_sigalrm = FALSE;
  (void) alarm(READ_TIMEOUT);
  
  reply = fb_readline(rmt_rd_fbp, &rep_len);

  /**
   ** If an interrupt error or connection reset error occurs, we don't
   ** want to log it -- since they are both "common" errors they are not
   ** "worthy" of being logged (unless we're in debug mode).  :-)
   **/
  if (fb_err(rmt_rd_fbp) && (Debug || (errno != EINTR && errno != ECONNRESET)))
    logmsg(LOG_WARNING, fname, "%s%s: read() error while reading reply: %s",
	   Host.sysname, errmsg(errno));
  
  ignore_sigalrm = TRUE;
  (void) alarm(0);			/* reset alarm clock */
  
#endif /* HAVE_SELECT */
  
  if (reply == NULL) {
    close_connection(DONT_SEND_QUIT_MSG);
    return NULL;
  }
  
  /*
   * Make sure that what the remote sent us had a CRLF at the end
   * of the line, and then null it out.
   */
  if (rep_len > 2 && *(cp = &reply[rep_len - 1]) == '\r')
    *cp = '\0';
  else {
    logmsg(LOG_WARNING, fname, "%s%s: bad reply from remote: %s",
	   Host.sysname, reply);
    close_connection(SEND_QUIT_MSG);
    return NULL;	/* error reading from remote */
  }
  
  dlogmsg(LOG_DEBUG, "", "%s%s", reply);
  
  /*
   * Skip any non-digits leading the response code 
   * and then convert the code from ascii to integer for
   * return from this routine.
   */
  cp = reply;
  while(*cp != '\0' && isascii(*cp) && !isdigit(*cp))
    cp++;	/* skip anything leading */
  
  if (*cp == '\0' || !isascii(*cp)) {
    logmsg(LOG_WARNING, fname, "%s%s: No response code in reply: %s",
	   Host.sysname, reply);
    close_connection(SEND_QUIT_MSG);
    return NULL;	/* error reading from remote */
  }
  
  *resp = atoi(cp);
  
  return reply;
}
  
char *
  converse(line, resp)
char *line;
int *resp;
{
  char *reply;
  
  *resp = FAIL;			/* Assume we're going to fail */
  
  if (!send_command(line, DO_LOG_ERROR))
    return NULL;
  
  while(((reply = read_reply(resp)) != NULL) &&
	(*resp >= 100) && (*resp < 200))
    ;
  
  return reply;
}


/*
 * open_connection - Contact the remote server and set up the two global
 *                   FILE pointers to that descriptor.
 *
 */
Boolean
  open_connection(remote_server, transport)
char *remote_server;
int transport;
{
  static char *fname = "open_connection: ";
  
  int		socket0, socket1;
  char	*reply;
  int		resp;
  
  switch(transport) {
  case T_IP_TCP:
    socket0 = get_tcp_conn(remote_server, "nntp");
    break;
    
  case T_DKHOST:
#ifdef HAVE_DK_H
    socket0 = get_dk_conn(remote_server);
#else
    fail(fname, "%sno DKHOST support compiled in");
#endif
    break;
    
  case T_DECNET:
#ifdef HAVE_DNET_CONN
    (void) signal(SIGPIPE, SIG_IGN);
    socket0 = dnet_conn(remote_server, "NNTP", SOCK_STREAM, 0, 0, 0, 0);
    if (socket0 < 0) {
      switch(errno) {
      case EADDRNOTAVAIL:
	socket0 = NOHOST;
	break;
      case ESRCH:
	socket0 = NOSERVICE;
	break;
      }
    }
#else
    fail(fname, "%sno DECNET support compiled in");
#endif
    break;
  }
  
  if (socket0 < 0) {
    switch(socket0) {
    case NOHOST:
      logmsg(LOG_WARNING, fname, "%s%s: host unknown", Host.sysname);
      return FALSE;
    case NOSERVICE:
      logmsg(LOG_WARNING, fname, "%s%s: service unknown: nntp",
	     Host.sysname);
      return FALSE;
    case FAIL:
      logmsg(LOG_NOTICE, fname, "%s%s: socket(): %s", Host.sysname,
	     errmsg(errno));
      return FALSE;
    }
  }
  
  if ((socket1 = dup(socket0)) < 0) {
    logmsg(LOG_WARNING, fname, "%s%s: dup(%d): %s", Host.sysname, socket0,
	   errmsg(errno));
    CLOSE(socket0);
    return FALSE;
  }
  
  if ((rmt_wr_fp = fdopen(socket1, "w")) == NULL) {
    logmsg(LOG_WARNING, fname, E_fdopen, Host.sysname, socket1, "w",
	   errmsg(errno));
    CLOSE(socket0);
    CLOSE(socket1);
    return FALSE;
  }
  
  rmt_rd_fbp = fb_fdopen(socket0);
  
  reply = read_reply(&resp);
  ++Stats.connects;

  if ((reply == NULL) || ((resp != OK_CANPOST) && (resp != OK_NOPOST))) {

    if (reply != NULL)
      logmsg(LOG_NOTICE, fname, "%s%s: greeted us with %s", Host.sysname,
	     reply);

    CLOSE(socket0);
    fb_close(rmt_rd_fbp);
    FCLOSE(rmt_wr_fp);
    return FALSE;
  }
  
  Time.begin_real = time(NULL);
  Host.connected = TRUE;
  Idle_time = 0L;
  return TRUE;
}


/*
 * close the connection with the remote server.
 *
 * We trap SIGPIPE because the socket might already be gone.
 */
void
  close_connection(send_quit)
Boolean	send_quit;
{
  register RETSIGTYPE (*pstate)() = signal(SIGPIPE, SIG_IGN);
  int resp;
  
  Host.connected = FALSE;
  Stats.since_close = 0L;
  Idle_time = 0L;
  
  if (send_quit && (rmt_wr_fp != NULL)) {
    dlogmsg(LOG_DEBUG, "", "%s>>> QUIT");

    (void) fprintf(rmt_wr_fp, "QUIT\r\n");
    (void) fflush(rmt_wr_fp);

    /**
     ** I don't care what they say to me; this is just being polite.
     **/
    if (!ferror(rmt_wr_fp))
      (void) read_reply(&resp);
  }
  
  CLOSE(fb_fileno(rmt_rd_fbp));
  fb_close(rmt_rd_fbp);
  FCLOSE(rmt_wr_fp);
  (void) signal(SIGPIPE, pstate);
  
  if (Time.begin_real > 0) {
    Time.end_real = time(NULL);
    Time.elapsed = Time.end_real - Time.begin_real;
  } else
    Time.elapsed = 0;
  
  Time.begin_real = 0;
  
  if (Log_close)
    (void) log_stats();
  
  return;
}


/**
 ** send the contents of an open file descriptor to the remote,
 ** with appropriate RFC822 filtering (e.g. CRLF line termination,
 ** and dot escaping). Return FALSE if something went wrong.
 **/
Boolean
  send_connection(fbp)
FileBuf *fbp;
{
  static char		*fname = "send_connection: ";
#ifndef HAVE_MMAP
  register char	*cp;
#else
  register int	c;
  register int	nl = TRUE;	/* assume we start on a new line */
  register char *mbufr, *mptr;
  long msize;
  struct stat sbuf;
#endif
  
  if (fbp == NULL) {
    dlogmsg(LOG_DEBUG, fname, "%s%s: fbp was null?");
    return FALSE;
  }
  
#ifdef HAVE_MMAP
  
  /**
   ** I'm using putc() instead of fputc();
   ** why do a subroutine call when you don't have to?
   ** Besides, this ought to give the C preprocessor a work-out.
   **/
  
#define	PUTC(c) \
  if (putc(c, rmt_wr_fp) == EOF) \
  { \
    (void) munmap (mbufr, sbuf.st_size); \
    dlogmsg(LOG_DEBUG, fname, "%s%s: putc(rmt_wr_fp): %s", \
	    errmsg(errno)); \
    close_connection(DONT_SEND_QUIT_MSG); \
    return FALSE; \
  }
  
  /* map the article into memory */
  (void) fstat(fb_fileno(fbp), &sbuf);
  if ((mbufr = mmap (0, sbuf.st_size, PROT_READ, MMAP_ARGS,
		     fb_fileno(Article.fbp), 0)) == (char *)FAIL) {
    dlogmsg(LOG_DEBUG, fname, "%s%s: mmap(%s) failed: %s", Article.filename,
	    errmsg(errno));
    return send_command(".", DO_LOG_ERROR);
  }
  
  Idle_time = 0L;
  
  mptr = mbufr;         /* start of article in memory */
  msize = sbuf.st_size; /* size of article (bytes) */
  while(msize-- > 0) {
    c = *mptr++;
    switch(c) {
    case '\n':
      PUTC('\r');		/* \n -> \r\n */
      PUTC(c);
      nl = TRUE;		/* for dot escaping */
      break;
      
    case '.':
      if (nl) {
	PUTC(c);	/* add a dot */
	nl = FALSE;
      }
      PUTC(c);
      break;
      
    default:
      PUTC(c);
      nl = FALSE;
      break;
    }
  }
  
  if (!nl) {
    PUTC('\r');
    PUTC('\n');
  }
  (void) munmap (mbufr, sbuf.st_size);
  
#else /* !HAVE_MMAP */
  
  Idle_time = 0L;
  
  while((cp = fb_readline(fbp, NULL)) != NULL) {
    if (*cp == '.')
      putc('.', rmt_wr_fp);
    if ((fputs(cp, rmt_wr_fp) == EOF) || (fputs("\r\n", rmt_wr_fp) == EOF)) {
      dlogmsg(LOG_DEBUG, fname, "%s%s: fputs(rmt_wr_fp): %s",
	      errmsg(errno));
      close_connection(DONT_SEND_QUIT_MSG);
      return FALSE;
    }
  }
  
  if (fb_error(fbp))
    logmsg(LOG_WARNING, fname, "%s%s: error reading %s: %s",
	   Host.sysname, Article.filename, errmsg(errno));
  
  if (fflush(rmt_wr_fp) != 0) {
    dlogmsg(LOG_DEBUG, fname, "%s%s: fflush(rmt_wr_fp): %s", errmsg(errno));
    close_connection(DONT_SEND_QUIT_MSG);
    return FALSE;
  }
  
#endif /* HAVE_MMAP */
  
  return send_command(".", DO_LOG_ERROR);
}
