/**
 ** Routine that sends the news article to the remote host.
 **
 ** $Id: sendnews.c,v 1.19 1994/01/09 18:46:33 alden Exp $
 **
 ** $Log: sendnews.c,v $
 ** Revision 1.19  1994/01/09  18:46:33  alden
 ** Removed trailing '\n' from logmsg() calls -- syslog() will add them for us
 **
 ** Revision 1.18  1994/01/04  01:19:58  alden
 ** Check return status of log_stats() -- if negative then we want
 **     to break out of the send_news() loop and return
 **
 ** Revision 1.17  1993/12/22  00:56:42  root
 ** Added "Retry_fails" definition
 **
 ** Revision 1.16  1993/12/21  23:32:21  alden
 ** Removed extraneous logmsg() in send_news()
 **
 ** Revision 1.15  1993/12/20  14:41:09  alden
 ** Replaced "Host.name" with "Host.sysname" in fail() and logmsg() calls
 **
 ** Revision 1.14  1993/12/20  14:13:24  alden
 ** If Article.timeid can't be set, assume it's 0
 **
 ** Revision 1.13  1993/11/27  22:08:59  alden
 ** Don't SEND_QUIT_MSG if the remote site sent ERR_GOODBYE
 **
 ** Revision 1.12  1993/11/19  20:43:22  alden
 ** Added code to handle start/stop debugging
 **
 ** Revision 1.11  1993/11/10  01:51:10  alden
 ** Changed all occurrences of log() to logmsg().
 ** Modified to handle unexpected connection closings properly
 **
 ** Revision 1.10  1993/10/25  18:28:23  root
 ** Modified accepted/rejected logging logic (hopefully it is correct  :-)
 **
 ** Revision 1.9  1993/10/21  17:35:02  root
 ** Removed the LOOKUP_ARTICLE dependencies
 **
 ** Revision 1.8  1993/10/13  20:26:48  root
 ** modified to deal with new '-y' option
 **
 ** Revision 1.7  1993/10/12  21:28:48  root
 ** Moved Delay sleep code into send_news()
 **
 ** Revision 1.6  1993/05/05  18:41:35  root
 ** Only rename() bfile.tmp if you are Queue_backlog'ing
 **
 ** Revision 1.5  1993/05/05  14:34:38  root
 ** Changed "queueing the backlog" message
 **
 ** Revision 1.4  1993/05/04  23:38:42  alden
 ** Cleaned up tabs
 **
 ** Revision 1.3  1993/04/20  00:34:12  alden
 ** Modified close_connection() call
 ** Changed behavior so if read_reply() fails to read a reply, then the article
 **     isn't considered a "failure", instead it is resent
 **
 ** Revision 1.2  1993/04/17  23:05:21  alden
 ** Fixed problem with read_reply timeout
 **
 ** Revision 1.1  1993/03/30  13:19:32  alden
 ** Initial revision
 **
 **
 **
 **/
#include "conf.h"
#include "readline.h"
#include "nntplink.h"
#include "nntp.h"
#include "strfuns.h"

extern Boolean Abort_signaled;
extern int Close_after;
extern Boolean Debug;
extern Boolean Debug_signaled;
extern long Failure_naptime;
extern int Log_after;
extern Boolean One_shot;
extern Boolean Open_art_first;
extern pid_t Prog_pid;
extern Boolean Queue_backlog;
extern FILE *Queue_fp;
extern Boolean Reset_signaled;
extern Boolean Retry_fails;
extern long Success_time;
extern long Delay;

extern void abort_nntplink();
extern void begin_debugging();
extern void check_sleep();
extern void close_article();
extern Boolean get_next_art();
extern void close_connection();
extern void close_history();
extern void end_debugging();
extern void fail();
extern char *get_history_entry();
extern void logmsg();
extern Boolean log_stats();
extern Boolean open_article();
extern Boolean open_connection();
extern char *read_reply();
extern Boolean send_connection();
extern char *send_ihave_msg();
extern void update_batchfile();

extern char *E_rename;

Boolean
  send_news()
{
  static char *fname = "send_news: ";
  Boolean failure = FALSE;
  int code;
  char *reply;
  time_t sleep_time;
  
  while(!Abort_signaled &&
	!Reset_signaled &&
	!failure &&
	get_next_art()) {

    if (Debug_signaled)
      if (Debug)
	end_debugging();
      else
	begin_debugging(NULL, Prog_pid);
    
    if (Open_art_first && !open_article())
      close_article();
    else {
      if (Delay > 0) {
	if (Article.timeid == 0) {

	  if (Article.filename == NULL)
	    Article.filename = get_history_entry(Article.mesgid, WANT_FILENAME,
						 &Article.timeid);
	  else
	  (void) get_history_entry(Article.mesgid, WANT_FILENAME,
				   &Article.timeid);
    
	  /*
	   * If we failed then we want to close the history file and retry
	   * since it's possible that the history file got aged.
	   */
	  if (Article.timeid == 0) {
	    close_history();
	    if (Article.filename == NULL)
	      Article.filename = get_history_entry(Article.mesgid,
						   WANT_FILENAME,
						   &Article.timeid);
	    else
	      (void) get_history_entry(Article.mesgid, WANT_FILENAME,
				       &Article.timeid);

	    if (Article.timeid == 0) {
	      logmsg(LOG_INFO, fname, "%s%s: History lookup failed for %s",
		     Host.sysname, Article.mesgid);
	    }
	  }
	}

	if (Article.timeid > 0) {
	  sleep_time = Article.timeid + Delay - time(NULL);
	  if (sleep_time > 1) {
	    dlogmsg(LOG_DEBUG, fname, "%s sleeping %d seconds...",
		    sleep_time);
	    check_sleep(sleep_time, FALSE);
	  }
	}
      }

      if (!Host.connected && !open_connection(Host.name, Host.transport))
	return FALSE;

      /**
       ** Now that we're ready to offer an article, we want to check to see if
       ** we've been Queue_backlog'ing, if so, then we want to close that file
       ** and rename it.
       **/
      if (Queue_fp) {
	dlogmsg(LOG_DEBUG, fname, "%sEnd queueing the backlog");
	FCLOSE(Queue_fp);
	if (Queue_backlog && (rename(Batchfile.tmp, Batchfile.nname) == FAIL))
	  fail(fname, E_rename, Host.sysname, Batchfile.tmp, Batchfile.nname,
	       errmsg(errno));
      }

      reply = send_ihave_msg(&code);

      /**
       ** If "reply == FAIL", then the remote side shut down.
       **/
      if (reply == NULL)
	failure = TRUE;
      else
	switch(code) {
	case CONT_XFER:
	  if (!send_connection(Article.fbp)) {
	    failure = TRUE;
	    Stats.failed++;
	  } else {
	    if ((reply = read_reply(&code)) != NULL) {
	      if (Article.err != NULL)
		free(Article.err);
	      Article.err = strsave(reply);
	    }

	    /**
	     ** If read_reply returns FAIL then the remote site closed
	     ** the connection.  If the response code was "FAIL" then
	     ** there was a "local" or network problem, and we don't
	     ** know if the article made it or not.  Therefore, we want
	     ** to attempt to resend this article without incrementing
	     ** the "attempts to send" counter.
	     **/
	    if (code == FAIL) {
	      failure = TRUE;
	      Stats.failed++;
	    } else if ((code != OK_XFERED) && (code != ERR_XFERRJCT)) {
	      if (!Retry_fails)
		Article.count++;
	      failure = TRUE;
	      Stats.failed++;
	      close_connection(
		       (code==ERR_GOODBYE) ? DONT_SEND_QUIT_MSG:SEND_QUIT_MSG);
	    } else
	      Stats.accepted++;
	  }
	  break;
	  
	case ERR_NOFILE:
	  if ((reply = read_reply(&code)) == NULL) {
	    failure = TRUE;
	  } else if ((code != OK_XFERED) && (code != ERR_XFERRJCT) &&
		     (code != ERR_XFERFAIL)) {
	    logmsg(LOG_NOTICE, fname, "%s%s: bad reply: %s",
		   Host.sysname, errmsg(errno));
	    failure = TRUE;
	  }

	  break;
	  
	case ERR_UNKNOWN:
	case ERR_NOFILE_CONT:
	case ERR_NOMSGID:
	case ERR_GOTIT:
	  break;
	  
	case ERR_COMMAND:
	  failure = TRUE;
	  close_connection(SEND_QUIT_MSG);
	  break;
	  
	case ERR_GOODBYE:
	case FAIL:
	  failure = TRUE;
	  if (Host.connected)
	    close_connection(DONT_SEND_QUIT_MSG);
	  break;
	  
	default:
	  logmsg(LOG_NOTICE, fname, "%s%s: sent IHAVE %s, received: %s",
		 Host.sysname, Article.mesgid, reply);
	  failure = TRUE;
	}
      
      if (!failure) {
	close_article();
	Failure_naptime = NAPTIME;
	Success_time = time(NULL);
      } else
	break;
    }
    
    if ((Stats.since_close >= Close_after) && Host.connected)
      close_connection(SEND_QUIT_MSG);
    
    if ((Stats.offered >= Log_after) && (!log_stats()))
      break;
  }
  
  if (Abort_signaled)
    abort_nntplink();
  
  if (Reset_signaled)
    update_batchfile();
  
  return !failure;
}
