/**
 ** Routines that deal with the batchfile.
 **
 ** $Id: batchfile.c,v 1.32 1994/06/07 13:24:23 root Exp $
 **
 ** $Log: batchfile.c,v $
 ** Revision 1.32  1994/06/07  13:24:23  root
 ** write_batchentry() occasionally tryed to deref a null pointer
 **     (Article.mesgid could be null) -- now it handles it properly.
 **
 ** Revision 1.31  1994/05/02  13:26:42  root
 ** Changed Abort_signaled to Term_signaled
 ** Added Int_signaled to tell update_batchfile() to rewrite the batchfile
 ** Removed extraneous code from update_batchfile() inside if (STDIN)
 **
 ** Revision 1.30  1994/04/30  20:37:20  root
 ** update_batchfile() no longer rewrites the batchfile if an abort was
 ** signaled.
 **
 ** Revision 1.29  1994/04/18  13:15:57  alden
 ** Modifed rewrite_batchfile() so it removes <bfile.tmp> if it had
 ** problems when creating it
 **
 ** Revision 1.28  1994/02/11  14:23:21  root
 ** Modified fail() call to include sleep_time (if necessary)
 **
 ** Revision 1.27  1994/01/09  18:44:13  alden
 ** Removed trailing '\n' from logmsg() calls -- syslog() will add them for us
 **
 ** Revision 1.26  1993/12/20  14:40:53  alden
 ** Replaced "Host.name" with "Host.sysname" in fail() and logmsg() calls
 **
 ** Revision 1.25  1993/12/20  13:52:01  alden
 ** Added close_history() to update_batchfile()
 **
 ** Revision 1.24  1993/11/22  14:20:38  alden
 ** postconf.h undef's _POSIX_VERSION now if sequent is defined
 **
 ** Revision 1.23  1993/11/19  20:41:59  alden
 ** Only scan Batchdir for <batchfile.pid> if Use_nntp_ext isn't defined
 **
 ** Revision 1.22  1993/11/12  01:14:51  alden
 ** Moved code which rewrites the batchfile into rewrite_batchfile() so it could
 **     be used in log_stats()
 **
 ** Revision 1.21  1993/11/10  18:17:19  alden
 ** Fixed bug in update_batchfile() that didn't save current article if the
 **     Article filename was NULL -- the check should be if the mesgid is NULL!
 **
 ** Revision 1.20  1993/11/10  13:14:27  root
 ** Made some of the logmsg() calls into dlogmsg() calls
 **
 ** Revision 1.19  1993/11/10  02:19:47  alden
 ** Removed "#ifdef _AIX..." -- hoping _ALL_SOURCE takes care of it
 **
 ** Revision 1.18  1993/11/10  01:45:17  alden
 ** Changed all occurrences of log() to logmsg().
 **
 ** Revision 1.17  1993/10/26  19:20:29  alden
 ** Removed the LOOKUP_ARTICLE dependencies
 ** Added definition of write_link_datafile()
 ** Fixed bug whereby Batchfile.use wasn't being reset to TRUE if Batchfile.fbp
 **     was defined
 **
 ** Revision 1.16  1993/10/13  20:26:29  root
 ** modified to deal with new '-y' option
 **
 ** Revision 1.15  1993/10/12  21:18:12  root
 ** Use BATCH_SEP for the batchfile seperator (dependent on BNEWS/CNEWS/INN)
 ** Take out extra logmsg() info (don't log if articles aren't queued)
 **
 ** Revision 1.14  1993/09/30  22:33:47  alden
 ** Take out check for empty batchfile in open_batchfile() -- not needed
 **
 ** Revision 1.13  1993/09/22  23:21:43  root
 ** If "sequent" is defined, then assume it isn't _POSIX_VERSION compliant  :-)
 **
 ** Revision 1.12  1993/06/04  19:29:36  alden
 ** Modified to only check return status of closedir() if ! VOID_CLOSEDIR
 **
 ** Revision 1.11  1993/06/04  19:23:07  alden
 ** Fixed logic error in update_batchfile()
 **
 ** Revision 1.10  1993/05/17  15:03:52  root
 ** Modified parse_entry() call to include type of entry
 **
 ** Revision 1.9  1993/05/12  13:05:47  alden
 ** Added () inside an if
 **
 ** Revision 1.8  1993/05/11  18:23:53  alden
 ** Fixed loop in update_batchfile() whereby if Reset_signaled is TRUE and
 **     check_sleep() is called, then check_sleep would call update_batchfile()
 **     which would call check_sleep(), etc.
 **
 ** Revision 1.7  1993/05/05  18:40:17  root
 ** Modified update_batchfile() so it removes 0 length batchfiles before
 **     return()'ing
 **
 ** Revision 1.6  1993/05/04  23:37:14  alden
 ** Cleaned up tabs
 **
 ** Revision 1.5  1993/05/04  20:41:02	alden
 ** Modified update_batchfile() to rewrite the batchfile even if we are in
 **	the One_shot mode
 **
 ** Revision 1.4  1993/04/19  21:41:50	alden
 ** Changed the wording on a couple of dlogmsg() messages
 **
 ** Revision 1.3  1993/04/17  23:03:31	alden
 ** Changed logmsg() messages to include filename
 ** Changed check_batchfile() to look for <batchfile>.<pid> and send it if
 **	found
 **
 ** Revision 1.2  1993/04/16  14:44:09	alden
 ** Changed update_batchfile() to call write_link_datafile() so the datafile
 **	will contain the updated information.
 **
 ** Revision 1.1  1993/03/30  13:19:13	alden
 ** Initial revision
 **
 **
 **/
#include "conf.h"
#include "readline.h"
#include "nntplink.h"
#include "strfuns.h"

#include <sys/stat.h>

extern char *E_closedir;
extern char *E_fopen;
extern char *E_fseek;
extern char *E_fstat;
extern char *E_open;
extern char *E_read;
extern char *E_rename;
extern char *E_unlink;
extern char *E_write;

extern Boolean Debug;
extern long Delay;
extern long Dtablesize;
extern long Idle_time;
extern int Input_from;
extern Boolean Int_signaled;
extern Boolean One_shot;
extern pid_t Prog_pid;
extern Boolean Queue_backlog;
extern FILE *Queue_fp;
extern Boolean Reset_signaled;
extern Boolean Term_signaled;
extern Boolean Use_nntp_ext;

extern void close_article();
extern void check_sleep();
extern void fail();
extern void logmsg();
extern Boolean parse_entry();
extern void write_link_datafile();

Boolean open_batchfile();
void rewrite_batchfile();
void write_batchentry();

int No_articles = 0;

void
  check_batchfile()
{
  static char *fname = "check_batchfile: ";
  struct stat statb;
  char *bf;
  int bf_len, pid;
  DIR *bd;
  struct dirent *bdp;
  
  Batchfile.use = FALSE;
  
  if (!One_shot) {
    
    if (open_batchfile(Batchfile.nname, FALSE)) {
      Batchfile.nntp_in_use = TRUE;
      return;
    }
    
    if (!Use_nntp_ext && ((bd = opendir(Batchfile.dir)) != NULL)) {
      
      bf = basename(Batchfile.name);
      bf_len = strlen(bf);
      
      for (bdp = readdir(bd); bdp != NULL; bdp = readdir(bd)) {
	
	if ((strncmp(bdp->d_name, bf, bf_len) != MATCH) ||
	    (bdp->d_name[bf_len] != '.') ||
	    ((pid = atoi(&bdp->d_name[bf_len + 1])) < 1))
	  continue;
	
	/**
	 ** If the pid is one, then we assume the user made the file
	 ** since a pid of 1 is usually reserved for <init>, otherwise
	 ** we check to see if the process actually exists before we
	 ** try to process the batchfile.
	 **/
	
	if ((pid == 1) ||
	    ((kill(pid, 0) == FAIL) && (errno != EPERM))) {
	  
	  bf = str3save(Batchfile.dir, "/", bdp->d_name);
	  
	  dlogmsg(LOG_DEBUG, fname, "%sFound %s - renaming to %s",
		  bdp->d_name, basename(Batchfile.nname));
	  
	  /**
	   ** Rename <batchfile.oldpid> to <batchfile.mypid>. If it
	   ** fails and the error is ENOENT, then that probably
	   ** means some other nntplink process already grabbed it.
	   **/
	  
	  if (rename(bf, Batchfile.nname) != FAIL) {
	    free(bf);
	    break;
	  } else if (errno != ENOENT)
	    fail(fname, 0, E_rename, Host.sysname, bf, Batchfile.nname,
		 errmsg(errno));
	  
	  free(bf);
	}
      }

#ifdef VOID_CLOSEDIR
      closedir(bd);
#else
      if (closedir(bd) == FAIL)
	fail(fname, 0, E_closedir, Host.sysname, Batchfile.dir, errmsg(errno));
#endif
      
      if (open_batchfile(Batchfile.nname, FALSE)) {
	Batchfile.nntp_in_use = TRUE;
	return;
      }
    }
    
    if (Input_from & FLG_BATCHFILE) {
      
      dlogmsg(LOG_DEBUG, fname, "%swaiting for %s: ",
	      basename(Batchfile.name));
      
      while (stat(Batchfile.name, &statb) == FAIL)
	check_sleep(Batchfile.nap_time, FALSE);
      
      dlogmsg(LOG_DEBUG, "", "%sGot it");
    }
  }
  
  if ((Input_from & FLG_BATCHFILE) || !One_shot)
    if (open_batchfile(Batchfile.name, !(Input_from & FLG_BATCHFILE)))
      return;
  
  if (One_shot && (Input_from & FLG_BATCHFILE))
    fail(fname, 0, "%s%s: %s not found on a One-shot", Host.sysname,
	 Batchfile.name);
  
  return;
}


void
  update_batchfile()
{
  static char *fname = "update_batchfile: ";
  char *entry, *filename = NULL, *mesgid = NULL;
  time_t timeid = 0;
  struct stat batch_stat, log_stat;
  Boolean sig_reset;
  int fd;
  
  No_articles = 0;
  sig_reset = Reset_signaled;
  Reset_signaled = FALSE;
  
  if (sig_reset && ((Input_from & FLG_FUNNEL) || (Input_from & FLG_LOGFILE)))
    logmsg(LOG_INFO, fname, "%s%s: Resetting to use new %s", Host.sysname,
	   ((Input_from & FLG_FUNNEL) ? "funnel file" : "logfile"));
  
  if (Debug)
    if ((Input_from & FLG_FUNNEL) || (Input_from & FLG_LOGFILE))
      logmsg(LOG_DEBUG, fname, "%sUpdating %s from %s",
	     basename(Batchfile.nname), basename(Datafile.name));
    else if (Input_from & FLG_STDIN)
      logmsg(LOG_DEBUG, fname, "%sUpdating %s from <stdin>",
	     basename(Batchfile.nname));
    else
      logmsg(LOG_DEBUG, fname, "%sUpdating %s from %s",
	     basename(Batchfile.nname), basename(Batchfile.name));
  
  if (Queue_backlog) {
    if (Queue_fp == NULL) {
      
      dlogmsg(LOG_DEBUG, fname, "%sBegin queueing the backlog");
      if ((rename(Batchfile.nname, Batchfile.tmp) == FAIL) && errno != ENOENT)
	fail(fname, 0, E_rename, Host.sysname, Batchfile.nname, Batchfile.tmp,
	     errmsg(errno));
      
      if ((Queue_fp = fopen(Batchfile.tmp, "a")) == NULL)
	fail(fname, 0, E_fopen, Host.sysname, Batchfile.tmp, "a",
	     errmsg(errno));
    }
  } else if ((Batchfile.fbp != NULL) && (sig_reset || Int_signaled)) {

    rewrite_batchfile(Batchfile.nname);
  
  }
  
  if ((Queue_fp == NULL) && ((Queue_fp = fopen(Batchfile.nname, "a")) == NULL))
    fail(fname, 0, E_fopen, Host.sysname, Batchfile.nname, "a", errmsg(errno));
  
  if (Article.filename != NULL || Article.mesgid != NULL) {
    No_articles++;
    write_batchentry(Queue_fp);
    close_article();
  }
  
  if ((Input_from & FLG_FUNNEL) || (Input_from & FLG_LOGFILE)) {
    
    if (Datafile.fbp != NULL) {

      while ((entry = fb_readline(Datafile.fbp, NULL)) != NULL)
	if (parse_entry(entry, Input_from)) {
	  No_articles++;
	  write_batchentry(Queue_fp);
	}
      
      if (fb_error(Datafile.fbp))
	logmsg(LOG_INFO, fname, E_read, Host.sysname, Datafile.name,
	       errmsg(errno));
      
      Article.filename = Article.mesgid = NULL;
      Article.timeid = 0;
      
      Datafile.offset = fb_tell(Datafile.fbp);
      close(fb_fileno(Datafile.fbp));
      fb_close(Datafile.fbp);
      Datafile.fbp = NULL;
    }
    
    (void) fflush(Queue_fp);
    
    if (ferror(Queue_fp))
      fail(fname, 0, "%s%s: copy to %s failed: %s", Host.sysname,
	   Batchfile.nname, errmsg(errno));
    
    FCLOSE(Queue_fp);
    
    if (!One_shot && !Term_signaled) {
      while ((fd = open(Datafile.name, O_RDONLY)) == FAIL)
	check_sleep(Datafile.nap_time, FALSE);
      
      Datafile.fbp = fb_fdopen(fd);
      
      if (fstat(fd, &log_stat) == FAIL)
	fail(fname, 0, E_fstat, Host.sysname, Datafile.name, errmsg(errno));
      else
	if (Datafile.inode == log_stat.st_ino) {
	  if (fb_seek(Datafile.fbp, Datafile.offset, 0) == FAIL)
	    logmsg(LOG_WARNING, fname, E_fseek, Host.sysname, Datafile.name,
		   errmsg(errno));
	} else
	  Datafile.inode = log_stat.st_ino;
    }
    
  } else if (Input_from & FLG_STDIN) {
    
    while((entry = fb_readline(Stdin, NULL)) != NULL)
      if (parse_entry(entry, Input_from)) {
	No_articles++;
	write_batchentry(Queue_fp);
      }
    
    if (fb_error(Stdin) &&
#ifdef _POSIX_VERSION
	(errno != EAGAIN))
#else
        (errno != EWOULDBLOCK))
#endif
      logmsg(LOG_INFO, fname, "%s%s: error reading stdin: %s", Host.sysname,
	     errmsg(errno));
    
    (void) fflush(Queue_fp);
    
    if (ferror(Queue_fp))
      fail(fname, 0, "%s%s: copy to %s failed: %s", Host.sysname,
	   Batchfile.nname, errmsg(errno));
    
    if (Term_signaled) {
      FCLOSE(Queue_fp);

      if (Queue_backlog) {
	dlogmsg(LOG_DEBUG, fname, "%sEnd queueing the backlog");
	if ((rename(Batchfile.tmp, Batchfile.nname) == FAIL) &&
	    errno != ENOENT)
	  fail(fname, 0, E_rename, Host.sysname, Batchfile.tmp,
	       Batchfile.nname,
	       errmsg(errno));
      }
    }
  }
  
  Batchfile.use = Batchfile.nntp_in_use = FALSE;
    
  if (stat(Batchfile.nname, &batch_stat) != FAIL)
    if (batch_stat.st_size == 0) {
      
      if ((unlink(Batchfile.nname) == FAIL) && errno != ENOENT)
	logmsg(LOG_WARNING, fname, E_unlink, Host.sysname, Batchfile.nname,
	       errmsg(errno));
      
    } else if (!Queue_backlog && !One_shot &&
	       (Batchfile.fbp == NULL)) 
      if (open_batchfile(Batchfile.nname, FALSE)) {
	if (Debug && !Term_signaled)
	  logmsg(LOG_DEBUG, fname, "%s%s: Switching to %s", Host.sysname,
		 basename(Batchfile.name));
      } else
	fail(fname, 0, E_fopen, Host.sysname, Batchfile.nname, "r",
	     errmsg(errno));

  if (Batchfile.fbp != NULL)
    Batchfile.use = Batchfile.nntp_in_use = TRUE;
  
  if ((Input_from & FLG_STDIN) && fb_eof(Stdin) && !Term_signaled) {
    logmsg(LOG_INFO, fname, "%s%s: EOF on stdin, exiting", Host.sysname);
    Term_signaled = TRUE;
  } else if ((sig_reset || Term_signaled) && (No_articles > 1) && Debug)
    logmsg(LOG_DEBUG, fname, "%s%s: %s Updated, %d articles queued",
	   Host.sysname, basename(Batchfile.name), No_articles);
  
  if (Term_signaled || sig_reset) {
    close_history();
    if (!One_shot)
      write_link_datafile(Prog_pid);
  }
  
  return;
}


Boolean
  open_batchfile(bfile, do_rename)
char *bfile;
Boolean do_rename;
{
  static char *fname = "open_batchfile: ";
  struct stat statb;
  int fd;
  
  if (stat(bfile, &statb) == FAIL)
    return FALSE;
  
  dlogmsg(LOG_DEBUG, fname, "%sFound %s - ", basename(bfile));
  
  if (do_rename) {
    
    dlogmsg(LOG_DEBUG, "", "%srenaming to %s", basename(Batchfile.nname));
    
    if (rename(bfile, Batchfile.nname) == FAIL)
      fail(fname, 0, E_rename, Host.sysname, bfile, Batchfile.nname,
	   errmsg(errno));
    
    if (open_batchfile(Batchfile.nname, FALSE)) {
      Batchfile.nntp_in_use = TRUE;
      return TRUE;
    } else
      return FALSE;
  }
  
  dlogmsg(LOG_DEBUG, "", "%sopening");
  
  if ((fd = open(bfile, O_RDONLY)) == FAIL)
    fail(fname, 0, E_open, Host.sysname, bfile, "r", errmsg(errno));
  
  Batchfile.use = TRUE;
  
  Batchfile.fbp = fb_fdopen(fd);
  
  if ((statb.st_ino == Batchfile.inode) &&
      (Batchfile.offset != 0) &&
      (fb_seek(Batchfile.fbp, Batchfile.offset, 0) == FAIL))
    logmsg(LOG_WARNING, fname, E_fseek, Host.sysname, bfile, errmsg(errno));
  else
    Batchfile.inode = statb.st_ino;
  
  return TRUE;
}


void
  rewrite_batchfile(bfile)
char *bfile;
{
  FILE *tmpfp;
  char *entry;
  static char *fname = "rewrite_batchfile: ";
  int fd;

  dlogmsg(LOG_DEBUG, fname, "%sRewriting current batchfile to %s", bfile);
  
  if ((tmpfp = fopen(Batchfile.tmp, "w")) == NULL)
    fail(fname, 0, E_fopen, Host.sysname, Batchfile.tmp, "w", errmsg(errno));
  
  fd = fb_fileno(Batchfile.fbp);
  
  if (Article.filename != NULL || Article.mesgid != NULL) {
    No_articles++;
    write_batchentry(tmpfp);
    close_article();
  }
  
  while ((entry = fb_readline(Batchfile.fbp, NULL)) != NULL) {
    No_articles++;
    if (fprintf(tmpfp, "%s\n", entry) == FAIL) {

      FCLOSE(tmpfp);
      close(fb_fileno(Batchfile.fbp));
      fb_close(Batchfile.fbp);

      if ((unlink(Batchfile.tmp) == FAIL) && errno != ENOENT)
	logmsg(LOG_WARNING, fname, E_unlink, Host.sysname, Batchfile.tmp,
	       errmsg(errno));

      fail(fname, 0, E_write, Host.sysname, Batchfile.tmp, errmsg(errno));
    }
  }
  
  FCLOSE(tmpfp);
  close(fb_fileno(Batchfile.fbp));
  fb_close(Batchfile.fbp);
  Batchfile.fbp = NULL;
  Batchfile.offset = Batchfile.inode = 0;
  
  if (rename(Batchfile.tmp, bfile) == FAIL)
    fail(fname, 0, E_rename, Host.sysname, Batchfile.tmp, bfile,
	 errmsg(errno));
  
  return;
}


void
  write_batchentry(fp)
FILE *fp;
{
  if (Article.filename != NULL)
    fprintf(fp, "%s", Article.filename);
  else
    fprintf(fp, "/");

  if (Article.mesgid != NULL)
    fprintf(fp, "%c%s", BATCH_SEP, Article.mesgid);

  if (Delay && Article.timeid)
    fprintf(fp, "%c%lu", BATCH_SEP, Article.timeid);

  fprintf(fp, "\n");
}
