/**
 ** Routines that deal with the datafile.
 **
 ** $Id: datafile.c,v 1.20 1994/05/02 13:28:13 root Exp $
 **
 ** $Log: datafile.c,v $
 ** Revision 1.20  1994/05/02  13:28:13  root
 ** Changed Abort_signaled to Term_signaled
 **
 ** Revision 1.19  1994/02/11  14:24:01  root
 ** Modified fail() call to include sleep_time (if necessary)
 **
 ** Revision 1.18  1994/01/09  18:44:36  alden
 ** Removed trailing '\n' from logmsg() calls -- syslog() will add them for us
 **
 ** Revision 1.17  1993/12/20  14:40:55  alden
 ** Replaced "Host.name" with "Host.sysname" in fail() and logmsg() calls
 **
 ** Revision 1.16  1993/11/30  14:06:20  alden
 ** Ignored TERM in update_link_datafile -- now it exits properly
 **
 ** Revision 1.15  1993/11/27  22:07:44  alden
 ** Fixed a typo ("attempts * 10" should have been "attempts * 30")
 **
 ** Revision 1.14  1993/11/19  17:35:14  alden
 ** Modified update_batchfile() so kill() waits MAX_NAPTIME before giving up
 **
 ** Revision 1.13  1993/11/10  01:45:39  alden
 ** Changed all occurrences of log() to logmsg().
 **
 ** Revision 1.12  1993/11/09  14:48:52  alden
 ** Don't print warning message about different pid, just go ahead an exit
 **
 ** Revision 1.11  1993/11/04  20:12:17  root
 ** Added "-I" (ignore pid in datafile) option
 **
 ** Revision 1.10  1993/10/21  17:22:43  root
 ** Added definition for fail()
 **
 ** Revision 1.9  1993/10/13  20:26:37  root
 ** modified to deal with new '-y' option
 **
 ** Revision 1.8  1993/10/01  00:09:34  root
 ** Fixed bug where I was calling atof() with a NULL pointer
 **
 ** Revision 1.7  1993/05/12  13:05:08  alden
 ** Modified to work on #*@#* NeXTs
 **
 ** Revision 1.6  1993/05/11  12:16:34  alden
 ** Only call close() and fb_close() if link_fd is FAIL
 **
 ** Revision 1.5  1993/05/11  00:06:18  alden
 ** Kill old nntplink if "-k" was specified
 **
 ** Revision 1.4  1993/05/04  23:37:27  alden
 ** Cleaned up tabs
 **
 ** Revision 1.3  1993/04/17  22:28:57  root
 ** Changed logmsg() messages to include filename
 **
 ** Revision 1.2  1993/04/16  14:36:57  alden
 ** Removed a bogus line (which was already commented out)
 **
 ** Revision 1.1  1993/03/30  13:19:14  alden
 ** Initial revision
 **
 **
 **/
#include "conf.h"
#include "readline.h"
#include "nntplink.h"
#include "patchlevel.h"

#include <signal.h>

extern char *E_open;
extern char *E_fseek;
extern char *E_fopen;

extern Boolean Debug;
extern int Input_from;
extern Boolean Kill_oldlink;
extern pid_t Prog_pid;
extern long Report_time;
extern long Success_time;
extern Boolean Term_signaled;

extern void fail();
extern void logmsg();

/*
 * Format of the datafile for nntplink:
 *
 *  Line  Description
 *  --------------------------------------------------------------------------
 *   1    Pid of current nntplink (or -999 if none)
 *   2    Version of the datafile (3.0 currently)
 *   3    Inode of the current batchfile
 *   4    Offset into the current batchfile
 *   5    Inode of the current datafile (either logfile or funnel file)
 *   6    Offset into the current datafile
 *   7    Last time we had a successful transmission of an article
 *   8    Last time we reported a failure
 */


void
  write_link_datafile(pid)
pid_t pid;
{
  static char *fname = "write_link_datafile: ";
  FILE *link_fp;
  FileBuf *link_fbp;
  int link_fd;
  char *line;
  pid_t tpid = -1;
  
  dlogmsg(LOG_DEBUG, fname, "%swriting current info to %s",
	  basename(Host.datafile));
  
  if ((link_fd = open(Host.datafile, O_RDONLY)) == FAIL) {
    dlogmsg(LOG_WARNING, fname, E_fopen, Host.sysname, Host.datafile, "r",
	    errmsg(errno));
    return;
  }
  
  link_fbp = fb_fdopen(link_fd);
  
  if ((line = fb_readline(link_fbp, NULL)) == NULL) {
    dlogmsg(LOG_WARNING, fname, "%s%s: error reading pid from datafile: %s",
	    Host.sysname, errmsg(errno));
  } else {
    tpid = (pid_t)atof(line);
  }

  close(link_fd);
  fb_close(link_fbp);
  
  if (Prog_pid != tpid) {
    if (pid != -999) 
      dlogmsg(LOG_INFO, fname, "%s%s: pid in datafile has changed, exiting",
	      Host.sysname);
    Term_signaled = TRUE;
  } else {
    if ((link_fp = fopen(Host.datafile, "w")) == NULL) {
      dlogmsg(LOG_WARNING, fname, E_fopen, Host.sysname, Host.datafile, "w",
	      errmsg(errno));
      return;
    }
    
    fprintf(link_fp, "%d\n", pid);
    fprintf(link_fp, "%.1f\n", DATAFILE_VERSION);
    fprintf(link_fp, "%lu\n", (unsigned long)Batchfile.inode);
    if (Batchfile.fbp == NULL)
      fprintf(link_fp, "0\n");
    else
      fprintf(link_fp, "%ld\n", Batchfile.offset);
    fprintf(link_fp, "%lu\n", (unsigned long)Datafile.inode);
    if (Datafile.fbp == NULL)
      if (Datafile.offset > 0)
	fprintf(link_fp, "%ld\n", (long)Datafile.offset);
      else
	fprintf(link_fp, "0\n");
    else
      fprintf(link_fp, "%ld\n", (long)fb_tell(Datafile.fbp));
    fprintf(link_fp, "%ld\n", Success_time);
    fprintf(link_fp, "%ld\n", Report_time);
    FCLOSE(link_fp);
  }
}


void
  update_link_datafile()
{
  static char *fname = "update_link_datafile: ";
  int link_fd;
  FileBuf *link_fbp;
  char *line;
  float version;
  int errflg = 0;
  pid_t old_pid = 0;
  FILE *link_fp;
  int attempts = 0;
  
  dlogmsg(LOG_DEBUG, fname, "%supdating current info in %s",
	  basename(Host.datafile));

  for(;;) {
    if ((link_fd = open(Host.datafile, O_RDONLY)) == FAIL) {
      logmsg(LOG_WARNING, fname, E_open, Host.sysname, Host.datafile, "r",
	     errmsg(errno));
      Success_time = Report_time = time(NULL);
      errflg++;
      break;
    } else {
      link_fbp = fb_fdopen(link_fd);
      
      if ((line = fb_readline(link_fbp, NULL)) == NULL) {
	dlogmsg(LOG_WARNING, fname,
		"%s%s: error reading pid from datafile: %s",
		Host.sysname, errmsg(errno));
	errflg++;
	break;
      } else if (((old_pid = (pid_t)atof(line)) != 0) &&
		 (old_pid != Prog_pid) &&
		 (kill(old_pid, 0) == 0)) {
	if (Kill_oldlink) {
	  
	  dlogmsg(LOG_DEBUG, fname, "%skilling old nntplink(%d)", old_pid);
	  
	  while((kill(old_pid, SIGTERM) == 0) &&
		((attempts++ * 30) < (MAX_NAPTIME + 60)) &&
		!Term_signaled)
	    sleep(30);

	  if ((attempts * 30) >= MAX_NAPTIME)
	    fail(fname, 0, "%s%s: nntplink(%d) already running - won't die",
		 Host.sysname, old_pid);
	  /**
	   ** If we receive EINTR, then somehow we received a signal (such as
	   ** interrupt or quit) and therefore we want to simply exit.
	   **/
	  else if ((errno == EINTR) || Term_signaled) {
	    logmsg(LOG_WARNING, fname,
		   "%s%s: received TERM signal -- exiting", Host.sysname);
	    my_exit(0);
	  /**
	   ** Die if the error returned is anything other than "bad
	   ** pid" (ESRCH).
	   **/
	  } else if (errno != ESRCH)
	      fail(fname, 0,
		   "%s%s: nntplink(%d) already running - kill() returns: %s",
		   Host.sysname, old_pid, errmsg(errno));
	  
	  dlogmsg(LOG_DEBUG, fname, "%sDone");

	} else {
	  sleep(5);
	  fail(fname, 0, "%s%s: nntplink(%d) already running",
	       Host.sysname, old_pid);
	}
      } else
	break;

    }
    close(link_fd);
    fb_close(link_fbp);
  }

  if (!errflg)
    if ((line = fb_readline(link_fbp, NULL)) == NULL) {
      logmsg(LOG_WARNING, fname,
	     "%s%s: error reading version from datafile: %s",
	     Host.sysname, errmsg(errno));
      errflg++;
    } else {
      version = (float)(((int)(atof(line) * 10)) / 10.0);
      if (version!= DATAFILE_VERSION) {
	logmsg(LOG_WARNING, fname,
	    "%s%s: datafile version(%.1f) not same as current version(%.1f)",
	       Host.sysname, version, DATAFILE_VERSION);
	errflg++;
      }
    }
  
  if (!errflg)
    if ((line = fb_readline(link_fbp, NULL)) == NULL) {
      logmsg(LOG_WARNING, fname,
	     "%s%s: error reading batchfile inode from datafile: %s",
	     Host.sysname, errmsg(errno));
      errflg++;
    } else
      Batchfile.inode = (ino_t)atol(line);
  
  if (!errflg)
    if (((line = fb_readline(link_fbp, NULL)) == NULL) ||
	((Batchfile.offset = (off_t)atol(line)) < 0)) {
      logmsg(LOG_WARNING, fname,
	     "%s%s: error reading batchfile offset from datafile: %s",
	     Host.sysname, errmsg(errno));
      errflg++;
    }
  
  if (!errflg)
    if ((line = fb_readline(link_fbp, NULL)) == NULL) {
      logmsg(LOG_WARNING, fname,
	     "%s%s: error reading inode from datafile: %s",
	     Host.sysname, errmsg(errno));
      errflg++;
    } else
      Datafile.inode = (ino_t)atol(line);
  
  if (!errflg)
    if (((line = fb_readline(link_fbp, NULL)) == NULL) ||
	((Datafile.offset = (off_t)atol(line)) < 0)) {
      logmsg(LOG_WARNING, fname,
	     "%s%s: error reading offset from datafile: %s",
	     Host.sysname, errmsg(errno));
      errflg++;
    }
  
  if (!errflg)
    if (((line = fb_readline(link_fbp, NULL)) == NULL) ||
	((Success_time = atol(line)) < 0)) {
      logmsg(LOG_WARNING, fname,
	     "%s%s: error reading success time from datafile: %s",
	     Host.sysname, errmsg(errno));
      errflg++;
    }
  
  if (!errflg)
    if (((line = fb_readline(link_fbp, NULL)) == NULL) ||
	((Report_time = atol(line)) < 0)) {
      logmsg(LOG_WARNING, fname,
	     "%s%s: error reading report time from datafile: %s",
	     Host.sysname, errmsg(errno));
      errflg++;
    }
  
  if (link_fd != FAIL) {
    close(link_fd);
    fb_close(link_fbp);
  }
  
  if ((link_fp = fopen(Host.datafile, "w")) == NULL) {
    dlogmsg(LOG_WARNING, fname, E_fopen, Host.sysname, Host.datafile, "w",
	    errmsg(errno));
  } else {
    fprintf(link_fp, "%d\n", Prog_pid);
    fprintf(link_fp, "%.1f\n", DATAFILE_VERSION);
    fprintf(link_fp, "%lu\n", (unsigned long)Batchfile.inode);
    fprintf(link_fp, "%ld\n", (long)Batchfile.offset);
    fprintf(link_fp, "%lu\n", (unsigned long)Datafile.inode);
    fprintf(link_fp, "%ld\n", (long)Datafile.offset);
    fprintf(link_fp, "%ld\n", Success_time);
    fprintf(link_fp, "%ld\n", Report_time);
    FCLOSE(link_fp);
  }
  
  return;
}
