/**
 ** Routine that logs the statistics.
 **
 ** $Id: logstats.c,v 1.16 1994/05/02 13:28:42 root Exp $
 **
 ** $Log: logstats.c,v $
 ** Revision 1.16  1994/05/02  13:28:42  root
 ** Replaced Abort_signaled with Term_signaled
 ** logstats() now renames the batchfile if inputing from batchfile and not
 **     nntp_in_use, and Term_signaled
 **
 ** Revision 1.15  1994/02/20  18:41:33  alden
 ** Changed CLEAR_BATCHFILE into BATCHFILE_MOVE & BATCHFILE_REWRITE
 **
 ** Revision 1.14  1994/02/11  14:25:05  root
 ** Modified fail() call to include sleep_time (if necessary)
 **
 ** Revision 1.13  1994/01/09  18:45:19  alden
 ** Removed trailing '\n' from logmsg() calls -- syslog() will add them for us
 **
 ** Revision 1.12  1994/01/04  01:18:21  alden
 ** log_stats() now returns FALSE if get_next_art() fails, TRUE otherwise
 **
 ** Revision 1.11  1993/12/22  00:55:11  root
 ** Fixed bug which caused Article.count to get reset to 0, instead of being
 **     reset back to its original value (after closing/reopening batchfile)
 **
 ** Revision 1.10  1993/12/20  14:41:05  alden
 ** Replaced "Host.name" with "Host.sysname" in fail() and logmsg() calls
 **
 ** Revision 1.9  1993/11/19  20:44:06  alden
 ** Cleaned up some if() statements  :-)
 **
 ** Revision 1.8  1993/11/12  01:17:33  alden
 ** Fixed code dealing with "-c <n>" so it really does rewrite the batchfile
 **     after <n> articles have been offered to the remote site
 **
 ** Revision 1.7  1993/11/10  01:48:18  alden
 ** Changed all occurrences of log() to logmsg().
 **
 ** Revision 1.6  1993/05/04  23:38:20  alden
 ** Cleaned up tabs
 **
 ** Revision 1.5  1993/05/04  20:42:10  alden
 ** Modified log_stats() to not rename the batchfile if Term_signaled
 **
 ** Revision 1.4  1993/04/28  14:34:31  alden
 ** Changed logmsg(LOG_NOTICE, ...) to logmsg(STATS_LOGLEVEL, ...)
 **
 ** Revision 1.3  1993/04/17  22:28:57  root
 ** Changed logmsg() messages to include filename
 **
 ** Revision 1.2  1993/04/16  13:06:15  alden
 ** Add <sys/times.h> inclusion
 **
 ** Revision 1.1  1993/03/30  13:19:21  alden
 ** Initial revision
 **
 **
 **/
#include "conf.h"
#include "readline.h"
#include "nntplink.h"

#ifdef HAVE_GETRUSAGE
#include <sys/resource.h>
#else
#include <sys/param.h>
#endif

#include <sys/stat.h>

#ifdef HAVE_SYS_TIME_H
#ifdef HAVE_SYS_TIMES_H
#include <sys/times.h>
#endif
#endif

extern Boolean Debug;
extern int Input_from;
extern int Log_after;
extern Boolean One_shot;
extern pid_t Prog_pid;
extern Boolean Term_signaled;

extern char *E_rename;

extern void fail();
extern void logmsg();
extern void write_link_datafile();

static int logged_stats = 0;

Boolean redo_batchfile();

int Batchfile_move = BATCHFILE_MOVE;
Boolean Report_stats = TRUE;
int Batchfile_rewrite = BATCHFILE_REWRITE;


Boolean
  log_stats()
{
  static char *fname = "log_stats: ";
  long elapsed;
  struct stat statb;
  Boolean success = TRUE;
  
#ifdef HAVE_GETRUSAGE
  struct rusage self, kids;
  
  (void) getrusage(RUSAGE_SELF, &self);
  (void) getrusage(RUSAGE_CHILDREN, &kids);
  
  Time.end_user = (double)(self.ru_utime.tv_sec + kids.ru_utime.tv_sec +
			   self.ru_utime.tv_usec/1000000. +
			   kids.ru_utime.tv_usec/1000000.);
  
  Time.end_sys = (double)(self.ru_stime.tv_sec + kids.ru_stime.tv_sec +
			  self.ru_stime.tv_usec/1000000. +
			  kids.ru_stime.tv_usec/1000000.);
#else /* !HAVE_GETRUSAGE */
  struct tms cpu;
  
  (void) times(&cpu);
  
  Time.end_user = (double)(cpu.tms_utime + cpu.tms_cutime) / HZ;
  Time.end_sys  = (double)(cpu.tms_stime + cpu.tms_cstime) / HZ;
#endif	/* HAVE_GETRUSAGE */
  
  Time.end_real = time(NULL);

  if (Report_stats) {
    
    logmsg(STATS_LOGLEVEL, fname,
    "%s%s stats %lu offered %lu accepted %lu rejected %lu failed %lu connects",
	   Host.sysname, Stats.offered, Stats.accepted, Stats.rejected,
	   Stats.failed, Stats.connects);
    
    elapsed = Time.elapsed;
    Time.elapsed = 0;
    /* Only update Time.begin_real if connection is currently open */
    if (Time.begin_real) {
      elapsed += Time.end_real - Time.begin_real;
      Time.begin_real = Time.end_real;
    }
    
    logmsg(STATS_LOGLEVEL, fname,
	   "%s%s xmit user %.1f system %.1f elapsed %ld",
	   Host.sysname, (Time.end_user - Time.begin_user),
	   (Time.end_sys - Time.begin_sys), elapsed);
    
    Time.begin_user = Time.end_user;
    Time.begin_sys = Time.end_sys;
    Stats.offered = Stats.accepted = Stats.rejected =
      Stats.failed = Stats.connects = 0;
  }
  
  if (!One_shot) {

    if ((Input_from & FLG_BATCHFILE) &&
	!Batchfile.nntp_in_use &&
	(((logged_stats >= Batchfile_move) && (Batchfile_move > 0)) ||
	 Term_signaled)) {
      
      Batchfile.nntp_in_use = TRUE;
      dlogmsg(LOG_DEBUG, fname, "%srenaming %s to %s",
	      basename(Batchfile.name), basename(Batchfile.nname));
      
      if (rename(Batchfile.name, Batchfile.nname) == FAIL)
	fail(fname, 0, E_rename, Host.sysname, Batchfile.name,
	     Batchfile.nname, errmsg(errno));
    }

    if (Term_signaled)
      write_link_datafile(-999);
    else {
      logged_stats += Log_after;

      if (Batchfile.fbp == NULL) {

	logged_stats = 0;

      } else {

	if (Batchfile.nntp_in_use) {

	  if ((logged_stats >= Batchfile_rewrite) &&
	      (Batchfile_rewrite > 0)) {

	    if ((Input_from & FLG_BATCHFILE) &&
		(stat(Batchfile.name, &statb) == FAIL)) {

	      dlogmsg(LOG_DEBUG, fname, "%s%s doesn't exist, can't rewrite %s",
		      basename(Batchfile.name), basename(Batchfile.nname));

	    } else
	      success = redo_batchfile();
	  }
	} else if (!(Input_from & FLG_BATCHFILE) &&
		   (logged_stats >= Batchfile_rewrite) &&
		   (Batchfile_rewrite > 0))
	  success = redo_batchfile();
      }
      write_link_datafile(Prog_pid);
    }
  }
  return success;
}


Boolean
redo_batchfile()
{
  static char *fname = "redo_batchfile: ";
  int article_count = -1;

  logged_stats = 0;

  dlogmsg(LOG_DEBUG, fname, "%srewriting %s",
	  Batchfile.nntp_in_use ? Batchfile.nname : Batchfile.name);

  if (Article.filename != NULL || Article.mesgid != NULL)
    article_count = Article.count;

  rewrite_batchfile(Batchfile.nntp_in_use ? Batchfile.nname : Batchfile.name);

  if (open_batchfile(Batchfile.nntp_in_use ? Batchfile.nname : Batchfile.name,
		     FALSE)) {

    if (!get_next_art())
      return FALSE;

    if (article_count != -1)
      Article.count = article_count;
  }

  return TRUE;
}
