/**
 ** Miscellaneous routines.
 **
 ** $Id: misc.c,v 1.15 1994/02/22 13:50:02 alden Exp $
 **
 ** $Log: misc.c,v $
 ** Revision 1.15  1994/02/22  13:50:02  alden
 ** Don't turn debugging on if fopen(debug_file, "a") fails.
 **
 ** Revision 1.14  1994/02/11  14:32:55  root
 ** Added sleep_time to fail() call -- this replaces sleep_fail()
 ** Fixed fail() routine
 **
 ** Revision 1.13  1994/01/09  18:45:27  alden
 ** Removed trailing '\n' from logmsg() calls -- syslog() will add them for us
 ** Added sleep_fail() -- logmsg(), check_sleep(60), and then fail
 **
 ** Revision 1.12  1994/01/04  01:19:09  alden
 ** log_stats() now Boolean, but we (void) it since we're exiting
 **
 ** Revision 1.11  1993/11/19  20:43:34  alden
 ** Added begin_debugging() and end_debugging()
 **
 ** Revision 1.10  1993/11/10  01:48:33  alden
 ** Changed all occurrences of log() to logmsg().
 **
 ** Revision 1.9  1993/10/25  18:27:38  root
 ** Changed parameters for close_connection()
 **
 ** Revision 1.8  1993/10/21  18:03:40  alden
 ** Moved varargs inclusion into postconf.h
 **
 ** Revision 1.7  1993/05/12  14:16:27  alden
 ** Added check for HAVE_VALUES_H
 **
 ** Revision 1.6  1993/05/04  23:38:24  alden
 ** Cleaned up tabs
 **
 ** Revision 1.5  1993/05/04  20:42:37  alden
 ** Modified abort_nntplink to call update_batchfile() unless you're input'ing
 **     from BATCHFILE
 **
 ** Revision 1.4  1993/04/28  22:23:33  alden
 ** Removed an (unneeded) definition for Entry_sleep
 **
 ** Revision 1.3  1993/04/20  00:32:46  alden
 ** modified close_connection() call
 **
 ** Revision 1.2  1993/04/16  13:22:49  alden
 ** Moved <varargs.h> inclusion before "conf.h" (needed on Solaris)
 **
 ** Revision 1.1  1993/03/30  13:19:22  alden
 ** Initial revision
 **
 **
 **
 **/
#include "conf.h"
#include "readline.h"
#include "nntplink.h"
#include "strfuns.h"

#ifdef HAVE_VALUES_H
#include <values.h>
#endif

extern char *E_fopen;

extern Boolean Debug;
extern Boolean Debug_signaled;
extern FILE *Debugfp;
extern long Idle_time;
extern int Input_from;
extern Boolean Log_close;
extern Boolean One_shot;
extern pid_t Prog_pid;
extern int Stdin_fd_flags;

extern void close_connection();
extern char *emalloc();
extern void logmsg();
extern Boolean log_stats();
extern void update_batchfile();
extern void write_link_datafile();

void my_exit();

#ifdef HAVE_VARARGS_H
void
  logmsg(va_alist)
va_dcl
{
  va_list ap;
  char *fname, *format;
  char *buf;
  int priority;
  
  va_start(ap);
  priority = va_arg(ap, int);
  fname = va_arg(ap, char *);
  format = va_arg(ap, char *);
  format += 2;
  
  if (Debug) {
    if (fputs(fname, Debugfp) == EOF) {
      fprintf(stderr, "logmsg: fputs failed - %s\n", errmsg(errno));
      my_exit(FAIL);
    }
    if (vfprintf(Debugfp, format, ap) == EOF) {
      fprintf(stderr, "logmsg: vfprintf failed - %s\n", errmsg(errno));
      my_exit(FAIL);
    }
    if (fputc('\n', Debugfp) == EOF) {
      fprintf(stderr, "logmsg: fputs failed - %s\n", errmsg(errno));
      my_exit(FAIL);
    }
    fflush(Debugfp);
  } else {
    buf = emalloc(1024);
    vsprintf(buf, format, ap);
    syslog(priority, "%s", buf);
    free(buf);
  }
  
  va_end(ap);
  return;
}

void
  fail(va_alist)
va_dcl
{
  va_list ap;
  char *fname, *format, buf[2048];
  int sleep_time;
  
  va_start(ap);

  fname = va_arg(ap, char *);
  sleep_time = va_arg(ap, int);
  format = va_arg(ap, char *);
  format += 2;

  if (Debug) {
    if (fputs(fname, Debugfp) == EOF) {
      fprintf(stderr, "fail: fputs failed - %s\n", errmsg(errno));
      my_exit(FAIL);
    }
    if (vfprintf(Debugfp, format, ap) == EOF) {
      fprintf(stderr, "fail: vfprintf failed - %s\n", errmsg(errno));
      my_exit(FAIL);
    }
    fflush(Debugfp);
  } else {
    vsprintf(buf, format, ap);
    syslog(LOG_ERR, "%s", buf);
    free(buf);
  }

  va_end(ap);

  if (sleep_time > 0)
    check_sleep(sleep_time, FALSE);

  my_exit(FAIL);
}

#else /* !HAVE_VARARGS_H */

/*VARARGS*/
void
  logmsg(priority, fname, format, x1, x2, x3, x4, x5, x6, x7, x8)
int priority;
char *fname;
char *format, *x1, *x2, *x3, *x4, *x5, *x6, *x7, *x8;
{
  if (Debug) {
    
    if (fprintf(Debugfp, format, fname, x1, x2, x3, x4, x5, x6, x7, x8)
	== EOF) {
      fprintf(stderr, "logmsg: fprintf failed - %s\n", errmsg(errno));
      my_exit(FAIL);
    }
    if (fputc('\n', Debugfp) == EOF) {
      fprintf(stderr, "logmsg: fputs failed - %s\n", errmsg(errno));
      my_exit(FAIL);
    }
    fflush(Debugfp);
    
  } else
    syslog(priority, format, "", x1, x2, x3, x4, x5, x6, x7, x8);
  
  return;
}


/*VARARGS*/
void
  fail(fname, sleep_time, format, x1, x2, x3, x4, x5, x6, x7, x8)
char *fname;
int sleep_time;
char *format, *x1, *x2, *x3, *x4, *x5, *x6, *x7, *x8;
{
  
  logmsg(LOG_ERR, fname, format, x1, x2, x3, x4, x5, x6, x7, x8);

  if (sleep_time > 0)
    check_sleep(sleep_time, FALSE);

  my_exit(FAIL);
}

#endif /* !HAVE_VARARGS_H */

void
  abort_nntplink()
{
  static char *fname = "abort_nntplink: ";
  
  dlogmsg(LOG_DEBUG, fname, "%sexiting");
  
  if ( !(Input_from & FLG_BATCHFILE) || One_shot)
    update_batchfile();
  
  if (Host.connected) {
    close_connection(DONT_SEND_QUIT_MSG);
    if (!Log_close)
      (void) log_stats();
  } else
    (void) log_stats();
  
  my_exit(0);
}


void
  my_exit(status)
int status;
{
  if (Input_from & FLG_STDIN)
    fcntl(fb_fileno(Stdin), F_SETFL, Stdin_fd_flags);
  exit(status);
}


char *
  itoa(num)
int num;
{
  static char *numbers = "0123456789";
  static char *neg = "-";
  register char *bp, *negative = neg + 1;
  char *buf = emalloc(MAXDIGS);
  
  if (num == 0)
    return "0";
  else if (num > MAXINT)
    return NULL;
  
  *(bp = buf + MAXDIGS) = '\0';
  
  if ((long)num < 0) {
    negative--;
    num = -num;
  }
  
  do {
    *--bp = *(numbers + (num % 10));
    num /= 10;
  } while (num);
  
  if (*negative)
    *--bp = *negative;
  
  return bp;
}


void
  begin_debugging(file, pid)
char *file;
pid_t pid;
{
  static char *fname = "begin_debugging: ";
  char *debug_file;

  Debug = TRUE;
  Debug_signaled = FALSE;

  if (file == NULL)
    debug_file = str2save("/tmp/nntplink.", itoa(pid));
  else
    debug_file = str2save(file, itoa(pid));

  if ((Debugfp = fopen(debug_file, "a")) == NULL) {
    Debug = FALSE;
    logmsg(LOG_WARNING, fname, E_fopen, debug_file, "a", errmsg(errno));
  }

  return;
}


void
  end_debugging()
{
  Debug = FALSE;
  Debug_signaled = FALSE;

  FCLOSE(Debugfp);

  return;
}
