#include "conf.h"
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/time.h>
#ifdef FAKESYSLOG
#include "fsyslog.h"
#else
#include <syslog.h>
#endif
#include "readline.h"
#include "nntplink.h"

extern char *E_fcntl;
extern char *E_fstat;
extern char *E_fseek;

extern Boolean Autobackground;
extern Boolean Debug;
extern long Idle_time;
extern int Input_from;
extern Boolean One_shot;
extern int Prog_pid;
extern char *Prog_name;
extern long Success_time;

extern void check_batchfile();
extern void check_sleep();
extern char *errmsg();
extern void fail();
extern void log();
extern void read_link_datafile();
extern void sighandler();
extern void write_link_datafile();

void
  setup_proc()
{
    static char *fname = "setup_proc: ";
    int fd, stdin_fd_flags;
    struct stat statb;
    int pid, mypgrp;

    Time.begin_real = Success_time = time(NULL);

    
    if (Autobackground) {
	if ((pid = fork()) < 0)
	  fail(fname, "%s%s: fork failed\n", Prog_name);

	if (pid != 0)			/* Parent exits */
	  exit(0);

	if ((mypgrp = setsid()) < 0)		/* break old association */
	  fail(fname, "%s: setsid() failed\n");

	/*
	 * We're going to set the signals no matter what since we are
	 * backgrounding the job.
	 */
	signal(SIGINT, sighandler);
	signal(SIGHUP, sighandler);
	signal(SIGTERM, sighandler);
	signal(SIGPIPE, sighandler);
#ifdef SIGURG
	signal(SIGURG, sighandler);
#endif
    } else {
	/*
	 * We only want to muck with the signal handlers if they've been set
	 * to ignore since since we are running as an interactive process
	 */
	if (signal(SIGINT, SIG_IGN) != SIG_IGN)
	  signal(SIGINT, sighandler);
	if (signal(SIGHUP, SIG_IGN) != SIG_IGN)
	  signal(SIGHUP, sighandler);
	if (signal(SIGTERM, SIG_IGN) != SIG_IGN)
	  signal(SIGTERM, sighandler);
	if (signal(SIGPIPE, SIG_IGN) != SIG_IGN)
	  signal(SIGPIPE, sighandler);
#ifdef SIGURG
	if (signal(SIGURG, SIG_IGN) != SIG_IGN)
	  signal(SIGURG, sighandler);
#endif
    }

    Prog_pid = getpid();

#ifdef LOG_NFACILITIES
    (void) openlog(Prog_name, LOG_PID, SYSLOG);
#else
    (void) openlog(Prog_name, LOG_PID);
#endif

    if (!One_shot) {
	read_link_datafile();
	write_link_datafile();
    }

    if (Input_from & FLG_LOGFILE) {

	dlog(LOG_DEBUG, fname, "%swaiting for logfile: ");

	if (One_shot) {
	    if ((fd = open(Logfile.name, O_RDONLY)) == FAIL)
	      fail(fname, "%s%s: %s not found on a One-shot\n", Host.name,
		   Logfile.name);
	} else
	  while ((fd = open(Logfile.name, O_RDONLY)) == FAIL)
	    check_sleep(Logfile.nap_time);

	dlog(LOG_DEBUG, "", "%sgot it\n");

	if (fstat(fd, &statb) == FAIL)
	  fail(fname, E_fstat, Host.name, Logfile.name, errmsg(errno));

	Logfile.fbp = fb_fdopen(fd);
	if ((statb.st_ino == Logfile.inode) && (Logfile.offset != 0) &&
	    (fb_seek(Logfile.fbp, Logfile.offset, 0) == FAIL))
	  log(LOG_WARNING, fname, E_fseek, Host.name, Logfile.name,
	      errmsg(errno));
	else
	  Logfile.inode = statb.st_ino;

	if (!One_shot) {
	    write_link_datafile();
	    check_batchfile();
	}
    } else if (Input_from & FLG_STDIN) {

	Stdin.fbp = fb_fdopen(fileno(stdin));
	if ((stdin_fd_flags = fcntl(fb_fileno(Stdin.fbp), F_GETFL, 0)) == FAIL)
	  fail(fname, E_fcntl, Host.name, "<stdin>", "F_GETFL", errmsg(errno));

	Stdin.fd_flags_block = stdin_fd_flags & ~O_NDELAY;
	Stdin.fd_flags_nblock = stdin_fd_flags | O_NDELAY;

	check_batchfile();

    } else
      check_batchfile();

    return;
}


#ifndef IS_POSIX
/*
 * The TIOCNOTTY breaks the association with the old controlling terminal,
 * but incompletely on Irix.  The setpgrp seems to be necessary on at least
 * SunOS 4.0 to form the association with the new controlling terminal, and
 * can be either the SysV or 4BSD setpgrp.
 *
 * Note that this was written by Mark Moraes (mark@cs.toronto.edu) - thanks
 * Mark.   :-)
 */
int
setsid()
{
	register int fd;
	int mypgrp = getpid();

	/* disassociate from old controlling terminal */
	fd = open("/dev/tty", O_RDWR);
	if (fd >= 0) {
		/* NB: Irix doesn't zero this process's pgrp */
#ifdef TIOCNOTTY
		(void) ioctl(fd, TIOCNOTTY, (char *)0);
#endif
		(void) close(fd);
	}
	/* getpgrp() == 0 here generally */
	/* make ourselves leader of new process group */
	if (setpgrp(0, mypgrp) < 0)	/* redundancy */
		return -1;
	return mypgrp;
}
#endif
