/* Subroutines for insn-output.c for Motorola 68000 family.
   Copyright (C) 1987 Free Software Foundation, Inc.

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdio.h>
extern FILE *asm_out_file;

static rtx find_addr_reg ();

osk_data_ref_p (x)
     rtx x;
{
  if (GET_CODE (x) == DATA_REF)
    return 1;
  else if (GET_CODE (x) == CONST)
    return osk_data_addr_p (XEXP (x, 0));
  else
    return 0;
}

osk_data_addr_p (x)
     rtx x;
{
  if (GET_CODE (x) == DATA_REF)
    return 1;
  else if (GET_CODE (x) == PLUS)
    return osk_data_ref_p (XEXP (x, 0)) || osk_data_ref_p (XEXP (x, 1));
  else
    return 0;
}

osk_symbol_ref_p (x)
     rtx x;
{
  if (GET_CODE (x) == SYMBOL_REF)
    return 1;
  else if (GET_CODE (x) == CONST)
    return osk_symbol_addr_p (XEXP (x, 0));
  else
    return 0;
}

osk_symbol_addr_p (x)
     rtx x;
{
  if (GET_CODE (x) == SYMBOL_REF)
    return 1;
  else if (GET_CODE (x) == PLUS)
    return osk_symbol_ref_p (XEXP (x, 0)) || osk_symbol_ref_p (XEXP (x, 1));
  else
    return 0;
}

/* Change %m in TEMPLATE to %p (resp. %v) if the corresponding
   operand is a memory refference via SYMBOL_REF (resp. DATA_REF). 

   Under the TARGET_REMOTE, MODIFY_MEMREF does not output %v.  Instead, it
   outputs templates of instruction to load 32bit offset into d2 or d3, and
   outputs template by replacing %mN in TEMPLATE to (a6,d2.l) or (a6,d3.l). 

   If there is no operand with SYMBOL_REF or DATA_REF, then `m' of %m is
   simply removed.  So `m' is never contained in the output template.

   Real need of MODIFY_MEMREF is support of TARGET_REMOTE, otherwise
   it is better to process %m in PRINT_OPERAND.  (A.S.)                */

struct offset_reg {
  char *name;
  int  opno;
};

#define MAX_OFFSET_REGS 2

static struct offset_reg regv [MAX_OFFSET_REGS] = {{"d2", 0}, {"d3", 0}};

static char new_template [MAX_OFFSET_REGS * 32 + 64];

char *
modify_memref (template, operands)
     char *template;
     rtx  *operands;
{
  char inst_body [64];		/* template for instruction */
  char load_offset [32];	/* template for loading offset */
  char *q, ctrl_char;
  rtx  opr, x;
  int  regc, opno, i;

  q = inst_body;
  new_template [0] = '\0';
  regc = 0;

  while (*template)
    {
      if (*template != '%')
	{
	  *q++ = *template++;
	  continue;
	}
      ++template;
      if (*template == '\0' || *template != 'm')
	{
	  *q++ = '%';
	  if (*template == '\0')
	    break;
	  *q++ = *template++;
	  continue;
	}

      ctrl_char = '*';
      opno = atoi (++template);
      opr = operands [opno];

      if (GET_CODE (opr) == MEM)
	{
	  x = XEXP (opr, 0);

	  if (osk_data_ref_p (x))
	    if (TARGET_REMOTE && !TARGET_68020)
	      {
		if (regc >= MAX_OFFSET_REGS)
		  abort ();

	        ctrl_char = '\0';
		sprintf (q, "(a6,%s%%.l)", regv[regc].name);
		regv[regc++].opno = opno;
		q += 10;
		while (*template >= '0' && *template <= '9')
		  template++;
	      }
	    else
	      ctrl_char = 'v';
	  else if (osk_symbol_ref_p (x))
	    ctrl_char = 'p';
	}

      if (ctrl_char)
	{
	  *q++ = '%';
	  if (ctrl_char != '*')
	    *q++ = ctrl_char;
	  while (*template >= '0' && *template <= '9')
	    *q++ = *template++;
	}
    }

  *q = '\0';

  if (TARGET_REMOTE && !TARGET_68020 && regc > 0)
    for (i = 0; i < regc; ++i)
      {
	sprintf (load_offset, "move%%.l %%i%d,%s\n\t", regv[i].opno, regv[i].name);
	strcat (new_template, load_offset);
      }

  strcat (new_template, inst_body);

  return new_template;
}

char *
output_btst (operands, countop, dataop, insn, signpos)
     rtx *operands;
     rtx countop, dataop;
     rtx insn;
     int signpos;
{
  operands[0] = countop;
  operands[1] = dataop;

  if (GET_CODE (countop) == CONST_INT)
    {
      register int count = INTVAL (countop);
      /* If COUNT is bigger than size of storage unit in use,
	 advance to the containing unit of same size.  */
      if (count > signpos)
	{
	  int offset = (count & ~signpos) / 8;
	  count = count & signpos;
	  operands[1] = dataop = adj_offsettable_operand (dataop, offset);
	}
      if (count == signpos)
	cc_status.flags = CC_NOT_POSITIVE | CC_Z_IN_NOT_N;
      else
	cc_status.flags = CC_NOT_NEGATIVE | CC_Z_IN_NOT_N;

      if (count == 31
	  && next_insns_test_no_inequality (insn))
	return modify_memref ("tst%.l %m1", operands);
      if (count == 15
	  && next_insns_test_no_inequality (insn))
	return modify_memref ("tst%.w %m1", operands);
      if (count == 7
	  && next_insns_test_no_inequality (insn))
	return modify_memref ("tst%.b %m1", operands);
      cc_status.flags = CC_NOT_NEGATIVE;
    }
  return modify_memref ("btst %0,%m1", operands);
}

/* Convert from host double precision real number format
   to IEEE single and double precision real number formats.
   Julian Blake, CERN, 1211 Geneva 23, Switzerland.
   April 1984, modified October 1990. */

#ifndef HOST_IEEE_FLOATING_POINT
extern double frexp();  /* Extract exponent from real number */

extern double ldexp();  /* Load exponent into real number */
#endif

/* Convert to IEEE single precision floating point format */
conv_ieee32(hval,ival)
double hval;
unsigned long int *ival;
{
#ifndef HOST_IEEE_FLOATING_POINT
        if (hval == 0.0) {             
                *ival = 0;
        } else {
		long int expon;
                register double frac;
		register unsigned long int first = 0;
		register short int pos = 23;

		/* treat sign */
                if (hval < 0.0) {
                        hval = -hval;
			first = 256;
                }
		/* scale and round */
		frac = frexp(hval,&expon);
		if (expon <= -126) {
			/* unnormalized fraction */
			frac = ldexp(hval,126);
			frac += ldexp(0.5,-23);
			if (frac >= 1.0) {
				/* normalized, after all */
				frac -= 1.0;
				++first;
			}
		} else {
			/* normalized fraction */
			frac += ldexp(0.5,-24);
			first += expon + 127;
			if (frac < 1.0) {
				frac += frac;
				--first;
			}
			/* remove leading bit of mantissa */
			frac -= 1.0;
		}
		/* add bits of fraction to sign and exponent */
                do {
                        frac += frac;
                        first += first;
                        if (frac >= 1.0) {
                                frac -= 1.0;
                                ++first;    
                        }
                        --pos;    
                } while (pos != 0);
		*ival = first;
        }
#else /* Host uses IEEE floating point number formats */
	union {float f; unsigned long int i;} u;

	u.f = hval;
	*ival = u.i;
#endif
}

/* Convert to IEEE double precision floating point format */
conv_ieee64(hval,ival1,ival2)
double hval;
unsigned long int *ival1, *ival2;
{
#ifndef HOST_IEEE_FLOATING_POINT
        if (hval == 0.0) {
                *ival1 = 0;
                *ival2 = 0;
        } else {
		long int expon;
                register double frac;
		register unsigned long int first = 0;
		register unsigned long int second = 0;
		register short int pos = 20;

		/* treat sign */
                if (hval < 0.0) {
                        hval = -hval;
                        first = 2048;
                }
		/* scale and round */
		frac = frexp(hval,&expon);
		frac += ldexp(0.5,-53);
		first += expon + 1023;
		if (frac < 1.0) {
			frac += frac;
			--first;
		}
		/* remove leading bit of mantissa */
                frac -= 1.0;
		/* add bits of fraction to sign and exponent */
                do {
                        frac += frac;
                        first += first;
                        if (frac >= 1.0) {
                                frac -= 1.0;
                                ++first;    
                        }
                        --pos;    
                } while (pos != 0);
                pos = 32;
                do {
                        frac += frac;
                        second += second;
                        if (frac >= 1.0) {
                                frac -= 1.0;
                                ++second;         
                        }
                        --pos;         
                } while (pos != 0);
                *ival1 = first;
                *ival2 = second;
        }
#else /* Host uses IEEE floating point number formats */
	union {double d; unsigned long int i[2];} u;

	u.d = hval;
	*ival1 = u.i[0];
	*ival2 = u.i[1];
#endif
}

/* End of CERN conversion routines */

/* Return the best assembler insn template
   for moving operands[1] into operands[0] as a fullword.  */

static char *
singlemove_string (operands)
     rtx *operands;
{
  rtx op0 = operands[0], op1 = operands[1];

  if (GET_CODE (op0) == MEM)
    {
      if (op1 == const0_rtx)
	return modify_memref ("clr%.l %m0", operands);

      return modify_memref ("move%.l %m1,%m0", operands);
    }
  else if (REG_P (op0))
    {
      if (GET_CODE (op1) == CONST_INT)
	if (DATA_REG_P (op0))
	  {
	    if (INTVAL (op1) < 128 && INTVAL (op1) >= -128)
	      return "moveq%.l %1,%0";
	  }
	else if (ADDRESS_REG_P (op0))
	  {
	    if (op0 == const0_rtx)
	      return "suba%.l %0,%0";
	    if (INTVAL (op1) < 0x8000 && INTVAL (op1) >= -0x8000)
	      return "move%.w %1,%0";
	  }
    }

  return modify_memref ("move%.l %m1,%0", operands);
}

/* Output assembler code to perform a doubleword move insn
   with operands OPERANDS.  */

char *
output_move_double (operands)
     rtx *operands;
{
  enum { REGOP, OFFSOP, MEMOP, PUSHOP, POPOP, CNSTOP, RNDOP } optype0, optype1;
  rtx latehalf[2];
  rtx addreg0 = 0, addreg1 = 0;

  /* First classify both operands.  */

  if (REG_P (operands[0]))
    optype0 = REGOP;
  else if (offsettable_memref_p (operands[0]))
    optype0 = OFFSOP;
  else if (GET_CODE (XEXP (operands[0], 0)) == POST_INC)
    optype0 = POPOP;
  else if (GET_CODE (XEXP (operands[0], 0)) == PRE_DEC)
    optype0 = PUSHOP;
  else if (GET_CODE (operands[0]) == MEM)
    optype0 = MEMOP;
  else
    optype0 = RNDOP;

  if (REG_P (operands[1]))
    optype1 = REGOP;
  else if (CONSTANT_P (operands[1])
	   || GET_CODE (operands[1]) == CONST_DOUBLE)
    optype1 = CNSTOP;
  else if (offsettable_memref_p (operands[1]))
    optype1 = OFFSOP;
  else if (GET_CODE (XEXP (operands[1], 0)) == POST_INC)
    optype1 = POPOP;
  else if (GET_CODE (XEXP (operands[1], 0)) == PRE_DEC)
    optype1 = PUSHOP;
  else if (GET_CODE (operands[1]) == MEM)
    optype1 = MEMOP;
  else
    optype1 = RNDOP;

  /* Check for the cases that the operand constraints are not
     supposed to allow to happen.  Abort if we get one,
     because generating code for these cases is painful.  */

  if (optype0 == RNDOP || optype1 == RNDOP)
    abort ();

  /* Use movem instruction to load or store a register pair.  */

  if (optype0 == REGOP
      && (optype1 == OFFSOP || optype1 == MEMOP || optype1 == POPOP))
    {
      char template [MAX_OFFSET_REGS * 32 + 64];

      sprintf (template, "movem%%.l %%m1,%s/%s",
	       reg_names[REGNO (operands[0])],
	       reg_names[REGNO (operands[0]) + 1]);
      return modify_memref (template, operands);
    }
  if ((optype0 == OFFSOP || optype0 == MEMOP || optype0 == PUSHOP)
      && optype1 == REGOP)
    {
      char template [MAX_OFFSET_REGS * 32 + 64];

      sprintf (template, "movem%%.l %s/%s,%%m0",
	       reg_names[REGNO (operands[1])],
	       reg_names[REGNO (operands[1]) + 1]);
      return modify_memref (template, operands);
    }

  /* If one operand is decrementing and one is incrementing
     decrement the former register explicitly
     and change that operand into ordinary indexing.  */

  if (optype0 == PUSHOP && optype1 == POPOP)
    {
      operands[0] = XEXP (XEXP (operands[0], 0), 0);
      output_asm_insn ("subq%.l %#8,%0", operands);
      operands[0] = gen_rtx (MEM, DImode, operands[0]);
      optype0 = OFFSOP;
    }
  if (optype0 == POPOP && optype1 == PUSHOP)
    {
      operands[1] = XEXP (XEXP (operands[1], 0), 0);
      output_asm_insn ("subq%.l %#8,%1", operands);
      operands[1] = gen_rtx (MEM, DImode, operands[1]);
      optype1 = OFFSOP;
    }

  /* If an operand is an unoffsettable memory ref, find a register
     we can increment temporarily to make it refer to the second word.  */

  if (optype0 == MEMOP)
    addreg0 = find_addr_reg (XEXP (operands[0], 0));

  if (optype1 == MEMOP)
    addreg1 = find_addr_reg (XEXP (operands[1], 0));

  /* Ok, we can do one word at a time.
     Normally we do the low-numbered word first,
     but if either operand is autodecrementing then we
     do the high-numbered word first.

     In either case, set up in LATEHALF the operands to use
     for the high-numbered word and in some cases alter the
     operands in OPERANDS to be suitable for the low-numbered word.  */

  if (optype0 == REGOP)
    latehalf[0] = gen_rtx (REG, SImode, REGNO (operands[0]) + 1);
  else if (optype0 == OFFSOP)
    latehalf[0] = adj_offsettable_operand (operands[0], 4);
  else
    latehalf[0] = operands[0];

  if (optype1 == REGOP)
    latehalf[1] = gen_rtx (REG, SImode, REGNO (operands[1]) + 1);
  else if (optype1 == OFFSOP)
    latehalf[1] = adj_offsettable_operand (operands[1], 4);
  else if (optype1 == CNSTOP)
    {
      if (CONSTANT_P (operands[1]))
	latehalf[1] = const0_rtx;
      else if (GET_CODE (operands[1]) == CONST_DOUBLE)
	{
	  union {double d; int i[2];} u;
	  unsigned long int ival1, ival2;

	  u.i[0] = CONST_DOUBLE_LOW (operands[1]);
	  u.i[1] = CONST_DOUBLE_HIGH (operands[1]);
	  conv_ieee64 (u.d, &ival1, &ival2);
	  latehalf[1] = gen_rtx (CONST_INT, VOIDmode, ival2);
	  operands[1] = gen_rtx (CONST_INT, VOIDmode, ival1);
	}
    }
  else
    latehalf[1] = operands[1];

  /* If insn is effectively movd N(sp),-(sp) then we will do the
     high word first.  We should use the adjusted operand 1 (which is N+4(sp))
     for the low word as well, to compensate for the first decrement of sp.  */
  if (optype0 == PUSHOP
      && REGNO (XEXP (XEXP (operands[0], 0), 0)) == STACK_POINTER_REGNUM
      &&  reg_overlap_mentioned_p (stack_pointer_rtx, operands[1]))
    operands[1] = latehalf[1];

  /* If one or both operands autodecrementing,
     do the two words, high-numbered first.  */

  /* Likewise,  the first move would clobber the source of the second one,
     do them in the other order.  This happens only for registers;
     such overlap can't happen in memory unless the user explicitly
     sets it up, and that is an undefined circumstance.  */

  if (optype0 == PUSHOP || optype1 == PUSHOP
      || (optype0 == REGOP && optype1 == REGOP
	  && REGNO (operands[0]) == REGNO (latehalf[1])))
    {
      /* Make any unoffsettable addresses point at high-numbered word.  */
      if (addreg0)
	output_asm_insn ("addq%.l %#4,%0", &addreg0);
      if (addreg1)
	output_asm_insn ("addq%.l %#4,%0", &addreg1);

      /* Do that word.  */
      output_asm_insn (singlemove_string (latehalf), latehalf);

      /* Undo the adds we just did.  */
      if (addreg0)
	output_asm_insn ("subq%.l %#4,%0", &addreg0);
      if (addreg1)
	output_asm_insn ("subq%.l %#4,%0", &addreg1);

      /* Do low-numbered word.  */
      return singlemove_string (operands);
    }

  /* Normal case: do the two words, low-numbered first.  */

  output_asm_insn (singlemove_string (operands), operands);

  /* Make any unoffsettable addresses point at high-numbered word.  */
  if (addreg0)
    output_asm_insn ("addq%.l %#4,%0", &addreg0);
  if (addreg1)
    output_asm_insn ("addq%.l %#4,%0", &addreg1);

  /* Do that word.  */
  output_asm_insn (singlemove_string (latehalf), latehalf);

  /* Undo the adds we just did.  */
  if (addreg0)
    output_asm_insn ("subq%.l %#4,%0", &addreg0);
  if (addreg1)
    output_asm_insn ("subq%.l %#4,%0", &addreg1);

  return "";
}

/* Return a REG that occurs in ADDR with coefficient 1.
   ADDR can be effectively incremented by incrementing REG.  */

static rtx
find_addr_reg (addr)
     rtx addr;
{
  while (GET_CODE (addr) == PLUS)
    {
      if (GET_CODE (XEXP (addr, 0)) == REG)
	addr = XEXP (addr, 0);
      else if (GET_CODE (XEXP (addr, 1)) == REG)
	addr = XEXP (addr, 1);
      else if (CONSTANT_P (XEXP (addr, 0)))
	addr = XEXP (addr, 1);
      else if (CONSTANT_P (XEXP (addr, 1)))
	addr = XEXP (addr, 0);
      else
	abort ();
    }
  if (GET_CODE (addr) == REG)
    return addr;
  abort ();
}

char *
output_move_const_double (operands)
     rtx *operands;
{
  int code = standard_68881_constant_p (operands[1]);

  if (code != 0)
    {
      static char buf[40];

      sprintf (buf, "fmovecr %%#0x%x,%%0", code & 0xff);
      return buf;
    }
  return "fmove%.d %1,%0";
}

char *
output_move_const_single (operands)
     rtx *operands;
{
  int code = standard_68881_constant_p (operands[1]);

  if (code != 0)
    {
      static char buf[40];

      sprintf (buf, "fmovecr %%#0x%x,%%0", code & 0xff);
      return buf;
    }
  return "fmove%.s %f1,%0";
}

/* Return nonzero if X, a CONST_DOUBLE, has a value that we can get
   from the "fmovecr" instruction.
   The value, anded with 0xff, gives the code to use in fmovecr
   to get the desired constant.  */

int
standard_68881_constant_p (x)
     rtx x;
{
  union {double d; int i[2];} u;
  register double d;
  u.i[0] = CONST_DOUBLE_LOW (x);
  u.i[1] = CONST_DOUBLE_HIGH (x);
  d = u.d;

  if (d == 0)
    return 0x0f;
  /* Note: there are various other constants available
     but it is a nuisance to put in their values here.  */
  if (d == 1)
    return 0x32;
  if (d == 10)
    return 0x33;
  if (d == 100)
    return 0x34;
  if (d == 10000)
    return 0x35;
  if (d == 1e8)
    return 0x36;
  if (GET_MODE (x) == SFmode)
    return 0;
  if (d == 1e16)
    return 0x37;
  /* larger powers of ten in the constants ram are not used
     because they are not equal to a `double' C constant.  */
  return 0;
}
