/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/*
*                            
*      Name        : scomp.c 
*                            
*      Version     : scomp.c,v 0.8 90/10/14 
*                            
*      Description : synthetic program to call gcc-syn repeatedly in a 
*                    similar fashion to the make program when it compiles
*                    gcc. This program forks processes in sequence to 
*                    execute the synthetic gcc program which in turn forks 
*                    processes to execute programs that synthsize the 
*                    programs that make up the compiler process.   
*                    
*      Written by  : Brad Nichols                                         
*                    bbn@maxine.wpi.edu                              
*
*      e-mail      : mach@cs.wpi.edu
*                   
*      Address     : Mach Research Group       
*                    Worcester Polytechnic Institute                
*                    CS Dept                                        
*                    100 Institute Road,                            
*                    Worcester MA 01609.                            
*                    U.S.A                                         
*                    (508) 831-5357                                 
*                                                                   
*      Rev History : 07/01/91
*                    Version 0.9 
*                    Jonas A. Lehmann (jlehmann@cs.wpi.edu)
*/                                                                 
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/


/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/* Program begins                                                        */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/


/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/* Includes                                                              */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/

#include <stdio.h>
#include <sys/types.h> 
#include <sys/wait.h>


/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/* SELECT TIMING METHOD                                                  */
/* To select any particular timing  method make sure that the            */
/* corresponding #define is uncommented and is the only timing method    */
/* selected.                                                             */
/*                                 systems

    timing          MACH 2.5    MACH 3.0   SCO V.3   SYS V.4
    ---------------------------------------------------------
    gettimeofday        X          X                    X
    getrusage           X          X                    
    ftime               X          X          X
    time                X          X          X         X
    times               X          X          X         X
    clock                                     X         X
*/
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/


#define GETTIMEOFDAY            /* ACTIVE TIMING METHOD */

/* #define GETRUSAGE */
/* #define FTIME     */
/* #define TIME      */
/* #define TIMES     */
/* #define CLOCK     */

/* #define VFORK     */


/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/* Constants                                                             */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/

#define NUM_GCC_CALLS   18  /* Number of times to invoke sgcc */



/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/* DEFINE TIMING PARAMETERS AND INCLUDES                                 */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/

#ifdef GETTIMEOFDAY
#include <sys/time.h>     
  struct timeval  gtime_start, gtime_end;
  struct timezone gtzp;
  double          gtime_elapsed;
  int             gtime_diff;
  int             time_diff;
#endif
#ifdef GETRUSAGE  
#include <sys/time.h>      
#include <sys/resource.h>  
  struct rusage   rusage_start, rusage_end;
  double          rusage_utime, rusage_stime;
  int             rusage_udiff, rusage_sdiff;
#endif
#ifdef FTIME
#include <sys/timeb.h>     
  struct timeb    ftime_start, ftime_end;
  unsigned long   ftime_sec1, ftime_sec2, ftime_millisec1, ftime_millisec2;
  unsigned long   ftime_diff;
#endif
#ifdef TIME
#include <sys/times.h>    
  time_t          time_start, time_end;
  int             time_diff;
#endif
#ifdef TIMES
#include <sys/times.h>    
#include <sys/param.h>        
  struct tms      times_start, times_end;
  long            times_utime, times_stime;
  int             times_udiff, times_sdiff;
#endif 
#ifdef CLOCK
  long            clock_start, clock_end;
  int             clock_diff;
#endif


/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/
/* main()                                                                */
/* Start Timer -> execute sgcc -> stop timer -> repeat                   */
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/

main(argc, argv)
     int   argc;
     char *argv[];
{
  int i,child;
  union wait *statusp;
  


  /* START TIMING */

#ifdef GETTIMEOFDAY
  gettimeofday (&gtime_start, &gtzp);
#endif
#ifdef GETRUSAGE
  getrusage(RUSAGE_SELF, &rusage_start);
#endif
#ifdef FTIME
  ftime(&ftime_start);
#endif 
#ifdef TIME
  time(&time_start);
#endif
#ifdef TIMES
  times(&times_start);
#endif
#ifdef CLOCK
  clock_start = clock();
#endif




  /* call sgcc as many times as defined by constant NUM_GCC_CALLS */

  for (i = 0; i < NUM_GCC_CALLS; i++) 
    {
      if ((child = fork()) == 0)  
	{
	  execvp("./sgcc",argv); 
	}
      else {
	wait(statusp);
      }
    }

  



  /* STOP TIMING, COMPUTE AND PRINT RESULT(S) */

#ifdef GETTIMEOFDAY
  gettimeofday (&gtime_end, &gtzp);
  gtime_elapsed = (1000*gtime_end.tv_sec + gtime_end.tv_usec/1000) -
    (1000*gtime_start.tv_sec + gtime_start.tv_usec/1000);
  gtime_diff = gtime_elapsed;
  time_diff = gtime_diff;
  printf("gettimeofday results:\n");
  printf("  Elaped time = %d msec\n",gtime_diff);
#endif
  
#ifdef GETRUSAGE
  getrusage(RUSAGE_SELF, &rusage_end);
  rusage_utime = (1000*rusage_end.ru_utime.tv_sec +
		  rusage_end.ru_utime.tv_usec/1000) - 
		    (1000*rusage_start.ru_utime.tv_sec +
		     rusage_start.ru_utime.tv_usec/1000);
  rusage_udiff = rusage_utime;
  rusage_stime = (1000*rusage_end.ru_stime.tv_sec +
		  rusage_end.ru_stime.tv_usec/1000)  -
		    (1000*rusage_start.ru_stime.tv_sec + 
		     rusage_start.ru_stime.tv_usec/1000);
  rusage_sdiff = rusage_stime;
  printf("getrusage results:\n");
  printf("  User time   = %d msec\n",rusage_udiff);
  printf("  System time = %d msec\n",rusage_sdiff);
  time_diff = rusage_udiff + rusage_sdiff;
#endif
  
#ifdef FTIME
  ftime(&ftime_end);
  ftime_sec1 = ftime_start.time;
  ftime_millisec1 = ftime_start.millitm;
  ftime_millisec1 += ftime_sec1*1000;
  ftime_sec2 = ftime_end.time;
  ftime_millisec2 = ftime_end.millitm;
  ftime_millisec2 += ftime_sec2*1000;
  ftime_diff += ftime_millisec2 - ftime_millisec1;
  time_diff = ftime_diff;
  printf("ftime results:\n");
  printf("  Elaped time = %d msec\n",ftime_diff);
#endif
  
#ifdef TIME
  time(&time_end);
  time_diff = time_end - time_start;
  printf("time results:\n");
  printf("  Elaped time = %d sec\n",time_diff);
#endif
  
#ifdef TIMES
  times(&times_end);
  times_utime = 1000*(times_end.tms_utime 
		      - times_start.tms_utime)/HZ;
  times_udiff = times_utime;
  times_stime = 1000*(times_end.tms_stime 
		      - times_start.tms_stime)/HZ;
  times_sdiff = times_stime;
  printf("times results:\n");
  printf("  User time   = %d msec\n",times_udiff);
  printf("  System time = %d msec\n",times_sdiff);
#endif
  
#ifdef CLOCK
  clock_end = clock();
  clock_diff = (clock_end - clock_start)/1000;
  printf("clock results:\n");
  printf("  CPU time    = %d msec\n",clock_diff);
#endif
  
}
