/* CHU      Marcus Leech  VE3MDL   Aug 1987 */
/* Copyright 1987 By Marcus Leech.
 * Permission is granted to any individual or institution to use, copy, or
 *  redistribute this program provided it is not sold for profit, and
 *  provided this copyright notice is retained in this and all copies of
 *  this program.
 */
/* This program reads and understands the timecode format
 *  produced by the CHU broadcast signal.  This signal provides
 *  an atomic time standard to most places in North and South
 *  America on 3.330 7.335 and 14.670 MHZ, mode A3. During
 *  seconds 31 thru 39 the time is broadcast using Bell 103 AFSK.
 *  The time code looks like this:
 *  6.d|d.d|h.h|m.m|s.s|6.d|d.d|h.h|m.m|s.s
 *  The time code is repeated twice for error detection. Each . represents
 *    one nibble boundary.  The nibbles get transmitted reversed, so you
 *    have to flip them to make sense of the time code.  Each nibble is a BCD
 *    digit.
 * It takes one argument, the input device, and some optional flags:
 *  chu [-adst] <device>
 *  -
 *  a adjust  attempt to adjust the system clock to CHU time
 *  d daemon  run continuously as a daemon
 *  s show    show semi-raw timecode
 *  t tune    show chars that wouldn't get considered for time code
 *  m <min>   set minimum adjustment to <min> (centiseconds)
 *  j <jit>   set allowable jitter to <jit> (centiseconds)
 *
 * When used as a system-time-adjuster, you need to set the system time
 *  to within 5sec of the correct time. The program will drag the
 *  clock into exact synchronism.
 */
/* This source contains support for VMS, MSDOS, BSD-UNIX, and SYSV-UNIX
 */
#include <stdio.h>
#ifndef VMS
#ifndef MSDOS
#include <sys/types.h>
#include <sgtty.h>
#include <time.h>
#define CHUREAD read
#else
#include <time.h>
#include <dos.h>
#define CHUREAD ourread
#endif
#else VMS
#include <types.h>
#include <time.h>
#endif VMS

/* Macros to fetch individual nibbles. */
#define hinib(x) (((x) >> 4) & 0x0F)
#define lonib(x) ((x) & 0x0F)

/* Macro to restore correct ordering within a byte. */
#define byte(h,l) (((l) << 4)|(h))

/* Parameters of the moving-average and smoothing algorithm
 */
#define AVGSIZ 5	/* Size of the moving average array. */
#define TJITTER 5	/* Maximum allowable jitter in array. */
int tjitt = TJITTER;
#define TMIN 20		/* Minimum centisecond adjustment we'll make. */
int tmin = TMIN;

#define TMAX 500	/* Maximum centisecond adjustment we'll make. */
#define GROSS (2*60*100)/* Errors larger than this constitute a gross error. */

int movavg[AVGSIZ];	  /* Array to compute moving-average. */
int avgpos;		  /* Current position in moving-average. */
#define TWEAK 50	  /* Fudge factor in centiseconds. */
                          /* CHU code is skewed by 0.5 seconds. */
int fudge = 0;

#define iabs(x) ((x < 0) ? -x : x)
#define max(x,y) ((x > y) ? x : y)

char sample1[5];  /* The first time code. */
char sample2[5];  /* The second (error checking) time code. */
#ifdef MSDOS
int parport = 0x3BC;
int coravg;
int ncor;
#endif MSDOS

char CopyRight[] = "Copyright 1987 Marcus Leech";

main (argc, argv)
int argc;
char **argv;
{
    char c, *p;
#ifdef MSDOS
    struct time dostime;
#endif MSDOS
#ifndef VMS
#ifndef MSDOS
    struct sgttyb sgb;            /* For fiddling with tty line params. */
#endif MSDOS
#endif VMS
    int line;                     /* Filedescriptor/port for the tty line.*/
    int i;
    struct tm *localtime(), *ltp, *gmtime();  /* To break out the time. */
    time_t now, time();
    long gettdiff ();

#if defined(HPUX) || defined(BSD)
    struct timeval tvb;
    struct timezone tzb;
    struct timeval t1, t2;
    int ptavg = 0;
    int ptcnt = 0;
#endif

#ifdef VMS
    long t0[2], t1[2], t2[2];
#endif VMS
    int amm, ass, mmss, diff;  /* Current difference, CHU conversion bufs. */
    char *devstr;  /* Input device. */
    char *parstr;  /* (MSDOS)  where is parallel port? */
    
    /* Command-line flags. */
    int adjust;
    int show;
    int tune;
    int daemon;
    int tjit, tavg;
    daemon = 0;

    setbuf (stdout, NULL);
    adjust = show = tune = 0;
    devstr = "/dev/null";
    parstr = "000000";

    /* Handle command-line options. */
    for (i = 1; i < argc; i++)
    {
        if (argv[i][0] == '-')
        {
            p = &argv[i][1];
            while (*p)
            {
                switch (*p)
                {
                case 'a':
                    adjust++;
                    break;
                case 't':
                    tune++;
                    break;
                case 's':
                    show++;
                    break;
                case 'd':
                    daemon++;
                    break;
		case 'j':
		    tjitt = atoi (argv[i+1]);
		    i++;
		    break;
		case 'm':
		    tmin = atoi (argv[i+1]);
		    i++;
		    break;
		case 'f':
		    fudge = atoi (argv[i+1]);
		    i++;
		    break;
                default:
                    fprintf (stderr, "unknown flag '%c'\n", *p);
                    exit (1);
                }
                p++;
            }
        }
        else
        {
            strcpy (devstr, argv[i]);
	    strcpy (parstr, argv[i+1]);
	    break;
        }
    }

    /* Open input device. */
#ifndef MSDOS
    if (strcmp (devstr, "stdin") == 0)
    {
        line = 0;
    }
    else
    {
        line = open (devstr, 0);
    }
    if (line == -1)
    {
        fprintf (stderr, "Unable to open %s\n", devstr);
        exit (1);
    }
#else MSDOS
    line = atoi (devstr) - 1;
    sscanf (parstr, "%x", &parport);
#endif MSDOS

#ifndef VMS
#ifndef MSDOS
    /* Set up 8-bit datapath, at 30CPS. */
    if (line != 0)
    {
        gtty (line, &sgb);
        sgb.sg_ispeed = sgb.sg_ospeed = B300;
        sgb.sg_flags |= RAW;
        sgb.sg_flags &= ~ECHO;
        stty (line, &sgb);
    }

#endif MSDOS
#endif VMS

    /* Read forever, waiting for the synchronizing BCD 6 digit to appear. */
    for (;;)
    {
        if (CHUREAD (line, &c, 1) <= 0)
	{
		exit (1);
	}
        /* We have a syncronizing digit. Grab two samples and compare. */
        if (lonib(c) == 6)
        {
#if defined(HPUX) || defined(BSD)
	    gettimeofday (&t1, (struct timezone *)NULL);
#endif
            /* Get first sample. */
            sample1[0] = byte(hinib(c),lonib(c));
            for (i = 1; i < 5; i++)
            {
                CHUREAD (line, &c, 1);
                sample1[i] = byte(hinib(c),lonib(c));
            }
            /* Get second sample. */
            for (i = 0; i < 5; i++)
            {
                CHUREAD (line, &c, 1);
                sample2[i] = byte(hinib(c),lonib(c));
            }
#ifdef MSDOS
	    updpartime ();
	    gettime (&dostime);
	    updstats (&dostime);
#endif MSDOS
            /* If samples match, we have a valid time code. */
            if (compare (sample1, sample2, 5) == 0)
            {
                /* Show the code (if -s).  The high-order nibble in the
                 *  first byte is the synch digit, so it gets masked out
                 *  for printing.
                 */
                if (show)
                {
                    fprintf (stdout, "TC: ");
                    for (i = 0; i < 5; i++)
                    {
                        fprintf (stdout, "%02x", sample1[i]);
                    }
                    fprintf (stdout, "\n");
                }
                /* Fetch UTC (GMT). */
		/* Local time and CHU time are converted into
		 *  hundreths of a second.
		 *  Both MSDOS and VMS can deal with this directly--
		 *  generic UNIX deals in units of seconds.
		 * We convert between when we need to.
		 */
                /* Convert (CHU) time code minutes and seconds into
                 *   binary.
                 */
                amm = (hinib(sample1[3]) * 10) + lonib(sample1[3]);
                ass = (hinib(sample1[4]) * 10) + lonib(sample1[4]);
                /* Compute the difference. */
#if defined(HPUX) || defined(BSD)
		diff = (int)gettdiff (amm, ass, TWEAK+fudge, &tvb);
#else
		diff = (int)gettdiff (amm, ass, TWEAK+fudge, &now);
#endif HPUX
#if defined(HPUX) || defined(BSD)
                now = tvb.tv_sec;
#endif HPUX
		/* Error--ignore. */
                if (iabs(diff) > GROSS)
                {
		    ltp = localtime (&now);
                    fprintf (stderr, "%02d-%02d-%02d %02d:%02d:%02d ",
                        ltp->tm_year, ltp->tm_mon + 1, ltp->tm_mday,
                        ltp->tm_hour, ltp->tm_min, ltp->tm_sec);
                    fprintf (stderr, "TERROR %d\n", diff);
                    if (!daemon)
                    {
                        break;
                    }
                    continue;
                }
		/* Compute moving average, and jitter. */
		movavg[avgpos] = diff;
                avgpos++;
		/* Handle wraparound of avgpos. */
		if (avgpos >= AVGSIZ)
		{
			avgpos = 0;
		}
		tavg = average (movavg, AVGSIZ);
		tjit = jitter (movavg,  AVGSIZ, iabs(tavg));
		if (show)
		{
			fprintf (stderr, "tjit %d tavg %d\n", tjit, tavg);
		}
		/* Unix time format is in seconds. */
                /* Only do it if there IS a (reasonable) difference. */
		/* If jitter is tolerable, and average adjustment
		 *  is reasonable, adjust the system time.
		 */
#ifdef MSDOS
		coravg = tavg;
#endif MSDOS
#if defined(HPUX) || defined(BSD)
                gettimeofday (&t2, (struct timezone *)NULL);
                ptavg += millidiff (&t2, &t1);
                ptcnt++;
#endif HPUX
		if ((tjit > tjitt) && (iabs(tavg) >= tmin) &&
		    (iabs(tavg) <= TMAX) )
		{
			ltp = localtime (&now);
			fprintf (stderr, "%02d/%02d/%02d %02d:%02d:%02d ",
				ltp->tm_year, ltp->tm_mon+1, ltp->tm_mday,
				ltp->tm_hour, ltp->tm_min, ltp->tm_sec);
			fprintf (stderr,
                                "jitter exceeds %d(%d) correction is %d\n",
				tjitt, tjit, tavg);
		}
                else if ((tjit <= tjitt) && (iabs(tavg) >= tmin) &&
		    (iabs(tavg) <= TMAX))
                {
		    time_t then;
#ifdef MSDOS
		    int mdiff, sdiff;
		    int hdiff;
		    int sgn;
		    int hundreds; /* ti_hund is UNSIGNED, so we need this! */
#endif MSDOS

		    then = now;
		    diff = tavg;
		    now += (diff / 100);
#ifdef MSDOS
		    ncor++;
#endif MSDOS
#if !defined(HPUX) && !defined(BSD)
		    /* Round up to the next second. */
		    if (iabs(diff % 100) >= 55)
		    {
			    now += diff < 0 ? -1 : 1;
		    }
#endif
                    /* What we'd REALLY like here is a system call of
                     * the form:   stime (*time, ticks)
                     * that would allow you to set the system tick
                     * counter to <ticks> in centiseconds.  Too bad.
                     *-->stime (&now, TWEAK);<--
                     */
#ifndef VMS
#ifndef EUNICE
#ifndef MSDOS
		    /* For UNIX, remember that now has already been
		     *   adjusted by the right number of seconds.
		     */
#if defined(HPUX) || defined (BSD)
		    tvb.tv_sec = now;
		    tvb.tv_usec  += (diff % 100) * 10000;
		    settimeofday (&tvb, &tzb);
#else
                    stime (&now);
#endif
#else
		    /* MSDOS can handle hundredths of a second. */
		    gettime (&dostime);
		    sgn = diff < 0 ? -1 : 1;
		    diff = iabs(diff);
		    mdiff = diff / 6000;
		    sdiff = diff - (mdiff * 6000);
		    sdiff /= 100;
		    hdiff = diff - ((mdiff*6000) + (sdiff * 100));
		    dostime.ti_min += (sgn * mdiff);
		    dostime.ti_sec += (sgn * sdiff);
		    /* ti_hund, is unsigned, so we havta use a temporary
		     *  that's signed, to allow the following code to work.
		     */
		    hundreds = (signed) dostime.ti_hund;
		    hundreds += (sgn * hdiff);
		    if (hundreds >= 100)
		    {
		        hundreds -= 100;
			dostime.ti_sec++;
		    }
		    if (hundreds < 0)
		    {
			hundreds += 100;
			dostime.ti_sec--;
		    }
		    dostime.ti_hund = hundreds;
		    settime (&dostime);
		    diff *= sgn;
#endif MSDOS
#endif EUNICE
#else VMS
		    /* VMS can handle microseconds, so we convert. */
                    t1[0] = 1000 * ((diff));
                    t1[1] = 0;
                    if (diff < 0)
                    {
                        t1[1] = -1;
                    }
                    sys$gettim (t0);
                    lib$addx (t0, t1, t2);
                    sys$setime (t2);
#endif VMS
                    ltp = localtime (&then);
                    fprintf (stderr, "%02d/%02d/%02d %02d:%02d:%02d to ",
			ltp->tm_year, ltp->tm_mon + 1, ltp->tm_mday,
                        ltp->tm_hour, ltp->tm_min, ltp->tm_sec);
		    ltp = localtime (&now);
		    fprintf (stderr, "%02d:%02d:%02d (%d)\n",
		        ltp->tm_hour, ltp->tm_min, ltp->tm_sec, diff);
#if defined(HPUX) || defined(BSD)
                    fprintf (stderr, "%02d:%02d:%02d (proc time %d usecs)\n",
                        ltp->tm_hour, ltp->tm_min, ltp->tm_sec, (ptavg/ptcnt)*100);
                    ptcnt = 0;
                    ptavg = 0;
#endif HPUX
                }
                if (!daemon)
                {
                    break;
                }
            }
        }
        else if (tune)
        {
                fprintf (stdout, "FT: %c (%02x)\n", c, (unsigned)c);
        }
    }
}
/* Compare two byte-arrays (s1, s2) of length cnt. */
compare (s1, s2, cnt)
char *s1;
char *s2;
int cnt;
{
    int i;
    for (i = 0; i < cnt; i++)
    {
        if (*s1++ != *s2++)
        {
            return (1);
        }
    }
    return (0);
}
#ifdef VMS
struct tm *
gmtime (tod)
time_t *tod;
{
    int hh;
    int mm;
    char s;
    int secdiff, sgn;
    time_t utc;
    char *p, *getenv ();
    p = getenv ("UTCDISP");
    sscanf (p, "%c %02d:%02d", &s, &hh, &mm);
    utc = *tod;
    secdiff = (hh * 3600) + (mm * 60);
    sgn = (s == '+') ? 1 : -1;
    secdiff *= sgn;
    utc += secdiff;
    return (localtime (&utc));
}
#endif VMS
#ifdef MSDOS
/* This function reads from the serial port. */
int _rxbufs[2] = {0x3F8, 0x2F8};
int _rxsts[2] = {0x3FD, 0x2FD};
/* During TIMEOUT, it does a TOD update. */
int
ourread (ln, d, cnt)
int ln;
char *d;
int cnt;
{
        int sts;
	int bioscom ();
	int ocnt;
	ocnt = cnt;
#define TIMEOUT (1<<12)
	while (ocnt)
	{
		while ((sts = getser (ln)) == TIMEOUT)
		{
			updpartime ();
		}
		*d++ = sts & 0xFF;
		ocnt--;
	}
	return (cnt - ocnt);
}
int
getser (line)
int line;
{
	int cnt;
	int sts;

	cnt = 0;
	while (cnt < 5)
	{
		sts = inportb (_rxsts[line]);
		if ((sts & 0x01) != 0)
		{
			break;
		}
		cnt++;
	}
	if (cnt >= 5)
	{
		return (TIMEOUT);
	}
	else
	{
		return (inportb (_rxbufs[line]) & 0xFF);
	}
}
/* Write current seconds and quarter-seconds to parallel port. */
int lastten = 0;
updpartime ()
{
	struct time dostime;

	gettime (&dostime);
	wrtpar (dostime.ti_hour, 0);
	wrtpar (dostime.ti_min, 1);
	wrtpar (dostime.ti_sec, 2);
	wrtpar (dostime.ti_hund/10, 3);
	if ((lastten != (dostime.ti_hund/20)) &&
		(dostime.ti_sec <= 29 || dostime.ti_sec >= 41) )
	{
		if (kbhit ())
		{
			fprintf (stderr, "new fudge value: ");
			fscanf (stdin, "%d", &fudge);
		}
		updstats (&dostime);
		lastten = dostime.ti_hund / 20;
	}
}
updstats (dt)
struct time *dt;
{
	struct tm *gmtime(), *ltp;

	fprintf (stderr, "%02d:%02d:%02d.%01d %03d %04d\r",
		dt->ti_hour, dt->ti_min, dt->ti_sec,
		dt->ti_hund/10, coravg, ncor);
}
wrtpar (val, reg)
int val;
int reg;
{
	char outbyte;
	outbyte = (char)(val << 2);
	outbyte |= (char)(reg & 0x03);
	outportb (parport+2, 0x01);
	outportb (parport, outbyte);
	outportb (parport+2, 0x00);
}
#endif MSDOS
average (ary, n)
int ary[];
int n;
{
	int avg;
	int i;

	avg = 0;
	/* Pass one compute average. */
	for (i = 0; i < n; i++)
	{
		avg += ary[i];
	}
	return (avg / n);
}
jitter (ary, n, avg)
int ary[];
int n;
int avg;
{
	int i, m;
	int d;
	d = 0;
	m = 0;

	for (i = 0; i < n; i++)
	{
		d = iabs(ary[i]) - avg;
		d = iabs(d);
		m = max (m, d);
	}
	return (m);
}
long
gettdiff (mm, ss, hh, now)
int mm;
int ss;
int hh;
#if defined(HPUX) || defined(BSD)
struct timeval *now;
#else
time_t *now;
#endif HPUX
{
	long chutime;
	long ourtime;
#ifdef MSDOS
	struct time dostime;
#else MSDOS
	struct tm *gmtime ();
	struct tm *ltp;
#endif MSDOS

#ifndef MSDOS
#if defined(HPUX) || defined(BSD)
	struct timezone tzb;
#endif
#if defined(HPUX) || defined(BSD)
	gettimeofday (now, &tzb);
	ltp = gmtime (&(now->tv_sec));
#else
	time (now);
	ltp = gmtime (now);
#endif
	ourtime = ltp->tm_min * 6000L;
	ourtime += ltp->tm_sec * 100L;
#if defined (HPUX) || defined(BSD)
	ourtime += (now->tv_usec / 10000);
#else
	ourtime += 25;
#endif
#else MSDOS
	gettime (&dostime);
	ourtime = dostime.ti_min * 6000;
	ourtime += dostime.ti_sec * 100;
	ourtime += dostime.ti_hund;
#endif MSDOS
	chutime = mm * 6000L;
	chutime += ss * 100;
	chutime += hh;
	return (chutime - ourtime);
}
#if defined(HPUX) || defined(BSD)
int
millidiff (t2, t1)
struct timeval *t1, *t2;
{
	int secdiff;
	int microdiff;

	secdiff = (t2->tv_sec - t1->tv_sec)*10000;
	microdiff = (t2->tv_usec - t1->tv_usec)/100;
	return (secdiff + microdiff);
}
#endif HPUX
