/*
	NTP routines borrowed from the xntp (ntpq) sources.
*/

#include <stdio.h>
#include <strings.h>
#include <ctype.h>
#include <signal.h>
#include <setjmp.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <netinet/in.h>
#include <netdb.h>

#include "ntp_fp.h"
#include "ntp.h"
#include "ntp_control.h"
#include "xtw.h"

/*
 * Some variables used and manipulated locally
 */

struct timeval tvout = { DEFTIMEOUT, 0 };       /* time out for reads */
struct timeval tvsout = { DEFSTIMEOUT, 0 };     /* secondary time out */
l_fp delay_time;                                /* delay time */
char currenthost[LENHOSTNAME];                  /* current host name */
struct sockaddr_in hostaddr = { 0 };            /* host address */
int showhostnames = 1;                          /* show host names by default */

int sockfd;                                     /* fd socket is openned on */
int havehost = 0;                               /* set to 1 when host open */
struct servent *server_entry = NULL;            /* server entry for ntp */

/*
 * Sequence number used for requests.  It is incremented before
 * it is used.
 */
u_short sequence;

/*
 * Holds data returned from queries.  Declare buffer long to be sure of
 * alignment.
 */
#define MAXFRAGS        24              /* maximum number of fragments */
#define DATASIZE        (MAXFRAGS*480)  /* maximum amount of data */
long pktdata[DATASIZE/sizeof(long)];

/*
 * Holds association data for use with the &n operator.
 */
struct association acache[MAXASSOC];
int numassoc = 0;               /* number of cached associations */

extern char *progname;
int debug = 0;

/*
 * Keyid used for authenticated requests.  Obtained on the fly.
 */
u_long info_auth_keyid;

/*
 * Flag which indicates we should always send authenticated requests
 */
int always_auth = 0;

/*
 * Packet version number we use
 */
u_char pktversion = NTP_VERSION;

/*
 * openhost - open a socket to a host
 */
int
openhost(hname)
	char *hname;
{
	u_long netnum;
#ifdef SO_RCVBUF
	int rbufsize;
#endif
	char temphost[MAXHOSTNAMELEN];
	int getnetnum();

	if (server_entry == NULL) {
		server_entry = getservbyname("ntp", "udp");
		if (server_entry == NULL) {
			(void) fprintf(stderr, "%s: ntp/udp: unknown service\n",
			    progname);
			exit(1);
		}
		if (debug > 2)
			printf("Got ntp/udp service entry\n");
	}

	if (!getnetnum(hname, &netnum, temphost))
		return FALSE;
	
	if (debug > 2)
		printf("Opening host %s\n", temphost);

	if (havehost == 1) {
		if (debug > 2)
			printf("Closing old host %s\n", currenthost);
		(void) close(sockfd);
		havehost = 0;
	}
	(void) strcpy(currenthost, temphost);

	hostaddr.sin_family = AF_INET;
	hostaddr.sin_port = server_entry->s_port;
	hostaddr.sin_addr.s_addr = netnum;

	sockfd = socket(AF_INET, SOCK_DGRAM, 0);
	if (sockfd == -1)
		error("socket", "", "");
	
#ifdef SO_RCVBUF
	rbufsize = DATASIZE + 2048;	/* 2K for slop */
	if (setsockopt(sockfd, SOL_SOCKET, SO_RCVBUF,
	    &rbufsize, sizeof(int)) == -1)
		error("setsockopt", "", "");
#endif

	if (connect(sockfd, (char *)&hostaddr, sizeof(hostaddr)) == -1)
		error("connect", "", "");
	
	havehost = TRUE;
	return TRUE;
}


/*
 * sendpkt - send a packet to the remote host
 */
int
sendpkt(xdata, xdatalen)
	char *xdata;
	int xdatalen;
{
	if (debug >= 3)
		printf("Sending %d octets\n", xdatalen);

	if (write(sockfd, xdata, xdatalen) == -1) {
		warning("write to %s failed", currenthost, "");
		return -1;
	}

	if (debug >= 4) {
		int first = 8;
		printf("Packet data:\n");
		while (xdatalen-- > 0) {
			if (first-- == 0) {
				printf("\n");
				first = 7;
			}
			printf(" %02x", *xdata++);
		}
		printf("\n");
	}
	return 0;
}



/*
 * getresponse - get a (series of) response packet(s) and return the data
 */
int
getresponse(opcode, associd, rstatus, rsize, rdata, timeo)
	int opcode;
	int associd;
	u_short *rstatus;
	int *rsize;
	char **rdata;
	int timeo;
{
	struct ntp_control rpkt;
	struct timeval tvo;
	u_short offsets[MAXFRAGS+1];
	u_short counts[MAXFRAGS+1];
	u_short offset;
	u_short count;
	int numfrags;
	int seenlastfrag;
	int firstpkt;
	fd_set fds;
	int n;

	/*
	 * This is pretty tricky.  We may get between 1 and MAXFRAG packets
	 * back in response to the request.  We peel the data out of
	 * each packet and collect it in one long block.  When the last
	 * packet in the sequence is received we'll know how much data we
	 * should have had.  Note we use one long time out, should reconsider.
	 */
	*rsize = 0;
	if (rstatus)
		*rstatus = 0;
	*rdata = (char *)pktdata;

	numfrags = 0;
	seenlastfrag = 0;
	firstpkt = 1;

	FD_ZERO(&fds);

again:
	if (firstpkt)
		tvo = tvout;
	else
		tvo = tvsout;
	
	FD_SET(sockfd, &fds);
	n = select(sockfd+1, &fds, (fd_set *)0, (fd_set *)0, &tvo);

	if (debug >= 1)
		printf("select() returns %d\n", n);

	if (n == -1) {
		warning("select fails", "", "");
		return -1;
	}
	if (n == 0) {
		/*
		 * Timed out.  Return what we have
		 */
		if (firstpkt) {
			if (timeo)
				(void) fprintf(stderr,
				    "%s: timed out, nothing received\n",
				    currenthost);
			return ERR_TIMEOUT;
		} else {
			if (timeo)
				(void) fprintf(stderr,
				    "%s: timed out with incomplete data\n",
				    currenthost);
			if (debug) {
				printf("Received fragments:\n");
				for (n = 0; n < numfrags; n++)
					printf("%4d %d\n", offsets[n],
					    counts[n]);
				if (seenlastfrag)
					printf("last fragment received\n");
				else
					printf("last fragment not received\n");
			}
			return ERR_INCOMPLETE;
		}
	}

	n = read(sockfd, (char *)&rpkt, sizeof(rpkt));
	if (n == -1) {
		warning("read", "", "");
		return -1;
	}


	/*
	 * Check for format errors.  Bug proofing.
	 */
	if (n < CTL_HEADER_LEN) {
		if (debug)
			printf("Short (%d byte) packet received\n", n);
		goto again;
	}
	if (PKT_VERSION(rpkt.li_vn_mode) != NTP_VERSION
	    && PKT_VERSION(rpkt.li_vn_mode) != NTP_OLDVERSION) {
		if (debug)
			printf("Packet received with version %d\n",
			    PKT_VERSION(rpkt.li_vn_mode));
		goto again;
	}
	if (PKT_MODE(rpkt.li_vn_mode) != MODE_CONTROL) {
		if (debug)
			printf("Packet received with mode %d\n",
			    PKT_MODE(rpkt.li_vn_mode));
		goto again;
	}
	if (!CTL_ISRESPONSE(rpkt.r_m_e_op)) {
		if (debug)
			printf("Received request packet, wanted response\n");
		goto again;
	}

	/*
	 * Check opcode and sequence number for a match.
	 * Could be old data getting to us.
	 */
	if (ntohs(rpkt.sequence) != sequence) {
		if (debug)
			printf(
			    "Received sequnce number %d, wanted %d\n",
			    ntohs(rpkt.sequence), sequence);
		goto again;
	}
	if (CTL_OP(rpkt.r_m_e_op) != opcode) {
		if (debug)
			printf(
		"Received opcode %d, wanted %d (sequence number okay)\n",
			 CTL_OP(rpkt.r_m_e_op), opcode);
		goto again;
	}

	/*
	 * Check the error code.  If non-zero, return it.
	 */
	if (CTL_ISERROR(rpkt.r_m_e_op)) {
		int errcode;

		errcode = (ntohs(rpkt.status) >> 8) & 0xff;
		if (debug && CTL_ISMORE(rpkt.r_m_e_op)) {
			printf("Error code %d received on not-final packet\n",
			    errcode);
		}
		if (errcode == CERR_UNSPEC)
			return ERR_UNSPEC;
		return errcode;
	}

	/*
	 * Check the association ID to make sure it matches what
	 * we sent.
	 */
	if (ntohs(rpkt.associd) != associd) {
		if (debug)
			printf("Association ID %d doesn't match expected %d\n",
			    ntohs(rpkt.associd), associd);
	/*
	 * Hack for silly fuzzballs which, at the time of writing,
	 * return an assID of sys.peer when queried for system variables.
	 */
#ifdef notdef
		goto again;
#endif
	}

	/*
	 * Collect offset and count.  Make sure they make sense.
	 */
	offset = ntohs(rpkt.offset);
	count = ntohs(rpkt.count);

	if (debug >= 4) {
		int shouldbesize;
		u_long key;
		u_long *lpkt;

		/*
		 * Usually we ignore authentication, but for debugging purposes
		 * we watch it here.
		 */
		shouldbesize = CTL_HEADER_LEN + count;
		while (shouldbesize & 0x3) {
			shouldbesize++;
		}

		if (n & 0x3) {
			printf("Packet not padded, size = %d\n", n);
		} if ((n - shouldbesize) >= MAC_LEN) {
			printf(
		"Packet shows signs of authentication (total %d, data %d)\n",
			    n, shouldbesize);
			lpkt = (u_long *)&rpkt;
			printf("%08lx %08lx %08lx %08lx %08lx %08lx\n",
			    ntohl(lpkt[(n - MAC_LEN)/sizeof(u_long) - 3]),
			    ntohl(lpkt[(n - MAC_LEN)/sizeof(u_long) - 2]),
			    ntohl(lpkt[(n - MAC_LEN)/sizeof(u_long) - 1]),
			    ntohl(lpkt[(n - MAC_LEN)/sizeof(u_long)]),
			    ntohl(lpkt[(n - MAC_LEN)/sizeof(u_long) + 1]),
			    ntohl(lpkt[(n - MAC_LEN)/sizeof(u_long) + 2]));
			key = ntohl(lpkt[(n - MAC_LEN) / sizeof(u_long)]);
			printf("Authenticated with keyid %lu\n", key);
			if (key != 0 && key != info_auth_keyid) {
				printf("We don't know that key\n");
			} else {
				if (authdecrypt(key, (u_long *)&rpkt,
				    (n - MAC_LEN))) {
					printf("Auth okay!\n");
				} else {
					printf("Auth failed!\n");
				}
			}
		}
	}

	if (debug >= 2)
		printf("Got packet, size = %d\n", n);
	if (count > (n-CTL_HEADER_LEN)) {
		if (debug)
			printf(
			  "Received count of %d octets, data in packet is %d\n",
			    count, n-CTL_HEADER_LEN);
		goto again;
	}
	if (count == 0 && CTL_ISMORE(rpkt.r_m_e_op)) {
		if (debug)
			printf("Received count of 0 in non-final fragment\n");
		goto again;
	}
	if (offset + count > sizeof(pktdata)) {
		if (debug)
			printf("Offset %d, count %d, too big for buffer\n");
		return ERR_TOOMUCH;
	}
	if (seenlastfrag && !CTL_ISMORE(rpkt.r_m_e_op)) {
		if (debug)
			printf("Received second last fragment packet\n");
		goto again;
	}

	/*
	 * So far, so good.  Record this fragment, making sure it doesn't
	 * overlap anything.
	 */
	if (debug >= 2)
		printf("Packet okay\n");;
	if (firstpkt) {
		firstpkt = 0;
		numfrags = 1;
		counts[0] = count;
		offsets[0] = offset;
	} else {
		for (n = 0; n < numfrags; n++) {
			if (offset == offsets[n])
				goto again;	/* duplicate */
			if (offset < offsets[n])
				break;
		}

		if (numfrags == MAXFRAGS) {
			if (debug)
				printf("Number of fragments exceeds maximum\n");
			return ERR_TOOMUCH;
		}

		if (n == numfrags) {
			/*
			 * Goes at end
			 */
			if (offsets[n-1] + counts[n-1] > offset)
				goto overlap;
		} else {
			register int i;

			if (n != 0) {
				if (offsets[n-1] + counts[n-1] > offset)
					goto overlap;
			}
			if (numfrags != 1 && offset + count > offsets[n+1])
				goto overlap;
			
			for (i = numfrags; i > n; i--) {
				offsets[i] = offsets[i-1];
				counts[i] = counts[i-1];
			}
		}
		offsets[n] = offset;
		counts[n] = count;
		numfrags++;
	}

	/*
	 * Got that stuffed in right.  Figure out if this was the last.
	 * Record status info out of the last packet.
	 */
	if (!CTL_ISMORE(rpkt.r_m_e_op)) {
		seenlastfrag = 1;
		if (rstatus != 0)
			*rstatus = ntohs(rpkt.status);
	}

	/*
	 * Copy the data into the data buffer.
	 */
	bcopy((char *)rpkt.data, (char *)pktdata + offset, count);
	/*
	 * Make sure it ends in a null
	 */
	rpkt.data[count] = '\0';

	/*
	 * If we've seen the last fragment, look for holes in the sequence.
	 * If there aren't any, we're done.
	 */
	if (seenlastfrag && offsets[0] == 0) {
		for (n = 1; n < numfrags; n++) {
			if (offsets[n-1] + counts[n-1] != offsets[n])
				break;
		}
		if (n == numfrags) {
			*rsize = offsets[numfrags-1] + counts[numfrags-1];
			return 0;
		}
	}
	goto again;

overlap:
	/*
	 * Print debugging message about overlapping fragments
	 */
	if (debug)
		printf("Overlapping fragments returned in response\n");
	goto again;
}


/*
 * sendrequest - format and send a request packet
 */
int
sendrequest(opcode, associd, auth, qsize, qdata)
	int opcode;
	int associd;
	int qsize;
	char *qdata;
	int auth;
{
	struct ntp_control qpkt;
	int pktsize;
	extern void authencrypt();
	extern void authusekey();
	extern int auth_havekey();
	extern char *getpass();
/*	u_long getkeyid();	*/

	/*
	 * Check to make sure the data will fit in one packet
	 */
	if (qsize > CTL_MAX_DATA_LEN) {
		(void) fprintf(stderr,
		    "***Internal error!  qsize (%d) too large\n",
		    qsize);
		return 1;
	}

	/*
	 * Fill in the packet
	 */
	qpkt.li_vn_mode = PKT_LI_VN_MODE(0, pktversion, MODE_CONTROL);
	qpkt.r_m_e_op = (u_char)opcode & CTL_OP_MASK;
	qpkt.sequence = htons(sequence);
	qpkt.status = 0;
	qpkt.associd = htons((u_short)associd);
	qpkt.offset = 0;
	qpkt.count = htons((u_short)qsize);

	/*
	 * If we have data, copy it in and pad it out to a 32
	 * bit boundary.
	 */
	if (qsize > 0) {
		bcopy(qdata, (char *)qpkt.data, qsize);
		pktsize = qsize + CTL_HEADER_LEN;
		while (pktsize & (sizeof(u_long)-1)) {
			qpkt.data[qsize++] = 0;
			pktsize++;
		}
	} else {
		pktsize = CTL_HEADER_LEN;
	}

	/*
	 * If it isn't authenticated we can just send it.  Otherwise
	 * we're going to have to think about it a little.
	 */
	if (!auth && !always_auth) {
		return sendpkt((char *)&qpkt, pktsize);
	} else {
		char *pass;

		/*
		 * Pad out packet to a multiple of 8 octets to be sure
		 * receiver can handle it.
		 */
		while (pktsize & ((sizeof(u_long)<<1)-1)) {
			qpkt.data[qsize++] = 0;
			pktsize++;
		}

		/*
		 * Get the keyid and the password if we don't have one.
		 */
		if (info_auth_keyid == 0) {
/*			info_auth_keyid = getkeyid("Keyid: ");	*/
			if (info_auth_keyid == 0) {
				(void) fprintf(stderr,
				   "Keyid must be defined, request not sent\n");
				return 1;
			}
		}
		if (!auth_havekey(info_auth_keyid)) {
			pass = getpass("Password: ");
			if (*pass != '\0')
				authusekey(info_auth_keyid, 3, pass);
		}
		if (auth_havekey(info_auth_keyid)) {
			/*
			 * Stick the keyid in the packet where
			 * cp currently points.  Cp should be aligned
			 * properly.  Then do the encryptions.
			 */
			*(u_long *)(&qpkt.data[qsize]) = htonl(info_auth_keyid);
			authencrypt(info_auth_keyid, (u_long *)&qpkt,
			    pktsize);
			return sendpkt((char *)&qpkt, pktsize + MAC_LEN);
		} else {
			(void) fprintf(stderr,
			    "No password, request not sent\n");
			return 1;
		}
	}
	/*NOTREACHED*/
}


/*
 * doquery - send a request and process the response
 */
int
doquery(opcode, associd, auth, qsize, qdata, rstatus, rsize, rdata)
	int opcode;
	int associd;
	int auth;
	int qsize;
	char *qdata;
	u_short *rstatus;
	int *rsize;
	char **rdata;
{
	int res;
	int done;

	/*
	 * Check to make sure host is open
	 */
	if (!havehost) {
		(void) fprintf(stderr, "***No host open\n");
		return -1;
	}

	done = 0;
	sequence++;

again:
	/*
	 * send a request
	 */
	res = sendrequest(opcode, associd, auth, qsize, qdata);
	if (res != 0)
		return res;
	
	/*
	 * Get the response.  If we got a standard error, print a message
	 */
	res = getresponse(opcode, associd, rstatus, rsize, rdata, done);

	if (res > 0) {
		if (!done && (res == ERR_TIMEOUT || res == ERR_INCOMPLETE)) {
			if (res == ERR_INCOMPLETE) {
				/*
				 * better bump the sequence so we don't
				 * get confused about differing fragments.
				 */
				sequence++;
			}
			done = 1;
			goto again;
		}
		switch(res) {
		case CERR_BADFMT:
			(void) fprintf(stderr,
		  	   "***Server reports a bad format request packet\n");
			break;
		case CERR_PERMISSION:
			(void) fprintf(stderr,
			    "***Server disallowed request (authentication?)\n");
			break;
		case CERR_BADOP:
			(void) fprintf(stderr,
			    "***Server reports a bad opcode in request\n");
			break;
		case CERR_BADASSOC:
			(void) fprintf(stderr,
			    "***Association ID %d unknown to server\n",associd);
			break;
		case CERR_UNKNOWNVAR:
			(void) fprintf(stderr,
			   "***A request variable was unknown to the server\n");
			break;
		case CERR_BADVALUE:
			(void) fprintf(stderr,
			    "***Server indicates a request variable was bad\n");
			break;
		case ERR_UNSPEC:
			(void) fprintf(stderr,
			    "***Server returned an unspecified error\n");
			break;
		case ERR_TIMEOUT:
			(void) fprintf(stderr, "***Request timed out\n");
			break;
		case ERR_INCOMPLETE:
			(void) fprintf(stderr,
			    "***Response from server was incomplete\n");
			break;
		case ERR_TOOMUCH:
			(void) fprintf(stderr,
			    "***Buffer size exceeded for returned data\n");
			break;
		default:
			(void) fprintf(stderr,
			    "***Server returns unknown error code %d\n", res);
			break;
		}
	}
	return res;
}


/*
 * getnetnum - given a host name, return its net number
 *	       and (optional) full name
 */
int
getnetnum(host, num, fullhost)
	char *host;
	u_long *num;
	char *fullhost;
{
	struct hostent *hp;
	int decodenetnum();

	if (decodenetnum(host, num)) {
		if (fullhost != 0) {
			(void) sprintf(fullhost,
			    "%d.%d.%d.%d", ((htonl(*num)>>24)&0xff),
			    ((htonl(*num)>>16)&0xff), ((htonl(*num)>>8)&0xff),
			    (htonl(*num)&0xff));
		}
		return 1;
	} else if ((hp = gethostbyname(host)) != 0) {
		bcopy(hp->h_addr, (char *)num, sizeof(u_long));
		if (fullhost != 0)
			(void) strcpy(fullhost, hp->h_name);
		return 1;
	} else {
		(void) fprintf(stderr, "***Can't find host %s\n", host);
		return 0;
	}
	/*NOTREACHED*/
}

/*
 * decodenetnum - return a net number (this is crude, but careful)
 */
int
decodenetnum(num, netnum)
	char *num;
	u_long *netnum;
{
	register char *cp;
	register char *bp;
	register int i;
	register int temp;
	register int eos;
	char buf[80];		/* will core dump on really stupid stuff */

	cp = num;
	*netnum = 0;

	if (*cp == '[') {
		eos = ']';
		cp++;
	} else {
		eos = '\0';
	}

	for (i = 0; i < 4; i++) {
		bp = buf;
		while (isdigit(*cp))
			*bp++ = *cp++;
		if (bp == buf)
			break;

		if (i < 3) {
			if (*cp++ != '.')
				break;
		} else if (*cp != eos)
			break;

		*bp = '\0';
		temp = atoi(buf);
		if (temp > 255)
			break;
		*netnum <<= 8;
		*netnum += temp;
	}
	
	if (i < 4)
		return 0;
	*netnum = htonl(*netnum);
	return 1;
}

/*
 * warning - print a warning message
 */
warning(fmt, st1, st2)
	char *fmt;
	char *st1;
	char *st2;
{
	(void) fprintf(stderr, "%s: ", progname);
	(void) fprintf(stderr, fmt, st1, st2);
	(void) fprintf(stderr, ": ");
	perror("");
}


/*
 * error - print a message and exit
 */
error(fmt, st1, st2)
	char *fmt;
	char *st1;
	char *st2;
{
	warning(fmt, st1, st2);
	exit(1);
}

/*
 * dogetassoc - query the host for its list of associations
 */
int
dogetassoc()
{
	u_short *datap;
	int res;
	int dsize;
	u_short rstatus;
	void sortassoc();

	res = doquery(CTL_OP_READSTAT, 0, 0, 0, (char *)0, &rstatus,
	    &dsize, (char **)&datap);
	
	if (res != 0)
		return 0;
	
	if (dsize == 0) {
		(void) fprintf(stderr, "No association ID's returned\n");
		return 0;
	}
	
	if (dsize & 0x3) {
		(void) fprintf(stderr,
	    "***Server returned %d octets, should be multiple of 4\n",
		    dsize);
		return 0;
	}

	numassoc = 0;
	while (dsize > 0) {
		acache[numassoc].assid = ntohs(*datap);
		datap++;
		acache[numassoc].status = ntohs(*datap);
		datap++;
		if (++numassoc >= MAXASSOC)
			break;
		dsize -= sizeof(u_short) + sizeof(u_short);
	}
	sortassoc();
	return 1;
}

/*
 * sortassoc - sort associations in the cache into ascending order
 */
void
sortassoc()
{
	int assoccmp();
	extern void qsort();

	if (numassoc > 1)
		qsort((char *)acache, numassoc,
		    sizeof(struct association), assoccmp);
}

/*
 * assoccmp - compare two associations
 */
int
assoccmp(ass1, ass2)
	struct association *ass1;
	struct association *ass2;
{
	if (ass1->assid < ass2->assid)
		return -1;
	if (ass1->assid > ass2->assid)
		return 1;
	return 0;
}

/*
 * nextvar - find the next variable in the buffer
 */
int
nextvar(datalen, datap, vname, vvalue)
	int *datalen;
	char **datap;
	char **vname;
	char **vvalue;
{
	register char *cp;
	register char *np;
	register char *cpend;
	static char name[MAXVARLEN];
	static char value[MAXVALLEN];

	cp = *datap;
	cpend = cp + *datalen;

	/*
	 * Space past commas and white space
	 */
	while (cp < cpend && (*cp == ',' || isspace(*cp)))
		cp++;
	if (cp == cpend)
		return 0;
	
	/*
	 * Copy name until we hit a ',', an '=', a '\r' or a '\n'.  Backspace
	 * over any white space and terminate it.
	 */
	np = name;
	while (cp < cpend && *cp != ',' && *cp != '='
	    && *cp != '\r' && *cp != '\n')
		*np++ = *cp++;
	while (isspace(*(np-1)))
		np--;
	*np = '\0';
	*vname = name;

	/*
	 * Check if we hit the end of the buffer or a ','.  If so we are done.
	 */
	if (cp == cpend || *cp == ',' || *cp == '\r' || *cp == '\n') {
		if (cp != cpend)
			cp++;
		*datap = cp;
		*datalen = cpend - cp;
		*vvalue = (char *)0;
		return 1;
	}

	/*
	 * So far, so good.  Copy out the value
	 */
	cp++;	/* past '=' */
	while (cp < cpend && (isspace(*cp) && *cp != '\r' && *cp != '\n'))
		cp++;
	np = value;
	while (cp < cpend && *cp != ',' && *cp != '\r' && *cp != '\n')
		*np++ = *cp++;
	while (np > value && isspace(*(np-1)))
		np--;
	*np = '\0';

	/*
	 * Return this.  All done.
	 */
	if (cp != cpend)
		cp++;
	*datap = cp;
	*datalen = cpend - cp;
	*vvalue = value;
	return 1;
}
