/*

	xtw.c - X client for NTP time warp visualization

*/

#include <stdio.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/telnet.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <math.h>

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <X11/cursorfont.h>

#include <ntp_fp.h>
#include <ntp.h>
#include <ntp_control.h>

#include "xtw.h"

#if defined(__convex__)
#define     M_PI    3.14159265358979323846
#endif


#define	ROTDELTA	.02	/* radians to rotate meter by */
#define	MAXRADIUS	12	/* max size of time indicator */
#define	MAXOFFSET	100	/* initialize size of 'scope */
#define	ZOOMDELTA	10	/* +/- on 'scope size */

/* X key modifier codes */

#define NORMAL_KEY      0
#define SHIFT_KEY       1    /* See page 202 of XLIB; must match modifiermap */
#define LOCK_KEY        2
#define CONTROL_KEY     3

/* Simple drawing modes */

#define	MODE_WRITE	0
#define	MODE_ERASE	1

/* External variables */

extern int errno;

/* Globals */

char	*display = NULL;
Window	myWin, helpWin = NULL;
Display	*dpy;
Colormap	map;
char	version[] = "xtw : The NTP Timewarp Visualizer - Version 1.1";
char	standard_font[] = "fixed";
int	win_x = 0;
int	win_y = 0;
int	win_w = DEFWINW;
int	win_h = DEFWINH;
int	radius = ((DEFWINW+DEFWINH)/4);
long	fg_pix, bg_pix, bd_pix, a_pix, s_pix, t_pix;
Visual	visual;
XSetWindowAttributes	xswa;
XGCValues		xgcv;
GC	gc;
XEvent	pe;
XExposeEvent		*ee;
XKeyPressedEvent	*kpe;
XKeyReleasedEvent	*kre;
XButtonPressedEvent	*be;
XConfigureEvent		*ce;
XFontStruct		*sf;
XMotionEvent		*me;
u_char	drawstring[MAXLINE];
KeySym	key, nkey;
XID	xerror;
int	ulx, uly, width, height;
Bool	exposures = False;
char	*fg = NULL;
char	*bg = NULL;
char	*bd = NULL;
char	*fn = NULL;
char	acolor[] = "plum";	/* axis */
char	scolor[] = "yellow";	/* symbol (square) */
char	tcolor[] = "green";	/* time trail */
int	bw;
char	*geom = NULL;
XColor	fg_def, fg_exact, bg_def, bg_exact, bd_def, bd_exact;
XColor	a_def, a_exact, s_def, s_exact, t_def, t_exact;
Status	status;
Cursor	cursor;
int	mydepth;
char	*progname;
char	**hosts;
int	nhosts;
int	allhosts, oallhosts;
int	offscreen, ooffscreen;
float	rotate;
float	maxoffset;
float	maxdelay;
struct	g_info ginfo[MAXGINFO];
int	pause = 30;

extern struct association acache[];
extern int numassoc;
extern int debug;

/*	Functions defined in this file	*/

void StartX(), InitScreen(), TickTock(), PlotTime();
void PlotClocks(), ReDraw(), PrintClock();
int FindGinfo();
void ScreenMode();
void ReScale(), UpdateScreen();
struct ht_info *AddTrail();
struct ht_info *BlowAway();
void ZeroTrails();

/*	Functions defined elsewhere */

extern int nextvar(), dogetassoc(), openhost();

main(argc,argv)
int argc;
char **argv;
{
	float vtmp;

	progname = argv[0];
	nhosts = argc-1;
	hosts = argv+1;
	allhosts = offscreen = 0;
	oallhosts = ooffscreen = -1;
	maxoffset = MAXOFFSET;
	maxdelay = 0;
	rotate = 0;
	bzero((char *) ginfo, sizeof(ginfo));

	if (nhosts < 1) {
	  fprintf(stderr, "Usage : %s host1 host2 ... hostN\n", progname);
	  exit(1);
	}

	StartX();
	InitScreen();
	TickTock();	/* build initial list */
	TickTock();	/* plot 'em for the first time */
	XFlush(dpy);
	signal(SIGALRM, PlotClocks);
	alarm(pause);

	while (TRUE == TRUE) {

	  XNextEvent(dpy, &pe);

	  switch (pe.type) {
	    case KeyPress :
	      kpe = (XKeyPressedEvent *) &pe;
	      key = XLookupKeysym(kpe, NORMAL_KEY);
	      switch (key) {
		case 'q' :	/* exit */
		  exit(0);
		  break;
		case '-' :	/* change probe delay (-) */
		  ScreenMode(MODE_ERASE, fg_pix);
		  sprintf(drawstring, "Probe Delay = %d secs   ", pause);
		  XDrawString(dpy, myWin, gc, 9, 20, drawstring, strlen(drawstring));
		  pause--;
		  if (pause == 0) pause = 1;
		  ScreenMode(MODE_WRITE, fg_pix);
		  sprintf(drawstring, "Probe Delay = %d secs   ", pause);
		  XDrawString(dpy, myWin, gc, 9, 20, drawstring, strlen(drawstring));
		  break;
		case '=' :	/* change probe delay (+) */
		  ScreenMode(MODE_ERASE, fg_pix);
		  sprintf(drawstring, "Probe Delay = %d secs   ", pause);
		  XDrawString(dpy, myWin, gc, 9, 20, drawstring, strlen(drawstring));
		  pause++;
		  ScreenMode(MODE_WRITE, fg_pix);
		  sprintf(drawstring, "Probe Delay = %d secs   ", pause);
		  XDrawString(dpy, myWin, gc, 9, 20, drawstring, strlen(drawstring));
		  break;
		case 'i' :	/* zoom In */
		  vtmp = maxoffset;
		  ScreenMode(MODE_ERASE, fg_pix);
		  sprintf(drawstring, "Radius = %3.0f msecs   ", maxoffset);
		  XDrawString(dpy, myWin, gc, 9, 9, drawstring, strlen(drawstring));
		  maxoffset -= ZOOMDELTA;
		  if (maxoffset < ZOOMDELTA) maxoffset = 5;
		  ScreenMode(MODE_WRITE, fg_pix);
		  sprintf(drawstring, "Radius = %3.0f msecs   ", maxoffset);
		  XDrawString(dpy, myWin, gc, 9, 9, drawstring, strlen(drawstring));
		  ReScale(vtmp, maxoffset);
		  ReDraw();
		  XFlush(dpy);
		  break;
		case 'o' :	/* zoom Out */
		  vtmp = maxoffset;
		  ScreenMode(MODE_ERASE, fg_pix);
		  sprintf(drawstring, "Radius = %3.0f msecs   ", maxoffset);
		  XDrawString(dpy, myWin, gc, 9, 9, drawstring, strlen(drawstring));
		  maxoffset += ZOOMDELTA;
		  ScreenMode(MODE_WRITE, fg_pix);
		  sprintf(drawstring, "Radius = %3.0f msecs   ", maxoffset);
		  XDrawString(dpy, myWin, gc, 9, 9, drawstring, strlen(drawstring));
		  ReScale(vtmp, maxoffset);
		  ReDraw();
		  XFlush(dpy);
		  break;
		case 'r' :	/* redraw (cleanup) screen */
		  ReDraw();
		  XFlush(dpy);
		  break;
		case 'z' :	/* zero happy trails */
		  ZeroTrails();
		  ReDraw();
		  XFlush(dpy);
		  break;
		default :
		  TickTock();
		  rotate += ROTDELTA;
	      }
	      break;
	    case KeyRelease :
	      break;
	    case ButtonPress :
	      be = (XButtonPressedEvent *) &pe;
	      switch (be->button) {
		case 1 :
		  PrintClock(be->x, be->y);
		  break;	  
		case 2 :
		  break;
		case 3 :
		  break;
	      }
	      break;
	    case MappingNotify :
	      printf("Mapping notify event seen\n");
	      break;
	    case Expose :
	      ee = (XExposeEvent *) &pe;
	      while (ee->count) {
		XNextEvent(dpy, &pe);
		ee = (XExposeEvent *) &pe;
	      }
	      break;
	    case MotionNotify :
	      me =(XMotionEvent *) &pe;
	      break;

	    case ConfigureNotify :
	      ce = (XConfigureEvent *) &pe;
	      win_x = ce->x;
	      win_y = ce->y;
	      win_w = ce->width;
	      win_h = ce->height;
	      radius = ((win_h+win_w)/4);
	      ReDraw();
	      break;
	    case CirculateNotify :
	    case DestroyNotify :
	    case GravityNotify :
	    case MapNotify :
	    case ReparentNotify :
	    case UnmapNotify :
	      break;
#ifdef	OPENWINDOWS
	    case EnterNotify :
	      XSetInputFocus(dpy, myWin, RevertToParent, CurrentTime);
	      break;
	    case LeaveNotify :
	      XSetInputFocus(dpy, PointerRoot, RevertToNone, CurrentTime);
	      break;
#endif
	    default :
	      fprintf(stderr, "Unknown X event type : %d\n", pe.type);
	  }
	}
}

void PrintClock(x, y)
{
	int i;
	int cx, cy, r;
	int res;
	char *datap;
	int dsize;
	u_short rstatus;

	for (i = 0; i < allhosts; i++) {
	  r = ginfo[i].r;
	  cx = ginfo[i].x+radius;
	  cy = ginfo[i].y+radius;

	  if ((x >= cx-r) && (x <= cx+r) &&
	      (y >= cy-r) && (y <= cy+r)) {
	    printf("host = %s, associd = %d, info : \n", ginfo[i].host, ginfo[i].associd);
	    XBell(dpy, 100);
	    XFlush(dpy);
	    openhost(ginfo[i].host);
	    if (!dogetassoc()) {
	      continue;
	    }
	    res = doquery(CTL_OP_READSTAT, ginfo[i].associd, 0, 0,
	      (char *) 0, &rstatus, &dsize, &datap);
	    printf(datap);
	  }
	}
}

void PlotClocks()
{
	signal(SIGALRM, PlotClocks);
	alarm(0);
	TickTock();
	XFlush(dpy);
	rotate += ROTDELTA;
	alarm(pause);
}

void ZeroTrails()
{
	int i;
	struct ht_info *p;

	for (i = 0; i < MAXGINFO; i++) {
	  ginfo[i].head = BlowAway(ginfo[i].head);
	}
}

struct ht_info *BlowAway(p)
struct ht_info *p;
{
	if (p == NULL) return(NULL);
	BlowAway(p->next);
	free(p);
	return(NULL);
}

void TickTock()
{
	int i, j;
	char *p;
	char *datap;
	char *name;
	char *value;
	int res, dsize;
	int newhosts;
	int old;
	u_short rstatus;
	u_short bogus;
	float estdelay, estoffset, estdisp, newmaxoffset, newmaxdelay;

	newhosts = 0;	/* potentially re-create unit circle spread */
	newmaxoffset = 0;
	newmaxdelay = 0;
	offscreen = 0;
	i = 0;
	p = *hosts;
/*	debug = 9;	*/

	ScreenMode(MODE_WRITE, s_pix);
	strcpy(drawstring, "** Probing **");
	XDrawString(dpy, myWin, gc, 9, 30, drawstring, strlen(drawstring));
	XFlush(dpy);

	while (i < nhosts && p != NULL) {
	  openhost(p);
	  if (!dogetassoc()) {
	    return;
	  }
	  for (j = 0; j < numassoc; j++) {
	    if (CTL_PEER_STATVAL(acache[j].status) & 
		(CTL_PST_CONFIG | CTL_PST_REACH)) {
	      res = doquery(CTL_OP_READSTAT, acache[j].assid, 0, 0,
		(char *) 0, &rstatus, &dsize, &datap);
	      if (res > 0) {
		if (debug > 1) {
		  fprintf(stderr, "READSTAT failed to host %s, assoc %d -- ignoring.\n",
		    p, acache[j].assid);
		}
		continue;
	      }

	      estdelay = estoffset = estdisp = old = 0;
	      bogus = FALSE;
	      while (nextvar(&dsize, &datap, &name, &value)) {
		if (!strcmp(name, "refid") && !strcmp(value, "0.0.0.0")) {
		  bogus = TRUE;	/* unreachable */
		  break;
		}
		if (!strcmp(name, "estdelay")) {	
		  old++;
		  sscanf(value, "%f", &estdelay);
		}
		if (!strcmp(name, "estoffset")) {
		  old++;
		  sscanf(value, "%f", &estoffset);
		}
		if (!strcmp(name, "estdisp")) {
		  old++;
		  sscanf(value, "%f", &estdisp);
		}
		/*
	          Ntp version 3 variables.
		 */
		if (!old) {
		    if (!strcmp(name, "delay")) 
				sscanf(value, "%f", &estdelay);
	 	    if (!strcmp(name, "offset")) 
				sscanf(value, "%f", &estoffset);
		    if (!strcmp(name, "dispersion")) 
				sscanf(value, "%f", &estdisp);
		}
	      }
	      if (bogus == FALSE) {
	        newhosts += 1;
	        newmaxoffset = MAX(newmaxoffset, fabs(estoffset));
	        newmaxdelay = MAX(newmaxdelay, fabs(estdelay));
	        if (debug > 1) {
	          printf("%s.%d : delay = %f, offset = %f, disp = %f\n",
		    p, acache[j].assid, estdelay, estoffset, estdisp);
	        }
	        if (allhosts > 0) {
		  PlotTime(p, acache[j].assid, newhosts, 
		    estdelay, estoffset, estdisp);
		}
	      }
	    }
	  }
	  p = hosts[++i];
	}
	newmaxoffset += (newmaxoffset/radius)*10;

	/* 
	   See if anything is different enough to warrant a full
	   replot.
	*/

	if (newmaxdelay != maxdelay) {
	  maxdelay = newmaxdelay;
	}
	if (abs(allhosts-newhosts) > 1) {
	  if (allhosts > 0) {
	    printf("Number of hosts changed from %d to %d -- redrawing!\n",
	      allhosts, newhosts);
	  }
	  allhosts = newhosts;
	  ReDraw();
	  XFlush(dpy);
	}
	ScreenMode(MODE_ERASE, fg_pix);
	strcpy(drawstring, "** Probing **");
	XDrawString(dpy, myWin, gc, 9, 30, drawstring, strlen(drawstring));
	sprintf(drawstring, "%d chimers   ", oallhosts);
	XDrawString(dpy, myWin, gc, 9, win_h-30, drawstring, strlen(drawstring));
	sprintf(drawstring, "%d offscreen   ", ooffscreen);
	XDrawString(dpy, myWin, gc, 9, win_h-20, drawstring, strlen(drawstring));
	UpdateScreen();

}

void UpdateScreen()
{
	ScreenMode(MODE_WRITE, fg_pix);
	sprintf(drawstring, "%d chimers   ", allhosts);
	XDrawString(dpy, myWin, gc, 9, win_h-30, drawstring, strlen(drawstring));
	oallhosts = allhosts;
	sprintf(drawstring, "%d offscreen   ", offscreen);
	XDrawString(dpy, myWin, gc, 9, win_h-20, drawstring, strlen(drawstring));
	ooffscreen = offscreen;
}

void ReScale(old, new)
float old, new;
{
	int i;
	float factor;
	struct ht_info *p;

	factor = (old/new);
	for (i = 0; i < MAXGINFO; i++) {
	  if (ginfo[i].host[0] == '\0') continue;
	  ginfo[i].x = ginfo[i].x*factor;
	  ginfo[i].y = ginfo[i].y*factor;
	  for (p = ginfo[i].head; p != NULL; p = p->next) {
	    p->x = p->x*factor;
	    p->y = p->y*factor;
	  }
	}
}


void ReDraw()
{
	int i;
	int ox, oy;
	struct ht_info *p;

	XClearArea(dpy, myWin, 0, 0, win_w, win_h, False);
	InitScreen();

	offscreen = 0;
	for (i = 0; i < MAXGINFO; i++) {
	  if (ginfo[i].host[0] == '\0') continue;
	  if (((ginfo[i].x+ginfo[i].r) > radius) || ((ginfo[i].y+ginfo[i].r) > radius)) {
	    offscreen++;
	    continue;
	  }
	  ScreenMode(MODE_WRITE, s_pix);
	  XDrawRectangle(dpy, myWin, gc, ginfo[i].x+radius-ginfo[i].r,
	    ginfo[i].y+radius-ginfo[i].r, ginfo[i].r*2, ginfo[i].r*2);
	  ScreenMode(MODE_WRITE, t_pix);
	  XDrawPoint(dpy, myWin, gc, ginfo[i].x+radius, ginfo[i].y+radius);
	  if (ginfo[i].head != NULL) {
	    ox = ginfo[i].head->x;	/* oldest X, Y pair */
	    oy = ginfo[i].head->y;	
	    for (p = ginfo[i].head->next; p != NULL; p = p->next) {
	      XDrawLine(dpy, myWin, gc, ox+radius, oy+radius, 
	        p->x+radius, p->y+radius);
	      ox=p->x;
	      oy=p->y;
	    }
	  }
	}
	ScreenMode(MODE_WRITE, fg_pix);
	UpdateScreen();
}

void PlotTime(host, associd, hn, delay, offset, disp)
char *host;
int associd;
int hn;
float delay, offset, disp;
{
	int x, y, size;
	double t1, t2, t3;
	double angle;
	int gi;
	struct ht_info *p;
	int ox, oy;

	t1 = hn-1;
	t2 = allhosts;
	t3 = (radius-10)*(fabs(offset)/maxoffset);
	angle = ((t1+rotate)/t2)*((double)M_PI*2.0);

	x = t3 * cos(angle);
	y = t3 * sin(angle);

	/*
	  Size of circle denotes scaled delay ("space" distance), where
	  a smaller size indicates (naturally) a farther away host.

	  Location of circle denotes scaled offset ("time" distance), where
	  location away from center indicates greater time offsets.

	*/

	size = MAXRADIUS-(MAXRADIUS*((fabs(delay)/maxdelay)));
	if (size == 0) size = 1;

	/*
	  Perform own clipping.
	*/

	if (((x+size) > radius) || ((y+size) > radius)) {
	  if (debug > 1) {
	    printf("Host %s assoc %d delay is off-screen @ (%d,%d) offset %f - ignoring.\n",
	      host, associd, x, y, offset);
	  }
	  offscreen++;
	  return;
	}

	gi = FindGinfo(host, associd, delay, offset, x, y, size);

	/*
	  Remove the old square, if previously plotted.
	*/

	if (ginfo[gi].head != NULL) {	/* history trail exists */
	  ScreenMode(MODE_ERASE, fg_pix);
	  XDrawRectangle(dpy, myWin, gc, ginfo[gi].x+radius-ginfo[gi].r, 
	    ginfo[gi].y+radius-ginfo[gi].r, ginfo[gi].r*2, ginfo[gi].r*2);
	}

	/*
	  Update new position info (possibly redundant)
	*/

	ginfo[gi].head = AddTrail(ginfo[gi].head, x, y);
	strcpy(ginfo[gi].host, host);
	ginfo[gi].associd = associd;
	ginfo[gi].delay = fabs(delay);
	ginfo[gi].offset = fabs(offset);
	ginfo[gi].x = x;
	ginfo[gi].y = y;
	ginfo[gi].r = size;

	ScreenMode(MODE_WRITE, s_pix);
	XDrawRectangle(dpy, myWin, gc, x+radius-size,
	  y+radius-size, size*2, size*2);
	ScreenMode(MODE_WRITE, t_pix);
	XDrawPoint(dpy, myWin, gc, x+radius, y+radius);
	if (ginfo[gi].head != NULL) {
	  ox = ginfo[gi].head->x;	/* oldest X, Y pair */
	  oy = ginfo[gi].head->y;	
	  for (p = ginfo[gi].head->next; p != NULL; p = p->next) {
	    XDrawLine(dpy, myWin, gc, ox+radius, oy+radius, 
	      p->x+radius, p->y+radius);
	    ox=p->x;
	    oy=p->y;
	  }
	}
}

struct ht_info *AddTrail(h, x, y)
struct ht_info *h;
int x, y;
{
	struct ht_info *p, *q;

	if (h == NULL) {
	  if ((h = (struct ht_info *) malloc(sizeof(struct ht_info))) == NULL) {
	    fprintf(stderr, "Dynamic space exhausted!\n");
	    exit(1);
	  }
	  h->x = x;
	  h->y = y;
	  h->next = NULL;
	} 
	else {
	  for (p = h; p != NULL; p = p->next) {
	    q = p;
	  }
	  if ((p = (struct ht_info *) malloc(sizeof(struct ht_info))) == NULL) {
	    fprintf(stderr, "Dynamic space exhausted!\n");
	    exit(1);
	  }
	  p->x = x;
	  p->y = y;
	  p->next = NULL;
	  q->next = p;
	}
	return(h);
}

int FindGinfo(host, associd, delay, offset, x, y, r)
char *host;
int associd;
float delay, offset;
int x,y;
int r;
{
	int i;

	for (i = 0; i < MAXGINFO; i++) {
	  if (ginfo[i].host[0] == '\0') break;
	  if ((strcmp(host, ginfo[i].host) == 0) &&
	      (associd == ginfo[i].associd)) {
	    return(i);
	  }
	}

	/*
	  If not found, insert it.
	*/

	strcpy(ginfo[i].host, host);
	ginfo[i].associd = associd;
	ginfo[i].delay = delay;
	ginfo[i].offset = offset;
	ginfo[i].x = x;
	ginfo[i].y = y;
	ginfo[i].r = r;
	return(i);
}

void ScreenMode(mode, fg)
int mode;
long fg;
{
	switch (mode) {
	  case MODE_WRITE :
	    xgcv.foreground = fg;
	    xgcv.background = bg_pix;
	    break;
	  case MODE_ERASE :
	    xgcv.foreground = xgcv.background = bg_pix;
	    break;
	}
	XChangeGC(dpy, gc, GCForeground | GCBackground, &xgcv);
}

void StartX()
{
	int i, j, k;

	if (geom) {
	  XParseGeometry(geom, &win_x, &win_y, (u_int *) &win_w, (u_int *) &win_h);
	}

	if ((dpy = XOpenDisplay(display)) == NULL) {
	  printf("Cannot open display on %s\n",display);
	  exit(1);
	}
  
	if ((map = XDefaultColormap(dpy, DefaultScreen(dpy))) == NULL) {
	  printf("Cannot get default colormap\n");
	  exit(1);
	}

	status = XAllocNamedColor(dpy, map, acolor, &a_def, &a_exact);
	a_pix = status ? a_def.pixel : WhitePixel(dpy, DefaultScreen(dpy));
	status = XAllocNamedColor(dpy, map, scolor, &s_def, &s_exact);
	s_pix = status ? s_def.pixel : WhitePixel(dpy, DefaultScreen(dpy));
	status = XAllocNamedColor(dpy, map, tcolor, &t_def, &t_exact);
	t_pix = status ? t_def.pixel : WhitePixel(dpy, DefaultScreen(dpy));

	if (fg) {
	  status = XAllocNamedColor(dpy, map, fg, &fg_def, &fg_exact);
	  fg_pix = status ? fg_def.pixel : WhitePixel(dpy, DefaultScreen(dpy));
	} else
	  fg_pix = WhitePixel(dpy, DefaultScreen(dpy));
	if (bg) {
	  status = XAllocNamedColor(dpy, map, bg, &bg_def, &bg_exact);
	  bg_pix = status ? bg_def.pixel : BlackPixel(dpy, DefaultScreen(dpy));
	} else
	  bg_pix = BlackPixel(dpy, DefaultScreen(dpy));
	if (bd) {
	  status = XAllocNamedColor(dpy, map, bd, &bd_def, &bd_exact);
	  bd_pix = status ? bd_def.pixel : WhitePixel(dpy, DefaultScreen(dpy));
	} else
	  bd_pix = WhitePixel(dpy, DefaultScreen(dpy));

	if (fn == NULL) {
	  if ((sf = XLoadQueryFont(dpy, standard_font)) == NULL) {
	    fprintf(stderr, "Cannot load standard font : %s\n", standard_font);
	    exit(1);
	  }
	}
	else {
	  if ((sf = XLoadQueryFont(dpy, fn)) == NULL) {
	    fprintf(stderr, "Cannot load font : %s\n", fn);
	    exit(1);
	  }
	}

	visual.visualid = CopyFromParent;

	xswa.backing_store = Always;
	xswa.event_mask = ExposureMask | StructureNotifyMask | KeyPressMask | ButtonPressMask |
#ifdef	OPENWINDOWS
	  KeyReleaseMask | EnterWindowMask | LeaveWindowMask | PointerMotionMask;
#else
	  KeyReleaseMask | PointerMotionMask;
#endif
	xswa.background_pixel = bg_pix;
	xswa.border_pixel = bd_pix;
	xswa.cursor = None;

	mydepth = DefaultDepth(dpy, DefaultScreen(dpy));
#ifdef	DEBUG_X
	printf("parent depth = %d\n",DefaultDepth(dpy, DefaultScreen(dpy)));
#endif

	myWin = XCreateWindow(dpy, RootWindow(dpy, DefaultScreen(dpy)),
	  win_x, win_y, win_w, win_h, bw, mydepth,
	  InputOutput, &visual, CWEventMask | CWBackingStore | CWBorderPixel | CWBackPixel,
	  &xswa);
	if (myWin == NULL) {
	  fprintf(stderr, "Cannot open window on display\n");
	  exit(1);
	}
	XChangeProperty(dpy, myWin, XA_WM_NAME, XA_STRING, 8, PropModeReplace,
	  (u_char *) version, strlen(version));
	XMapWindow(dpy, myWin);

	xgcv.foreground = fg_pix;
	xgcv.background = bg_pix;
	xgcv.function = GXcopy;
	xgcv.plane_mask = AllPlanes;
	xgcv.fill_style = FillSolid;
	xgcv.graphics_exposures = False;
	xgcv.font = sf->fid;
	gc = XCreateGC(dpy, myWin, GCForeground | GCBackground | GCFunction | GCPlaneMask |
	  GCFillStyle | GCGraphicsExposures | GCFont, &xgcv);

	cursor    = XCreateFontCursor(dpy, XC_top_left_arrow);
	XDefineCursor(dpy, myWin, cursor);
}

void InitScreen()
{
	ScreenMode(MODE_WRITE, fg_pix);
	sprintf(drawstring, "Radius = %3.0f msecs", maxoffset);
	XDrawString(dpy, myWin, gc, 9, 9, drawstring, strlen(drawstring));
	sprintf(drawstring, "Probe Delay = %d secs   ", pause);
	XDrawString(dpy, myWin, gc, 9, 20, drawstring, strlen(drawstring));
	ScreenMode(MODE_WRITE, a_pix);
	XDrawArc(dpy, myWin, gc, 0, 0, win_w-1, win_h-1, 0, 360*64);
	XDrawLine(dpy, myWin, gc, 0, win_h/2, win_w, win_h/2);
	XDrawLine(dpy, myWin, gc, win_w/2, 0, win_w/2, win_h);
	ScreenMode(MODE_WRITE, fg_pix);
	sprintf(drawstring, "%3.0f", maxoffset);
	XDrawString(dpy, myWin, gc, 1, win_h/2-4, drawstring, strlen(drawstring));
}
