// $Header: Client.h,v 1.5 92/12/01 10:53:11 vern Locked $

#ifndef client_h
#define client_h

#include <signal.h>
#include <generic.h>
#include <sys/types.h>

#include "Glish/Value.h"

#ifdef __GNUG__
typedef SignalHandler SigHandler;
#else
typedef SIG_PF SigHandler;
#endif


class GlishEvent : public GlishObject {
    public:
	GlishEvent( char* arg_name, Value* arg_value )
		{ name = arg_name; value = arg_value; }

	~GlishEvent()
		{ delete name; Unref( value ); }

	char* name;
	Value* value;
	};


declare(List,int);

// Holds information regarding outbound "link" commands.
class EventLink;
declare(PList,EventLink);
typedef PList(EventLink) event_link_list;
declare(PDict,event_link_list);

class AcceptSocket;

class Client {
    public:
	// Client's are constructed by giving them the program's
	// argc and argv.  Any client-specific arguments are read
	// and stripped off.
	Client( int& argc, char** argv );

	virtual ~Client();

	// Wait for the next event to arrive and return a pointer to 
	// it.  The GlishEvent will be Unref()'d on the next call to
	// NextEvent(), so if the caller wishes to keep the GlishEvent
	// (or its Value) they must Ref() the GlishEvent (or its Value).
	//
	// If the sequencer connection has been broken then 0 is returned
	// (and the caller should terminate).
	GlishEvent* NextEvent();

	// Another version of NextEvent which can be passed an fd_set
	// returned by select() to aid in determining from where to
	// read the next event.
	GlishEvent* NextEvent( fd_set* mask );

	// Called by the main program (or whoever called NextEvent()) when
	// the current event is unrecognized.
	void Unrecognized();


	// Sends an event with the given name and value.
	void PostEvent( GlishEvent* event );
	void PostEvent( const char* event_name, const Value* event_value );

	// Sends an event with the given name and character string value.
	void PostEvent( const char* event_name, const char* event_value );

	// Sends an event with the given name, using a printf-style format
	// and an associated string argument.  For example,
	//
	//	client->PostEvent( "error", "couldn't open %s", file_name );
	//
	void PostEvent( const char* event_name, const char* event_fmt,
				const char* event_arg );


	// For any file descriptors this Client might read events from,
	// sets the corresponding bits in the passed fd_set.  The caller
	// may then use the fd_set in a call to select().
	void AddInputMask( fd_set* mask );

	// Returns true if the following mask indicates that input
	// is available for the client.
	int HasClientInput( fd_set* mask );

	// Returns true if the client was invoked by a Glish sequencer,
	// false if the client is running standalone.
	int HasSequencerConnection()	{ return have_sequencer_connection; }

    protected:
	friend void Client_signal_handler();

	// Returns the next event from the given fd.
	GlishEvent* GetEvent( int fd );

	// Called whenever a new fd is added (add_flag=true) or deleted
	// (add_flag=false) from those that the client listens to.  By
	// redefining this function, a Client subclass can keep track of
	// the same information that otherwise must be computed by calling
	// AddInputMask() prior to each call to select().
	virtual void FD_Change( int fd, bool add_flag );

	// Called asynchronously whenever a ping is received; the
	// sequencer sends a ping whenever a new event is ready and
	// the client was created with the <ping> attribute.
	virtual void HandlePing();

	// The following four member functions support local ("Pipe")
	// and remote ("Socket") "links".
	void OpenPipeSource( Value* v );
	void OpenPipeSink( Value* v );

	void OpenSocketSource( Value* v );
	void OpenSocketSink( Value* v );

	// Removes a link.
	void UnlinkSink( Value* v );

	// Decodes an event value describing an output link and returns
	// a pointer to a (perhaps preexisting) corresponding EventLink.
	// "want_active" states whether the EventLink must be active
	// or inactive.  "is_new" is true upon return if a new EventLink
	// was created.  Nil is returned upon an error.
	EventLink* AddOutputLink( Value* v, int want_active, int& is_new );


	char* client_name;
	int read_fd;
	int write_fd;
	bool have_sequencer_connection;
	AcceptSocket* accept_socket;

	List(int) input_links;
	PDict(event_link_list) output_links;

	GlishEvent* last_event;

	// Previous signal handler; used for <ping>'s.
	SigHandler former_handler;

	char local_host[64];
	};


extern GlishEvent* recv_event( int fd );

extern void send_event( int fd, const char* event_name,
			const Value* event_value );

#endif	/* client_h */
