/*
 * dfile.c - SCO OpenServer file processing functions for lsof
 */


/*
 * Copyright 1995 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1995 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dfile.c,v 1.4 97/10/23 15:47:46 abe Exp $";
#endif

#include "lsof.h"


/*
 * get_max_fd() - get maximum file descriptor plus one
 */

int
get_max_fd()
{

#if	defined(F_GETHFDO) || defined(_SC_OPEN_MAX)
	int nd;
#endif	/* defined(F_GETHFDO) || defined(_SC_OPEN_MAX) */

#if	defined(F_GETHFDO)
	if ((nd = fcntl(-1, F_GETHFDO, 0)) >= 0)
	    return(nd);
#endif	/* defined(F_GETHFDO) */

#if	defined(_SC_OPEN_MAX)
	if ((nd = sysconf(_SC_OPEN_MAX)) >= 0)
	    return(nd);
#endif	/* defined(_SC_OPEN_MAX) */

	return(getdtablesize());
}


/*
 * is_file_named() - is this file named?
 */

int
is_file_named(p, ty)
	char *p;			/* path name; NULL = search by device
					 * and inode (from *Lf) */
	int ty;				/* inode type */
{
	int f;
	struct sfile *s;
/*
 * Search the file chain.
 */
	for (f = 0, s = Sfile; s; s = s->next) {

	/*
	 * Search by path name, as requested.
	 */
		if (p != NULL) {
			if (strcmp(p, s->aname) == 0
			||  strcmp(p, s->name)  == 0)
			{
				f = 2;
				break;
			}
			continue;
		}
	/*
	 * If this is a stream, check for a clone device match.
	 */
		if (Lf->is_stream) {
			if (HaveCloneMajor && major(s->dev) == CloneMajor
			&&  Lf->dev_def && major(Lf->dev) == minor(s->dev)) {
				f = 3;
				break;
			}
		}
	/*
	 * Check for a regular file or directory -- the device and
	 * inode numbers must match.
	 */
		if (s->type && Lf->dev_def && Lf->inp_ty == 1) {
			if (Lf->dev == s->dev && (ino_t)Lf->inode == s->i) {
				f = Lf->is_stream ? 3 : 1;
				break;
			}
			continue;
		}
	/*
	 * Check for a file system match.
	 * Try to avoid matching character files to non-character devices.
	 */
		if (!s->type && Lf->dev_def && Lf->dev == s->dev) {
			if (!(ty == IFCHR && s->mode != S_IFCHR)) {
				f = 1;
				break;
			}
		}
	}
/*
 * Convert the name if a match occurred.
 */
	switch (f) {
	case 0:
		return(0);
	case 1:
		(void) strcpy(Namech, s->name);
		if (s->devnm)
			(void) sprintf(endnm(), " (%s)", s->devnm);
		break;
	case 2:
		(void) strcpy(Namech, p);
		break;
	/* case 3:		do nothing for stream matches */
	}
	s->f = 1;
	return(1);
}


/*
 * print_dev() - print dev
 */

char *
print_dev(lf)
	struct lfile *lf;		/* file whose device is to be printed */
{
	static char buf[128];

	(void) sprintf(buf, "%d,%d",
	    lf->is_nfs ? ((~(lf->dev >> 8)) & 0xff) : emajor(lf->dev),
	    eminor(lf->dev));
	return(buf);
}


/*
 * print_ino() - print inode
 */

char *
print_ino(lf)
	struct lfile *lf;		/* file whose device is to be printed */
{
	static char buf[128];

	(void) sprintf(buf, (lf->inode & 0x80000000) ? "%#x" : "%lu",
	    lf->inode);
	return(buf);
}


/*
 * process_file() - process file
 */

void
process_file(fp)
	struct file *fp;		/* kernel file structure address */
{
	struct file f;
	int flag;

	if (kread((KA_T)fp, (char *)&f, sizeof(f))) {
		(void) sprintf(Namech, "can't read file struct from %#x", fp);
		enter_nm(Namech);
		return;
	}
	Lf->off = (SZOFFTYPE)f.f_offset;

	if (f.f_count) {

	/*
	 * Construct access code.
	 */
		if ((flag = (f.f_flag & (FREAD | FWRITE))) == FREAD)
			Lf->access = 'r';
		else if (flag == FWRITE)
			Lf->access = 'w';
		else if (flag == (FREAD | FWRITE))
			Lf->access = 'u';
	/*
	 * Process structure.
	 */
		process_node((caddr_t)f.f_inode);
		return;

	}
	enter_nm("no more information");
}
