/*
* This file is part of the MGUI library
* Copyright 1996-1998 Vincenzo Morello
*
* This file describes the C++ frame for the C MGUI library.
*
* Version 1.4  14 Sep 1998
*
* All MGUI classes have the Cm suffix. Use something other,
* such as Ca, for suffixing application defined classes.
*
* All GUI Objects (buttons, labels, menu etc.) should be created via
* the new operator.
*
* All GUI Objects (both C and C++) are automatically destroyed when the
* CmShell instance, or any derived one, is deleted.
*
* The CmContainer class is a Superclass for CmShell and CmForm.
* All GUI object contructors have a pointer to this class as
* the first argument. This keeps the MGUI object hierarchy.
*
* Virtual methods are provided by objects that must call application
* actions (the callbacks in C MGUI).
* These action functions have a suffix of 'on', such as onActivate() for
* push button.
*
*/

#ifndef _MGUIPP_H
#define _MGUIPP_H

#include "mgui.h"

class CmCallback {
protected:
    virtual void dummy(void) {}
};

class CmMenuItem;

typedef void (CmCallback::*VOID_CB)(void);
typedef void (CmCallback::*EVENT_CB)(MEvent *);
typedef void (CmCallback::*LONG_CB)(long);
typedef void (CmCallback::*INT_CB)(int);
typedef void (CmCallback::*MENUITEM_CB)(int);
typedef void (CmCallback::*ITEMSEL_CB)(CmMenuItem *);
typedef void (CmCallback::*STRING_CB)(char *);
typedef void (CmCallback::*NEWBLOCK_CB)(SBL_NEW_BLOCK *);
typedef void (CmCallback::*ITEMSEEK_CB)(SBL_ITEM_SEEK *);
typedef void (CmCallback::*EDITVAL_CB)(EDIT_VAL *);
typedef void (CmCallback::*EDITCURS_CB)(EDIT_CURS *);
typedef void (CmCallback::*EDITCHANGE_CB)(EDIT_CHANGE *);
typedef void (CmCallback::*LISTACT_CB)(LIST_ACT *);
typedef void (CmCallback::*LISTSEL_CB)(LIST_SEL *);
typedef void (CmCallback::*DRAWAREAPREF_CB)(DRAWAREA_PREF *);

/**
   Application Class.
   Provides the application's starting and terminating
   methods. The application program must define a derived
   instance to provide the pure virtual start() method.
   The program can be terminated at any point by calling
   CmAppl::end(code);
*/

class CmAppl {
	friend void MGUIMain(int argc, char **argv);
	static CmAppl *app_inst;
protected:
	CmAppl(void);
public:
/// The program starts receiving the main's ordinary arguments argc and argv.
	virtual void start(int argc, char **argv)=0;
/// Terminates the program freeing allocated resources and then calling exit(code)
	static void end(int code);
	void logInit(int level, int nr, int nvr, int nc, int nvc, const char *fname);
	void enableCustomizing(void);
	void mainLoop(void);
};

/**
   Timeout Class.
*/

class CmTimeout : public CmCallback {
	friend void _TimeoutCB_(TIMEOUT_ID id, void *ud);
	TIMEOUT_ID tid;
	CARD32	msec;
	int continuous;
	int stopped;
	VOID_CB callback;
	CmCallback *owner;

public:
	CmTimeout(CmCallback *own, VOID_CB cb, CARD32 msec=0, int continuous=False);
	virtual ~CmTimeout(void);
	void start(CARD32 msec);
	void start(void);
	void stop(void);
};

/**
 Abstract Color Class. See derived classes
*/

class CmColor {
protected:
	MTColor id;
public:
/// Doesn't allocate the color, simply creates another id
	CmColor(MTColor cc=0) { id = cc;}
/// internal use
	operator MTColor() { return id; }
/// returns the color's R-G-B components.
	void getRGB(int *r, int *g, int *b);
};

/**
  Ordinary color class. This is a read-only shared color used to set
  widget's background/foreground and for drawing primitives.
 */
class CmSharedColor : public CmColor {
public:
	virtual ~CmSharedColor(void);
/// creates a color with the R-G-B components specified in parameters (r,g,b)
	CmSharedColor(unsigned r, unsigned g, unsigned b);
/// creates a color as a shade of the color c. shift specifies the shade intensity (darker if < 100, brighter if > 100)
	CmSharedColor(CmColor *c, int shift=100);
};

/**
  Color cell class. This is a read-write non-shared color. It is available
  only when the video system runs in palette mode (usually 16 or 256 colors).
 */
class CmRWColor : public CmColor {
public:
	CmRWColor(unsigned r, unsigned g, unsigned b);
	virtual ~CmRWColor(void);
	void setRGB(int r, int g, int b);
};

/**
 IMAGE Class
*/

class CmImage : public MIMAGE_DESCR {
protected:
	void allocImage(int imgw, int imgh, int np);
public:
	CmImage(const char *fname);
	CmImage(int imgw, int imgh, int np);
	virtual ~CmImage(void);

	void install(void);
	void uninstall(void);
	int  getWidth(void)   { return img_w; }
	int  getHeight(void)  { return img_h; }
	int  getNColors(void) { return ncolors; }
	int  getNPlanes(void) { return nplanes; }
	void getPixelRGB(int x, int y, int *pr, int *pg, int *pb);
	void setPixelRGB(int x, int y, int r, int g, int b);
	int  getPixelIndex(int x, int y);
	void setPixelIndex(int x, int y, int i);
	void getCMapEntry(int i, int *pr, int *pg, int *pb);
	void setCMapEntry(int i, int r, int g, int b);
};

typedef CmImage CmBMPImage;

class CmGIFImage : public CmImage {
public:
	CmGIFImage(const char *fname);
};

class CmTIFFImage : public CmImage {
public:
	CmTIFFImage(const char *fname);
};

class CmCursor {
	MTCursor cur;
public:
	operator MTCursor() { return cur;}
	CmCursor(CmImage *bits, int xs, int ys) {cur=MCreateCursor(bits,xs,ys);}
};

class CmFont {
	MTFont font;
	int loaded;
public:
	operator MTFont() { return font;}
	CmFont(int ff=DEFAULT_FONT)
		{ font = ff; loaded=0;}
	CmFont(const char *family, int h, int bold=0, int italic=0)
		{ font = MLoadFont(family,h,bold,italic); loaded=1;}
	~CmFont()
		{ if (loaded) MUnloadFont(font);}
	int textCharX(const char *text, int x) { return MTextCharX(text,font,x);}
	int textHeight(const char *text)  { return MTextHeight(text,font);}
	int charHeight(char car)    { return MCharHeight(car,font);}
	int textWidth(const char *text)   { return MTextWidth(text,font);}
	int textNWidth(const char *text, int len) {return MTextNWidth(text,font,len);}
	int charWidth(char car)     { return MCharWidth(car,font);}
};

class CmShell;
class CmForm;
class CmContainer;
class CmObject;

/**
 Abstract Object Class
*/

class CmObject : public CmCallback {

	friend class CmOptionEdit;
	friend void _ObjectDestroyCB_(MOBJECT p, void *a, void *ud);
	friend void _EventPreProcessCB_(MOBJECT p, MEvent *pe, void *ud);

	void *user_data; 

protected:
	MOBJECT  obj;
	EVENT_CB eventCB;

	void reserved(void);
	virtual ~CmObject(void);
public:
	operator MOBJECT() {return obj; };

	inline void drawText(int x, int y, int w, int h, const char *text,
			MTFont font, CmColor *fgc, int align);
	inline void drawNText(int x, int y, int w, int h, const char *text,
			MTFont font, CmColor *fgc, int align, int n);
	inline void drawBGText(int x, int y, int w, int h, const char *text,
			MTFont font, CmColor *bgc, CmColor *fgc, int align);
	inline void drawBGNText(int x, int y, int w, int h, const char *text,
			MTFont font, CmColor *bgc, CmColor *fgc, int align, int n);
	inline void drawShadowedText(int x, int y, int w, int h, const char *text,
			MTFont font, int sh, CmColor *fg1, CmColor *fg2, int align);
	inline void drawShadowedNText(int x, int y, int w, int h, const char *text,
			MTFont font, int sh, CmColor *fg1, CmColor *fg2, int align, int n);

	inline void drawPoint(int x, int y, CmColor *col);
	inline void drawLine(int x1, int y1, int x2, int y2, CmColor *col);
	inline void drawPolyline(MPOINT *points, int n, CmColor *col);
	inline void drawFilledPolyline(MPOINT *points, int n, CmColor *col);
	inline void drawBox(int x, int y, int w, int h, CmColor *bgc);
	inline void drawFilledBox(int x, int y, int w, int h, CmColor *bgc);
	inline void drawCircle(int x, int y, int r, CmColor *col);
	inline void drawFilledCircle(int x, int y, int r, CmColor *col);
	inline void drawArc(int x, int y, int r, int a1, int a2, CmColor *col);
	inline void drawFilledArc(int x, int y, int r, int a1, int a2, CmColor *c);
	inline void drawImage(CmImage *gi, int img_x, int img_y, int w, int h,
			int x, int y);

	inline void drawXorLine(int x1, int y1, int x2, int y2);
	inline void drawXorBox(int x, int y, int w, int h);

	inline void setSensitive(void);
	inline void setUnsensitive(void);
	inline void setSensitivity(int true_false);
	inline void setSelectable(void);
	inline void setUnselectable(void);
	inline void setColor(CmColor *bgc, CmColor *fgc);
	inline void setBackgroundRGB(unsigned r, unsigned g, unsigned b);
	inline void setForegroundRGB(unsigned r, unsigned g, unsigned b);
	inline void setText(const char *text);
	inline void setName(const char *name);
	inline void setFont(MTFont font, int resize_obj=0);
	inline void setShadow(int type, int in=0, int out=0);
	inline void setResize(int resize_w, int resize_h);
	inline void setWidth(int width);
	inline void setHeight(int height);
	inline void setAttachment(int top, int bottom, int left, int right);
	inline void setOffset(int t_off, int b_off, int l_off, int r_off);
	inline void setCursor(MTCursor cursor);
	inline void setPopupHelp(const char *str, int delay, int duration);
	inline void setLabelHelp(const char *str, CmObject *label);
	inline void setUserData(void *ud) { user_data = ud; };
	inline void changePopupHelpText(const char *str);
	inline void changeLabelHelpText(const char *str);
	inline void topAttach(int att_type, int off, CmObject *obj);
	inline void bottomAttach(int att_type, int off, CmObject *obj);
	inline void leftAttach(int att_type, int off, CmObject *obj);
	inline void rightAttach(int att_type, int off, CmObject *obj);
	inline void align(int h_align, int v_align);

	inline void getText(char *text);
	inline const char *getName(void);
	inline void *getUserData(void) { return user_data; };
	inline MTFont getFont(void);
	inline void getShadow(int *type, int *in, int *out);
	inline int isSensitive(void);
	inline int isSelectable(void);
	inline int isMapped(void);
	inline int getWidth(void);
	inline int getHeight(void);
	inline int getX(void);
	inline int getY(void);
	CmColor *getBackgroundColor(void);
	CmColor *getForegroundColor(void);
	CmContainer *getParent(void);
	CmShell *getShell(void);

	inline void resize(int w, int h);
	inline void redraw(void);
	inline void select(void);
	inline void map(void);
	inline void unmap(void);
	inline void enableOptions(const char *name, CARD32 mask);
	inline void saveOptions(void);
	inline void unlink(int resize_obj);
	inline void focusLink(CmObject *obj_to_link);
	void reparent(CmContainer *new_parent, CmObject *where);
	void link(CmContainer *parent, CmObject *where, int resize_parent);

	void setEventCallback(CmCallback *pp, EVENT_CB ff, CARD32 mask);
};

/**
 Abstract Container Class
*/

class CmContainer : public CmObject {
public:
	virtual void addChild(CmObject *child)=0;

	CmObject *getSelectedObject(void);
	CmObject *getFirstChild(void);
	CmObject *getNextChild(void);
	int		  getNChildren(void);
};

/**
 SHELL Object
*/

class CmShell : public CmContainer {
	friend void _WMCloseCB_(MOBJECT p, void *a, void *ud);
	VOID_CB closeCB;
public:
	~CmShell(void);
	CmShell(const char *text=0, int flags=0);

	inline Window getWindow(void);

	inline void realize(void);
	inline void realize(int x, int y);
	inline void realize(int x, int y, int w, int h);
	inline void unrealize(void);
	inline void raise(void);
	void addChild(CmObject *child);

	void setWMCloseCallback(CmCallback *pp, VOID_CB ff);
};

/**
 Abstract Form Class
*/

class CmForm : public CmContainer {
protected:
	CmForm(void){};
public:
	inline void setSpacing(int offset);
	inline void setFocusWrap(int true_false);
	inline void setFocusHighlight(int true_false);

	void addChild(CmObject *object);
};

/**
 RowForm Class
*/

class CmRowForm : public CmForm {
public:
	CmRowForm(CmContainer *parent);
};

/// ColForm Class
class CmColForm : public CmForm {
public:
	CmColForm(CmContainer *parent);
};

/// TableForm Class
class CmTableForm : public CmForm {
public:
    CmTableForm(CmContainer *parent, int n_columns);
    inline void showGrid(void);
    inline void hideGrid(void);
    inline void setRowHeight(int row, int height);
    inline void setColWidth(int col, int width);
    inline void setColFont(int col, MTFont f);
    inline void setColAlign(int col, int al);
    inline void setColBackgroundRGB(int col, int r, int g, int b);
    inline void setColForegroundRGB(int col, int r, int g, int b);
    inline void setCellText(int x, int y, const char *text);
    inline void getCellText(int x, int y, char *text);
    inline void setCellFont(int x, int y, MTFont f);
    inline void setCellAlign(int x, int y, int h_al, int v_al);
    inline void setCellBackgroundRGB(int x, int y, int r, int g, int b);
    inline void setCellForegroundRGB(int x, int y, int r, int g, int b);
    CmObject *getCellObject(int x, int y);
};

/**
 Abstract Paged Form
 */
class CmAbsPagedForm : public CmForm {
public:
	inline void setActivePage(int page);
};

class CmPagedRowForm : public CmAbsPagedForm {
public:
	CmPagedRowForm(CmContainer *parent, int n_pages);
};

class CmPagedColForm : public CmAbsPagedForm {
public:
	CmPagedColForm(CmContainer *parent, int n_pages);
};

class CmPagedRowFormExt : public CmAbsPagedForm {
public:
	CmPagedRowFormExt(CmContainer *parent, int n_pages);
};

class CmPagedColFormExt : public CmAbsPagedForm {
public:
	CmPagedColFormExt(CmContainer *parent, int n_pages);
};

/**
 * Abstract Clipped Form
 */
class CmAbsCForm : public CmForm {
public:
	inline void setClipX(int clip_x);
	inline void setClipY(int clip_y);
	inline int	getClipX(void);
	inline int	getClipY(void);
};

class CmScrolledForm : public CmAbsCForm {
public:
	CmScrolledForm(CmContainer *parent, int w, int h);
};

class CmClippedForm : public CmAbsCForm {
public:
	CmClippedForm(CmContainer *parent, int w, int h);
};

/**
 LABEL Object
*/

class CmLabel : public CmObject {
public:
	CmLabel(CmContainer *parent, const char *text=0, MTFont font=DEFAULT_FONT);
	void setMargin(int wm, int hm);
	void setHSpacing(int hs);
	void setAlignment(int align);
};

/**
 Abstract Edit Class
*/

class CmAbsEdit : public CmObject {
	friend void _EditNewValueCB_(MOBJECT p, char *text, void *ud);
	friend void _EditValueChangeCB_(MOBJECT p, EDIT_CHANGE *ec, void *ud);
	STRING_CB newValueCB;
	EDITCHANGE_CB valueChangeCB;
protected:
	CmAbsEdit(void){};
public:
	void changeText(int off, int len, const char *new_t, int chg_len);
	void getSubString(int start, int end, char *str);
	int  findSubStringForward(int off, const char *str, int icase);
	int  findSubStringBackward(int off, const char *str, int icase);
	void setSelection(int start, int end);
	void getSelection(int *pstart, int *pend);
	char *getAllocatedText(void);
	void setCursorPos(int pos);
	int  getCursorPos(void);

	void setNewValueCallback(CmCallback *, STRING_CB);
	void setValueChangeCallback(CmCallback *, EDITCHANGE_CB);
};

/**
 Abstract EditField Class
*/

class CmAbsEditField : public CmAbsEdit {
	friend void _EditActCB_(MOBJECT p, char *text, void *ud);
	friend void _EditValCB_(MOBJECT p, EDIT_VAL *ev, void *ud);
	friend void _EditHotKeyCB_(MOBJECT p, void *a, void *ud);
	STRING_CB  activateCB;
	EDITVAL_CB validateCB;
	VOID_CB    hotKeyCB;
protected:
	CmAbsEditField(void){};
public:
	void setFilter(int filter, const char *extra_chars);
	void setTextExt(const char *text, int validated, int notify);
	void setValidatedColors(CmColor *bgc, CmColor *fgc);
	void setInvalidatedColors(CmColor *bgc, CmColor *fgc);
	void setValidationStatus(int validated);
	void selectForReplace(void);
	const char *text(void);
	int  getFilter(void);
	int  isValidated(void);

	void setActivateCallback(CmCallback *, STRING_CB);
	void setHotKeyCallback(CmCallback *, VOID_CB, int);
	void setValidateCallback(CmCallback *, EDITVAL_CB);
};

/**
 EditField Classes
*/

class CmEditField : public CmAbsEditField {
public:
	CmEditField(CmContainer *parent, const char *text, int len,
		int clip_len=-1, MTFont font=DEFAULT_FONT);
};

class CmHiddenEditField : public CmAbsEditField {
public:
	CmHiddenEditField(CmContainer *parent, const char *text, int len,
		int clip_len=-1, MTFont font=DEFAULT_FONT, char shown='*');
};

class CmAbsEditWindow : public CmAbsEdit {
	friend void _EditCursorCB_(MOBJECT p, EDIT_CURS *ec, void *ud);
	EDITCURS_CB cursCB;
public:
	void setCursorXY(int x, int y);
	void getCursorXY(int *px, int *py);
	void setAutoIndent(int on_off);
	void setTabSize(int size);
	void setClipWindow(int clip_w, int clip_h);

	void setCursorCallback(CmCallback *, EDITCURS_CB);
};

/**
 EditWindow Classes
*/

class CmEditWindow : public CmAbsEditWindow {
public:
	CmEditWindow(CmContainer *parent, const char *text, int width, int height,
		MTFont font=DEFAULT_FONT);
};

class CmScrolledEditWindow : public CmAbsEditWindow {
public:
	CmScrolledEditWindow(CmContainer *parent, const char *text, int width, int height,
		MTFont font=DEFAULT_FONT);
};

/**
 Abstract Button Object
*/

class CmButton : public CmObject {
public:
	virtual void setImages(const char *normal, const char *on_arm=0, const char *unsens=0)=0;
};

/**
 Push Button Object
*/

class CmPushButton : public CmButton {
	friend void _PushButtonActivateCB_(MOBJECT p, void *a, void *ud);
	VOID_CB activateCB;
public:
	CmPushButton(CmContainer *p, const char *text, MTFont font=DEFAULT_FONT);
	CmPushButton(CmContainer *p, CmImage *norm, CmImage *arm=0, CmImage *uns=0);
	CmPushButton(CmContainer *p, const char *norm, const char *arm, const char *uns=0);
	void setRepeatDelay(int first, int repeat);
	void setImages(const char *normal, const char *armed=0, const char *when_unsens=0);
	void setMargin(int wm, int hm);
	void setHSpacing(int hs);
	void setAlignment(int align);
	void setAccelerator(int key);

	void setCallback(CmCallback *owner, VOID_CB cb);
};

/**
 Toggle Button Object
*/

class CmToggleButton : public CmButton {
	friend void _ToggleButtonActivateCB_(MOBJECT p, MVALUE a, void *ud);
	INT_CB activateCB;
public:
	CmToggleButton(CmContainer *par, const char *text, MTFont font=DEFAULT_FONT);
	CmToggleButton(CmContainer *p, CmImage *n, CmImage *arm=0, CmImage *uns=0);
	CmToggleButton(CmContainer *par, const char *normal, const char *on_arm, const char *uns=0);

	void setImages(const char *normal, const char *on_arm=0, const char *uns=0);
	void arm(void);
	void disarm(void);

	int isArmed(void);

	void setCallback(CmCallback *owner, INT_CB cb);
};

/**
 Radio Button Object
*/

class CmRadioButton : public CmButton {
	friend void _RadioButtonActivateCB_(MOBJECT p, MVALUE a, void *ud);
	INT_CB activateCB;
public:
	CmRadioButton(CmContainer *par, const char *text, MTFont font=DEFAULT_FONT);
	CmRadioButton(CmContainer *p, CmImage *norm, CmImage *arm=0, CmImage *uns=0);
	CmRadioButton(CmContainer *par, const char *normal, const char *on_arm, const char *when_unsens=0);

	void setImages(const char *normal, const char *on_arm=0, const char *when_unsens=0);
	void arm(void);
	void disarm(void);

	int isArmed(void);

	void setCallback(CmCallback *owner, INT_CB cb);
};

/**
 Pixmap Object
*/

class CmPixmap : public CmObject {
public:
	CmPixmap(CmContainer *par, CmImage *normal);
	CmPixmap(CmContainer *par, const char *fname);

	void setImage(CmImage *normal);
	void setImage(const char *fname);
};

/**
 DrawArea Object
*/
class CmDrawArea : public CmObject {
	friend void _DrawAreaExposeCB_(MOBJECT, MEvent *, void *);
	friend void _DrawAreaInputCB_(MOBJECT, MEvent *, void *);
	friend void _DrawAreaResizeCB_(MOBJECT, DRAWAREA_PREF *, void *);
	EVENT_CB exposeCB;
	EVENT_CB inputCB;
	DRAWAREAPREF_CB resizeCB;
public:
	CmDrawArea(CmContainer *parent, int w, int h);

	void setExposeCallback(CmCallback *, EVENT_CB);
	void setInputCallback(CmCallback *, EVENT_CB);
	void setResizeCallback(CmCallback *, DRAWAREAPREF_CB);
};

/**
 Status Bar Object
*/

class CmStatusBar : public CmObject {
public:
	CmStatusBar(CmContainer *parent, const char *text=0, int max=10, MTFont font=DEFAULT_FONT);

	void setMax(int max);
	void setWidth(int width);
	void setPos(int pos);
};


/**
 Scroll Bar Object
*/

class CmScrollBar : public CmObject {
	friend void _ScrollBarValueCB_(MOBJECT p, MVALUE a, void *ud);
	LONG_CB changeCB;
public:
	CmScrollBar(CmContainer *parent, int type, long n, long max);

	void setMax(long max);
	void setPos(long pos);
	void setN(long n);
	void setIncr(int incr);
	void setValues(long max, long n, long pos);
	long getMax(void);
	long getPos(void);
	long getN(void);

	void setCallback(CmCallback *, LONG_CB);
};

/**
 SFile Object
*/

class CmSFile : public CmObject {
public:
	CmSFile(CmContainer *parent, const char *fname, MTFont font=DEFAULT_FONT,
		int nrow=10, int ncol=20);

	void setFile(const char *fname);
        void CmSFile::setHMargin(int hm);
};

/**
 Abstract List Class
*/

class CmAbsList : public CmObject {
	friend class CmOptionEdit;
	friend void _ListActivateCB_(MOBJECT p, LIST_ACT *la, void *ud);
	friend void _ListPickCB_(MOBJECT p, LIST_ACT *la, void *ud);
        friend void _ListSelectionCB_(MOBJECT p, LIST_SEL *la, void *ud);
	LISTACT_CB activateCB;
	LISTACT_CB pickCB;
	LISTSEL_CB selectionCB;
protected:
	CmAbsList(void){};
public:
	void setSortCallback(STRCMP_CB cb);
	void setHead(const char *text);
	void setHeadColor(CmColor *bgc, CmColor *fgc);
	void setHMargin(int hm);
	void setItemCheckStatus(const char *item, int status);

	LIST_ACT *getFirstItem(void);
	LIST_ACT *getFirstSelectedItem(void);
	LIST_ACT *getNextItem(void);
	LIST_ACT *getNextSelectedItem(void);
	LIST_ACT *getFirstUnselectedItem(void);
	LIST_ACT *getNextUnselectedItem(void);
	long      getNItem(void);
	LIST_ACT *getItem(const char *item);
	LIST_ACT *getElement(long n);
	LIST_ACT *getSelected(void);
	int       getItemCheckStatus(const char *item);
	void	  getHead(char *buff);

	void selectItem(const char *item);
	void deselectItem(const char *item);
	void selectElement(long n);
	void deselectElement(long n);
	void freeze(void);
	void thaw(void);
	void reset(void);
	void clear(void);
	int  addItem(const char *item, void *data);
	int  delItem(const char *item);
	int  addElement(long n, const char *item, void *data);
	int  delElement(long n);
	void changeItem(const char *old, const char *new_item, void *u_data);
	void changeElement(long n, const char *new_item, void *u_data);
	void enableMultiSelection(void);

	void setCallback(CmCallback *, LISTACT_CB);
	void setPickCallback(CmCallback *, LISTACT_CB);
	void setSelectionCallback(CmCallback *, LISTSEL_CB);
};

class CmSList : public CmAbsList {
public:
	CmSList(CmContainer *parent, MTFont font, int row_len,
		int nrow, int ncol);
};

class CmEditList : public CmAbsList {
public:
	CmEditList(CmContainer *parent, MTFont font, EL_TEMPLATE *elt,
		int ne, int nrow, int ncol);
	int     addItem(char **argv, void *u_data);
	int     addItemFromEdits(void *u_data);
	void    changeElement(long n, char **new_ele, void *data);
	void    delElement(long n);
	void    seekItem(char **argv, long *pos, void **u_data);
	void    getItemField(const char *item, int ii, char *buff);
	void    getEditText(int ii, char *buff);
	void    setEditText(int ii, const char *buff);
};

class CmClippedSList : public CmAbsList {
	friend void _ClippedListNewBlockCB_(MOBJECT, SBL_NEW_BLOCK *, void *);
	friend void _ClippedListItemSeekCB_(MOBJECT, SBL_ITEM_SEEK *, void *);
	NEWBLOCK_CB newBlockCB;
	ITEMSEEK_CB itemSeekCB;
public:
	CmClippedSList(CmContainer *parent,
		CmCallback *owner, NEWBLOCK_CB nb, ITEMSEEK_CB is,
		MTFont font, int nrow, int ncol, int xc, long xn=0);
};

/**
 OptionEdit Class
*/

class CmOptionEdit : public CmAbsEditField {
	CmAbsList *slist;
public:

	CmOptionEdit(CmContainer *parent, const char *text, int len, int clip_len=-1,
		int nrow=10, MTFont font=DEFAULT_FONT);
	~CmOptionEdit(void);
	CmAbsList *getSList(void) { return slist; };
};

/**
 Menu Class
*/

class CmMenuBar : public CmObject {
public:
	CmMenuBar(CmContainer *parent, MTFont font);

	void setAcceleratorKey(int key);
};

class CmMenu;

class CmMenuLine {
protected:
	friend class CmMenu;
	MENU_ITEM id;
	MOBJECT pdobj;
	CmMenuLine *next;
	virtual ~CmMenuLine(void) {}
};

class CmMenuSeparator : public CmMenuLine {
public:
	CmMenuSeparator(CmMenu *pd);
};

class CmAbsMenuItem : public CmMenuLine, public CmCallback {
	friend class CmMenu;
	friend void _MenuItemActivateCB_(MENU_ITEM item, MVALUE status, void *ud);
	MENUITEM_CB itemCB;
	CmCallback *owner;
protected:
	CmAbsMenuItem(CmMenu *pd, CmCallback *ff, MENUITEM_CB cb);
public:

	void setText(const char *text);
	void setSensitive(void);
	void setUnsensitive(void);
	void setAcceleratorKey(int key, const char *a_text=0);
	int  setCheckStatus(int true_false);
	int  getCheckStatus(void);
	const char *getText(void);
	CmMenu *getMenu(void);
	int  toggleCheckStatus(void);
};

class CmMenuItem : public CmAbsMenuItem {
public:
	CmMenuItem(CmMenu *pd, const char *text, CmCallback *ff=0, MENUITEM_CB cb=0, int acc=0, const char *at=0);
};

class CmMenuRadioItem : public CmAbsMenuItem {
public:
	CmMenuRadioItem(CmMenu *pd, const char *text, CmCallback *ff=0, MENUITEM_CB cb=0, int acc=0, const char *at=0);
};

class CmMenuToggleItem : public CmAbsMenuItem {
public:
	CmMenuToggleItem(CmMenu *pd, const char *text, CmCallback *ff=0, MENUITEM_CB cb=0, int acc=0, const char *at=0);
};

/**
 The container for menu items.
 Can be created as a stand alone container (popup), a container for a menu bar,
 a container as child of an item (subpulldown).
 */
class CmMenu : public CmObject {
	friend class CmMenuSeparator;
	friend class CmAbsMenuItem;
	CmMenuLine *list;
	ITEMSEL_CB itemSelCB;
public:
	CmMenu(CmMenuBar *menu, const char *title);
	CmMenu(CmMenuItem *it);
	CmMenu(MTFont font=HELV_MEDIUM);
	virtual ~CmMenu(void);
	void setText(const char *text);
	const char *getText(void);
	void popup(int x, int y);

	void setItemSelectionCallback(CmCallback *, ITEMSEL_CB);
};

/*******************************************************************/
// INLINE FUNCTIONS
/*******************************************************************/

/*
 * Inline CmObject functions
 */

inline void CmObject::drawText(int x, int y, int w, int h, const char *text,
		MTFont font, CmColor *fgc, int align)
{
	MDrawText(obj, x, y, w, h, text, font, *fgc, align);
}

inline void CmObject::drawNText(int x, int y, int w, int h, const char *text,
		MTFont font, CmColor *fgc, int align, int n)
{
	MDrawNText(obj, x, y, w, h, text, font, *fgc, align, n);
}

inline void CmObject::drawBGText(int x, int y, int w, int h, const char *text,
		MTFont font, CmColor *bgc, CmColor *fgc, int align)
{
	MDrawBGText(obj, x, y, w, h, text, font, *bgc, *fgc, align);
}

inline void CmObject::drawBGNText(int x, int y, int w, int h, const char *text,
		MTFont font, CmColor *bgc, CmColor *fgc, int align, int n)
{
	MDrawBGNText(obj, x, y, w, h, text, font, *bgc, *fgc, align, n);
}

inline void CmObject::drawShadowedText(int x, int y, int w, int h, const char *text,
		MTFont font, int sh, CmColor *fgc1, CmColor *fgc2, int align)
{
	MDrawShadowedText(obj, x, y, w, h, text, font, sh, *fgc1, *fgc2, align);
}

inline void CmObject::drawShadowedNText(int x, int y, int w, int h, const char *text,
		MTFont font, int sh, CmColor *fgc1, CmColor *fgc2, int align, int n)
{
	MDrawShadowedNText(obj, x, y, w, h, text, font, sh, *fgc1, *fgc2, align, n);
}

inline void CmObject::drawPoint(int x, int y, CmColor *col)
{
	MDrawPoint(obj, x, y, *col);
}

inline void CmObject::drawLine(int x1, int y1, int x2, int y2, CmColor *col)
{
	MDrawLine(obj, x1, y1, x2, y2, *col);
}

inline void CmObject::drawPolyline(MPOINT *points, int n, CmColor *col)
{
	MDrawPolyline(obj, points, n, *col);
}

inline void CmObject::drawFilledPolyline(MPOINT *points, int n, CmColor *col)
{
	MDrawFilledPolyline(obj, points, n, *col);
}

inline void CmObject::drawBox(int x, int y, int w, int h, CmColor *col)
{
	MDrawBox(obj, x, y, w, h, *col);
}

inline void CmObject::drawFilledBox(int x, int y, int w, int h, CmColor *col)
{
	MDrawFilledBox(obj, x, y, w, h, *col);
}

inline void CmObject::drawCircle(int x, int y, int r, CmColor *col)
{
	MDrawCircle(obj, x, y, r, *col);
}

inline void CmObject::drawFilledCircle(int x, int y, int r, CmColor *col)
{
	MDrawFilledCircle(obj, x, y, r, *col);
}

inline void CmObject::drawArc(int x, int y, int r, int a1, int a2, CmColor *col)
{
	MDrawArc(obj, x, y, r, a1, a2, *col);
}

inline void CmObject::drawFilledArc(int x, int y, int r, int a1, int a2, CmColor *c)
{
	MDrawFilledArc(obj, x, y, r, a1, a2, *c);
}

inline void CmObject::drawImage(CmImage *gi, int img_x, int img_y, int w, int h,
		int x, int y)
{
	MDrawImage(obj, gi, img_x, img_y, w, h, x, y);
}

inline void CmObject::drawXorLine(int x1, int y1, int x2, int y2)
{
	MDrawXorLine(obj, x1, y1, x2, y2);
}

inline void CmObject::drawXorBox(int x, int y, int w, int h)
{
	MDrawXorBox(obj, x, y, w, h);
}

inline void CmObject::setSensitive(void)
{
	MObjectSetSensitive(obj);
}

inline void CmObject::setUnsensitive(void)
{
	MObjectSetUnsensitive(obj);
}

inline void CmObject::setSensitivity(int true_false)
{
	MObjectSetSensitivity(obj, true_false);
}

inline void CmObject::setSelectable(void)
{
	MObjectSetSelectable(obj);
}

inline void CmObject::setUnselectable(void)
{
	MObjectSetUnselectable(obj);
}

inline void CmObject::setColor(CmColor *bgc, CmColor *fgc)
{
	MObjectSetColor(obj, *bgc, *fgc);
}

inline void CmObject::setBackgroundRGB(unsigned r, unsigned g, unsigned b)
{
	MObjectSetBackgroundRGB(obj, r, g, b);
}

inline void CmObject::setForegroundRGB(unsigned r, unsigned g, unsigned b)
{
	MObjectSetForegroundRGB(obj, r, g, b);
}

inline void CmObject::setText(const char *text)
{
	MObjectSetText(obj, text);
}

inline void CmObject::setName(const char *name)
{
	MObjectSetName(obj, name);
}

inline void CmObject::setFont(MTFont font, int resize)
{
	MObjectSetFont(obj, font, resize);
}

inline void CmObject::setShadow(int type, int in, int out)
{
	MObjectSetShadow(obj, type, in, out);
}

inline void CmObject::setResize(int resize_w, int resize_h)
{
	MObjectSetResize(obj, resize_w, resize_h);
}

inline void CmObject::align(int h_align, int v_align)
{
	MObjectAlign(obj, h_align, v_align);
}

inline void CmObject::setCursor(MTCursor cursor)
{
	MObjectSetCursor(obj, cursor);
}

inline void CmObject::setPopupHelp(const char *str, int delay, int duration)
{
	MObjectSetPopupHelp(obj, str, delay, duration);
}

inline void CmObject::setLabelHelp(const char *str, CmObject *label)
{
	MObjectSetLabelHelp(obj, str, label->obj);
}

inline void CmObject::changePopupHelpText(const char *str)
{
	MObjectChangePopupHelpText(obj, str);
}

inline void CmObject::changeLabelHelpText(const char *str)
{
	MObjectChangeLabelHelpText(obj, str);
}

inline void CmObject::getText(char *text)
{
	MObjectGetText(obj, text);
}

inline int CmObject::getWidth(void)
{
	return MObjectWidth(obj);
}

inline int CmObject::getHeight(void)
{
	return MObjectHeight(obj);
}

inline int CmObject::getX(void)
{
	return MObjectX(obj);
}

inline MTFont CmObject::getFont(void)
{
	return MObjectGetFont(obj);
}

inline int CmObject::getY(void)
{
	return MObjectY(obj);
}

inline void CmObject::resize(int w, int h)
{
	MObjectResize(obj, w, h);
}

inline void CmObject::redraw(void)
{
	MObjectRedraw(obj);
}

inline void CmObject::select(void)
{
	MSelectObject(obj);
}

inline void CmObject::map(void)
{
	MObjectMap(obj);
}

inline void CmObject::unmap(void)
{
	MObjectUnmap(obj);
}

inline int CmObject::isSensitive(void)
{
	return MObjectGetSensitivity(obj);
}

inline void CmObject::enableOptions(const char *name, CARD32 mask)
{
	MObjectEnableOptions(obj, name, mask);
}

inline void CmObject::saveOptions(void)
{
	MObjectSaveOptions(obj);
}

inline void CmObject::unlink(int resize)
{
	MDeleteObject(obj, resize);
}

inline void CmObject::setAttachment(int top, int bottom, int left, int right)
{
	MObjectSetAttachment(obj, top, bottom, left, right);
}

inline void CmObject::setOffset(int t_off, int b_off, int l_off, int r_off)
{
	MObjectSetOffset(obj, t_off, b_off, l_off, r_off);
}

inline void CmObject::setWidth(int width)
{
	MObjectSetWidth(obj, width);
}

inline void CmObject::setHeight(int height)
{
	MObjectSetHeight(obj, height);
}

inline void CmObject::focusLink(CmObject *obj2)
{
	MFormFocusLink(obj, obj2->obj);
}

inline void CmObject::topAttach(int att_type, int off, CmObject *p)
{
	MTopAttach(obj, att_type, off, p);
}

inline void CmObject::bottomAttach(int att_type, int off, CmObject *p)
{
	MBottomAttach(obj, att_type, off, p);
}

inline void CmObject::leftAttach(int att_type, int off, CmObject *p)
{
	MLeftAttach(obj, att_type, off, p);
}

inline void CmObject::rightAttach(int att_type, int off, CmObject *p)
{
	MRightAttach(obj, att_type, off, p);
}

/*
 * Inline CmShell functions
 */
inline Window CmShell::getWindow(void)
{
	return MShellGetWindow(obj);
}

inline void CmShell::realize(void)
{
	MShellRealize(obj);
}

inline void CmShell::realize(int x, int y)
{
	MShellRealizeXY(obj, x, y);
}

inline void CmShell::realize(int x, int y, int w, int h)
{
	MShellRealizeXYWH(obj, x, y, w, h);
}

inline void CmShell::unrealize(void)
{
	MShellUnrealize(obj);
}

inline void CmShell::raise(void)
{
	MShellRaise(obj);
}

/*
 *  Inline CmForm functions
 */
inline void CmForm::setSpacing(int offset)
{
	MFormSetSpacing(obj, offset);
}

inline void CmForm::setFocusWrap(int true_false)
{
	MFormSetFocusWrap(obj, true_false);
}

inline void CmForm::setFocusHighlight(int true_false)
{
	MFormSetFocusHighlight(obj, true_false);
}

inline void CmAbsCForm::setClipX(int clip_x)
{
	MFormSetClipX(obj, clip_x);
}

inline void CmAbsCForm::setClipY(int clip_y)
{
	MFormSetClipY(obj, clip_y);
}

inline int CmAbsCForm::getClipX(void)
{
	return MFormGetClipX(obj);
}

inline int CmAbsCForm::getClipY(void)
{
	return MFormGetClipY(obj);
}

inline void CmAbsPagedForm::setActivePage(int page)
{
	MPagedFormSetActivePage(obj, page);
}

inline void CmTableForm::showGrid(void)
{
    MTableFormShowGrid(obj);
}

inline void CmTableForm::hideGrid(void)
{
    MTableFormHideGrid(obj);
}

inline void CmTableForm::setColFont(int col, MTFont f)
{
    MTableFormSetColFont(obj, col, f);
}

inline void CmTableForm::setRowHeight(int row, int height)
{
    MTableFormSetRowHeight(obj, row, height);
}

inline void CmTableForm::setColWidth(int col, int width)
{
    MTableFormSetColWidth(obj, col, width);
}

inline void CmTableForm::setColAlign(int col, int al)
{
    MTableFormSetColAlign(obj, col, al);
}

inline void CmTableForm::setColBackgroundRGB(int col, int r, int g, int b)
{
    MTableFormSetColBackgroundRGB(obj, col, r, g, b);
}

inline void CmTableForm::setColForegroundRGB(int col, int r, int g, int b)
{
    MTableFormSetColForegroundRGB(obj, col, r, g, b);
}

inline void CmTableForm::setCellText(int x, int y, const char *text)
{
    MTableFormSetCellText(obj, x, y, text);
}

inline void CmTableForm::getCellText(int x, int y, char *text)
{
    MTableFormGetCellText(obj, x, y, text);
}

inline void CmTableForm::setCellFont(int x, int y, MTFont f)
{
    MTableFormSetCellFont(obj, x, y, f);
}

inline void CmTableForm::setCellAlign(int x, int y, int h_al, int v_al)
{
    MTableFormSetCellAlign(obj, x, y, h_al, v_al);
}

inline void CmTableForm::setCellBackgroundRGB(int x, int y, int r, int g, int b)
{
    MTableFormSetCellBackgroundRGB(obj, x, y, r, g, b);
}

inline void CmTableForm::setCellForegroundRGB(int x, int y, int r, int g, int b)
{
    MTableFormSetCellForegroundRGB(obj, x, y, r, g, b);
}

inline int CmEditList::addItem(char **argv, void *u_data)
{
	return MEditListAddItem(obj, argv, u_data);
}

inline int CmEditList::addItemFromEdits(void *u_data)
{
	return MEditListAddItemFromEdits(obj, u_data);
}

inline void CmEditList::changeElement(long n, char **new_ele, void *data)
{
	MEditListChangeElement(obj, n, new_ele, data);
}

inline void CmEditList::delElement(long n)
{
	MEditListDelElement(obj, n);
}

inline void CmEditList::seekItem(char **argv, long *pos, void **u_data)
{
	MEditListSeekItem(obj, argv, pos, u_data);
}

inline void CmEditList::getItemField(const char *item, int ii, char *buff)
{
	MEditListGetItemField(obj, item, ii, buff);
}

inline void CmEditList::getEditText(int ii, char *buff)
{
	MEditListGetEditText(obj, ii, buff);
}

inline void CmEditList::setEditText(int ii, const char *buff)
{
	MEditListSetEditText(obj, ii, buff);
}

inline void CmAppl::logInit(int level, int nr, int nvr, int nc, int nvc, const char *fname)
{
	MLogInit(level, nr, nvr, nc, nvc, fname);
}

inline void CmAppl::enableCustomizing(void)
{
	MEnableCustomizing();
}

inline void CmAppl::mainLoop(void)
{
	MMainLoop();
}

#endif
