/* p12_pkey.c */
/* Copyright (C) 1997-8 Dr S N Henson (shenson@bigfoot.com) 
 * All Rights Reserved.
 * Any software using this code must include the following message in its
 * startup code or documentation and in any advertising material:
 * "This Product includes cryptographic software written by Dr S N Henson
 *  (shenson@bigfoot.com)"
 */
#include <stdio.h>
#include <stdlib.h>
#include <err.h>
#include <rand.h>
#include "pkcs12.h"

/* Extract a private key from a PKCS8 structure */

EVP_PKEY *PKCS82PKEY (p8)
PKCS8_PRIV_KEY_INFO *p8;
{
	EVP_PKEY *pkey;
	RSA *rsa;
	DSA *dsa;
	ASN1_INTEGER *dsapriv;
	X509_ALGOR *a;
	STACK *ndsa;
	BN_CTX *ctx;
	unsigned char *p;
	int plen, pkeylen;

	switch (p8->broken) {
		case PKCS8_OK:
		p = p8->pkey->value.octet_string->data;
		pkeylen = p8->pkey->value.octet_string->length;
		break;

		case PKCS8_NO_OCTET:
		p = p8->pkey->value.sequence->data;
		pkeylen = p8->pkey->value.sequence->length;
		break;

		default:
		PKCS12err(PKCS12_F_PKCS12_PKCS82PKEY,PKCS12_R_PKCS8_UNKNOWN_BROKEN_TYPE);
		return NULL;
		break;
	}
	if (!(pkey = EVP_PKEY_new())) {
		PKCS12err(PKCS12_F_PKCS12_PKCS82PKEY,ERR_R_MALLOC_FAILURE);
		return NULL;
	}
	a = p8->pkeyalg;
	switch (OBJ_obj2nid(a->algorithm))
	{
		case NID_rsaEncryption:
		if (!(rsa = d2i_RSAPrivateKey (NULL, &p, pkeylen))) {
			PKCS12err(PKCS12_F_PKCS12_PKCS82PKEY, PKCS12_R_DECODE_ERROR);
			return NULL;
		}
		EVP_PKEY_assign_RSA (pkey, rsa);
		break;
		
		case NID_dsa:
		/* PKCS#8 DSA is weird: you just get a private key integer
	         * and parameters in the AlgorithmIdentifier the pubkey must
		 * be recalculated.
		 */
	
		/* Check for broken Netscape Database DSA PKCS#8, UGH! */
		if(*p == (V_ASN1_SEQUENCE|V_ASN1_CONSTRUCTED)) {
		    if(!(ndsa = PKCS12_struct_unpack(p, pkeylen, 
					(char *(*)())d2i_ASN1_INTEGER))) {
			PKCS12err(PKCS12_F_PKCS12_PKCS82PKEY, PKCS12_R_DECODE_ERROR);
			return NULL;
		    }
		    if(sk_num(ndsa) != 2 ) {
			PKCS12err(PKCS12_F_PKCS12_PKCS82PKEY, PKCS12_R_DECODE_ERROR);
			sk_pop_free(ndsa, ASN1_STRING_free);
			return NULL;
		    }
		    dsapriv = (ASN1_INTEGER *) sk_pop(ndsa);
		    sk_pop_free(ndsa, ASN1_STRING_free);
		} else if (!(dsapriv=d2i_ASN1_INTEGER (NULL, &p, pkeylen))) {
			PKCS12err(PKCS12_F_PKCS12_PKCS82PKEY, PKCS12_R_DECODE_ERROR);
			return NULL;
		}
		/* Retrieve parameters */
		if (a->parameter->type != V_ASN1_SEQUENCE) {
			PKCS12err(PKCS12_F_PKCS12_PKCS82PKEY, PKCS12_R_NO_DSA_PARAMETERS);
			return NULL;
		}
		p = a->parameter->value.sequence->data;
		plen = a->parameter->value.sequence->length;
		if (!(dsa = d2i_DSAparams (NULL, &p, plen))) {
			PKCS12err(PKCS12_F_PKCS12_PKCS82PKEY, PKCS12_R_DECODE_ERROR);
			return NULL;
		}
		/* We have parameters now set private key */
		if (!(dsa->priv_key = ASN1_INTEGER_to_BN(dsapriv, NULL))) {
			PKCS12err(PKCS12_F_PKCS12_PKCS82PKEY,PKCS12_R_BN_DECODE_ERROR);
			DSA_free (dsa);
			return NULL;
		}
		/* Calculate public key (ouch!) */
		if (!(dsa->pub_key = BN_new())) {
			PKCS12err(PKCS12_F_PKCS12_PKCS82PKEY,ERR_R_MALLOC_FAILURE);
			DSA_free (dsa);
			return NULL;
		}
		if (!(ctx = BN_CTX_new())) {
			PKCS12err(PKCS12_F_PKCS12_PKCS82PKEY,ERR_R_MALLOC_FAILURE);
			DSA_free (dsa);
			return NULL;
		}
			
		if (!BN_mod_exp(dsa->pub_key, dsa->g,
						 dsa->priv_key, dsa->p, ctx)) {
			
			PKCS12err(PKCS12_F_PKCS12_PKCS82PKEY,PKCS12_R_BN_PUBKEY_ERROR);
			BN_CTX_free (ctx);
			DSA_free (dsa);
			return NULL;
		}

		EVP_PKEY_assign_DSA (pkey, dsa);
		BN_CTX_free (ctx);
		break;

		default:
		PKCS12err(PKCS12_F_PKCS12_PKCS82PKEY, PKCS12_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM);
		PKCS12_add_obj_error(a->algorithm);
		EVP_PKEY_free (pkey);
		return NULL;
	}
	return pkey;
}

/* Turn a private key into a PKCS8 structure */

PKCS8_PRIV_KEY_INFO *PKEY2PKCS8(pkey)
EVP_PKEY *pkey;
{
	PKCS8_PRIV_KEY_INFO *p8;
	ASN1_INTEGER *dpkey;
	unsigned char *p, *q;
	int len;
	if (!(p8 = PKCS8_PRIV_KEY_INFO_new())) {	
		PKCS12err(PKCS12_F_PKCS12_PKEY2PKCS8,ERR_R_MALLOC_FAILURE);
		return NULL;
	}
	ASN1_INTEGER_set (p8->version, 0);
	if (!(p8->pkeyalg->parameter = ASN1_TYPE_new ())) {
		PKCS12err(PKCS12_F_PKCS12_PKEY2PKCS8,ERR_R_MALLOC_FAILURE);
		PKCS8_PRIV_KEY_INFO_free (p8);
		return NULL;
	}
	switch (EVP_PKEY_type(pkey->type)) {
		case EVP_PKEY_RSA:

		M_ASN1_OBJECT_set(p8->pkeyalg->algorithm, NID_rsaEncryption);
		p8->pkeyalg->parameter->type = V_ASN1_NULL;
		if (!PKCS12_pack_octet ((char *)pkey, i2d_PrivateKey,
					 &p8->pkey->value.octet_string)) {
			PKCS12err(PKCS12_F_PKCS12_PKEY2PKCS8,ERR_R_MALLOC_FAILURE);
			PKCS8_PRIV_KEY_INFO_free (p8);
			return NULL;
		}
		break;

		case EVP_PKEY_DSA:
		M_ASN1_OBJECT_set(p8->pkeyalg->algorithm, NID_dsa);

		/* get paramaters and place in AlgorithmIdentifier */
		len = i2d_DSAparams (pkey->pkey.dsa, NULL);
		if (!(p = Malloc(len))) {
			PKCS12err(PKCS12_F_PKCS12_PKEY2PKCS8,ERR_R_MALLOC_FAILURE);
			PKCS8_PRIV_KEY_INFO_free (p8);
			return NULL;
		}
		q = p;
		i2d_DSAparams (pkey->pkey.dsa, &q);
		p8->pkeyalg->parameter->type = V_ASN1_SEQUENCE;
		p8->pkeyalg->parameter->value.sequence = ASN1_STRING_new();
		ASN1_STRING_set(p8->pkeyalg->parameter->value.sequence, p, len);
		Free(p);
		/* Get private key into an integer and pack */
		if (!(dpkey = BN_to_ASN1_INTEGER (pkey->pkey.dsa->priv_key, NULL))) {
			PKCS12err(PKCS12_F_PKCS12_PKEY2PKCS8,PKCS12_R_ENCODE_ERROR);
			PKCS8_PRIV_KEY_INFO_free (p8);
			return NULL;
		}
		
		if (!PKCS12_pack_octet ((char *)dpkey, i2d_ASN1_INTEGER,
					 &p8->pkey->value.octet_string)) {
			PKCS12err(PKCS12_F_PKCS12_PKEY2PKCS8,ERR_R_MALLOC_FAILURE);
			ASN1_INTEGER_free (dpkey);
			PKCS8_PRIV_KEY_INFO_free (p8);
			return NULL;
		}
		ASN1_INTEGER_free (dpkey);
		break;

		default:
		PKCS12err(PKCS12_F_PKCS12_PKEY2PKCS8, PKCS12_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM);
		PKCS8_PRIV_KEY_INFO_free (p8);
		return NULL;
	}
	p8->pkey->type = V_ASN1_OCTET_STRING;
	RAND_seed (p8->pkey->value.octet_string->data,
					 p8->pkey->value.octet_string->length);
	return p8;
}

PKCS8_PRIV_KEY_INFO *PKCS8_set_broken(p8, broken)
PKCS8_PRIV_KEY_INFO *p8;
int broken;
{
	switch (broken) {

		case PKCS8_OK:
		p8->broken = PKCS8_OK;
		return p8;
		break;

		case PKCS8_NO_OCTET:
		p8->broken = PKCS8_NO_OCTET;
		p8->pkey->type = V_ASN1_SEQUENCE;
		return p8;
		break;

		default:
		PKCS12err(PKCS12_F_PKCS12_PKCS8_SET_BROKEN,PKCS12_R_PKCS8_UNKNOWN_BROKEN_TYPE);
		return NULL;
		break;
		
	}
}


