/* pkcs12.c */
/* Copyright (C) 1997-8 Dr S N Henson (shenson@bigfoot.com) 
 * All Rights Reserved.
 * Any software using this code must include the following message in its
 * startup code or documentation and in any advertising material:
 * "This Product includes cryptographic software written by Dr S N Henson
 *  (shenson@bigfoot.com)"
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pem.h>
#include <err.h>
#include "pkcs12.h"

#define STANDALONE

#ifdef STANDALONE
#define MAIN main
BIO *bio_err=NULL;
#define EXIT return
#else
#include "apps.h"
#undef PROG
#define PROG pkcs12_main
#endif

EVP_CIPHER *enc;

#define _ITER_ 1000

#define NOKEYS		0x1
#define NOCERTS 	0x2
#define INFO		0x4
#define CLCERTS		0x8
#define CACERTS		0x10

int get_cert_chain (X509 *cert, STACK **chain);
int dump_certs_keys_p12 (BIO *out, PKCS12 *p12, unsigned char *pass, int passlen, int options);
int dump_certs_pkeys_bags (BIO *out, STACK *bags, unsigned char *pass, int passlen, int options);
int dump_certs_pkeys_bag (BIO *out, PKCS12_SAFEBAG *bags, unsigned char *pass, int passlen, int options);
int print_attrib (BIO *out, PKCS12_SAFEBAG *bag);
int alg_print (BIO *x, X509_ALGOR *alg);
int pk8_usage_print(BIO *x, PKCS8_PRIV_KEY_INFO *p8);
int pk8_attr_print(BIO *x, PKCS8_PRIV_KEY_INFO *p8);

int MAIN(argc, argv)
int argc;
char **argv;
{
    char *infile=NULL, *outfile=NULL, *keyname = NULL;	
    char *certfile=NULL;
    BIO *in=NULL, *out = NULL, *inkey = NULL, *certsin = NULL;
    char **args;
    char *name = NULL;
    PKCS12 *p12 = NULL;
    char pass[50], macpass[50];
    int export_cert = 0;
    int options = 0;
    int chain = 0;
    int badarg = 0;
    int iter = _ITER_;
    int maciter = 1;
    int twopass = 0;
    unsigned char keytype = 0;
    int cert_pbe = NID_pbe_WithSHA1And40BitRC2_CBC;
    int ret = 1;
    STACK *canames = NULL;
#ifndef STANDALONE
    apps_startup();
#endif


    enc = EVP_des_ede3_cbc();
    if (bio_err == NULL ) bio_err = BIO_new_fp (stderr, BIO_NOCLOSE);

    args = argv + 1;

    while (*args) {
	if (*args[0] == '-') {
		if (!strcmp (*args, "-nokeys")) options |= NOKEYS;
		else if (!strcmp (*args, "-keyex")) keytype = KEY_EX;
		else if (!strcmp (*args, "-keysig")) keytype = KEY_SIG;
		else if (!strcmp (*args, "-nocerts")) options |= NOCERTS;
		else if (!strcmp (*args, "-clcerts")) options |= CLCERTS;
		else if (!strcmp (*args, "-cacerts")) options |= CACERTS;
		else if (!strcmp (*args, "-noout")) options |= (NOKEYS|NOCERTS);
		else if (!strcmp (*args, "-info")) options |= INFO;
		else if (!strcmp (*args, "-chain")) chain = 1;
		else if (!strcmp (*args, "-twopass")) twopass = 1;
		else if (!strcmp (*args, "-descert"))
    			cert_pbe = NID_pbe_WithSHA1And3_Key_TripleDES_CBC;
		else if (!strcmp (*args, "-export")) export_cert = 1;
		else if (!strcmp (*args, "-des")) enc=EVP_des_cbc();
		else if (!strcmp (*args, "-idea")) enc=EVP_idea_cbc();
		else if (!strcmp (*args, "-des3")) enc = EVP_des_ede3_cbc();
		else if (!strcmp (*args, "-noiter")) iter = 1;
		else if (!strcmp (*args, "-maciter")) maciter = _ITER_;
		else if (!strcmp (*args, "-nodes")) enc=NULL;
		else if (!strcmp (*args, "-inkey")) {
		    if (args[1]) {
			args++;	
			keyname = *args;
		    } else badarg = 1;
		} else if (!strcmp (*args, "-certfile")) {
		    if (args[1]) {
			args++;	
			certfile = *args;
		    } else badarg = 1;
		} else if (!strcmp (*args, "-name")) {
		    if (args[1]) {
			args++;	
			name = *args;
		    } else badarg = 1;
		} else if (!strcmp (*args, "-caname")) {
		    if (args[1]) {
			args++;	
			if (!canames) canames = sk_new(NULL);
			sk_push(canames, *args);
		    } else badarg = 1;
		} else if (!strcmp (*args, "-in")) {
		    if (args[1]) {
			args++;	
			infile = *args;
		    } else badarg = 1;
		} else if (!strcmp (*args, "-out")) {
		    if (args[1]) {
			args++;	
			outfile = *args;
		    } else badarg = 1;
		} else badarg = 1;

	} else badarg = 1;
	args++;
    }

    if (badarg) {
	BIO_printf (bio_err, "PKCS#12 program version 0.52\n");
	BIO_printf (bio_err, "Usage: pkcs12 [options]\n");
	BIO_printf (bio_err, "where options are\n");
	BIO_printf (bio_err, "-export       output PKCS12 file\n");
	BIO_printf (bio_err, "-chain        add certificate chain\n");
	BIO_printf (bio_err, "-inkey file   private key if not infile\n");
	BIO_printf (bio_err, "-certfile f   add all certs in f\n");
	BIO_printf (bio_err, "-name \"name\"  use name as friendly name\n");
	BIO_printf (bio_err, "-caname \"nm\"  use nm as CA friendly name (can be used more than once).\n");
	BIO_printf (bio_err, "-in  infile   input filename\n");
	BIO_printf (bio_err, "-out outfile  output filename\n");
	BIO_printf (bio_err, "-noout        don't output anything, just verify.\n");
	BIO_printf (bio_err, "-nocerts      don't output certificates.\n");
	BIO_printf (bio_err, "-clcerts      only output client certificates.\n");
	BIO_printf (bio_err, "-cacerts      only output CA certificates.\n");
	BIO_printf (bio_err, "-nokeys       don't output private keys.\n");
	BIO_printf (bio_err, "-info         give info about PKCS#12 structure.\n");
	BIO_printf (bio_err, "-des          encrypt private keys with DES\n");
	BIO_printf (bio_err, "-des3         encrypt private keys with triple DES (default)\n");
	BIO_printf (bio_err, "-idea         encrypt private keys with idea\n");
	BIO_printf (bio_err, "-nodes        don't encrypt private keys\n");
	BIO_printf (bio_err, "-noiter       don't use encryption iteration\n");
	BIO_printf (bio_err, "-maciter      use MAC iteration\n");
	BIO_printf (bio_err, "-twopass      separate MAC, encryption passwords\n");
	BIO_printf (bio_err, "-descert      encrypt PKCS#12 certificates with triple DES (default RC2-40)\n");
	BIO_printf (bio_err, "-keyex        set MS key exchange type\n");
	BIO_printf (bio_err, "-keysig       set MS key signature type\n");
    	goto end;
    }

    ERR_load_crypto_strings();
#ifdef STANDALONE
    SSLeay_add_all_algorithms();
    PKCS12_lib_init();
#endif

#ifdef CRYPTO_MDEBUG
CRYPTO_mem_ctrl(CRYPTO_MEM_CHECK_ON);
#endif

    in = BIO_new (BIO_s_file());
    out = BIO_new (BIO_s_file());

    if (!infile) BIO_set_fp (in, stdin, BIO_NOCLOSE);
    else {
        if (BIO_read_filename (in, infile) <= 0) {
	    perror (infile);
	    goto end;
	}
    }

   if (certfile) {
    	certsin = BIO_new (BIO_s_file());
        if (BIO_read_filename (certsin, certfile) <= 0) {
	    perror (certfile);
	    goto end;
	}
    }

    if (keyname) {
    	inkey = BIO_new (BIO_s_file());
        if (BIO_read_filename (inkey, keyname) <= 0) {
	    perror (keyname);
	    goto end;
	}
     }

    if (!outfile) BIO_set_fp (out, stdout, BIO_NOCLOSE);
    else {
        if (BIO_write_filename (out, outfile) <= 0) {
	    perror (outfile);
	    goto end;
	}
    }
    if (twopass) {
	if(EVP_read_pw_string (macpass, 50, "Enter MAC Password:", export_cert)) {
    	    BIO_printf (bio_err, "Can't read Password\n");
    	    goto end;
       	}
    }

if (export_cert) {
	EVP_PKEY *key;
	STACK *bags, *safes;
	PKCS12_SAFEBAG *bag;
	PKCS8_PRIV_KEY_INFO *p8;
	PKCS7 *authsafe;
	X509 *cert;
	char *catmp;
	unsigned char keyid[] = { 1, 0, 0, 0};
	/* Get private key so we can match it to a certificate */
	key = PEM_read_bio_PrivateKey(inkey ? inkey : in, NULL, NULL);
	if (!inkey) BIO_reset(in);
	if (!key) {
		BIO_printf (bio_err, "Error loading private key\n");
		ERR_print_errors(bio_err);
		goto end;
	}
	
	cert = PEM_read_bio_X509(in, NULL, NULL);

	if (!cert) {
		BIO_printf (bio_err, "Error loading certificate key\n");
		ERR_print_errors(bio_err);
		goto end;
	}

	bags = sk_new (NULL);
	bag = M_PKCS12_x5092certbag (cert);
	if (!bag) {
		ERR_print_errors (bio_err);
		goto end;
	}
        if (name) PKCS12_add_friendlyname (bag, name, -1);
	PKCS12_add_localkeyid (bag, keyid, 4);
	sk_push (bags, (char *)bag);
	/* If chaining get chain */
	if (chain) {
        	int vret, i;
		X509 *x509tmp;
		STACK *chain;
		vret = get_cert_chain (cert, &chain);
		if (vret) {
			BIO_printf (bio_err, "Error %s getting chain.\n",
					X509_verify_cert_error_string(vret));
			goto end;
		}
		/* Exclude verified certificate */
		for (i = 1; i < sk_num (chain) ; i++) {
			x509tmp = (X509 *) sk_value (chain, i);
			bag = M_PKCS12_x5092certbag (x509tmp);
			if ((catmp = sk_shift(canames))) 
        			PKCS12_add_friendlyname (bag, catmp, -1);
		        sk_push (bags, (char *)bag);
		}
			
    	}

	X509_free (cert);

	/* Add any more certificates asked for */
	if (certsin) {
	    STACK *sk;
	    X509_INFO *xi;
	    int i;
	    sk = PEM_X509_INFO_read_bio (certsin, NULL, NULL);
	    if (sk != NULL) {
		for (i = 0; i < sk_num(sk); i++) {
		    xi = (X509_INFO *)sk_value(sk, i);
		    if (xi->x509) {
			bag = M_PKCS12_x5092certbag (xi->x509);
			if ((catmp = sk_shift(canames))) 
        			PKCS12_add_friendlyname (bag, catmp, -1);
		        sk_push (bags, (char *)bag);
		    }
		}
	    }
	    sk_pop_free (sk, X509_INFO_free);
	    BIO_free(certsin);
 	}

	if (canames) sk_free(canames);

	if(EVP_read_pw_string (pass, 50, "Enter Export Password:", 1)) {
	    BIO_printf (bio_err, "Can't read Password\n");
	    goto end;
        }
	if (!twopass) strcpy(macpass, pass);
	/* Turn certbags into encrypted authsafe */
	authsafe = PKCS12_pack_p7encdata (cert_pbe, pass, -1, NULL, 0,
								 iter, bags);
	sk_pop_free(bags, PKCS12_SAFEBAG_free);

	if (!authsafe) {
		ERR_print_errors (bio_err);
		goto end;
	}

	safes = sk_new (NULL);
	sk_push (safes, (char *)authsafe);		

	/* Make a shrouded key bag */
	p8 = PKEY2PKCS8 (key);
	EVP_PKEY_free(key);
	if(keytype) PKCS8_add_keyusage(p8, keytype);
	bag = PKCS12_MAKE_SHKEYBAG (NID_pbe_WithSHA1And3_Key_TripleDES_CBC,
			pass, -1, NULL, 0, iter, p8);
	PKCS8_PRIV_KEY_INFO_free(p8);
        if (name) PKCS12_add_friendlyname (bag, name, -1);
	PKCS12_add_localkeyid (bag, keyid, 4);
	bags = sk_new(NULL);
	sk_push (bags, (char *)bag);
	/* Turn it into unencrypted safe bag */
	authsafe = PKCS12_pack_p7data (bags);
	sk_pop_free(bags, PKCS12_SAFEBAG_free);
	sk_push (safes, (char *)authsafe);

	p12 = PKCS12_init (NID_pkcs7_data);

	M_PKCS12_pack_authsafes (p12, safes);

	sk_pop_free(safes, PKCS7_free);

	PKCS12_set_mac (p12, macpass, -1, NULL, 0, maciter, NULL);

	i2d_PKCS12_bio (out, p12);

	PKCS12_free(p12);

	ret = 0;
	goto end;
	
}

    if (!(p12 = d2i_PKCS12_bio (in, NULL))) {
	ERR_print_errors(bio_err);
	goto end;
    }

    if(EVP_read_pw_string (pass, 50, "Enter Import Password:", 0)) {
	BIO_printf (bio_err, "Can't read Password\n");
	goto end;
    }

    if (!twopass) strcpy(macpass, pass);

    if (options & INFO) BIO_printf (bio_err, "MAC Iteration %ld\n", p12->mac->iter ? ASN1_INTEGER_get (p12->mac->iter) : 1);
    if (!PKCS12_verify_mac (p12, macpass, -1)) {
      BIO_printf (bio_err, "Mac verify errror: invalid password?\n");
      goto end;
     } else BIO_printf (bio_err, "MAC verified OK\n");

    if (!dump_certs_keys_p12 (out, p12, pass, -1, options)) {
	BIO_printf(bio_err, "Error outputting keys and certificates\n");
	ERR_print_errors (bio_err);
	goto end;
    }
    PKCS12_free(p12);
    ret = 0;
    end:
#ifdef CRYPTO_MDEBUG
    BIO_free(in);
    BIO_free(out);
	ERR_print_errors (bio_err);
    CRYPTO_mem_leaks(bio_err);
#endif
    EXIT(ret);
}

int dump_cert_text (out, x)
BIO *out;
X509 *x;
{
	char buf[256];
	X509_NAME_oneline(X509_get_subject_name(x),buf,256);
	BIO_puts(out,"subject=");
	BIO_puts(out,buf);

	X509_NAME_oneline(X509_get_issuer_name(x),buf,256);
	BIO_puts(out,"\nissuer= ");
	BIO_puts(out,buf);
	BIO_puts(out,"\n");
        return 0;
}

int print_attrib (out, bag)
BIO *out;
PKCS12_SAFEBAG *bag;
{
	ASN1_TYPE *attrib;
	char *name;
	int i;
	name = PKCS12_get_friendlyname(bag);
        if(bag->attrib) BIO_printf(out, "Attribute count %d\n", sk_num(bag->attrib));
	if (name) {
		BIO_printf (out, "Friendly Name: %s\n", name);
		Free (name);
	}
	attrib = PKCS12_get_attr (bag, NID_localKeyID);
	if (attrib) {
		BIO_printf (out, "Local Key ID: ");
		for (i = 0; i < attrib->value.octet_string->length; i++) {
			BIO_printf (out, "%02X ",
					 attrib->value.octet_string->data[i]);
		}
		BIO_printf (out, "\n");
	}
	return 0;
}

int dump_certs_keys_p12 (out, p12, pass, passlen, options)
BIO *out;
PKCS12 *p12;
unsigned char *pass;
int passlen;
int options;
{
	STACK *asafes, *bags;
	int i, bagnid;
	PKCS7 *p7;
	if (!( asafes = M_PKCS12_unpack_authsafes (p12))) return 0;
	for (i = 0; i < sk_num (asafes); i++) {
		p7 = (PKCS7 *) sk_value (asafes, i);
		bagnid = OBJ_obj2nid (p7->type);
		if (bagnid == NID_pkcs7_data) {
			bags = M_PKCS12_unpack_p7data (p7);
			if (options & INFO) BIO_printf (bio_err, "PKCS7 Data\n");
		} else if (bagnid == NID_pkcs7_encrypted) {
			if (options & INFO) {
				BIO_printf (bio_err, "PKCS7 Encrypted data: ");
				alg_print (bio_err, 
					p7->d.encrypted->enc_data->algorithm);
			}
			bags = M_PKCS12_unpack_p7encdata (p7, pass, passlen);
		} else continue;
		if (!bags) return 0;
	    	if (!dump_certs_pkeys_bags (out, bags, pass, passlen, 
							 options)) {
			sk_pop_free (bags, PKCS12_SAFEBAG_free);
			return 0;
		}
		sk_pop_free (bags, PKCS12_SAFEBAG_free);
	}
	sk_pop_free (asafes, PKCS7_free);
	return 1;
}

int dump_certs_pkeys_bags (out, bags, pass, passlen, options)
BIO *out;
STACK *bags;
unsigned char *pass;
int passlen;
int options;
{
	int i;
	for (i = 0; i < sk_num (bags); i++) {
		if (!dump_certs_pkeys_bag (out,
			 (PKCS12_SAFEBAG *)sk_value (bags, i), pass, passlen,
					 		options)) return 0;
	}
	return 1;
}

int dump_certs_pkeys_bag (out, bag, pass, passlen, options)
BIO *out;
PKCS12_SAFEBAG *bag;
unsigned char *pass;
int passlen;
int options;
{
	EVP_PKEY *pkey;
	PKCS8_PRIV_KEY_INFO *p8;
	X509 *x509;
	
	switch (M_PKCS12_bag_type(bag))
	{
	case NID_keyBag:
		if (options & INFO) BIO_printf (bio_err, "Key bag\n");
		if (options & NOKEYS) return 1;
		print_attrib (out, bag);
		p8 = bag->value.keybag;
		if (!(pkey = PKCS82PKEY (p8))) return 0;
		pk8_attr_print(out, p8);
		PEM_write_bio_PrivateKey (out, pkey, enc, NULL, 0, NULL);
		EVP_PKEY_free(pkey);
	break;

	case NID_pkcs8ShroudedKeyBag:
		if (options & INFO) {
			BIO_printf (bio_err, "Shrouded Keybag: ");
			alg_print (bio_err, bag->value.shkeybag->algor);
		}
		if (options & NOKEYS) return 1;
		print_attrib (out, bag);
		if (!(p8 = M_PKCS12_decrypt_skey (bag, pass, passlen)))
				return 0;
		if (!(pkey = PKCS82PKEY (p8))) return 0;
		pk8_attr_print(out, p8);
		PKCS8_PRIV_KEY_INFO_free(p8);
		PEM_write_bio_PrivateKey (out, pkey, enc, NULL, 0, NULL);
		EVP_PKEY_free(pkey);
	break;

	case NID_certBag:
		if (options & INFO) BIO_printf (bio_err, "Certificate bag\n");
		if (options & NOCERTS) return 1;
                if (PKCS12_get_attr(bag, NID_localKeyID)) {
			if (options & CACERTS) return 1;
		} else if (options & CLCERTS) return 1;
		print_attrib (out, bag);
		if (M_PKCS12_cert_bag_type(bag) != NID_x509Certificate )
								 return 1;
		if (!(x509 = M_PKCS12_certbag2x509(bag))) return 0;
		dump_cert_text (out, x509);
		PEM_write_bio_X509 (out, x509);
		X509_free(x509);
	break;

	case NID_safeContentsBag:
		if (options & INFO) BIO_printf (bio_err, "Safe Contents bag\n");
		print_attrib (out, bag);
		return dump_certs_pkeys_bags (out, bag->value.safes, pass,
							    passlen, options);
					
	default:
		BIO_printf (bio_err, "Warning unsupported bag type: ");
		i2a_ASN1_OBJECT (bio_err, bag->type);
		BIO_printf (bio_err, "\n");
		return 1;
	break;
	}
	return 1;
}

/* Given a single certificate return a verified chain or NULL if error */

/* Hope this is OK .... */

int get_cert_chain (cert, chain)
X509 *cert;
STACK **chain;
{
	X509_STORE *store;
	X509_STORE_CTX store_ctx;
	STACK *chn;
	int i;
	X509 *x;
	store = X509_STORE_new ();
	X509_STORE_set_default_paths (store);
	X509_STORE_CTX_init(&store_ctx, store, cert, NULL);
	if (X509_verify_cert(&store_ctx) <= 0) {
		i = X509_STORE_CTX_get_error (&store_ctx);
		goto err;
	}
	chn =  sk_dup(X509_STORE_CTX_get_chain (&store_ctx));
	for (i = 0; i < sk_num(chn); i++) {
		x = (X509 *)sk_value(chn, i);
		CRYPTO_add(&x->references,1,CRYPTO_LOCK_X509);
	}
	i = 0;
	*chain = chn;
err:
	X509_STORE_CTX_cleanup(&store_ctx);
	X509_STORE_free(store);
	
	return i;
}	

int alg_print (x, alg)
BIO *x;
X509_ALGOR *alg;
{
	PBEPARAM *pbe;
	unsigned char *p;
	p = alg->parameter->value.sequence->data;
	pbe = d2i_PBEPARAM (NULL, &p, alg->parameter->value.sequence->length);
	BIO_printf (bio_err, "%s, Iteration %d\n", 
	OBJ_nid2ln(OBJ_obj2nid(alg->algorithm)), ASN1_INTEGER_get(pbe->iter));
	PBEPARAM_free (pbe);
	return 0;
}

int pk8_attr_print(x, p8)
BIO *x;
PKCS8_PRIV_KEY_INFO *p8;
{
	if(p8->attributes) BIO_printf(x, "Private Key Attribute Count %d\n",
							sk_num(p8->attributes));
	else return 0;
	return pk8_usage_print(x, p8);
}

int pk8_usage_print(x, p8)
BIO *x;
PKCS8_PRIV_KEY_INFO *p8;
{
	ASN1_TYPE *attr;
	if(!(attr = PKCS8_get_attr(p8, NID_key_usage))) {
		BIO_printf(x, "No Known Attributes\n");
		return 0;
	}
	if(attr->type != V_ASN1_BIT_STRING) {
		BIO_printf(x, "Invalid usage type\n");
		return 0;
	}
	BIO_printf(x, "Key Usage Value is 0x%X\n", 
					attr->value.bit_string->data[0]);
	return 0;
}

