#!/bin/sh
##
##  mkcert.sh -- Make SSL Certificate for `make certificate' command
##  Copyright (c) 1998 Ralf S. Engelschall, All Rights Reserved. 
##

#   parameters
ssleay="$1"
type="$2"
crt="$3"
key="$4"

#   configuration
#   WE ARE CALLED FROM THE PARENT DIR!
sslcrtdir="../conf/ssl.crt"
sslcsrdir="../conf/ssl.csr"
sslkeydir="../conf/ssl.key"

#   processing
case $type in
    dummy)
        echo ""
        echo "Generating self-signed Snake Oil certificate [DUMMY]"
        echo "WARNING: Do not use this for real-life/production systems"
        echo ""
        cp $sslcrtdir/snakeoil.crt $sslcrtdir/server.crt
        cp $sslkeydir/snakeoil.key $sslkeydir/server.key
        echo "Ok, the following files were created:"
        ls -l $sslcrtdir/server.crt $sslkeydir/server.key
        echo ""
        echo "You should use these for testing purposes only and replace"
        echo "them immediately after you decided to run a real server."
        echo ""
        ;;
    test)
        echo ""
        echo "Generating test certificate signed by Snake Oil CA [TEST]"
        echo "WARNING: Do not use this for real-life/production systems"
        echo "______________________________________________________________"
        echo ""
        echo "STEP 1: Generating RSA private key (1024 bit)"
        echo "======"
        $ssleay genrsa -out $sslkeydir/server.key 1024
        echo ""
        echo "STEP 2: Generating X.509 certificate signing request"
        echo "======"
        $ssleay req -new -days 365 \
                    -key $sslkeydir/server.key \
                    -out $sslcsrdir/server.csr
        echo ""
        echo "STEP 3: Generating X.509 certificate signed by Snake Oil CA"
        echo "======"
        rm -f serial.txt
        echo '01' >serial.txt
        $ssleay x509 -CA $sslcrtdir/snakeoil.crt \
                     -CAkey $sslkeydir/snakeoil.key \
                     -CAserial serial.txt \
                     -in $sslcsrdir/server.csr -req \
                     -out $sslcrtdir/server.crt
        rm -f serial.txt
        echo ""
        echo "STEP 4: Enrypting RSA private key with a pass phrase for security"
        echo "======"
        $ssleay rsa -des3 -in $sslkeydir/server.key -out $sslkeydir/server.key.crypt
        cp $sslkeydir/server.key.crypt $sslkeydir/server.key
        rm -f $sslkeydir/server.key.crypt
        echo "______________________________________________________________"
        echo ""
        echo "Ok, the following files were created:"
        ls -l $sslcrtdir/server.crt $sslcsrdir/server.csr $sslkeydir/server.key
        echo ""
        echo "You now can use the server.crt file with the SSLCertificateFile directive"
        echo "and the server.key file with the SSLCertificateKeyFile directive. The"
        echo "server.csr file can be send to an official CA to request a real certificate"
        echo "(not signed by Snake Oil CA) which can replace server.crt later."
        echo ""
        ;;
    custom)
        echo ""
        echo "Generating custom certificate signed by own CA [CUSTOM]"
        echo "______________________________________________________________"
        echo ""
        echo "STEP 1: Generating RSA private key for CA (1024 bit)"
        echo "======"
        $ssleay genrsa -out $sslkeydir/ca.key 1024
        echo ""
        echo "STEP 2: Generating X.509 certificate signing request for CA"
        echo "======"
        $ssleay req -new -days 365 \
                    -key $sslkeydir/ca.key \
                    -out $sslcsrdir/ca.csr
        echo ""
        echo "STEP 3: Generating X.509 certificate for CA signed by itself"
        echo "======"
        $ssleay x509 -req -signkey $sslkeydir/ca.key \
                     -in  $sslcsrdir/ca.csr \
                     -out $sslcrtdir/ca.crt
        echo ""
        echo "STEP 4: Generating RSA private key for SERVER (1024 bit)"
        echo "======"
        $ssleay genrsa -out $sslkeydir/server.key 1024
        echo ""
        echo "STEP 5: Generating X.509 certificate signing request for SERVER"
        echo "======"
        $ssleay req -new -days 365 \
                    -key $sslkeydir/server.key \
                    -out $sslcsrdir/server.csr
        echo ""
        echo "STEP 6: Generating X.509 certificate signed by own CA"
        echo "======"
        rm -f serial.txt
        echo '01' >serial.txt
        $ssleay x509 -CA $sslcrtdir/ca.crt \
                     -CAkey $sslkeydir/ca.key \
                     -CAserial serial.txt \
                     -in $sslcsrdir/server.csr -req \
                     -out $sslcrtdir/server.crt
        rm -f serial.txt
        echo ""
        echo "STEP 7: Enrypting RSA private key of CA with a pass phrase for security"
        echo "======"
        $ssleay rsa -des3 -in $sslkeydir/ca.key -out $sslkeydir/ca.key.crypt
        cp $sslkeydir/ca.key.crypt $sslkeydir/ca.key
        rm -f $sslkeydir/ca.key.crypt
        echo ""
        echo "STEP 8: Enrypting RSA private key of SERVER with a pass phrase for security"
        echo "======"
        $ssleay rsa -des3 -in $sslkeydir/server.key -out $sslkeydir/server.key.crypt
        cp $sslkeydir/server.key.crypt $sslkeydir/server.key
        rm -f $sslkeydir/server.key.crypt
        echo "______________________________________________________________"
        echo ""
        echo "Ok, the following files were created:"
        ls -l $sslcrtdir/ca.crt $sslcsrdir/ca.csr $sslkeydir/ca.key \
              $sslcrtdir/server.crt $sslcsrdir/server.csr $sslkeydir/server.key
        echo ""
        echo "You now can use the server.crt file with the SSLCertificateFile directive,"
        echo "the server.key file with the SSLCertificateKeyFile directive and the"
        echo "ca.crt file with the SSLCACertificateFile directive. The server.csr"
        echo "file can be send to an official CA to request a certificate"
        echo "(not signed by your own CA) which can replace server.crt later."
        echo ""
        ;;
    existing)
        echo ""
        echo "Using existing custom certificate [EXISTING]"
        echo ""
        if [ ".$crt" = . ]; then
            echo "mkcert.sh: No certificate file given" 1>&2
            exit 1
        fi
        if [ ! -f "$crt" ]; then
            echo "mkcert.sh: Cannot find certificate file: $crt" 1>&2
            exit 1
        fi
        if [ ".$key" != . ]; then
            if [ ! -f "$key" ]; then
                echo "mkcert.sh: Cannot find private key file: $key" 1>&2
                exit 1
            fi
            cp $crt $sslcrtdir/server.crt
            cp $key $sslkeydir/server.key
        else
            sed -e '/-----BEGIN CERTIFICATE/,/-----END CERTIFICATE/p' -e '/.*/d' \
                <$crt >$sslcrtdir/server.crt
            sed -e '/-----BEGIN RSA PRIVATE KEY/,/-----END RSA PRIVATE KEY/p' -e '/.*/d' \
                <$crt >$sslkeydir/server.key
        fi
        echo "Ok, the following files were created:"
        ls -l $sslcrtdir/server.crt $sslkeydir/server.key
        echo ""
        echo "Congratulations that you establish your server with real certificates."
        echo ""
        ;;
esac

