/* ====================================================================
 * Copyright (c) 1998 Ralf S. Engelschall. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * 4. The names "mod_ssl" must not be used to endorse or promote
 *    products derived from this software without prior written
 *    permission. For written permission, please contact
 *    rse@engelschall.com.
 *
 * 5. Products derived from this software may not be called "mod_ssl"
 *    nor may "mod_ssl" appear in their names without prior
 *    written permission of Ralf S. Engelschall.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * THIS SOFTWARE IS PROVIDED BY RALF S. ENGELSCHALL ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RALF S. ENGELSCHALL OR
 * HIS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */


#include "mod_ssl.h"


/*  _________________________________________________________________
**
**  Module Initialization
**  _________________________________________________________________
*/

/*
 *  Per-module initialization
 */
void ssl_init_Module(server_rec *s, pool *p)
{
    char buf[MAX_STRING_LEN];
    char *cp;
    char *cp2;
    server_rec *s2;
    SSLConfigRec *pConfig;

    ssl_g_nInitCount++;

    ssl_g_pServer = s;
    ssl_g_pPool   = p;

    /* 
     *  try to fix the configuration and open the dedicated SSL
     *  logfile as early as possible 
     */
    for (s2 = s; s2 != NULL; s2 = s2->next) {
        pConfig = mySrvConfig(s2);

        /* Fix up stuff that may not have been set */
        if (pConfig->bEnabled == UNSET)
            pConfig->bEnabled = FALSE;
        if (pConfig->nVerifyClient == VERIFY_UNSET)
            pConfig->nVerifyClient = VERIFY_NONE;
        if (pConfig->bFakeBasicAuth == UNSET)
            pConfig->bFakeBasicAuth = FALSE;
        if (pConfig->nPassPhraseDialogType == SSL_PPTYPE_UNSET)
            pConfig->nPassPhraseDialogType = SSL_PPTYPE_BUILTIN;

        /* Open the dedicated SSL logfile */
        ssl_log_open(s2, p);
    }

    if (ssl_g_nInitCount == 1)
        ssl_log(s, SSL_LOG_INFO, "Init: 1st startup round (still not detached)");
    else if (ssl_g_nInitCount == 2)
        ssl_log(s, SSL_LOG_INFO, "Init: 2nd startup round (already detached)");
    else
        ssl_log(s, SSL_LOG_INFO, "Init: %d%s restart round (already detached)",
                ssl_g_nInitCount-2, (ssl_g_nInitCount-2) == 1 ? "st" : "nd");

    /*
     *  initialize SSLeay
     */
    ssl_log(s, SSL_LOG_INFO, "Init: Initializing SSLeay library");
    __SSLeay SSL_load_error_strings();
    __SSLeay SSLeay_add_ssl_algorithms();

    /*  
     *  The Apache API does two initializations of modules.  While we want to
     *  initialize only in the second one we have to do the Pass Phrase dialog
     *  in the first one.  Because between the first and the second round
     *  Apache detaches from the terminal.
     */
    if (ssl_g_nInitCount == 1) {
        ssl_init_HandlePassPhrase(s, p);
        /* 
         * immediately return because we go on with
         * initialization in the 2nd startup round.
         */
        return;
    }
    if (ssl_g_nInitCount > 2) {
        ssl_init_HandlePassPhraseOnRestart(s, p);
    }

    /*
     * Warn the user that he should use the session cache.
     * But we can operate without it, of course.
     */
    if (!ssl_g_nSessionCacheMode == SSL_SCMODE_UNSET) {
        ssl_log(s, SSL_LOG_WARN, 
                "Init: Session Cache is not configured [hint: SSLSessionCacheMode]");
        ssl_g_nSessionCacheMode = SSL_SCMODE_NONE;
    }

    /*
     *  initialize the mutex handling and session caching
     */
    ssl_mutex_init(s, p);
    ssl_scache_init(s, p);

    /*
     *  pre-generate the temporary RSA key
     */
    if (ssl_g_pRSATmpKey == NULL) {
        ssl_log(s, SSL_LOG_INFO, "Init: Generating temporary (512 bit) RSA key");
#if SSLEAY_VERSION_NUMBER >= 0x0900
        ssl_g_pRSATmpKey = __SSLeay RSA_generate_key(512, RSA_F4, NULL, NULL);
#else
        ssl_g_pRSATmpKey = __SSLeay RSA_generate_key(512, RSA_F4, NULL);
#endif
    }

    /*
     *  initialize servers
     */
    ssl_log(s, SSL_LOG_INFO, "Init: Initializing servers for SSL");
    for (; s != NULL; s = s->next) {
        pConfig = mySrvConfig(s);

        if (pConfig->bEnabled)
            ssl_log(s, SSL_LOG_INFO, 
                    "Init: Configuring server %s:%d for SSL protocol",
                    s->server_hostname, s->port);

        /* If we are disabled skip this server but
           make sure the port is initialized correctly */
        if (!pConfig->bEnabled) {
            if (!s->port)
                s->port = DEFAULT_HTTP_PORT;
            continue;
        }
        else {
            if (!s->port)
                s->port = DEFAULT_HTTPS_PORT;
        }

        /* 
         * Now check for important parameters and the
         * possibility that the user forgot to set them.
         */
        if (pConfig->nSessionCacheTimeout == 0) {
            ssl_log(s, SSL_LOG_ERROR,
                    "Init: No SSL Session Cache Timeout set [hint: SSLSessionCacheTimeout]");
            ap_clear_pool(p);
            ssl_die();
        }
        if (!pConfig->szCertificateFile) {
            ssl_log(s, SSL_LOG_ERROR,
                    "Init: No SSL Certificate set for server %s:%d [hint: SSLCertificateFile]",
                    s->server_hostname, s->port);
            ap_clear_pool(p);
            ssl_die();
        }

        /*
         * Read the server certificate and key
         */
        ssl_init_GetCertAndKey(s, p, pConfig);
    }

    /*
     * Optionally wipe out the PassPhrase if
     * `SSLPassPhraseCaching off' is active
     */
    pConfig = mySrvConfig(ssl_g_pServer);
    if (ssl_g_bPassPhrase) {
        if (   ssl_g_szPassPhrase != NULL
            && pConfig->bPassPhraseCaching == FALSE) {
            memset(buf, 0, (unsigned int)strlen(ssl_g_szPassPhrase));
            ssl_g_szPassPhrase = NULL;
            ssl_log(ssl_g_pServer, SSL_LOG_INFO, "Init: Wiping out pass phrase from memory");
        }
        else {
            ssl_log(ssl_g_pServer, SSL_LOG_INFO, "Init: Caching pass phrase in memory");
        }
    }

    /*
     *  Announce mod_ssl and SSLeay in HTTP Server field
     *  as ``mod_ssl/X.X.X SSLeay/X.X.X''
     */
    ap_snprintf(buf, sizeof(buf), "mod_ssl/%s", MOD_SSL_VERSION);
    ap_add_version_component(buf);
    cp = __SSLeay SSLeay_version(SSLEAY_VERSION);
    for (; !isdigit(*cp); cp++);
    ap_cpystrn(buf, "SSLeay/", sizeof(buf));
    for (cp2 = buf + strlen(buf); *cp != ' ' && *cp != NUL;)
        *cp2++ = *cp++;
    *cp2 = NUL;
    ap_add_version_component(ap_pstrdup(p, buf));
    ssl_log(ssl_g_pServer, SSL_LOG_INFO, "Init: Announce ourself as mod_ssl/%s %s",
            MOD_SSL_VERSION, buf);

    return;
}

void ssl_init_Child(server_rec *s, pool *p)
{
     /* open the mutex lockfile */
     ssl_mutex_open(s, p);
     return;
}

void ssl_init_HandlePassPhrase(server_rec *s, pool *p)
{
    server_rec *s2;
    SSLConfigRec *pConfig;
    char szPath[MAX_STRING_LEN];
    FILE *fp;
    RSA *prsaKey;

    /*  look for SSL servers  */
    ssl_g_bPassPhrase = FALSE;
    for (s2 = s; s2 != NULL; s2 = s2->next) {
        pConfig = mySrvConfig(s2);
        if (pConfig->bEnabled && pConfig->szCertificateFile) {

            /* 
             * Ok, this is an SSL server, so find the public key...
             */
            if (pConfig->szKeyFile != NULL) {
                if (*pConfig->szKeyFile == '/')
                    ap_cpystrn(szPath, pConfig->szKeyFile, sizeof(szPath));
                else
                    ap_snprintf(szPath, sizeof(szPath), "%s/private/%s", 
                                __SSLeay X509_get_default_cert_area(), 
                                pConfig->szKeyFile);
                if ((fp = ap_pfopen(p, szPath, "r")) == NULL) {
                    ssl_log(s, SSL_LOG_ERROR|SSL_ADD_ERRNO, 
                            "Can't open SSL server private key file %s, nor %s",
                            pConfig->szCertificateFile, szPath);
                    ap_clear_pool(p);
                    ssl_die();
                }
            }
            else {
                ap_cpystrn(szPath, pConfig->szCertificateFile, sizeof(szPath));     
                if ((fp = ap_pfopen(p, szPath, "r")) == NULL) {
                    ap_snprintf(szPath, sizeof(szPath), "%s/%s", 
                                __SSLeay X509_get_default_cert_dir(),
                                pConfig->szCertificateFile);
                    if ((fp = ap_pfopen(p, szPath, "r")) == NULL) {
                        ssl_log(s, SSL_LOG_ERROR|SSL_ADD_ERRNO, 
                                "Can't open SSL server certificate file %s, nor %s",
                                pConfig->szCertificateFile, szPath);
                        ap_clear_pool(p);
                        ssl_die();
                    }
                }
            }

            /*
             * Fine, try to read it and remember the pass phrase
             * in the callback function
             */
            prsaKey = __SSLeay RSA_new();
            if (!__SSLeay PEM_read_RSAPrivateKey(fp, &prsaKey, 
                                                 ssl_init_HandlePassPhrase_CB)) {
                ssl_log(s, SSL_LOG_ERROR|SSL_ADD_SSLEAY, 
                        "Unable to read private key file %s", szPath);
                ap_clear_pool(p);
                ssl_die();
            }
            ap_pfclose(p, fp);

            if (ssl_g_bPassPhrase) {
                break;
            }
        }
    }
    return;
}

#define STDERR_FILENO_STORE 10

int ssl_init_HandlePassPhrase_CB(char *buf, int bufsize, int w)
{
    int i;
    int len = -1;
    SSLConfigRec *pConfig = mySrvConfig(ssl_g_pServer);
    pool *p = ssl_g_pPool;

    /*
     * Builtin dialog
     */
    if (pConfig->nPassPhraseDialogType == SSL_PPTYPE_BUILTIN) {
        char *prompt;

        ssl_log(ssl_g_pServer, SSL_LOG_INFO, "Init: Requesting pass phrase via builtin terminal dialog");

        /* 
         * Reconnect STDERR to terminal (here STDOUT) because
         * at our init stage Apache already connected STDERR
         * to the general error logfile.
         */
        dup2(STDERR_FILENO, STDERR_FILENO_STORE);
        dup2(STDOUT_FILENO, STDERR_FILENO);

        /*
         * Display a header to inform the user about what program
         * he actually speaks to, which modules is responsible
         * for this terminal dialog and why to the hell he
         * has to enter something...
         */
        fprintf(stderr, "%s mod_ssl/%s (Startup Dialog)\n",
                SERVER_BASEVERSION, MOD_SSL_VERSION);
        fprintf(stderr, "Your private key files are protected for security reasons.\n");

        /* 
         * Emulate the SSLeay internal pass phrase dialog
         * (see crypto/pem/pem_lib.c:def_callback() for details)
         */
        prompt = "Enter PEM pass phrase:";
        for (;;) {
            if ((i = EVP_read_pw_string(buf, bufsize, prompt, w)) != 0) {
                PEMerr(PEM_F_DEF_CALLBACK,PEM_R_PROBLEMS_GETTING_PASSWORD);
                memset(buf, 0, (unsigned int)bufsize);
                return (-1);
            } 
            len = strlen(buf);
            if (len < 4)
                fprintf(stderr, "Error: Phrase is too short, needs to be at least %d chars.\n", 4);
            else
                break;
        }

        /*
         * Restore STDERR to Apache error logfile
         */
        dup2(STDERR_FILENO_STORE, STDERR_FILENO);

    }

    /*
     * Filter program
     */
    else if (pConfig->nPassPhraseDialogType == SSL_PPTYPE_FILTER) {
        char *cmd;
        char *result;

        ssl_log(ssl_g_pServer, SSL_LOG_INFO, 
                "Init: Requesting pass phrase from dialog filter program (%s)",
                pConfig->szPassPhraseDialogPath);

        cmd = ap_psprintf(p, "%s", pConfig->szPassPhraseDialogPath);
        result = ssl_util_readfilter(ssl_g_pServer, p, cmd);
        ap_cpystrn(buf, result, bufsize);
        len = strlen(buf);
    }

    /*
     * Ok, we now have the pass phrase, so store it!
     * (at least temporarily, because it is destroyed later
     *  when `SSLPassPhraseCaching off' is active)
     */
    ssl_g_szPassPhrase = strdup(buf); /* REALLY strdup here, not ap_pstrdup! */
    ssl_g_bPassPhrase  = TRUE;

    return (len);
}

void ssl_init_HandlePassPhraseOnRestart(server_rec *s, pool *p)
{
    /* 
     * we only have to do something when a pass phrase is needed
     * and it isn't chached.
     */
    if (ssl_g_bPassPhrase) {
        SSLConfigRec *pConfig = mySrvConfig(s);
        if (pConfig->bPassPhraseCaching == FALSE) {
            /*
             * Ok, a pass phrase is needed but wasn't cached so
             * our only chance now is a dialog filter, because
             * the builtin terminal dialog no longer can be used
             * (Apache is detached)
             */
            if (pConfig->nPassPhraseDialogType == SSL_PPTYPE_FILTER
                && pConfig->szPassPhraseDialogPath != NULL         ) {
                /*
                 * Fine, let's re-run the filter...
                 */
                char *cmd;
                char *result;

                ssl_log(ssl_g_pServer, SSL_LOG_INFO, 
                        "Init: Requesting pass phrase (again) from dialog filter program",
                        pConfig->szPassPhraseDialogPath);

                cmd = ap_psprintf(p, "%s", pConfig->szPassPhraseDialogPath);
                result = ssl_util_readfilter(ssl_g_pServer, p, cmd);
                ssl_g_szPassPhrase = strdup(result); /* REALLY strdup here, not ap_pstrdup! */
            }
        }
    }
    return;
}

/*
 * Read the SSL Server Certificate and Key
 */
void ssl_init_GetCertAndKey(server_rec *s, pool *p, SSLConfigRec *pConfig)
{
    FILE *f;
    char szPath[MAX_STRING_LEN];
    int nVerify;

    /*  
     *  Check for problematic re-initializations
     */
    if (pConfig->px509Certificate) {
        ssl_log(s, SSL_LOG_ERROR, 
                "Init: Illegal attempt to re-initialise SSL for server %s:%d",
                s->server_hostname, s->port);
        ap_clear_pool(p);
        ssl_die();
    }

    /*
     *  Calculate SSLeay verify type
     */
    nVerify = SSL_VERIFY_NONE;
    switch (pConfig->nVerifyClient) {
        case VERIFY_REQUIRE:
            nVerify |= SSL_VERIFY_PEER|SSL_VERIFY_FAIL_IF_NO_PEER_CERT;
            break;
        case VERIFY_OPTIONAL:
        case VERIFY_OPTIONAL_NO_CA:
            nVerify |= SSL_VERIFY_PEER;
            break;
        default:
            break;
    }

    /*
     *  Create new SSL context and configure callbacks
     */
    ssl_log(s, SSL_LOG_INFO, "Init: Creating new SSL context for server %s:%d",
            s->server_hostname, s->port);
    pConfig->pSSLCtx = __SSLeay SSL_CTX_new(__SSLeay SSLv23_server_method());
    __SSLeay SSL_CTX_set_verify(pConfig->pSSLCtx, nVerify, 
                 ssl_callback_SSLVerify);
    __SSLeay SSL_CTX_sess_set_new_cb(pConfig->pSSLCtx, 
                 ssl_callback_NewSessionCacheEntry);
    __SSLeay SSL_CTX_sess_set_get_cb(pConfig->pSSLCtx, 
                 ssl_callback_GetSessionCacheEntry);
    __SSLeay SSL_CTX_sess_set_remove_cb(pConfig->pSSLCtx, 
                 ssl_callback_DelSessionCacheEntry);
    __SSLeay SSL_CTX_set_tmp_rsa_callback(pConfig->pSSLCtx, 
                 ssl_callback_TmpRSA);

    /*
     *  Configure required SSL Ciphers 
     */
    ssl_log(s, SSL_LOG_INFO, "Init: Configuring permitted SSL ciphers for server %s:%d",
            s->server_hostname, s->port);
    if (pConfig->szCipherSuite != NULL) {
        if (!__SSLeay SSL_CTX_set_cipher_list(pConfig->pSSLCtx, 
                                              pConfig->szCipherSuite)) {
            ssl_log(s, SSL_LOG_ERROR|SSL_ADD_SSLEAY, 
                    "Unable to configure permitted SSL ciphers");
            ap_clear_pool(p);
            ssl_die();
        }
    }

    /*
     * Configure SSL Client verification
     */
    ssl_log(s, SSL_LOG_INFO, "Init: Configuring client verification paths for server %s:%d",
            s->server_hostname, s->port);
    if (   ( (   pConfig->szCACertificateFile 
              || pConfig->szCACertificatePath)
           && !__SSLeay SSL_CTX_load_verify_locations(pConfig->pSSLCtx,
                                                      pConfig->szCACertificateFile,
                                                      pConfig->szCACertificatePath) )
        || !__SSLeay SSL_CTX_set_default_verify_paths(pConfig->pSSLCtx) ) {
        ssl_log(s, SSL_LOG_ERROR|SSL_ADD_SSLEAY, 
                "Unable to configure SSL verify locations");
        ap_clear_pool(p);
        ssl_die();
    }

    /*
     *  Read server certificate
     */
    ssl_log(s, SSL_LOG_INFO, "Init: Reading certificate file for server %s:%d",
            s->server_hostname, s->port);
    ap_cpystrn(szPath, pConfig->szCertificateFile, sizeof(szPath));     
    if ((f = ap_pfopen(p, szPath, "r")) == NULL) {
        ap_snprintf(szPath, sizeof(szPath), "%s/%s", 
                    __SSLeay X509_get_default_cert_dir(), 
                    pConfig->szCertificateFile);
        if ((f = ap_pfopen(p, szPath, "r")) == NULL) {
            ssl_log(s, SSL_LOG_ERROR|SSL_ADD_ERRNO, 
                    "Can't open SSL server certificate file %s, nor %s",
                    pConfig->szCertificateFile, szPath);
            ap_clear_pool(p);
            ssl_die();
        }
    }
    pConfig->px509Certificate = __SSLeay X509_new();
    if (!__SSLeay PEM_read_X509(f, &pConfig->px509Certificate, 
                                ssl_callback_PassPhrase)) {
        ssl_log(s, SSL_LOG_ERROR|SSL_ADD_SSLEAY, 
                "Unable to read SSL server certificate file %s", szPath);
        ap_clear_pool(p);
        ssl_die();
    }
    ap_pfclose(p, f);

    /*
     *  Read server private key 
     */
    ssl_log(s, SSL_LOG_INFO, "Init: Reading private key file for server %s:%d",
            s->server_hostname, s->port);
    if (pConfig->szKeyFile)
        if (*pConfig->szKeyFile == '/')
            ap_cpystrn(szPath, pConfig->szKeyFile, sizeof(szPath));
        else
            ap_snprintf(szPath, sizeof(szPath), "%s/private/%s", 
                    __SSLeay X509_get_default_cert_area(), pConfig->szKeyFile);
    /* Otherwise the path already contains the name of the certificate file */
    if ((f = ap_pfopen(p, szPath, "r")) == NULL) {
        ssl_log(s, SSL_LOG_ERROR|SSL_ADD_ERRNO, 
                "Can't open private key file %s", szPath);
        ap_clear_pool(p);
        ssl_die();
    }
    pConfig->prsaKey = __SSLeay RSA_new();
    if (!__SSLeay PEM_read_RSAPrivateKey(f, &pConfig->prsaKey, 
                                         ssl_callback_PassPhrase)) {
        ssl_log(s, SSL_LOG_ERROR|SSL_ADD_SSLEAY, 
                "Unable to read private key file %s", szPath);
        ap_clear_pool(p);
        ssl_die();
    }
    ap_pfclose(p, f);

    /* 
     *  Additionally load certificates which will be sent to the
     *  client on `SSLv3 write certificate request A'. This is
     *  optionally used by the clients to speedup the server
     *  authentication when SSLv3 certificate chaining is ised.
     *  Here the client loads intermediate certificates in the
     *  chain from the server. This defaults to
     *  SSLCACertificateFile. 
     */
    if (   pConfig->szCACertificateReqFile != NULL
        || pConfig->szCACertificateFile    != NULL)
        ssl_log(s, SSL_LOG_INFO, "Init: Reading CA certification file for server %s:%d",
                s->server_hostname, s->port);
    if (pConfig->szCACertificateReqFile != NULL)
        __SSLeay SSL_CTX_set_client_CA_list(pConfig->pSSLCtx,
            __SSLeay SSL_load_client_CA_file(pConfig->szCACertificateReqFile));
    else
        if (pConfig->szCACertificateFile != NULL)
            __SSLeay SSL_CTX_set_client_CA_list(pConfig->pSSLCtx,
                __SSLeay SSL_load_client_CA_file(pConfig->szCACertificateFile));

    return;
}


