/* ====================================================================
 * Copyright (c) 1998 Ralf S. Engelschall. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * 4. The names "mod_ssl" must not be used to endorse or promote
 *    products derived from this software without prior written
 *    permission. For written permission, please contact
 *    rse@engelschall.com.
 *
 * 5. Products derived from this software may not be called "mod_ssl"
 *    nor may "mod_ssl" appear in their names without prior
 *    written permission of Ralf S. Engelschall.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * THIS SOFTWARE IS PROVIDED BY RALF S. ENGELSCHALL ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RALF S. ENGELSCHALL OR
 * HIS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */


#include "mod_ssl.h"


/*  _________________________________________________________________
**
**  Session Cache Support (Common)
**  _________________________________________________________________
*/

/*  
 *  FIXME: There is no define in SSLeay, but SSLeay uses 1024*10,
 *  so 1024*20 should be ok.
 */
#define MAX_SESSION_DER 1024*20

void ssl_scache_init(server_rec *s, pool *p)
{
    if (ssl_g_nSessionCacheMode == SSL_SCMODE_DBM)
        ssl_scache_dbm_init(s, p);
    ssl_scache_expire();
    return;
}

void ssl_scache_store(SSL_SESSION *pSession, int timeout)
{
    SessionCacheInfo SCI;
    uchar buf[MAX_SESSION_DER];
    uchar *b;

    /* add the key */
    SCI.ucaKey = pSession->session_id;
    SCI.nKey   = pSession->session_id_length;

    /* transform the session into a data stream */
    SCI.ucaData    = b = buf;
    SCI.nData      = __SSLeay i2d_SSL_SESSION(pSession, &b);
    SCI.tExpiresAt = timeout;

    /* and store it... */
    if (ssl_g_nSessionCacheMode == SSL_SCMODE_DBM)
        ssl_scache_dbm_store(&SCI);

    return;
}

SSL_SESSION *ssl_scache_retrieve(uchar *id, int idlen)
{
    SSL_SESSION *pSession = NULL;
    SessionCacheInfo SCI;
    time_t tNow;

    /* create cache query */
    SCI.ucaKey     = id;
    SCI.nKey       = idlen;
    SCI.ucaData    = NULL;
    SCI.nData      = 0;
    SCI.tExpiresAt = 0;

    /* perform cache query */
    if (ssl_g_nSessionCacheMode == SSL_SCMODE_DBM)
        ssl_scache_dbm_retrieve(&SCI);

    /* return immediately if not found */
    if (SCI.ucaData == NULL)
        return NULL;

    /* check for expire time */
    tNow = time(NULL);
    if (SCI.tExpiresAt <= tNow) {
        if (ssl_g_nSessionCacheMode == SSL_SCMODE_DBM)
            ssl_scache_dbm_remove(&SCI);
        return NULL;
    }

    /* extract result and return it */
    pSession = __SSLeay d2i_SSL_SESSION(NULL, &SCI.ucaData, SCI.nData);
    return pSession;
}

void ssl_scache_remove(SSL_SESSION *pSession)
{
    SessionCacheInfo SCI;

    /* create cache query */
    SCI.ucaKey     = pSession->session_id;
    SCI.nKey       = pSession->session_id_length;
    SCI.ucaData    = NULL;
    SCI.nData      = 0;
    SCI.tExpiresAt = 0;

    /* perform remove */
    if (ssl_g_nSessionCacheMode == SSL_SCMODE_DBM)
        ssl_scache_dbm_remove(&SCI);

    return;
}

void ssl_scache_expire(void)
{
    if (ssl_g_nSessionCacheMode == SSL_SCMODE_DBM)
        ssl_scache_dbm_expire();
    return;
}

char *ssl_scache_id2sz(uchar *id, int idlen)
{
    static char str[(SSL_MAX_SSL_SESSION_ID_LENGTH+1)*2];
    char *cp;
    int n;

    cp = str;
    for (n = 0; n < idlen && n < SSL_MAX_SSL_SESSION_ID_LENGTH; n++) {
        ap_snprintf(cp, sizeof(str)-(cp-str), "%02X", id[n]);
        cp += 2;
    }
    *cp = NUL;
    return str;
}


/*  _________________________________________________________________
**
**  Session Cache Support (DBM)
**  _________________________________________________________________
*/

#define DBM_FILE_MODE ( S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH )

void ssl_scache_dbm_init(server_rec *s, pool *p)
{
    DBM *dbm;

    /*
     * for the DBM we need the data file
     */
    if (ssl_g_szSessionCacheDataFile == NULL) {
        ssl_log(s, SSL_LOG_ERROR, "SSLSessionCacheDataFile required");
        ssl_die();
    }

    /*
     * Open it once to create it and to make sure it
     * _can_ be created.
     */
    ssl_mutex_on();
    if ((dbm = sdbm_open(ssl_g_szSessionCacheDataFile, 
                        O_WRONLY|O_CREAT, DBM_FILE_MODE)) == NULL) {
        ssl_log(s, SSL_LOG_ERROR|SSL_ADD_ERRNO,
                "Cannot create SSLSessionCacheDataFile %s as a DBM file",
                ssl_g_szSessionCacheDataFile);
        ssl_mutex_off();
        return;
    }
    sdbm_close(dbm);
    ssl_mutex_off();

    return;
}

void ssl_scache_dbm_store(SessionCacheInfo *SCI)
{
    DBM *dbm;
    datum dbmkey;
    datum dbmval;

    /* create DBM key */
    dbmkey.dptr  = SCI->ucaKey;
    dbmkey.dsize = SCI->nKey;

    /* create DBM value */
    dbmval.dsize = sizeof(SCI->tExpiresAt)+SCI->nData;
    dbmval.dptr  = (uchar *)malloc(dbmval.dsize);
    memcpy(dbmval.dptr, &SCI->tExpiresAt, sizeof(SCI->tExpiresAt));
    memcpy(dbmval.dptr+sizeof(SCI->tExpiresAt), SCI->ucaData, SCI->nData);

    /* and store it to the DBM file */
    ssl_mutex_on();
    if ((dbm = sdbm_open(ssl_g_szSessionCacheDataFile, 
                        O_RDWR, DBM_FILE_MODE)) == NULL) {
        ssl_log(ssl_g_pCurrentConnection->server, SSL_LOG_ERROR|SSL_ADD_ERRNO,
                "Cannot open DBM file SSLSessionCacheDataFile %s for writing",
                ssl_g_szSessionCacheDataFile);
        ssl_mutex_off();
        return;
    }
    sdbm_store(dbm, dbmkey, dbmval, DBM_INSERT);
    sdbm_close(dbm);
    ssl_mutex_off();

    /* free temporary buffers */
    free(dbmval.dptr);

    return;
}

void ssl_scache_dbm_retrieve(SessionCacheInfo *SCI)
{
    DBM *dbm;
    datum dbmkey;
    datum dbmval;

    /* initialize result */
    SCI->ucaData    = NULL;
    SCI->nData      = 0;
    SCI->tExpiresAt = 0;

    /* create DBM key and values */
    dbmkey.dptr  = SCI->ucaKey;
    dbmkey.dsize = SCI->nKey;

    /* and fetch it from the DBM file */
    ssl_mutex_on();
    if ((dbm = sdbm_open(ssl_g_szSessionCacheDataFile, 
                        O_RDONLY, DBM_FILE_MODE)) == NULL) {
        ssl_log(ssl_g_pCurrentConnection->server, SSL_LOG_ERROR|SSL_ADD_ERRNO,
                "Cannot open DBM file SSLSessionCacheDataFile %s for reading",
                ssl_g_szSessionCacheDataFile);
        ssl_mutex_off();
        return;
    }
    dbmval = sdbm_fetch(dbm, dbmkey);
    sdbm_close(dbm);
    ssl_mutex_off();

    /* immediately return if not found */
    if (dbmval.dptr == NULL)
        return;

    /* copy over the information to the SCI */
    SCI->nData   = dbmval.dsize-sizeof(SCI->tExpiresAt);
    SCI->ucaData = (uchar *)malloc(dbmval.dsize-sizeof(SCI->tExpiresAt));
    memcpy(SCI->ucaData, dbmval.dptr+sizeof(SCI->tExpiresAt), 
           dbmval.dsize-sizeof(SCI->tExpiresAt));
    memcpy(&SCI->tExpiresAt, dbmval.dptr, sizeof(SCI->tExpiresAt));

    return;
}

void ssl_scache_dbm_remove(SessionCacheInfo *SCI)
{
    DBM *dbm;
    datum dbmkey;

    /* create DBM key and values */
    dbmkey.dptr  = SCI->ucaKey;
    dbmkey.dsize = SCI->nKey;

    /* and delete it from the DBM file */
    ssl_mutex_on();
    if ((dbm = sdbm_open(ssl_g_szSessionCacheDataFile, 
                        O_WRONLY, DBM_FILE_MODE)) == NULL) {
        ssl_log(ssl_g_pCurrentConnection->server, SSL_LOG_ERROR|SSL_ADD_ERRNO,
                "Cannot open DBM file SSLSessionCacheDataFile %s for writing",
                ssl_g_szSessionCacheDataFile);
        ssl_mutex_off();
        return;
    }
    sdbm_delete(dbm, dbmkey);
    sdbm_close(dbm);
    ssl_mutex_off();

    return;
}

void ssl_scache_dbm_expire(void)
{
    static int nExpireCalls = 0;
    DBM *dbm;
    datum dbmkey;
    datum dbmval;
    time_t tNow;
    time_t tExpiresAt;

    /*
     * It's to expensive to expire allways,
     * so do it only from time to time...
     */
    if (nExpireCalls++ < 100)
        return;
    else
        nExpireCalls = 0;

    ssl_mutex_on();
    if ((dbm = sdbm_open(ssl_g_szSessionCacheDataFile, 
                         O_RDWR, DBM_FILE_MODE)) == NULL) {
        ssl_log(ssl_g_pCurrentConnection->server, SSL_LOG_ERROR|SSL_ADD_ERRNO,
                "Cannot open DBM file SSLSessionCacheDataFile %s for expiring",
                ssl_g_szSessionCacheDataFile);
        ssl_mutex_off();
        return;
    }
    tNow = time(NULL);
    dbmkey = sdbm_firstkey(dbm);
    for ( ; dbmkey.dptr != NULL; dbmkey = sdbm_nextkey(dbm)) {
        dbmval = sdbm_fetch(dbm, dbmkey);
        memcpy(&tExpiresAt, dbmval.dptr, sizeof(tExpiresAt));
        if (tExpiresAt >= tNow)
            sdbm_delete(dbm, dbmkey);
    }
    sdbm_close(dbm);
    ssl_mutex_off();

    return;
}

