/* ====================================================================
 * Copyright (c) 1998 Ralf S. Engelschall. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * 4. The names "mod_ssl" must not be used to endorse or promote
 *    products derived from this software without prior written
 *    permission. For written permission, please contact
 *    rse@engelschall.com.
 *
 * 5. Products derived from this software may not be called "mod_ssl"
 *    nor may "mod_ssl" appear in their names without prior
 *    written permission of Ralf S. Engelschall.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * THIS SOFTWARE IS PROVIDED BY RALF S. ENGELSCHALL ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RALF S. ENGELSCHALL OR
 * HIS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */


#include "mod_ssl.h"


/*  _________________________________________________________________
**
**  SSL Engine Kernel
**  _________________________________________________________________
*/

/*
 *  Connect Handler:
 *  Connect SSL to the accepted socket
 *
 *  Usually we would need an Apache API hook which is triggered right after
 *  the socket is accepted for handling a new request. But Apache 1.3 doesn't
 *  provide such a hook, so we have to patch http_main.c and call this
 *  function directly.
 */
int SSL_HOOK_SetupConnection(conn_rec *conn)
{
    server_rec *srvr;
    BUFF *fb;
    SSLSrvConfigRec *pConfig;

    /*
     * Get context
     */
    srvr    = conn->server;
    fb      = conn->client;
    pConfig = mySrvConfig(srvr);

    /*
     * Immediately stop processing if SSL 
     * is disabled for this connection
     */
    if (pConfig == NULL || !pConfig->bEnabled) {
        fb->ssl = NULL;
        return TRUE;
    }

    /*
     * Remember the connection information for
     * later access inside callback functions
     */
    ssl_log(srvr, SSL_LOG_INFO, "Connection to child %d established (server %s:%d)",
            conn->child_num, srvr->server_hostname, srvr->port);

    /*
     * Globally remember some connection context
     */
    ssl_ModConfig->rCtx.pConn = conn;
    ssl_ModConfig->rCtx.pServ = conn->server;
    ssl_ModConfig->rCtx.pPool = conn->pool;

    /*
     * Create a new SSL connection with the configured server SSL context and
     * attach this to the socket. Additionally we register this attachment
     * so we can detach later.
     */
    fb->ssl = SSL_new(pConfig->pSSLCtx);
    SSL_set_fd(fb->ssl, fb->fd);
    ap_register_cleanup(conn->pool, (void *)conn, 
                        ssl_hook_CloseConnection, ssl_hook_CloseConnection);

    /*
     * Configure the server certificate and private key 
     * which should be used for this connection.
     */
    if (pConfig->szCertificateFile != NULL) {
        if (SSL_use_certificate(fb->ssl, pConfig->px509Certificate) <= 0) {
            ssl_log(conn->server, SSL_LOG_ERROR|SSL_ADD_SSLEAY,
                    "Unable to configure server certificate for connection");
            SSL_free(fb->ssl);
            fb->ssl = NULL;
            fb->flags |= B_EOF|B_EOUT;
            return FALSE;
        }
        if (SSL_use_RSAPrivateKey(fb->ssl, pConfig->prsaKey) <= 0) {
            ssl_log(conn->server, SSL_LOG_ERROR|SSL_ADD_SSLEAY,
                    "Unable to configure server private key for connection");
            SSL_free(fb->ssl);
            fb->ssl = NULL;
            fb->flags |= B_EOF|B_EOUT;
            return FALSE;
        }
    }

    /*
     * Now do the SSL handshake (or re-negotiation)
     */
    conn->client->szClientX509 = NULL;
    conn->client->nVerifyError = X509_V_OK;

    while (!SSL_is_init_finished(fb->ssl)) {

        if (SSL_accept(fb->ssl) <= 0) {
            ssl_log(srvr, SSL_LOG_ERROR|SSL_ADD_SSLEAY, 
                    "SSL connection acception failed");

            /*
             * Check for the case where SSLeay has recognized a HTTP request
             * line which means the client speaks plain HTTP on our HTTPS
             * port. Hmmmm...  At least for this error we can be more friendly
             * and try to provide him with a HTML error page. We have only one
             * problem: SSLeay has already read some bytes from the HTTP
             * request. So we have to skip the request line manually and
             * instead provide a faked one in order to continue the internal
             * Apache processing.
             */
            if (ERR_GET_REASON(ERR_peek_error()) == SSL_R_HTTP_REQUEST) {
                char ca[2];
                int rv;

                /* first: skip the remaining bytes of the request line */
                do {
                    do {
                        rv = read(fb->fd, ca, 1); 
                    } while (rv == -1 && errno == EINTR);
                } while (rv > 0 && ca[0] != '\012' /*LF*/);

                /* second: fake the request line */
                fb->inbase = ap_palloc(fb->pool, fb->bufsiz);
                ap_cpystrn(fb->inbase, "GET /mod_ssl:error:HTTP-request HTTP/1.0\r\n", 
                           fb->bufsiz);
                fb->inptr = fb->inbase;
                fb->incnt = strlen(fb->inptr);

                /* third: kick away the SSL stuff */
                SSL_set_shutdown(fb->ssl, SSL_SENT_SHUTDOWN|SSL_RECEIVED_SHUTDOWN); 
                while (!SSL_shutdown(fb->ssl));
                SSL_free(fb->ssl);
                fb->ssl = NULL;

                /* finally: let Apache go on with processing */
                return TRUE;
            }

            /* 
             * Ok, try to gracefully shutdown the connection:
             * - send an own shutdown message (be gracefully)
             * - don't wait for peer's shutdown message (deadloop)
             * - kick away the SSL stuff immediately
             * - block the socket, so Apache cannot operate any more
             */
            SSL_set_shutdown(fb->ssl, SSL_RECEIVED_SHUTDOWN); 
            while (!SSL_shutdown(fb->ssl));
            SSL_free(fb->ssl);
            fb->ssl = NULL;
            fb->flags |= B_EOF|B_EOUT;
            return FALSE;
        }

        /*
         * Check for failed client authentication
         */
        if (conn->client->nVerifyError != X509_V_OK) {
            ssl_log(srvr, SSL_LOG_ERROR|SSL_ADD_SSLEAY, 
                    "SSL client authentication failed");
            SSL_set_shutdown(fb->ssl, SSL_RECEIVED_SHUTDOWN); 
            while (!SSL_shutdown(fb->ssl));
            SSL_free(fb->ssl);
            fb->ssl = NULL;
            fb->flags |= B_EOF|B_EOUT;
            return FALSE;
        }

        /*
         * Remember the peer certificate when 
         * client authentication was done
         */
        if (pConfig->nVerifyClient != VERIFY_NONE) {
            char *s;
            X509 *xs;
            if ((xs = SSL_get_peer_certificate(fb->ssl)) != NULL) {
                s = X509_NAME_oneline(X509_get_subject_name(xs), NULL, 0);
                conn->client->szClientX509 = ap_pstrdup(conn->pool, s);
                free(s);
            }
        }

        /*
         * Make really sure that when a peer certificate
         * is required we really got one... (be paranoid)
         */
        if (   pConfig->nVerifyClient == VERIFY_REQUIRE
            && conn->client->szClientX509 == NULL      ) {
            ssl_log(srvr, SSL_LOG_ERROR, 
                    "No acceptable peer certificate available");
            SSL_set_shutdown(fb->ssl, SSL_RECEIVED_SHUTDOWN); 
            while (!SSL_shutdown(fb->ssl));
            SSL_free(fb->ssl);
            fb->ssl = NULL;
            fb->flags |= B_EOF|B_EOUT;
            return FALSE;
        }

        /*
         * Finally log the connection details
         */
        ssl_log(srvr, SSL_LOG_INFO, 
                "Client IP: %s, Protocol: %s, Cipher: %s (%s/%s bits)", 
                ssl_var_lookup(NULL, NULL, conn, NULL, "REMOTE_ADDR"),
                ssl_var_lookup(NULL, NULL, conn, NULL, "SSL_PROTOCOL"),
                ssl_var_lookup(NULL, NULL, conn, NULL, "SSL_CIPHER"),
                ssl_var_lookup(NULL, NULL, conn, NULL, "SSL_CIPHER_USEKEYSIZE"),
                ssl_var_lookup(NULL, NULL, conn, NULL, "SSL_CIPHER_ALGKEYSIZE"));
    }

    /* 
     * This should be a safe optimization...
     */
    SSL_set_read_ahead(fb->ssl, 1);

    return TRUE;
}

/*
 *  Close the SSL part of the socket connection
 */
void ssl_hook_CloseConnection(void *_conn)
{
    conn_rec *conn = _conn;
    SSL *ssl;

    /*
     * Optionally shutdown the SSL session
     */
    ssl = conn->client->ssl;
    if (ssl != NULL) {
        SSL_set_shutdown(ssl, SSL_SENT_SHUTDOWN|SSL_RECEIVED_SHUTDOWN);
        while (!SSL_shutdown(ssl));
        SSL_free(ssl);
        conn->client->ssl = NULL;
    }

    /* 
     * Delete connection context
     */
    ssl_ModConfig->rCtx.pConn = NULL;
    ssl_ModConfig->rCtx.pServ = NULL;
    ssl_ModConfig->rCtx.pPool = NULL;

    /*
     * And finally log the connection close
     */
    ssl_log(conn->server, SSL_LOG_INFO, "Connection to child %d closed (server %s:%d)",
            conn->child_num, conn->server->server_hostname, conn->server->port);

    return;
}

/*
 *  Post Read Request Handler
 */
int ssl_hook_ReadReq(request_rec *r)
{
    if (strEQn(r->uri, "/mod_ssl:", 9)) 
        r->handler = "mod_ssl:content-handler";
    return DECLINED;
}

/*
 *  Content Handler
 */
int ssl_hook_Handler(request_rec *r)
{
    int port;
    char *thisport;
    char *thisurl;

    if (strNEn(r->uri, "/mod_ssl:", 9))
        return DECLINED;

    if (strEQ(r->uri, "/mod_ssl:error:HTTP-request")) {
        thisport = "";
        port = ap_get_server_port(r);
        if (!ap_is_default_port(port, r))
            thisport = ap_psprintf(r->pool, ":%u", port);
        thisurl = ap_psprintf(r->pool, "https://%s%s/",
                              ap_get_server_name(r), thisport);

        ap_table_setn(r->notes, "error-notes", ap_psprintf(r->pool,
                      "You're speaking plain HTTP to an SSL-enabled server.<BR>\n"
                      "Instead use HTTPS protocol to access this server, please.<BR>\n"
                      "<BLOCKQUOTE>Hint: <A HREF=\"%s\">%s</A></BLOCKQUOTE>", 
                      thisurl, thisurl));
        r->status_line = "500 SSL Engine Error";
    }

    return HTTP_INTERNAL_SERVER_ERROR;
}

/*
 *  Auth Handler:
 *  Fake a Basic authentication from the X509 client certificate.
 *
 *  This must be run fairly early on to prevent a real authentication from
 *  occuring, in particular it must be run before anything else that
 *  authenticates a user.  This means that the Module statement for this
 *  module should be LAST in the Configuration file.
 */
int ssl_hook_Auth(request_rec *r)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(r->server);
    char b1[MAX_STRING_LEN], b2[MAX_STRING_LEN];

    /*
     * We decline operation in various situations..
     */
    if (OPTIONAL_SSL)
        return DECLINED;
    if (!pConfig->bFakeBasicAuth)
        return DECLINED;
    if (r->connection->user)
        return DECLINED;
    if (!r->connection->client->szClientX509)
        return DECLINED;
    
    /*
     * Fake a password - which one would be immaterial, as, it seems, an empty
     * password in the users file would match ALL incoming passwords, if only
     * we were using the standard crypt library routine. Unfortunately, SSLeay
     * "fixes" a "bug" in crypt and thus prevents blank passwords from
     * working.  (IMHO what they really fix is a bug in the users of the code
     * - failing to program correctly for shadow passwords).  We need,
     * therefore, to provide a password. This password can be matched by
     * adding the string "xxj31ZMTZzkVA" as the password in the user file.
     * This is just the crypted variant of the word "password" ;-)
     */
    ap_snprintf(b1, sizeof(b1), "%s:password", 
                r->connection->client->szClientX509);
    ssl_util_uuencode(b2, b1);
    ap_snprintf(b1, sizeof(b1), "Basic %s", b2);
    ap_table_set(r->headers_in, "Authorization", b1);

    return DECLINED;
}

/*
 *  Access Handler
 */
int ssl_hook_Access(request_rec *r)
{
    SSLDirConfigRec *rec  = myDirConfig(r);
    SSLSrvConfigRec *pConfig = mySrvConfig(r->server);
    array_header *apRequirement;
    SSLRequirement *pRequirements;
    SSLRequirement *pRequirement;
    char *cp;
    int ok;
    int i;

    /*
     * Support for SSLRequireSSL directive
     */
    if (rec->bSSLRequired && !r->connection->client->ssl) {
        ap_log_reason("SSL connection required", r->filename, r);
        return FORBIDDEN;
    }

    /* 
     * Check to see if SSL protocol is on 
     */
    if (!pConfig->bEnabled || OPTIONAL_SSL)
        return DECLINED;

    /*
     * Check SSLRequire boolean expressions
     */
    apRequirement = rec->aRequirement;
    pRequirements  = (SSLRequirement *)apRequirement->elts;
    for (i = 0; i < apRequirement->nelts; i++) {
        pRequirement = &pRequirements[i];
        ok = ssl_expr_exec(r, pRequirement->mpExpr);
        if (ok < 0) {
            cp = ap_psprintf(r->pool, "Failed to execute SSL requirement expression: %s",
                             ssl_expr_get_error());
            ap_log_reason(cp, r->filename, r);
            return FORBIDDEN;
        }
        if (ok != 1) {
            ssl_log(r->server, SSL_LOG_INFO, 
                    "Access to %s denied for %s (requirement expression not fulfilled)",
                    r->filename, r->connection->remote_ip);
            ssl_log(r->server, SSL_LOG_INFO,
                    "Failed expression: %s", pRequirement->cpExpr);
            ap_log_reason("SSL requirement expression not fulfilled "
                          "(see SSL logfile for more details)", r->filename, r);
            return FORBIDDEN;
        }
    }

    /*
     * Else access is granted...
     */
    return OK;
}

/*
 *   Fixup Handler
 */

static const char *ssl_hook_Fixup_vars[] = {
    "SSL_VERSION_INTERFACE",
    "SSL_VERSION_LIBRARY",
    "SSL_PROTOCOL",
    "SSL_CIPHER",
    "SSL_CIPHER_USEKEYSIZE",
    "SSL_CIPHER_ALGKEYSIZE",
    "SSL_CLIENT_M_VERSION",
    "SSL_CLIENT_M_SERIAL",
    "SSL_CLIENT_V_START",
    "SSL_CLIENT_V_END",
    "SSL_CLIENT_S_DN",
    "SSL_CLIENT_S_DN_C",
    "SSL_CLIENT_S_DN_SP",
    "SSL_CLIENT_S_DN_L",
    "SSL_CLIENT_S_DN_O",
    "SSL_CLIENT_S_DN_OU",
    "SSL_CLIENT_S_DN_CN",
    "SSL_CLIENT_S_DN_Email",
    "SSL_CLIENT_I_DN",
    "SSL_CLIENT_I_DN_C",
    "SSL_CLIENT_I_DN_SP",
    "SSL_CLIENT_I_DN_L",
    "SSL_CLIENT_I_DN_O",
    "SSL_CLIENT_I_DN_OU",
    "SSL_CLIENT_I_DN_CN",
    "SSL_CLIENT_I_DN_Email",
    "SSL_CLIENT_A_KEY",
    "SSL_CLIENT_A_SIG",
#if 0
    "SSL_CLIENT_CERT",
    "SSL_CLIENT_CERT_CHAIN_0",
    "SSL_CLIENT_CERT_CHAIN_1",
    "SSL_CLIENT_CERT_CHAIN_2",
    "SSL_CLIENT_CERT_CHAIN_3",
    "SSL_CLIENT_CERT_CHAIN_4",
    "SSL_CLIENT_CERT_CHAIN_5",
#endif
    "SSL_SERVER_M_VERSION",
    "SSL_SERVER_M_SERIAL",
    "SSL_SERVER_V_START",
    "SSL_SERVER_V_END",
    "SSL_SERVER_S_DN",
    "SSL_SERVER_S_DN_C",
    "SSL_SERVER_S_DN_SP",
    "SSL_SERVER_S_DN_L",
    "SSL_SERVER_S_DN_O",
    "SSL_SERVER_S_DN_OU",
    "SSL_SERVER_S_DN_CN",
    "SSL_SERVER_S_DN_Email",
    "SSL_SERVER_I_DN",
    "SSL_SERVER_I_DN_C",
    "SSL_SERVER_I_DN_SP",
    "SSL_SERVER_I_DN_L",
    "SSL_SERVER_I_DN_O",
    "SSL_SERVER_I_DN_OU",
    "SSL_SERVER_I_DN_CN",
    "SSL_SERVER_I_DN_Email",
    "SSL_SERVER_A_KEY",
    "SSL_SERVER_A_SIG",
#if 0
    "SSL_SERVER_CERT",
#endif
    NULL
};

int ssl_hook_Fixup(request_rec *r)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(r->server);
    table *e = r->subprocess_env;
    char *var;
    char *val;
    int i;

    /* 
     * Check to see if SSL is on
     */
    if (!pConfig->bEnabled || OPTIONAL_SSL)
        return DECLINED;

    /*
     * Annotate the SSI/CGI environment with SSL information
     */
    ap_table_set(e, "HTTPS", "on"); /* the SSL flag! */

    for (i = 0; ssl_hook_Fixup_vars[i] != NULL; i++) {
        var = (char *)ssl_hook_Fixup_vars[i];
        val = ssl_var_lookup(r->pool, r->server, r->connection, r, var);
        if (!strIsEmpty(val))
            ap_table_set(e, var, val);
    }

    return DECLINED;
}

/*  _________________________________________________________________
**
**  SSLeay Callback Functions
**  _________________________________________________________________
*/

/*
 * Just a stub in case SSLeay again wants a pass phrase
 */
int ssl_callback_PassPhrase(char *buf, int bufsize, int dummy)
{
    ssl_log(ssl_ModConfig->rCtx.pServ, SSL_LOG_ERROR,
            "Ops, an unexpected request from SSLeay for a phrase phrase occured.");
    ssl_log(ssl_ModConfig->rCtx.pServ, SSL_LOG_INFO,
            "You should perform a hard server restart because the keyfiles seems to have changed.");
    return (-1);
}

/* 
 * Handle out the already generated RSA key...
 */
RSA *ssl_callback_TmpRSA(SSL *pSSL, int nExport)
{
    return ssl_ModConfig->pRSATmpKey;
}

/*
 * This SSLeay callback function is called when SSLeay
 * does client authentication and verifies the certificate chain.
 */
int ssl_callback_SSLVerify(int ok, X509_STORE_CTX *ctx)
{
    X509 *xs;
    int errnum;
    int errdepth;
    char *cp, *cp2;

    SSLSrvConfigRec *pConfig = mySrvConfig(ssl_ModConfig->rCtx.pConn->server);

    /*
     * Get verify ingredients
     */
    xs       = X509_STORE_CTX_get_current_cert(ctx);
    errnum   = X509_STORE_CTX_get_error(ctx);
    errdepth = X509_STORE_CTX_get_error_depth(ctx);

    /* 
     * Log verification information
     */
    cp  = X509_NAME_oneline(X509_get_subject_name(xs), NULL, 0);
    cp2 = X509_NAME_oneline(X509_get_issuer_name(xs),  NULL, 0);
    ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_DEBUG,
            "Certificate Verification: depth: %d, subject: %s, issuer: %s", 
            errdepth, cp != NULL ? cp : "-unknown-", 
            cp2 != NULL ? cp2 : "-unknown");
    free(cp);
    free(cp2);

    /*
     * Check for optionally acceptable non-verifiable issuer situation
     */
    if (   (   errnum == X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT
            || errnum == X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN
            || errnum == X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY 
            || errnum == X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE  )
        && pConfig->nVerifyClient == VERIFY_OPTIONAL_NO_CA            ) {
        ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_DEBUG,
                "Certificate Verification: Verifiable Issuer is configured as "
                "optional, therefore we're accepting the certificate");
        ok = TRUE;
    }

    /*
     * If we already know it's not ok, log the real reason
     */
    if (!ok) {
        ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_ERROR,
                "Certificate Verification: Error (%d): %s", errnum,
                X509_verify_cert_error_string(errnum));
        ssl_ModConfig->rCtx.pConn->client->szClientX509 = NULL;
        ssl_ModConfig->rCtx.pConn->client->nVerifyError = errnum;
    }

    /*
     * Finally check the depth of the certificate verification
     */
    if (errdepth >= pConfig->nVerifyDepth) {
        ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_ERROR,
                "Certificate Verification: Certificate Chain too long");
        ssl_ModConfig->rCtx.pConn->client->nVerifyError = X509_V_ERR_CERT_CHAIN_TOO_LONG;
        ok = FALSE;
    }

    /*
     * And finally signal SSLeay the (perhaps changed) state
     */
    return (ok);
}

/*
 *  This callback function is executed by SSLeay whenever a new SSL_SESSION is
 *  added to the internal SSLeay session cache. We use this hook to spread the
 *  SSL_SESSION also to the inter-process disk-cache to make share it with our
 *  other Apache pre-forked server processes.
 */
int ssl_callback_NewSessionCacheEntry(SSL *pSSL, SSL_SESSION *pNew)
{
    SSLSrvConfigRec *pConfig;
    long t = 0;

    /*
     * Set the timeout also for the internal SSLeay cache, because this way
     * our inter-process cache is consulted only when it's really necessary.
     */
    pConfig = mySrvConfig(ssl_ModConfig->rCtx.pConn->server);
    t = SSL_get_time(pNew) + pConfig->nSessionCacheTimeout;
    SSL_set_timeout(pNew, t);

    /*
     * Store the SSL_SESSION in the inter-process cache with the
     * same expire time, so it expires automatically there, too.
     */
    ssl_scache_store(pNew, t);

    /*
     * Log this cache operation
     */
    ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_DEBUG, 
            "Inter-Process Session Cache: request=SET id=%s timeout=%ds (session caching)",
            ssl_scache_id2sz(pNew->session_id, pNew->session_id_length),
            t-time(NULL));

    /* 
     * return 0 which means to SSLeay that the pNew is still
     * valid and was not freed by us with SSL_SESSION_free().
     */
    return 0;
}

/*
 *  This callback function is executed by SSLeay whenever a
 *  SSL_SESSION is looked up in the internal SSLeay cache and it
 *  was not found. We use this to lookup the SSL_SESSION in the
 *  inter-process disk-cache where it was perhaps stored by one
 *  of our other Apache pre-forked server processes.
 */
SSL_SESSION *ssl_callback_GetSessionCacheEntry(
    SSL *pSSL, unsigned char *id, int idlen, int *pCopy)
{
    SSL_SESSION *pSession;

    /*
     * Try to retrieve the SSL_SESSION from the inter-process cache
     */
    pSession = ssl_scache_retrieve(id, idlen);

    /*
     * Log this cache operation
     */
    if (pSession != NULL)
        ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_DEBUG, 
                "Inter-Process Session Cache: request=GET status=FOUND id=%s (session reuse)",
                ssl_scache_id2sz(id, idlen));
    else
        ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_DEBUG, 
                "Inter-Process Session Cache: request=GET status=MISSED id=%s (session renewal)",
                ssl_scache_id2sz(id, idlen));

    /*
     * Return NULL or the retrieved SSL_SESSION. But indicate (by
     * setting pCopy to 0) that the reference count on the
     * SSL_SESSION should not be incremented by the SSL library,
     * because we will no longer hold a reference to it ourself.
     */
    *pCopy = 0;
    return pSession;
}

/*
 *  This callback function is executed by SSLeay whenever a
 *  SSL_SESSION is removed from the the internal SSLeay cache.
 *  We use this to remove the SSL_SESSION in the inter-process
 *  disk-cache, too.
 */
void ssl_callback_DelSessionCacheEntry(
    SSL_CTX *ctx, SSL_SESSION *pSession)
{
    /*
     * Remove the SSL_SESSION from the inter-process cache
     */
    ssl_scache_remove(pSession);

    /*
     * Log this cache operation
     */
    ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_DEBUG, 
            "Inter-Process Session Cache: request=REM status=OK id=%s (session dead)",
            ssl_scache_id2sz(pSession->session_id, pSession->session_id_length));

    return;
}

