/*  Hey Emacs, switch to -*-C-*-, please 
------------

Copyright (C) 1997 Tage Stabell-Kul (tage@acm.org).  All rights
reserved.

This is a package, containing a library, that makes it possible for
programmers to include security functionality in their programs, in a
format compatible with PGP.

This library is free for commercial and non-commercial use as long as
the following conditions are aheared to:

Copyright remains Tage Stabell-Kul's, and as such any Copyright
notices in the code are not to be removed.
If this package is used in a product, Tage Stabell-Kul should be given
attribution as the author of the parts of the library used.
This can be in the form of a textual message at program startup or
in documentation (online or textual) provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:
   "This product includes software written by
    Tage Stabell-Kul (tage@acm.org)"

THIS SOFTWARE IS PROVIDED BY TAGE STABELL-KUL ``AS IS'' AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The licence and distribution terms for any publically available
version or derivative of this code cannot be changed.  i.e. this code
cannot simply be copied and put under another distribution licence
[including the GNU Public Licence.]

The reason behind this being stated in this direct manner is past
experience in code simply being copied and the attribution removed
from it and then being distributed as part of other packages. This
implementation was a non-trivial and unpaid effort.
*/

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "pgp_m.h"
#include "pgplib.h"


void
free_literal(PGPliteral_t *literal)
{
    if (literal == NULL )
	return;
    free(literal->filename);
    free(literal->data);
    memset(literal, 0, sizeof(PGPliteral_t));
}

/* We have a  file which contains a literal packet.  Read in
 * the info and fill out the struct, ut leave the pointer as NULL.
 * The FILE is advanced so that data can be read.  On error return -1,
 * on OK return 0, on syntax error return 1.
 */

int
parse_literal_file(FILE		*fin,
		   PGPliteral_t	*lit)
{
    int		offset;
    u_long	ul, ll;
    u_char	len[5];
    PGPliteral_t *literal;
    /* scratch */
    int 	i;

    if ( fin == NULL || lit == NULL )
	return(-1);

    i = fgetc(fin);
    if ( i == -1 )
	goto bad;
    /* CTB */
    len[0] = (u_char) i;
    ll = ctb_llength(len[0]);

    assert( ll == 4 );

    i = fread(len+1, 1, ll, fin);
    if ( i != ll )
	goto bad;

    literal->length = packet_length(len);
    i = fgetc(fin);
    if ( i == EOF )
	goto bad;
    literal->mode = (u_char) i;

    switch ( literal->mode ) {
    case MODE_BINARY:
    case MODE_TEXT:
    case MODE_LOCAL:
	break;
    default:
	goto bad;
    }

    /* Original filename */
    i = fgetc(fin);
    if ( i == EOF )
	goto bad;
    offset = 6 + i;
    literal->filename = calloc(i+1, 1);
    if ( literal->filename == NULL ) {
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }
    if ( fread(literal->filename, i, 1, fin) != 1 ) {
	goto bad;
    }
    i = fread(&ul, 4, 1, fin);
    if ( i != 1 )
	goto bad;
    offset += 4;
    literal->timestamp = ntohl(ul);

    /* Length of data is always 5 less than total */
    literal->datalen = literal->length - 5;

    return(0);

bad:
    free(literal->filename);
    return(1);
}


/* **************************************************************
 * A buffer contains a literal packet.  Parse and build structure.
 * ************************************************************** */

int
parse_literal_buffer(u_char	*inbuf,
		     int  	inbuflen,
		     PGPliteral_t *literal)
{
    int		count;
    char     	*tmp;
    /* scratch */
    int 	i;
    u_long	ul;

    if ( literal == NULL )
	return(-1);

    if ( ! is_ctb(inbuf[0]) || ! is_ctb_type(inbuf[0], CTB_LITERAL2_TYPE)) {
	fprintf(stderr, "Wrong type of packet\n");
	return(-1);
    }
    literal->length = packet_length(inbuf);
    if ( literal->length > inbuflen ) {
	fprintf(stderr, "Packet too long to fit in buffer\n");
	return(-1);
    }
    
    /* Skip CTB and length of length */
    count = 1 + ctb_llength(inbuf[0]);
    assert(count == 5 );

    literal->mode = inbuf[count];
    count++;

    switch ( literal->mode ) {
    case MODE_BINARY:
    case MODE_TEXT:
    case MODE_LOCAL:
	break;
    default:
	fprintf(stderr, "Unknown type of mode\n");
	return(-1);
    }

    /* Original filename */
    i = inbuf[count];
    count++;
    tmp = calloc(1, i+1);
    if ( tmp == NULL ) {
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }
    literal->filename = tmp;
    strncpy(literal->filename, inbuf+count, i);
    count += i;
    
    /* Timestamp */
    memcpy(&ul, inbuf+count, 4);
    literal->timestamp = ntohl(ul);
    count += 4;

    tmp = calloc(1, literal->length - count);
    if ( tmp == NULL ) {
	free(literal->filename);
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }
    literal->data = tmp;
    memcpy(literal->data, inbuf+count, literal->length - count);
    literal->datalen = literal->length-count;
 
    return(0);
}

/* 
 * Marshall a literal into a buffer.  Return the size of the buffer,
 * or 0 if not OK.
 */

int
literal_to_buf(PGPliteral_t	*literal,
	       u_char		**buf)
{
    u_char	header[5];
    int		header_len;
    int		plen;
    int		namelen;
    u_char	*tmp;
    
    if ( literal == NULL || *buf == NULL )
	return(0);

    tmp = malloc(literal->length);
    if ( tmp == NULL ) {
	fprintf(stderr, "Out of memory\n");
	return(0);
    }

    namelen = strlen(literal->filename);
    assert (namelen < 255);
    /* mode + namelength + timestamp */
    plen = 1+1+namelen+4+literal->datalen;

    header_len = make_ctb(CTB_LITERAL2_TYPE, plen, 1, header);
    assert(is_ctb(header[0]));
    assert(is_ctb_type(header[0], CTB_LITERAL2_TYPE));
    assert(header_len == 5);

    memcpy(tmp, header, header_len);
    tmp[header_len] = 'b';

    tmp[header_len+1] =  namelen;
    memcpy(tmp+header_len+2, literal->filename, namelen);
    memset(tmp+header_len+2+namelen, 0, 4);
    memcpy(tmp+header_len+2+namelen+4, literal->data, literal->datalen);
    assert(literal->length == plen + header_len);
    *buf = tmp;
    
    return(plen + header_len);
}

/*
 * Take a file name, create a literal struct;
 */

int
create_literal(char		*filename,
	       PGPliteral_t	*literal)
{
    u_char	header[5];
    int		header_len;
    int 	fd;
    int		plen;
    struct stat sbuf;
    /* scratch */
    int      	i;

    if ( filename == NULL || literal == NULL )
	return(-1);
    
    i = stat(filename, &sbuf);
    if ( i < 0 ) {
	perror(filename);
	return(-1);
    }
    literal->data = malloc(sbuf.st_size);
    if ( literal->data == NULL ) {
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }
    fd = open(filename,  O_RDONLY);
    if ( fd < 0 ) {
	perror(filename);
	free(literal->data);
	return(-1);
    }
    i = read(fd, literal->data, sbuf.st_size);
    if ( i != sbuf.st_size) {
	perror(filename);
	free(literal->data);
	return(-1);
    }
    close(fd);

    /* 'b' is default on UNIX systems, or what ? */
    literal->mode = 'b';
    literal->timestamp = 0;
    literal->filename = strdup(filename);
    literal->datalen = sbuf.st_size;
    
    plen = 1+1+strlen(filename)+4+sbuf.st_size;
    header_len = make_ctb(CTB_LITERAL2_TYPE, plen, 1, header);
    assert(is_ctb_type(header[0], CTB_LITERAL2_TYPE));
    assert(header_len == 5 );

    literal->length = header_len + plen;

    return(0);
}
