const char rcsid_kd_delete_c[] = "$Id: kd_delete.c,v 1.22 1999/05/31 20:21:10 marc Exp $";

/* 
 * Copyright (c) 1996, 1997, 1998, 1999, Marc Horowitz.  All rights reserved.
 * See the LICENSE file in the release for redistribution information.
 */


#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>

#include "database.h"
#include "globals.h"
#include "llist.h"
#include "kd_types.h"
#include "kd_internal.h"
#include "kd_search.h"
#include "pgputil.h"
#include "util.h"

typedef struct _dwfw_state {
   unsigned char entry[12];
   error *err;
} dwfw_state;

/* this isn't particularly efficient, but delete's aren't all that
   common. */

int delete_word_from_worddb(void *e, void *c)
{
   words_elem *we = (words_elem *) e;
   dwfw_state *s = (dwfw_state *) c;

   unsigned char word[256];
   DBT key, data;
   unsigned char *here, *last;
   int o, i, ret;

   memset(&key, 0, sizeof(key));
   memset(&data, 0, sizeof(data));

   for (i=0; i<we->len; i++)
      word[i] = tolower((we->ptr)[i]);

   key.data = (void *) word;
   key.size = (size_t) we->len;

   ret = (*(worddb->get))(worddb, NULL, &key, &data, 0);

   if (ret == DB_NOTFOUND) {
      s->err->fatal = 1;
      sprintf(s->err->buf,
	      "consistency error reading worddb for delete: %.*s not found ",
	      (int) we->len, we->ptr);
      fail();
   } else if (ret) {
      s->err->fatal = 1;
      sprintf(s->err->buf,
	      "error reading worddb entry for delete (errno = %d)", ret);
      fail();
   }

   if (data.size % 12) {
      /*
       * unknown error, db is broken??
       */
      s->err->fatal = 1;
      sprintf(s->err->buf,
	      "consistency error reading worddb for delete: %.*s not found ",
	      (int) we->len, we->ptr);
      fail();
      return(1);
   }

   here = NULL;

   /* find the matching entry */

   for (last = (unsigned char *) data.data;
	((last < (((unsigned char *) data.data) + data.size)) &&
	 (memcmp(last, zeros, 12) != 0));
	last += 12) {
      o = memcmp(last, s->entry, 12);

      if (o == 0) {
	 here = last;
	 break;
      }

      if (o > 0)
	 break;
   }

   /* bail if there isn't one */

   if (here == NULL) {
      char buf[1024];

      sprintf(buf,
	      "consistency error: word \"%.*s\" in key id %02X%02X%02X%02X\n"
	      "does not refer back to key", (int) key.size, (char *) key.data,
	      s->entry[8], s->entry[9], s->entry[10], s->entry[11]);
      log_error("delete_word_from_worddb", buf);

      return(1);
   }

   /* if there is only one useful entry, then remove the whole db
      entry */

   if ((here == data.data) &&
       ((data.size == 12) ||
	(memcmp(here+12, zeros, 12)))) {
      if ((*(worddb->del))(worddb, NULL, &key, 0)) {
	 s->err->fatal = 1;
	 s->err->str = "failed deleting worddb entry from database";
	 fail();
      }

      return(1);
   }

   /* find the end of the real entries */

   for (;
	((last < (((unsigned char *) data.data) + data.size)) &&
	 (memcmp(last, zeros, 12) != 0));
	last += 12)
      ;

   memmove(here+12, here, last-(here+12));
   memset(here, 0, 12);

   if ((*(worddb->put))(worddb, NULL, &key, &data, 0)) {
      s->err->fatal = 1;
      s->err->str = "failed replacing worddb entry in database";
      fail();
   }

   return(1);
}

int delete_key_from_worddb(void *e, llist *new_list, void *c, error *err)
{
   keys_elem *ke = (keys_elem *) e;
   xbuffer *xb = (xbuffer *) c;
   dwfw_state dwfws;
   char buf[128];

   kd_make_worddb_entry(ke, dwfws.entry);
   dwfws.err = err;

   if (!llist_iterate(&(ke->words), delete_word_from_worddb, &dwfws))
      return(0);

   sprintf(buf, "key id %02X%02X%02X%02X deleted\n",
	   ke->keyidbits.buf[4],
	   ke->keyidbits.buf[5],
	   ke->keyidbits.buf[6],
	   ke->keyidbits.buf[7]);

   if (!xbuffer_append_str(xb, buf))
      return(0);

   keys_elem_free(e, NULL);

   return(1);
}

int kd_delete_1(unsigned char *userid, long len, int flags,
		xbuffer *deleted, error *err)
{
   int ret;

   ret = kd_search_1(userid, len, KD_SEARCH_EXACT, -1,
		     NULL, delete_key_from_worddb, deleted, err);

   kd_sync();

   if (!ret)
      return(0);

   if (deleted->len == 0) {
      /* no matching keys in database */

      err->fatal = 0;
      err->str = "No matching keys in database";
      return(0);
   }

   return(1);
}

int kd_delete(unsigned char *userid, long len, int flags,
              unsigned char **ret, long *retlen)
{
   error err;
   xbuffer deleted;
   kd_txn tid;

   err.str = err.buf;

   xbuffer_alloc(&deleted);

   kd_log_start("kd_delete", userid, len, flags);

   if (kd_txn_begin(&tid, &err) &&
       kd_delete_1(userid, len, flags, &deleted, &err) &&
       kd_txn_commit(tid, &err)) {
      *ret = deleted.buf;
      *retlen = deleted.len;

      kd_log_finish("kd_delete", 1);

      return(1);
   }

   kd_txn_abort(tid, NULL);

   if (!err.fatal) {
      if (!(*ret = (unsigned char *) my_strdup(err.str))) {
         err.fatal = 1;
         err.str = "Failed allocating space for error string";
	 dabort();

         /* fall through to fatal error handler */
      } else {
         *retlen = strlen((char *) *ret);

	 kd_log_finish("kd_delete", 0);

         return(0);
      }
   }

   /* fatal errors */

   if (err.fatal) {
      log_fatal("ks_delete", err.str);
      /* never returns */
   }

   /* keep the compiler quiet */

   return(0);
}   
