const char rcsid_kd_index_c[] = "$Id: kd_index.c,v 1.14 1999/05/31 20:21:14 marc Exp $";

/* 
 * Copyright (c) 1996, 1997, 1998, 1999, Marc Horowitz.  All rights reserved.
 * See the LICENSE file in the release for redistribution information.
 */


#include <stdio.h>

#include "database.h"
#include "globals.h"
#include "llist.h"
#include "md5.h"
#include "kd_types.h"
#include "kd_internal.h"
#include "kd_search.h"

#include "shs.h"

typedef struct _gfu_state {
   unsigned char *ptr;
   long len;
} gfu_state;

int get_first_userid(void *e, void *c)
{
   keys_elem *ke = (keys_elem *) e;
   gfu_state *s = (gfu_state *) c;

   /* if there was likely to be more than one key in one of these
      lists, I'd deal with a truncation return, but it's not
      worth it */

   if (!s->ptr) {
      s->ptr = ke->primary->uidprint;
      s->len = ke->primary->uidplen;
   }

   return(1);
}

typedef struct _gi_state {
   int verbose;
   xbuffer *xb;
} gi_state;

int sigs_elem_genindex(void *e, void *c)
{
   sigs_elem *se = (sigs_elem *) e;
   gi_state *s = (gi_state *) c;
   llist keys;
   error err;
   gfu_state gfus;
   char buf[512];

   llist_alloc(&keys);
   err.str = err.buf;

   if (!kd_get_keys_by_keyid(se->keyid.buf, &keys, &err)) {
      if (err.fatal) {
	 llist_free(&keys);
	 return(0);
      }
   }

   if (llist_count(&keys)) {
      gfus.ptr = NULL;

      if (!llist_iterate(&keys, get_first_userid, (void *) &gfus))
	 return(0);

      sprintf(buf, "sig       %02X%02X%02X%02X             %.*s\n",
	      se->keyid.buf[se->keyid.len-4],
	      se->keyid.buf[se->keyid.len-3],
	      se->keyid.buf[se->keyid.len-2],
	      se->keyid.buf[se->keyid.len-1],
	      (int) gfus.len, gfus.ptr);
   } else {
      sprintf(buf, "sig       %02X%02X%02X%02X             (Unknown signator, can't be checked)\n",
	      se->keyid.buf[se->keyid.len-4],
	      se->keyid.buf[se->keyid.len-3],
	      se->keyid.buf[se->keyid.len-2],
	      se->keyid.buf[se->keyid.len-1]);
   }

   llist_iterate(&keys, keys_elem_free, NULL);
   llist_free(&keys);

   if (!xbuffer_append_str(s->xb, buf))
      return(0);

   return(1);
}

int userids_elem_genindex(void *e, void *c)
{
   userids_elem *ue = (userids_elem *) e;
   gi_state *s = (gi_state *) c;
   char buf[512];

   sprintf(buf, "                              %.*s\n",
	   (int) ue->uidplen, ue->uidprint);

   if (!xbuffer_append_str(s->xb, buf))
      return(0);

   if (s->verbose) {
      if (!llist_iterate(&(ue->sigs), sigs_elem_genindex, c))
	 return(0);
   }

   return(1);
}

typedef struct _keg_state {
   int flags;
   xbuffer *xb;
} keg_state;

int keys_elem_genindex(void *e, void *c)
{
   keys_elem *ke = (keys_elem *) e;
   keg_state *s = (keg_state *) c;
   gi_state gis;
   char buf[512];
   struct tm *c_tm;
   MD5_CTX md5ctx;
   SHS_CTX sha;
   unsigned char hash[20];
   int i;

   /* pgp does gmtime, so we do, too */
   c_tm = gmtime(&(ke->create_time));

   sprintf(buf, "pub%c%5d/%02X%02X%02X%02X %04d/%02d/%02d %s%.*s\n",
	   (ke->disabled?'-':' '),
	   (int) ke->modsigbits,
	   ke->keyidbits.buf[4],
	   ke->keyidbits.buf[5],
	   ke->keyidbits.buf[6],
	   ke->keyidbits.buf[7],
	   c_tm->tm_year+1900, c_tm->tm_mon+1, c_tm->tm_mday,
	   (ke->revocation.len?
	    "*** KEY REVOKED ***\n                              ":""),
	   (int) ke->primary->uidplen,
	   ke->primary->uidprint);

   if (!xbuffer_append_str(s->xb, buf))
      return(0);

   if (s->flags & KD_INDEX_FINGERPRINT) {
      if (ke->keytype == 16 || ke->keytype == 17) {
         shsInit(&sha);
         shsUpdate(&sha, ke->pubkey.buf, ke->pubkey.len);
         shsFinal(&sha, hash);
      } else {
         MD5Init(&md5ctx);
         MD5Update(&md5ctx, ke->modbits.buf, ke->modbits.len);
         MD5Update(&md5ctx, ke->expbits.buf, ke->expbits.len);
         MD5Final(hash, &md5ctx);
      }

      for (i=0; i<8; i++)
         sprintf(buf+i*3, "%02X ", hash[i]);
      buf[24] = ' ';
      for (i=8; i<16; i++)
	 sprintf(buf+1+i*3, "%02X ", hash[i]);
      if (ke->keytype == 16 || ke->keytype == 17) {
         buf[49] = ' ';
         for (i=16; i<20; i++)
	    sprintf(buf+2+i*3, "%02X ", hash[i]);
         buf[62] = '\n';
      } else {
         buf[48] = '\n';
      }

      if (!xbuffer_append_str(s->xb, "          Key fingerprint =  "))
	 return(0);
      if (ke->keytype == 16 || ke->keytype == 17) {
         if (!xbuffer_append(s->xb, (unsigned char *) buf, 63))
	    return(0);
      } else {
         if (!xbuffer_append(s->xb, (unsigned char *) buf, 49))
	    return(0);
      }
   }

   gis.verbose = (s->flags & KD_INDEX_VERBOSE);
   gis.xb = s->xb;

   if (s->flags & KD_INDEX_VERBOSE) {
      if (!llist_iterate(&(ke->primary->sigs), sigs_elem_genindex, &gis))
	 return(0);
   }

   if (!llist_iterate(&(ke->userids), userids_elem_genindex, &gis))
      return(0);

   return(1);
}

typedef struct _kegc_state {
   FILE *out;
   keg_state keg;
} kegc_state;

int keys_elem_genindex_cout(void *e, void *c)
{
   kegc_state *s = (kegc_state *) c;
   xbuffer buf;

   xbuffer_alloc(&buf);

   s->keg.xb = &buf;

   if (!keys_elem_genindex(e, &s->keg))
      fail();

   fwrite(buf.buf, buf.len, 1, s->out);

   xbuffer_free(&buf);

   return(1);
}

int kd_index_1(unsigned char *userid, long len, int flags, int maxkeys,
	       xbuffer *index, error *err)
{
   keg_state kegs;

   /* This is called violating abstractions in the interest of
      efficiency.  Whee. */

   if (flags & KD_INDEX_STDOUT) {
      kegc_state kegcs;

      kegcs.out = stdout;
      kegcs.keg.flags = flags;
      /* kegcs.keg.xb is filled in by keys_elem_genindex_cout */

      if (!kd_search_1(userid, len, flags & KD_SEARCH_FLAGS, maxkeys,
		       keys_elem_genindex_cout, NULL, &kegcs, err))
	 return(0);

      return(1);
   }

   kegs.flags = flags;
   kegs.xb = index;

   if (!kd_search_1(userid, len, flags & KD_SEARCH_FLAGS, maxkeys,
		    keys_elem_genindex, NULL, &kegs, err))
      return(0);

   if (index->len == 0) {
      /* no matching keys in database */

      err->fatal = 0;
      err->str = "No matching keys in database";
      return(0);
   }

   return(1);
}

int kd_index(unsigned char *userid, long len, int flags, int maxkeys,
	     unsigned char **ret, long *retlen)
{
   error err;
   xbuffer index;
   kd_txn tid;

   err.str = err.buf;
   xbuffer_alloc(&index);

   kd_log_start("kd_index", userid, len, flags);

   if (kd_txn_begin(&tid, &err) &&
       kd_index_1(userid, len, flags, maxkeys, &index, &err) &&
       kd_txn_commit(tid, &err)) {
      *ret = index.buf;
      *retlen = index.len;

      kd_log_finish("kd_index", 1);

      return(1);
   }

   kd_txn_abort(tid, NULL);

   if (!err.fatal) {
      if (!(*ret = (unsigned char *) my_strdup(err.str))) {
	 err.fatal = 1;
	 err.str = "Failed allocating space for error string";
	 dabort();

	 /* fall through to fatal error handler */
      } else {
	 *retlen = strlen((char *) *ret);

	 kd_log_finish("kd_index", 0);

	 return(0);
      }
   }

   /* fatal errors */

   if (err.fatal) {
      log_fatal("kd_index", err.str);
      /* never returns */
   }

   /* keep the compiler quiet */

   return(0);
}
