/* A filter that scans for lines of the form:
 * DCL_KEYWORD2(id, "string")
 * and "optimizes" them into a form requiring no run-time initialion
 * and compatible with gcc.
 */

#include <stdio.h>

#ifndef ENCODE_CHAR_2
#define ENCODE_CHAR_2 '_'
#endif


int HandleEscapes(srcPtr, dstPtr, endChar)
     char **srcPtr; /* input: quoted C string, buf[-1] assumed to be '"' */
     char **dstPtr; /* output: C escapes (and initial '"') have been removed */
     int endChar;
     /* returns: -1 on error, else length of string (in buf) */
     /* terminating '"' is set to '\0' to aid further parsing */
{
    register char *src = *srcPtr, *dst = *dstPtr;
    for (;;) {
	int val;
	char c = *src++;
	if (c == endChar) {
	    int length = dst - *dstPtr;
	    *dst = 0;
	    *srcPtr = src;
	    *dstPtr = dst;
	    return length;
	}
	else if (c == '\\') {
	    char c = *src++;
	    switch (c) {
	      case '\\': case '"': case '\'': case '?':
		*dst++ = c; break;
	      case 'n': *dst++ = '\n'; break;
	      case 'r': *dst++ = '\r'; break;
	      case 'b': *dst++ = '\b'; break;
	      case 't': *dst++ = '\t'; break;
	      case 'v': *dst++ = '\v'; break;
	      case 'f': *dst++ = '\f'; break;
	      case 'a': *dst++ = '\a'; break;
	      case 'x':
		val = 0;
		for (;; src++) {
		    c = src[0];
		    if (c >= '0' && c <= '9')
			val = 16 * val + c - '0';
		    else if (c >= 'A' && c <= 'F')
			val = 16 * val + c - 'A' + 10;
		    else if (c >= 'a' && c <= 'f')
			val = 16 * val + c - 'a' + 10;
		    else
			break;
		    
		}
		*dst++ = val;
		break;
	      default:
		if (c < '0' || c > '7') { *dst++ = c; break; /*goto fail?*/ }
		c -= '0';
		if (src[0] >= '0' && src[0] <= '7') {
		    c = c * 8 + *src++ - '0';
		    if (src[0] >= '0' && src[0] <= '7')
			c = c * 8 + *src++ - '0';
		}
		*dst++ = c;
		break;
	     }
	}
	else if (c < ' ' || c >= 127) goto fail;
	else  *dst++ = c;
    }
  fail:
    *srcPtr = src;
    *dstPtr = dst;
    return -1;
}

enum SymbolKind {
    StringKind,
    KeywordKind, /* Symbol in KEYWORD package */
    CLispKind, /* Symbol in COMMON-LISP package */
    SchemeKind, /* Symbol in SCHEME package */
    BuiltinKind, /* Symbol in BUILTIN package */
    SymbolKind, /* General symbol in name package. */
};

int
main(argc, argv)
     int argc; char** argv;
{
    int line_max = 500;
    enum SymbolKind kind;
    int encode_max = 100;
    register FILE *in = stdin;
    register FILE *out = stdout;
    char *string = (char*)malloc(line_max);
    char *line = (char*)malloc(line_max);
    char *encoded = (char*)malloc(encode_max);
    if (argc > 1) {
	in = fopen(argv[1], "r");
	if (in == NULL) {
	    fprintf(stderr, "Cannot open input file: %s\n", argv[1]);
	    exit(-1);
	}
    }
    fprintf(out, "/* DO NOT EDIT THIS FILE! */\n");
    fprintf(out, "/* IT HAS BEEN AUTOMATICALLY GENERATED BY: */\n");
    fprintf(out, "/* %s %s. */\n\n", argv[0], argc > 1 ? argv[1] : "<stdin>");
    for ( ;; ) {
	int line_len = 0;
	char *ptr, *string_ptr;
	int string_len;
	char *id_start, *id_comma;
	int c = getc(in);
	if (c == EOF)
	    return 0;
	while (c == ' ' || c == '\t') { putchar(c); c = getc(in); }
	/* Read remainder of line (including '\n', if any). */
	while (c != EOF) {
	    /* -1 to allow final '\0' */
	    if (line_len >= line_max - 1) {
		line_max += 500;
		line = (char*)realloc(line, line_max);
		string = (char*)realloc(string, line_max);
	    }
	    line[line_len++] = c;
	    if (c == '\n')
		break;
	    c = getc(in);
	}
	line[line_len] = '\0';
	if (strncmp(line, "DCL_KEYWORD2", 12) == 0) {
	    kind = KeywordKind;
	    ptr = line + 12;
	}
	else if (strncmp(line, "DCL_LISPSYM2", 12) == 0) {
	    kind = CLispKind;
	    ptr = line + 12;
	}
	else if (strncmp(line, "DclSchemeSym", 12) == 0) {
	    kind = SchemeKind;
	    ptr = line + 12;
	}
	else if (strncmp(line, "DclQ_Builtin", 12) == 0) {
	    kind = BuiltinKind;
	    ptr = line + 12;
	}
	else if (strncmp(line, "DclString", 9) == 0) {
	    kind = StringKind;
	    ptr = line + 9;
	}
	else
	    goto no_match;
	c = *ptr++;
	while (c == ' ' || c == '\t') c = *ptr++;
	if (c != '(') goto no_match;
	id_start = ptr;
	while (c != ',' && c != '\0') c = *ptr++;
	if (c != ',') goto no_match;
	id_comma = ptr-1;
	c = *ptr++;
	while (c == ' ' || c == '\t') c = *ptr++;
	if (c != '"') goto no_match;
	string_ptr = string;
	string_len = HandleEscapes(&ptr, &string_ptr, '"');
	if (string_len < 0) goto no_match;
	if (*ptr++ != ')') goto no_match;
	/* ok: - print it */
	*id_comma = '\0';
	if (2*string_len +10 >= encode_max) {
	    encode_max += 500;
	    encoded = (char*)realloc(encoded, encode_max);
	}
	encode_string_to_label(string, string_len, encoded);
#if 0
	if (strcmp(string, encoded) == 0)
	    printf("#ifdef __GNUC__\n");
#endif
	switch (kind) {
	  case KeywordKind:
	    printf("extern Symbol %s asm(\"_%cKW%s\")",
		   id_start, ENCODE_CHAR_2, encoded);
	    break;
	  case CLispKind:
	    printf("extern Symbol %s asm(\"_%ccl%s\")",
		   id_start, ENCODE_CHAR_2, encoded);
	    break;
	  case SchemeKind:
	    printf("extern Symbol %s asm(\"_%csc%s\")",
		   id_start, ENCODE_CHAR_2, encoded);
	    break;
	  case BuiltinKind:
	    printf("extern Symbol %s asm(\"_%cQB%s\")",
		   id_start, ENCODE_CHAR_2, encoded);
	    break;
	  case StringKind:
	    printf("extern StringC %s asm(\"_%cST%s\")",
		   id_start, ENCODE_CHAR_2, encoded);
	    break;
	  default:
	    abort();
	}

#if 0
	if (strcmp(string, encoded) == 0) {
	    printf("\n#else\n");
	    printf("extern Symbol KEY__%s;\n", encoded);
	    printf("#define %s KEY__%s\n", id_start, encoded);
	    printf("#endif");
	}
#endif
	fputs(ptr, stdout);
	continue;

      no_match:
	fwrite(line, 1, line_len, stdout);
    }
}
