#include "stdio.h"
#include "string.h"
#include "stdlib.h"

/* ========================================================================
DSK OUTPUT FILES

  The DSK output file supports 16 bit addresses both in program and
data space allowing direct placement of runtime code into a target.

+-- DSK characters:  K designates start of file
|                    1 designates code entry point
|                    9 designates load address
|
|          +-- DSK revision and output file name
|          |                                 +-- Checksum follows
^ |<-------------->|                         |    +-- line terminate
K_DSKA_1.01_TEST.DSK                         |    |
1FB007FB00F                                  ^    ^
9FB00B5588B55A9B55AAB55ABB55ACB55ADB55AEB55AF7AD3CF
9FB08BFFA8BFB01B67E0B60F07C379F
90400M0000M0000M123471234F
:              ^
^              |
|              +-- B designates load to program space
|                  M designates load to data space
|
+-- End of file record
======================================================================== */

typedef unsigned char byte;
typedef unsigned int word;

/* In this context, segments does not refer to linker segments like .text
or .data; here it refers to contiguous blocks in an Intel hex file. */

#define MAX_NUM_SEGMENTS  32
typedef struct
    {
    byte nsegs, checksum_ok;
    word starting_address[MAX_NUM_SEGMENTS],
         size[MAX_NUM_SEGMENTS],
         array_address[MAX_NUM_SEGMENTS];
    } hexfile_info;

/* Return values */
#define RETURN_OK        0
#define CANNOT_OPEN_FILE 1
#define BAD_CHECKSUM     2

/* ======================================================================= */
/*                               read_hexfile                              */
/* ======================================================================= */

FILE *f;
word start;
int nbytes, aptr;

/* "read_one_line" sets start, sets h->checksum status */

void read_one_line(hexfile_info *h, byte array[])
    {
    word a;
    byte b, i, checksum, alleged_checksum, filler;

    if (feof(f))
        {
        nbytes = 0;
        return;
        }
    fscanf(f, ":%02X%04X%02X", &nbytes, &a, &filler);
    start = a;
    checksum = (byte) (nbytes + a + (a >> 8) + filler);
    for (i = 0; i < nbytes; i++)
        {
        fscanf(f, "%02X", &b);
        array[aptr++] = b;
        checksum += b;
        }
    fscanf(f, "%02X\n", &alleged_checksum);
    checksum += alleged_checksum;
    if (checksum != 0)
        h->checksum_ok = 0;
    }

int read_hexfile(byte array[], hexfile_info *h, char fname[])
    {
    word sstart, astart, running_address;
    byte ns;

    if (strlen(fname))
        f = fopen(fname, "r");
    else
        f = stdin;
    if (f == NULL) return CANNOT_OPEN_FILE;
    h->checksum_ok = 1;
    h->nsegs = 0;
    aptr = 0;
    astart = 0;
    read_one_line(h, array);
    while (!feof(f))
        {
        sstart = start;
        astart = aptr - nbytes;
        running_address = start + nbytes;

        while (!feof(f))
            {
            read_one_line(h, array);
            if (running_address != start) break;
            running_address = start + nbytes;
            }

        ns = h->nsegs;
        h->array_address[ns] = astart;
        h->starting_address[ns] = sstart;
        h->size[ns] = (aptr - nbytes) - astart;
        if (h->size[ns] > 0) h->nsegs++;
        }
    fclose(f);
    if (h->checksum_ok) return RETURN_OK;
    return BAD_CHECKSUM;
    }

byte arraylo[2000], arrayhi[2000];
hexfile_info infolo, infohi;
FILE *outf, *mapfile;
int checksum = 0;

#define CODE_ENTRY  '1'
#define CHECKSUM    '7'
#define ADDRESS     '9'
#define PROGRAM     'B'
#define DATA        'M'

void append_word(char type, int content)
    {
    fprintf(outf, "%c%04X", type, content);
    if (type != ADDRESS) checksum += content;
    }

void finish_line(void)
    {
    append_word(CHECKSUM, checksum);
    fprintf(outf, "F\n");
    checksum = 0;
    }

void help(void)
    {
    printf(
        "Call this by saying:\n"
        "helper filename.map filename.lo filename.hi filename.dsk\n",
        "first argument is TI map file\n"
        "second argument is low byte input file, Intel hex\n"
        "third argument is high byte input file, Intel hex\n"
        "fourth argument is output file in DSK format\n"
        "Code entry point is assumed to be the label \"_c_int0\"\n"
        "If \"_c_int0\" is not found, code entry point "
            "is assumed to be at 0A00h\n"
        );
    exit(0);
    }

char name1[40], name2[40], line[100];

int find_c_beginning(void)
    {
    word num1, num2;

    while (!feof(mapfile))
        {
        fgets(line, 100, mapfile);
        if (sscanf(line, "%x %s %x %s", &num1, name1, &num2, name2) == 4)
            if (strcmp(name1, "_c_int0") == 0)
                {
                fclose(mapfile);
                return num1;
                }
        }
    printf("Didn't find \"_c_int0\" in map file!!\n");
    printf("Assuming default code entry point is 0A00h\n");
    return 0xA00;
    }

/* Note: Here is where we make assumptions about what is program and
what is data. */
#define program_memory(addr)   (a >= 0x800 && a < 0xC00)

void main(int argc, char *argv[])
    {
    word a, b, i, j;

    if (argc < 5) help();
    if ((mapfile = fopen(argv[1], "r")) == NULL) help();
    a = find_c_beginning();
    if (read_hexfile(arraylo, &infolo, argv[2]) != RETURN_OK)
        {
        printf("Problem reading %s\n", argv[2]);
        help();
        }
    if (read_hexfile(arrayhi, &infohi, argv[3]) != RETURN_OK)
        {
        printf("Problem reading %s\n", argv[3]);
        help();
        }
    if ((outf = fopen(argv[4], "w")) == NULL) help();
    fprintf(outf, "K_RANDOM_CRAPOLA_SAMPLE.DSK\n");
    append_word(CODE_ENTRY, a);
    finish_line();

    for (j = 0; j < infolo.nsegs; j++)
        {
        a = infolo.starting_address[j];
        b = infolo.array_address[j];
        for (i = 0; i < infolo.size[j]; i++)
            {
            if ((i & 7) == 0) append_word(ADDRESS, a + i);
            append_word(program_memory(a) ? PROGRAM : DATA,
                (((word) arrayhi[b + i]) << 8) | ((word) arraylo[b + i]));
            if ((i & 7) == 7 && i < infolo.size[j] - 1) finish_line();
            }
        finish_line();
        }
    fclose(outf);
    }
