// === Starter Application 1.0.0 === //

/* ========================================================================

   FUNCTION:  FailOnAVS()

   INPUTS:    avsFail - the numeric code resulting from CheckAVS() 
		     indicating what kind of AVS failure occurred, if any.
			* a code of  1  indicates an Address failure.
			* a code of  2  indicates a Zip Code failure.

   RETURNS:   true, if authorizations should fail because of the AVS failures (if any).
	      false, if the authorizations should succeed.

   SPECIAL NOTES:  This function depends on PROJECT properties set in "start.html" to set when 
		   authorizations should fail on AVS failures.  

		   project.failOnAddr - if "Y", function will indicate failures on Address mismatches
		   project.failOnZip - if "Y", function will indicate failures on Zip Code mismatches
		
   DESCRIPTION: 

   Function returns true indicating the merchant wants to cause the transaction to
   fail upon EITHER and address AVS failure OR a zip code AVS failure OR BOTH.  


======================================================================== */
  
function FailOnAVS( avsFail )
{

	var failFlag = false;

	// Check for Address failure.  

	if ( (avsFail == 1) && (project.failOnAddr.toUpperCase() == "Y") )
		failFlag = true;

	// Check for Zip Code failure.  

	if ( (avsFail == 2) && (project.failOnZip.toUpperCase() == "Y") )
		failFlag = true;

	return failFlag;

} // END FUNCTION FailOnAVS()



/* ======================================================

FUNCTION:  MatchShipToBill

INPUT:     request - the LiveWire REQUEST object resulting from a call
	from "purchase.html" and containing properties associated with
	the form elements contained within the form in "purchase.html".

RETURN:    Modified REQUEST object properties for shipping information
	to match the corresponding billing information.

DESCRIPTION:  This function is used to map billing properties of the REQUEST
	object to the shipping properties in the case a user has checked
	a box on the form indicating that Shipping Information is the same
	as Billing Information.  

SPECIAL NOTES:  If the form element names from "purchase.html" 
	referenced below are changed, this function must be 
	altered to reflect the changes to those HTML form-element 
	names, or mapping may cease to function correctly.

====================================================== */

function MatchShipToBill( request )
{

	request.shipName = request.billName;
	request.shipAddr1 = request.billAddr1;	
	request.shipAddr2 = request.billAddr2;	
	request.shipCity = request.billCity;	
	request.shipState = request.billState;	
	request.shipCountry = request.billCountry;	
	request.shipZip = request.billZip;	
	request.shipPhone = request.billPhone;	
	request.shipEmail = request.billEmail;

	// REQUEST is a global object, so any changes made within this function
	// are made globally, thus no return statement is necessary.

} // END FUNCTION MatchShipToBill



/*  ================================================================
    FUNCTION:  IsValidZip()
 
    INPUT:    zipstr - a string representing a 5-digit zip code

    RETURNS:  true, if the string is 5-digits long
		    
	      false, otherwise

    SPECIAL NOTES:  This function could be modified for foreign zip codes
		    or xxxxx-xxxx zip code notation.
    ================================================================ */

function IsValidZip( zipstr )
{
	
   var valid = true;

   // zipstr must be 5 characters long, and can only contain numbers from
   // 0 through 9
   //
   if (IsBlank(zipstr) || (zipstr.length != 5) || !IsNum(zipstr))
	valid = false;
   
   return valid;

} // END FUNCTION IsValidZip()



/*  ================================================================
    FUNCTION:  VerifyPurchaseData()
 
    INPUT:    request - a copy of the request object from the
		        "purchase.html" input form.

    CALLS:    PrintError() if one of the form elements is invalid
	      and must be re-entered.

    SPECIAL NOTES:  This function could be modified to validate additional
		    fields from "purchase.html".

    ================================================================ */

function VerifyPurchaseData( request )
{

    var errMsg = "";
    var billState = request.billState;

    // Get current month and year to compare to card expiration date entered
    var today = new Date();
    var thisMonth = today.getMonth() + 1;
    var thisYear = today.getYear() + 1900;
    var expMonth = request.cardExpMonth;

    if (request.cardExpMonth.substring(0,1) == "0")
	expMonth = request.cardExpMonth.charAt(1);

    if (IsBlank(request.billName))
	errMsg = "Please enter a name for the Card Holder.";

    else if (IsBlank(request.cardType))
	    errMsg = "Please enter a Credit Card Type.";

    else if (!IsNum(request.cardNumber) || !IsCardMatch(request.cardType, request.cardNumber)) 
	    errMsg = "Invalid Credit Card Number entered for card type " + request.cardType;

    else if (IsBlank(request.cardExpMonth))
	    errMsg = "Please enter an expiration month.";

    else if (!IsValidYear(request.cardExpYear))
	errMsg = "Invalid year entered for the card expiration date.";

    else if ((parseInt(expMonth) < thisMonth) && (parseInt(request.cardExpYear) == thisYear))
	errMsg = "This card has expired of you have entered the wrong expiration month.";

    else if (parseInt(request.cardExpYear) < thisYear)
	errMsg = "This card has expired of you have entered the wrong expiration year.";

    else if ( (IsBlank(request.billAddr1)) && (IsBlank(request.billAddr2)) )
	errMsg = "Please enter a street address.";

    else if (IsBlank(request.billCity))
	errMsg = "Please enter a city.";

    else if (IsBlank(billState) || (billState.length != 2))
	errMsg = "Please enter a state.";

    else if (!IsValidZip(request.billZip))
	errMsg = "Please enter a valid 5-digit zip code.";

    else if (request.shipSame == null)
		if (IsBlank(request.shipName))
			errMsg = "Please enter a name for the shipping recipient.";

		else if ( (IsBlank(request.shipAddr1)) && (IsBlank(request.shipAddr2)) )
			errMsg = "Please enter a shipping street address.";

    		else if (IsBlank(request.shipCity))
			errMsg = "Please enter a shipping city.";

    		else if (IsBlank(request.shipState))
			errMsg = "Please enter a shipping state/province.";

    		else if (IsBlank(request.shipZip))
			errMsg = "Please enter a valid shipping postal code.";

    

    if (errMsg != "")
	    PrintError("some purchase form data is invalid.", errMsg);



}  // END FUNCTION VerifyPurchaseData()


