/* verify.c: The opieverify() library function.

Portions of this software are Copyright 1996 by Craig Metz, All Rights
Reserved. The Inner Net Copyright Notice and License Agreement applies to
these portions of the software.

Portions of this software are Copyright 1995 by Randall Atkinson and Dan
McDonald, All Rights Reserved. All Rights under this copyright are assigned
to the U.S. Naval Research Laboratory (NRL). The NRL Copyright Notice and
License Agreement applies to this software.

        History:

        Modified by cmetz for OPIE 2.2. Work with a copy of response --
               opiestripcrlf modifies its argument in place. Ifdef around
               some headers and add stdlib.h. No longer stripcrlf the
               response. Inline and obselete opiestripcrlf. Initialize
               response. Test both hex and English per spec. Use symbolic
               fseek() whence values.
        Created at NRL for OPIE 2.2 from opiesubr2.c. Changed opiestrip_crlf
               to opiestripcrlf.
*/
#include "opie_cfg.h"

#include <stdio.h>
#if HAVE_STRING_H
#include <string.h>
#endif /* HAVE_STRING_H */
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */

#if HAVE_SETPRIORITY && HAVE_SYS_RESOURCE_H
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else /* TIME_WITH_SYS_TIME */
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else /* HAVE_SYS_TIME_H */
#include <time.h>
#endif /* HAVE_SYS_TIME_H */
#endif /* TIME_WITH_SYS_TIME */
#include <sys/resource.h>
#else /* HAVE_SETPRIORITY && HAVE_SYS_RESOURCE_H */
#if TM_IN_SYS_TIME
#include <sys/time.h>
#else /* TM_IN_SYS_TIME */
#include <time.h>
#endif /* TM_IN_SYS_TIME */
#endif /* HAVE_SETPRIORITY && HAVE_SYS_RESOURCE_H */

#include "opie.h"

/* Verify response to an opie challenge.

   Return codes:
   -1: Error of some sort; database unchanged
    0:  Verify successful, database updated
    1:  Verify failed, database unchanged
  
   The database file is always closed by this call.

   This function MUST be called exactly once in a pair with calls to 
   opiechallenge() in order to set and clear locks properly.

   This function always clears the internal state block. N.B. that the 
   Bellcore S/Key Version 1 software distribution looks inside the internal
   state block to find the current sequence number and do appropriate
   warnings. This interface should not be used with OPIE and will not be
   supported in the future. Use opiegetsequence() instead.
*/
int opieverify FUNCTION((mp, responsein), struct opie *mp AND char *responsein)
{
  char key[2][8];
  char fkey[2][8];
  int keyvalid[2];
  char filekey[8];
  time_t now;
  struct tm *tm;
  char tbuf[27];
  int rval = -1;
  char *cp;
  char *response = NULL;

  if (!mp->keyfile)
    goto invalid;

  if (!responsein)
    goto invalid;

  if (!(response = malloc(strlen(responsein) + 1)))
    goto invalid;

  strcpy(response, responsein);

  time(&now);
  tm = localtime(&now);
  strftime(tbuf, sizeof(tbuf), " %b %d,%Y %T", tm);

  keyvalid[0] = keyvalid[1] = 0;

  if (opieetob(key[0], response) == 1) {
    memcpy(fkey[0], key[0], sizeof(key[0]));
    opiehash(fkey[0], MDX);
    keyvalid[0]++;
  }

  if (!opieatob8(key[1], response)) {
    memcpy(fkey[1], key[1], sizeof(key[1]));
    opiehash(fkey[1], MDX);
    keyvalid[1]++;
  }

  /* In order to make the window of update as short as possible we must do
     the comparison here and if OK write it back otherwise the same password
     can be used twice to get in to the system. */

#if HAVE_SETPRIORITY && HAVE_SYS_RESOURCE_H
  setpriority(PRIO_PROCESS, 0, -4);	/* present only in BSD */
#endif /* HAVE_SETPRIORITY && HAVE_SYS_RESOURCE_H */

#if HAVE_FPURGE
  if (fpurge((FILE *)mp->keyfile))
    goto invalid;
#endif /* HAVE_FPURGE */
  /* reread the file record NOW */
  if (fseek((FILE *)mp->keyfile, mp->recstart, SEEK_SET))
    goto invalid;
  if (fgets(mp->buf, sizeof(mp->buf), (FILE *)mp->keyfile) != mp->buf)
    goto invalid;

  {
  char *cp;

  if (cp = strchr(mp->buf, '\n')) {
    if (*(cp - 1) == '\r')
      cp--;
    *cp = 0;
  }
  }

  if (!(mp->logname = strtok(mp->buf, " \t")))
    goto invalid;
  {
    int n;

    if (!(cp = strtok(NULL, " \t")))
      goto invalid;
    if (!(n = atoi(cp)))
      goto invalid;
    if (mp->n != n) {
      rval = 1;
      goto invalid;
    }
  }
  if (!(mp->seed = strtok(NULL, " \t")))
    goto invalid;
  if (!(mp->val = strtok(NULL, " \t")))
    goto invalid;
  /* And convert file value to hex for comparison */
  opieatob8(filekey, mp->val);

  if (keyvalid[0] && !memcmp(fkey[0], filekey, 8))
    goto valid;

  if (keyvalid[1] && !memcmp(fkey[1], filekey, 8)) {
    memcpy(key[0], key[1], 8);
    goto valid;
  }

  rval = 1;
  goto invalid;

valid:
  /* Update key in database by overwriting entire record. Note that we must
     write exactly the same number of bytes as in the original record (note
     fixed width field for N). */
  opiebtoa8(mp->val, key[0]);
  mp->n--;
  fseek((FILE *)mp->keyfile, mp->recstart, SEEK_SET);
  fprintf((FILE *)mp->keyfile, "%s %04d %-16s %s %-21s\n", mp->logname, mp->n,
          mp->seed, mp->val, tbuf);
  rval = 0;

invalid:
  if (mp->keyfile)
    fclose((FILE *)mp->keyfile);

#if HAVE_SETPRIORITY && HAVE_SYS_RESOURCE_H
  setpriority(PRIO_PROCESS, 0, 0);
#endif /* HAVE_SETPRIORITY && HAVE_SYS_RESOURCE_H */

  opieunlock();

  memset(mp, 0, sizeof(*mp));

  if (response)
    free(response);

  return rval;
}
